/*
	description:

		"C declarations for the Gobo Eiffel runtime."

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2005-2019, Eric Bezault and others"
	license: "MIT License"
	date: "$Date$"
	revision: "$Revision$"
*/

#ifndef GE_EIFFEL_H
#define GE_EIFFEL_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

/* Class name mapping as defined in the FreeELKS library. */
#ifndef EIF_INTEGER
#define EIF_INTEGER EIF_INTEGER_32
#endif
#ifndef EIF_CHARACTER
#define EIF_CHARACTER EIF_CHARACTER_8
#endif
#ifndef EIF_REAL
#define EIF_REAL EIF_REAL_32
#endif
#ifndef EIF_DOUBLE
#define EIF_DOUBLE EIF_REAL_64
#endif
#ifndef GE_ms
#define GE_ms(s,c) GE_ms8((s),(c))
#endif

#if defined(__USE_POSIX) || defined(__unix__) || defined(_POSIX_C_SOURCE)
#include <unistd.h>
#endif
#if !defined(WIN32) && \
	(defined(WINVER) || defined(_WIN32_WINNT) || defined(_WIN32) || \
	defined(__WIN32__) || defined(__TOS_WIN__) || defined(_MSC_VER) || \
	defined(__MINGW32__))
#define WIN32 1
#endif
#ifdef WIN32
#define EIF_WINDOWS 1
#include <windows.h>
#endif

#define BYTEORDER 0x1234

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <stddef.h>

/* Platform definition */
/* Unix definition */
#define EIF_IS_UNIX EIF_TRUE
/* Windows definition */
#ifdef EIF_WINDOWS
#define EIF_IS_WINDOWS EIF_TRUE
#undef EIF_IS_UNIX
#define EIF_IS_UNIX EIF_FALSE
#else
#define EIF_IS_WINDOWS EIF_FALSE
#endif
/* VMS definition */
#ifdef EIF_VMS
#define EIF_IS_VMS EIF_TRUE
#undef EIF_IS_UNIX
#define EIF_IS_UNIX EIF_FALSE
#else
#define EIF_IS_VMS EIF_FALSE
#endif
/* MAC definition */
#ifdef EIF_MAC
#define EIF_IS_MAC EIF_TRUE
#undef EIF_IS_UNIX
#define EIF_IS_UNIX EIF_FALSE
#else
#define EIF_IS_MAC EIF_FALSE
#endif
/* VxWorks definition */
#ifdef EIF_VXWORKS
#define EIF_IS_VXWORKS EIF_TRUE
#undef EIF_IS_UNIX
#define EIF_IS_UNIX EIF_FALSE
#else
#define EIF_IS_VXWORKS EIF_FALSE
#endif

#ifdef __cplusplus
extern "C" {
#endif

#ifdef _MSC_VER /* MSVC */
typedef signed char int8_t;
typedef signed short int16_t;
typedef signed int int32_t;
typedef signed __int64 int64_t;
typedef unsigned char uint8_t;
typedef unsigned short uint16_t;
typedef unsigned int uint32_t;
typedef unsigned __int64 uint64_t;
#else
#if defined (__BORLANDC__) && (__BORLANDC__ < 0x600) /* Borland before 6.0 */
typedef signed char int8_t;
typedef signed short int16_t;
typedef signed long int int32_t;
typedef signed __int64 int64_t;
typedef unsigned char uint8_t;
typedef unsigned short uint16_t;
typedef unsigned long int uint32_t;
typedef unsigned __int64 uint64_t;
#else
#include <inttypes.h>
#endif
#endif

/* Portable integer pointers */
#ifdef EIF_WINDOWS
#ifndef _INTPTR_T_DEFINED
#define _INTPTR_T_DEFINED
#ifdef _WIN64
typedef __int64 intptr_t;
#else
typedef int intptr_t;
#endif
#endif
#ifndef _UINTPTR_T_DEFINED
#define _UINTPTR_T_DEFINED
#ifdef _WIN64
  typedef unsigned __int64 uintptr_t;
#else
  typedef unsigned int uintptr_t;
#endif
#endif
#endif

/* C type for underlying integer type identifying object's dynamic type. */
typedef uint16_t EIF_TYPE_INDEX;

/*
 * Abstraction representing an Eiffel type.
 * It is made of a compiler type-id,
 * and of some annotations (attached/detachable/separate/variant/frozen).
 */
typedef struct eif_type {
	EIF_TYPE_INDEX id;
	EIF_TYPE_INDEX annotations;
} EIF_TYPE;

/*
 * Since EIF_TYPE and EIF_ENCODED_TYPE have the same size, the encoded version
 * is basically a memcpy version of the EIF_TYPE representation.
 * It is used to provide backward compatibility to most Eiffel and
 * C APIs manipulating types as an INTEGER.
 */
typedef int32_t EIF_ENCODED_TYPE;
typedef EIF_ENCODED_TYPE EIF_TYPE_ID;
#define EIF_NO_TYPE (EIF_TYPE_ID)(-1)

/* Basic Eiffel types */
typedef char EIF_BOOLEAN;
typedef unsigned char EIF_CHARACTER_8;
typedef uint32_t EIF_CHARACTER_32;
typedef int8_t EIF_INTEGER_8;
typedef int16_t EIF_INTEGER_16;
typedef int32_t EIF_INTEGER_32;
typedef int64_t EIF_INTEGER_64;
typedef uint8_t EIF_NATURAL_8;
typedef uint16_t EIF_NATURAL_16;
typedef uint32_t EIF_NATURAL_32;
typedef uint64_t EIF_NATURAL_64;
typedef void* EIF_POINTER;
typedef float EIF_REAL_32;
typedef double EIF_REAL_64;
typedef struct {EIF_TYPE_INDEX id; uint16_t flags;} EIF_ANY;
typedef EIF_ANY* EIF_REFERENCE;
typedef struct {EIF_TYPE_INDEX id; uint16_t flags; EIF_REFERENCE area; EIF_INTEGER count;} EIF_STRING;
typedef struct {EIF_TYPE_INDEX id; uint16_t flags; uint32_t offset; EIF_INTEGER count; EIF_INTEGER capacity;} EIF_SPECIAL;

#ifdef EIF_WINDOWS
typedef wchar_t EIF_NATIVE_CHAR;
#else
typedef char EIF_NATIVE_CHAR;
#endif
typedef EIF_NATIVE_CHAR* EIF_FILENAME;

#define EIF_VOID ((EIF_REFERENCE)0)
#define EIF_FALSE ((EIF_BOOLEAN)'\0')
#define EIF_TRUE ((EIF_BOOLEAN)'\1')
#define EIF_TEST(x) ((x) ? EIF_TRUE : EIF_FALSE)

#define EIF_IS_WORKBENCH EIF_FALSE
#define EIF_POINTER_DISPLAY "lX"

/* For INTEGER and NATURAL manifest constants */
#define GE_int8(x) x
#define GE_nat8(x) x
#define GE_int16(x) x
#define GE_nat16(x) x
#define GE_int32(x) x##L
#define GE_nat32(x) x##U
#if defined (_MSC_VER) && (_MSC_VER < 1400) /* MSC older than v8 */
#define GE_int64(x) x##i64
#define GE_nat64(x) x##ui64
#elif defined(__BORLANDC__) && (__BORLANDC__ < 0x600) /* Borland before 6.0 */
#define GE_int64(x) x##i64
#define GE_nat64(x) x##ui64
#else /* ISO C 99 */
#define GE_int64(x) x##LL
#define GE_nat64(x) x##ULL
#endif
#ifdef __LCC__
/* lcc-win32 reports a constant overflow for -21474836478. */
#define GE_min_int32 (-GE_int32(2147483647)-GE_int32(1))
#else
#define GE_min_int32 GE_int32(-2147483648)
#endif
#define GE_max_int32 GE_int32(2147483647)
#if defined(__LCC__) || defined(__GNUC__) || defined(__MINGW32__)
/* lcc-win32 reports a constant overflow for -9223372036854775808. */
/* gcc and mingw-win64 warn that integer constant is so large that it is unsigned. */
#define GE_min_int64 (-GE_int64(9223372036854775807)-GE_int64(1))
#else
#define GE_min_int64 GE_int64(-9223372036854775808)
#endif
#if defined(__LCC__) && !defined(_WIN64)
/* lcc-win32 does not consider 64 bit constants as constants in case statement. */
#define GE_case_int64(x) ((int32_t)(x))
#define GE_case_nat64(x) ((uint32_t)(x))
#else
#define GE_case_int64(x) (x)
#define GE_case_nat64(x) (x)
#endif

#ifdef _WIN64
#define GE_IS_64_BITS EIF_TRUE
#else
#define GE_IS_64_BITS EIF_TEST(sizeof(void*)==64)
#endif

/* Posix threads */
#if !defined(EIF_WINDOWS)
#define GE_USE_POSIX_THREADS
#endif

#ifdef _MSC_VER /* MSVC */
/* MSVC does not support ISO C 99's 'snprintf' from stdio.h */
#define snprintf(a,b,c,d) sprintf(a,c,d)
#endif

/*
 * Gobo compiler version.
 * Starts with 6080 (looks like GOBO) followed by 5 digits.
 */
#define GE_compiler_version() 608000001

/*
	Interoperability with ISE.
*/
#define RTI64C(x) GE_int64(x)
#define EIF_OBJECT EIF_REFERENCE
#define EIF_OBJ EIF_OBJECT
#define OVERHEAD sizeof(EIF_ANY)
/* Function pointer call to make sure all arguments are correctly pushed onto stack. */
/* FUNCTION_CAST is for standard C calls. */
/* FUNCTION_CAST_TYPE is for non-standard C calls. */
#define FUNCTION_CAST(r_type,arg_types) (r_type (*) arg_types)
#define FUNCTION_CAST_TYPE(r_type,call_type,arg_types) (r_type (call_type *) arg_types)
#define SIGBLOCK
#define SIGRESUME
#define rt_public				/* default C scope */
#define rt_private static		/* static outside a block means private */
#define rt_shared				/* data shared between modules, but not public */
typedef intptr_t rt_int_ptr;
typedef uintptr_t rt_uint_ptr;
#define RTMS(s) GE_str8(s)
#define RTMS_EX(s,c) GE_ms8((s),(c))

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to implement once features"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2017, Eric Bezault and others"
	license: "MIT License"
	date: "$Date$"
	revision: "$Revision$"
*/

#ifndef GE_ONCE_H
#define GE_ONCE_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Struct to keep track of the call status
 * and results of once features.
 */
typedef struct {
	EIF_BOOLEAN* boolean_value;
	EIF_REFERENCE* boolean_exception;
	unsigned char* boolean_status;
	unsigned int boolean_count;
	EIF_CHARACTER_8* character_8_value;
	EIF_REFERENCE* character_8_exception;
	unsigned char* character_8_status;
	unsigned int character_8_count;
	EIF_CHARACTER_32* character_32_value;
	EIF_REFERENCE* character_32_exception;
	unsigned char* character_32_status;
	unsigned int character_32_count;
	EIF_INTEGER_8* integer_8_value;
	EIF_REFERENCE* integer_8_exception;
	unsigned char* integer_8_status;
	unsigned int integer_8_count;
	EIF_INTEGER_16* integer_16_value;
	EIF_REFERENCE* integer_16_exception;
	unsigned char* integer_16_status;
	unsigned int integer_16_count;
	EIF_INTEGER_32* integer_32_value;
	EIF_REFERENCE* integer_32_exception;
	unsigned char* integer_32_status;
	unsigned int integer_32_count;
	EIF_INTEGER_64* integer_64_value;
	EIF_REFERENCE* integer_64_exception;
	unsigned char* integer_64_status;
	unsigned int integer_64_count;
	EIF_NATURAL_8* natural_8_value;
	EIF_REFERENCE* natural_8_exception;
	unsigned char* natural_8_status;
	unsigned int natural_8_count;
	EIF_NATURAL_16* natural_16_value;
	EIF_REFERENCE* natural_16_exception;
	unsigned char* natural_16_status;
	unsigned int natural_16_count;
	EIF_NATURAL_32* natural_32_value;
	EIF_REFERENCE* natural_32_exception;
	unsigned char* natural_32_status;
	unsigned int natural_32_count;
	EIF_NATURAL_64* natural_64_value;
	EIF_REFERENCE* natural_64_exception;
	unsigned char* natural_64_status;
	unsigned int natural_64_count;
	EIF_POINTER* pointer_value;
	EIF_REFERENCE* pointer_exception;
	unsigned char* pointer_status;
	unsigned int pointer_count;
	EIF_REAL_32* real_32_value;
	EIF_REFERENCE* real_32_exception;
	unsigned char* real_32_status;
	unsigned int real_32_count;
	EIF_REAL_64* real_64_value;
	EIF_REFERENCE* real_64_exception;
	unsigned char* real_64_status;
	unsigned int real_64_count;
	EIF_REFERENCE* reference_value;
	EIF_REFERENCE* reference_exception;
	unsigned char* reference_status;
	unsigned int reference_count;
	EIF_REFERENCE* procedure_exception;
	unsigned char* procedure_status;
	unsigned int procedure_count;
} GE_onces;

/*
 * Variable to keep track of the call status
 * and results of once-per-process features.
 */
extern GE_onces* GE_process_onces;

/*
 * Initialize `GE_process_onces'.
 */
extern void GE_init_onces(
	unsigned int a_boolean_count,
	unsigned int a_character_8_count,
	unsigned int a_character_32_count,
	unsigned int a_integer_8_count,
	unsigned int a_integer_16_count,
	unsigned int a_integer_32_count,
	unsigned int a_integer_64_count,
	unsigned int a_natural_8_count,
	unsigned int a_natural_16_count,
	unsigned int a_natural_32_count,
	unsigned int a_natural_64_count,
	unsigned int a_pointer_count,
	unsigned int a_real_32_count,
	unsigned int a_real_64_count,
	unsigned int a_reference_count,
	unsigned int a_procedure_count);

/*
 * Create a new 'GE_onces' struct which can deal with the
 * numbers of once features passed as argument.
 */
extern GE_onces* GE_new_onces(
	unsigned int a_boolean_count,
	unsigned int a_character_8_count,
	unsigned int a_character_32_count,
	unsigned int a_integer_8_count,
	unsigned int a_integer_16_count,
	unsigned int a_integer_32_count,
	unsigned int a_integer_64_count,
	unsigned int a_natural_8_count,
	unsigned int a_natural_16_count,
	unsigned int a_natural_32_count,
	unsigned int a_natural_64_count,
	unsigned int a_pointer_count,
	unsigned int a_real_32_count,
	unsigned int a_real_64_count,
	unsigned int a_reference_count,
	unsigned int a_procedure_count);

/*
 * Free memory allocated by `a_onces'.
 */
extern void GE_free_onces(GE_onces* a_onces);

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to implement class EXCEPTION"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2007-2018, Eric Bezault and others"
	license: "MIT License"
	date: "$Date$"
	revision: "$Revision$"
*/

#ifndef GE_EXCEPTION_H
#define GE_EXCEPTION_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif
#ifndef GE_ONCE_H
#include "ge_once.h"
#endif
#ifdef GE_USE_THREADS
#ifndef GE_THREAD_TYPES_H
#include "ge_thread_types.h"
#endif
#endif

#include <setjmp.h>

/*
 * On Linux glibc systems, we need to use sig* versions of jmp_buf,
 * setjmp and longjmp to preserve the signal handling context.
 * One way to detect this is if _SIGSET_H_types has
 * been defined in /usr/include/setjmp.h.
 * NOTE: ANSI only recognizes the non-sig versions.
 */
#if (defined(_SIGSET_H_types) && !defined(__STRICT_ANSI__))
#define GE_jmp_buf sigjmp_buf
#define GE_setjmp(x) sigsetjmp((x),1)
#define GE_longjmp(x,y) siglongjmp((x),(y))
#else
#define GE_jmp_buf jmp_buf
#define GE_setjmp(x) setjmp((x))
#define GE_longjmp(x,y) longjmp((x),(y))
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Predefined exception codes.
 */
#define GE_EX_VOID		1			/* Feature applied to void reference */
#define GE_EX_MEM		2			/* No more memory */
#define GE_EX_PRE		3			/* Pre-condition violated */
#define GE_EX_POST		4			/* Post-condition violated */
#define GE_EX_FLOAT		5			/* Floating point exception (signal SIGFPE) */
#define GE_EX_CINV		6			/* Class invariant violated */
#define GE_EX_CHECK		7			/* Check instruction violated */
#define GE_EX_FAIL		8			/* Routine failure */
#define GE_EX_WHEN		9			/* Unmatched inspect value */
#define GE_EX_VAR		10			/* Non-decreasing loop variant */
#define GE_EX_LINV		11			/* Loop invariant violated */
#define GE_EX_SIG		12			/* Operating system signal */
#define GE_EX_BYE		13			/* Eiffel run-time panic */
#define GE_EX_RESC		14			/* Exception in rescue clause */
#define GE_EX_OMEM		15			/* Out of memory (cannot be ignored) */
#define GE_EX_RES		16			/* Resumption failed (retry did not succeed) */
#define GE_EX_CDEF		17			/* Create on deferred */
#define GE_EX_EXT		18			/* External event */
#define GE_EX_VEXP		19			/* Void assigned to expanded */
#define GE_EX_HDLR		20			/* Exception in signal handler */
#define GE_EX_IO		21			/* I/O error */
#define GE_EX_SYS		22			/* Operating system error */
#define GE_EX_RETR		23			/* Retrieval error */
#define GE_EX_PROG		24			/* Developer exception */
#define GE_EX_FATAL		25			/* Eiffel run-time fatal error */
#define GE_EX_DOL		26			/* $ applied to melted feature */
#define GE_EX_ISE_IO	27			/* I/O error raised by the ISE Eiffel runtime */
#define GE_EX_COM		28			/* COM error raised by EiffelCOM runtime */
#define GE_EX_RT_CHECK	29			/* Runtime check error such as out-of-bound array access */
#define GE_EX_OLD		30			/* Old violation */
#define GE_EX_SEL		31			/* Serialization failure */
#define GE_EX_DIRTY		32			/* SCOOP processor dirty exception. */
#define GE_EX_NEX		32			/* Number of internal exceptions */

/*
 * String buffer used to build the exception trace.
 */
typedef struct GE_exception_trace_buffer_struct GE_exception_trace_buffer;
struct GE_exception_trace_buffer_struct {
	char* area;
	uint32_t count;
	uint32_t capacity;
};

/*
 * Information about the feature being executed.
 */
typedef struct GE_call_struct GE_call;
struct GE_call_struct {
#ifdef GE_USE_CURRENT_IN_EXCEPTION_TRACE
	void* object; /* Current object */
#endif
	const char* class_name;
	const char* feature_name;
	GE_call* caller; /* previous feature in the call chain */
};

/*
 * Context of features containing a rescue clause.
 */
typedef struct GE_rescue_struct GE_rescue;
struct GE_rescue_struct {
	GE_jmp_buf jb;
	GE_rescue* previous; /* previous context in the call chain */
};

/*
 * Information about the execution context.
 * One such struct per thread.
 */
typedef struct GE_context_struct GE_context;
struct GE_context_struct {
	GE_call* call; /* Call stack */
	uint32_t in_assertion; /* Is an assertion evaluated? */
	GE_rescue* last_rescue; /* Context of last feature entered containing a rescue clause */
	uint32_t in_rescue; /* Number of rescue clauses currently being executed */
	EIF_REFERENCE exception_manager; /* Exception manager */
	char raising_exception; /* Is an exception currently being raised? */
	char exception_trace_enabled; /* Should exception trace be displayed? */
	long exception_code; /* Code of the exception currently being raised, 0 otherwise */
	const char* exception_tag; /* Tag of the exception currently being raised, NULL otherwise */
	GE_exception_trace_buffer exception_trace_buffer; /* String buffer used to build the exception trace */
	GE_exception_trace_buffer last_exception_trace; /* Last non-routine-failure exception trace */
	int pre_ecma_mapping_status; /* Do we map old names to new name? (i.e. STRING to STRING_8, INTEGER to INTEGER_32, ...). */
#ifdef GE_USE_THREADS
	GE_thread_context* thread; /* Thread context */
	GE_onces* process_onces; /* Cache for status and results of onces-per-process */
	GE_onces* thread_onces; /* Status and results of onces-per-thread */
	void* wel_per_thread_data; /* WEL private data */
#endif
};

/*
 * Default initialization for `GE_context'.
 */
extern GE_context GE_default_context;

/*
 * Execution context of main thread.
 */
extern GE_context* GE_main_context;

/*
 * Execution context of current thread.
 */
extern GE_context* GE_current_context(void);

/*
 * Initialization of exception handling.
 */
extern void GE_init_exception(GE_context* context);

/*
 * Free memory allocated in `a_context' for exception handling.
 */
extern void GE_free_exception(GE_context* a_context);

/*
 * Pointer to function to create a new exception manager object
 * (of type ISE_EXCEPTION_MANAGER).
 */
extern EIF_REFERENCE (*GE_new_exception_manager)(EIF_BOOLEAN);

/*
 * Pointer to Eiffel routine ISE_EXCEPTION_MANAGER.init_exception_manager.
 */
extern void (*GE_init_exception_manager)(GE_context*, EIF_REFERENCE);

/*
 * Pointer to Eiffel routine ISE_EXCEPTION_MANAGER.last_exception.
 */
extern EIF_REFERENCE (*GE_last_exception)(GE_context*, EIF_REFERENCE);

/*
 * Pointer to Eiffel routine ISE_EXCEPTION_MANAGER.once_raise.
 */
extern void (*GE_once_raise)(GE_context*, EIF_REFERENCE, EIF_REFERENCE);

/*
 * Pointer to Eiffel routine ISE_EXCEPTION_MANAGER.set_exception_data.
 */
extern void (*GE_set_exception_data)(GE_context*, EIF_REFERENCE, EIF_INTEGER_32, EIF_BOOLEAN, EIF_INTEGER_32, EIF_INTEGER_32, EIF_REFERENCE, EIF_REFERENCE, EIF_REFERENCE, EIF_REFERENCE, EIF_REFERENCE, EIF_REFERENCE, EIF_INTEGER_32, EIF_BOOLEAN);

/*
 * Exception tag associated with `a_code'.
 */
extern char* GE_exception_tag(long a_code);

/*
 * Raise an exception with code `a_code'.
 */
extern void GE_raise(long a_code);

/*
 * Raise an exception with code `a_code' and message `msg'.
 */
extern void GE_raise_with_message(long a_code, const char* msg);

/*
 * Raise an exception from EXCEPTION_MANAGER.
 */
extern void GE_developer_raise(long a_code, char* a_meaning, char* a_message);

/*
 * Raise exception which was raised the first time a once routine
 * was executed when executing it again.
 */
extern void GE_raise_once_exception(GE_context* a_context, EIF_REFERENCE a_exception);

/*
 * Exception, if any, which was last raised in `a_context'.
 */
extern EIF_REFERENCE GE_last_exception_raised(GE_context* a_context);

/*
 * Jump to execute the rescue of the last routine with a rescue
 * in the call stack.
 */
extern void GE_jump_to_last_rescue(GE_context* a_context);

/*
 * Set `in_assertion' to 'not b'.
 * Return the opposite of previous value.
 */
extern EIF_BOOLEAN GE_check_assert(EIF_BOOLEAN b);

/*
 * Check whether the type id of `obj' is not in `type_ids'.
 * If it is, then raise a CAT-call exception. Don't do anything if `obj' is Void.
 * `nb' is the number of ids in `type_ids' and is expected to be >0.
 * `type_ids' is sorted in increasing order.
 * Return `obj'.
 */
#define GE_catcall(obj,type_ids,nb) GE_check_catcall((obj),(type_ids),(nb))
extern EIF_REFERENCE GE_check_catcall(EIF_REFERENCE obj, EIF_TYPE_INDEX type_ids[], int nb);

/*
 * Check whether `obj' is Void.
 * If it is, then raise a call-on-void-target exception.
 * Return `obj'.
 */
#define GE_void(obj) (!(obj)?GE_check_void(obj):(obj))
extern EIF_REFERENCE GE_check_void(EIF_REFERENCE obj);

/*
 * Check whether `ptr' is a null pointer.
 * If it is, then raise a no-more-memory exception.
 * Return `ptr'.
 */
#define GE_null(ptr) GE_check_null(ptr)
extern void* GE_check_null(void* ptr);

#ifdef EIF_WINDOWS
/*
 * Set default exception handler.
 */
extern void GE_set_windows_exception_filter(void);
#endif

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to manipulate native strings"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2013-2018, Eric Bezault and others"
	license: "MIT License"
	date: "$Date$"
	revision: "$Revision$"
*/

#ifndef GE_NATIVE_STRING_H
#define GE_NATIVE_STRING_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif

#include <string.h>
#ifdef EIF_WINDOWS
#ifdef __LCC__
/* With lcc-win32, stat.h should be included before wchar.h. */
#include <sys/stat.h>
#endif
#include <wchar.h>
#else
#include <sys/types.h>
#endif

#ifdef __cplusplus
extern "C" {
#endif

#ifdef EIF_WINDOWS

/* Macro used to manipulate native strings, i.e: (wchar_t*) */
#define GE_nstrlen wcslen /* size of string */
#define GE_nstrncpy wcsncpy /* Copy n characters of one string to another */
#define GE_nstrcpy wcscpy /* Copy one string to another */
#define GE_nstrncat wcsncat /* Append characters of a string */
#define GE_nstrcat wcscat /* Append a string */
#define GE_nstrstr wcsstr /* Return a pointer to the first occurrence of a search string in a string. */
#define GE_nmakestr(quote) L##quote /* Manifest Native string declaration */
#define GE_nstr_fopen _wfopen /* Open file using native string name */
#define GE_nstrcmp wcscmp /* Compare two strings. */
#define GE_nstrdup _wcsdup /* Duplicate string. */
#define GE_nstr_cat_ascii(dest, src) {							\
		int i;													\
		size_t dest_len, src_len;								\
		dest_len = rt_nstrlen (dest);							\
		src_len = strlen (src);									\
		for (i = 0; i < src_len; i++) {							\
			dest[dest_len + i] = (EIF_NATIVE_CHAR)src[i];		\
		}														\
		dest[dest_len + src_len] = (EIF_NATIVE_CHAR)0;			\
	}

#else /* not EIF_WINDOWS */

/* Macro used to manipulate native strings, i.e: (char*) */
#define GE_nstrlen strlen /* size of string */
#define GE_nstrncpy strncpy /* Copy n characters of one string to another */
#define GE_nstrcpy strcpy /* Copy one string to another */
#define GE_nstrncat strncat /* Append characters of a string */
#define GE_nstrcat strcat /* Append a string */
#define GE_nstrstr strstr /* Return a pointer to the first occurrence of a search string in a string. */
#define GE_nmakestr(quote) quote /* Manifest Native string declaration */
#define GE_nstr_fopen fopen /* Open file using native string name */
#define GE_nstrcmp strcmp /* Compare two strings. */
#define GE_nstrdup strdup /* Duplicate string. */
#define GE_nstr_cat_ascii strcat

#endif


#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to implement class ARGUMENTS"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2007-2017, Eric Bezault and others"
	license: "MIT License"
	date: "$Date$"
	revision: "$Revision$"
*/

#ifndef GE_ARGUMENTS_H
#define GE_ARGUMENTS_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

extern int GE_argc;
extern EIF_NATIVE_CHAR** GE_argv;

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to implement type information"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2016-2018, Eric Bezault and others"
	license: "MIT License"
	date: "$Date$"
	revision: "$Revision$"
*/

#ifndef GE_TYPES_H
#define GE_TYPES_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif
#ifndef GE_EXCEPTION_H
#include "ge_exception.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Type annotations.
 * When a type has no annotation, it means a detachable, non-separate, variant type.
 * In all other cases, there will be an annotation.
 */
#define ANNOTATION_MASK			0x007F	/* All possible annotations. */
#define ATTACHED_FLAG			0x0001
#define DETACHABLE_FLAG			0x0002	/* Only present when overriding an attached type. */
#define SEPARATE_FLAG			0x0004
#define VARIANT_FLAG			0x0008	/* Only present when overriding a frozen/poly type. */
#define UNUSABLE_FLAG			0x0010	/* Reserved for backward compatibility for storables. */
#define FROZEN_FLAG				0x0020
#define POLY_FLAG				0x0040

/*
 * Type flags.
 */
#define GE_TYPE_FLAG_SPECIAL		0x0010
#define GE_TYPE_FLAG_TUPLE			0x0020
#define GE_TYPE_FLAG_EXPANDED		0x0040
#define GE_TYPE_FLAG_DEFERRED		0x0080
#define GE_TYPE_FLAG_NONE			0x0100
#define GE_TYPE_FLAG_BASIC_MASK		0x000F /* One of "BOOLEAN", "CHARACTER_8", "CHARACTER_32", "INTEGER_8", "INTEGER_16", "INTEGER_32", "INTEGER_64", "NATURAL_8", "NATURAL_16", "NATURAL_32", "NATURAL_64", "POINTER", "REAL_32", "REAL_64" */
#define GE_TYPE_FLAG_BOOLEAN		0x0001
#define GE_TYPE_FLAG_CHARACTER_8	0x0002
#define GE_TYPE_FLAG_CHARACTER_32	0x0003
#define GE_TYPE_FLAG_INTEGER_8		0x0004
#define GE_TYPE_FLAG_INTEGER_16		0x0005
#define GE_TYPE_FLAG_INTEGER_32		0x0006
#define GE_TYPE_FLAG_INTEGER_64		0x0007
#define GE_TYPE_FLAG_NATURAL_8		0x0008
#define GE_TYPE_FLAG_NATURAL_16		0x0009
#define GE_TYPE_FLAG_NATURAL_32		0x000A
#define GE_TYPE_FLAG_NATURAL_64		0x000B
#define GE_TYPE_FLAG_POINTER		0x000C
#define GE_TYPE_FLAG_REAL_32		0x000D
#define GE_TYPE_FLAG_REAL_64		0x000E

/*
 * Convention for attribute types.
 * The values are in sync with REFLECTOR_CONSTANTS.
 */
#define GE_TYPE_KIND_INVALID		-1
#define GE_TYPE_KIND_POINTER		0
#define GE_TYPE_KIND_REFERENCE		1
#define GE_TYPE_KIND_CHARACTER_8	2
#define GE_TYPE_KIND_BOOLEAN		3
#define GE_TYPE_KIND_INTEGER_32		4
#define GE_TYPE_KIND_REAL_32		5
#define GE_TYPE_KIND_REAL_64		6
#define GE_TYPE_KIND_EXPANDED		7
#define GE_TYPE_KIND_INTEGER_8		9
#define GE_TYPE_KIND_INTEGER_16		10
#define GE_TYPE_KIND_INTEGER_64 	11
#define GE_TYPE_KIND_CHARACTER_32	12
#define GE_TYPE_KIND_NATURAL_8		13
#define GE_TYPE_KIND_NATURAL_16		14
#define GE_TYPE_KIND_NATURAL_32 	15
#define GE_TYPE_KIND_NATURAL_64 	16

/*
 * Object flags.
 */
#define GE_OBJECT_FLAG_MARKED		0x0001

/*
 * Ancestor relationship between two types X and Y.
 */
#ifdef GE_USE_ANCESTORS
typedef struct {
	EIF_TYPE_INDEX type_id; /* Type id of Y */
	EIF_BOOLEAN conforms; /* Does X conform to Y? */
	void (**qualified_calls)(); /* Function pointers, indexed by call id, when the static type of the target is Y and the dynamic type is X */
} GE_ancestor;
#endif

/*
 * Attribute.
 */
#ifdef GE_USE_ATTRIBUTES
typedef struct {
#ifdef GE_USE_ATTRIBUTE_NAME
	const char* name; /* Attribute name */
#endif
#ifdef GE_USE_ATTRIBUTE_TYPE_ID
	EIF_ENCODED_TYPE type_id; /* Static type id */
#endif
#ifdef GE_USE_ATTRIBUTE_OFFSET
	uint32_t offset; /* Address offset in object */
#endif
} GE_attribute;
#endif

/*
 * Type information.
 */
typedef struct {
	EIF_TYPE_INDEX type_id;
	uint16_t flags;
#ifdef GE_USE_TYPE_GENERATOR
	const char* generator; /* Generator class name */
#endif
#ifdef GE_USE_TYPE_NAME
	const char* name; /* Full type name */
#endif
#ifdef GE_USE_TYPE_GENERIC_PARAMETERS
	EIF_ENCODED_TYPE* generic_parameters;
	uint32_t generic_parameter_count;
#endif
#ifdef GE_USE_ANCESTORS
	GE_ancestor** ancestors;
	uint32_t ancestor_count;
#endif
#ifdef GE_USE_ATTRIBUTES
	GE_attribute** attributes;
	uint32_t attribute_count;
#endif
#ifdef GE_USE_TYPE_OBJECT_SIZE
	uint64_t object_size;
#endif
	EIF_REFERENCE (*new_instance)();
	void (*dispose)(GE_context*, EIF_REFERENCE);
} GE_type_info;

typedef struct {
	EIF_TYPE_INDEX id; /* Type id of the "TYPE [X]" object */
	EIF_INTEGER type_id; /* Type id of the type "X" */
	EIF_BOOLEAN is_special;
	void (*dispose)(GE_context*, EIF_REFERENCE);
	EIF_REFERENCE a1; /* internal_name */
	EIF_REFERENCE a2; /* internal_name_32 */
} EIF_TYPE_OBJ;

/*
 * Types indexed by type id.
 * Generated by the compiler.
 */
extern EIF_TYPE_OBJ GE_types[][2];
extern GE_type_info GE_type_infos[];

/*
 * Number of type infos in `GE_type_infos'.
 * Do not take into account the fake item at index 0.
 */
extern int GE_type_info_count;

/*
 * Encode a EIF_TYPE into a EIF_ENCODED_TYPE.
 * The lower part of EIF_ENCODED_TYPE contains the .id field,
 * and the upper part the .annotations.
 */
extern EIF_ENCODED_TYPE GE_encoded_type(EIF_TYPE a_type);

/*
 * Decode a EIF_ENCODED_TYPE into a EIF_TYPE.
 * The lower part of EIF_ENCODED_TYPE contains the .id field,
 * and the upper part the .annotations.
 */
extern EIF_TYPE GE_decoded_type(EIF_ENCODED_TYPE a_type);

/*
 * Type with `a_id' and `a_annotations'.
 */
extern EIF_TYPE GE_new_type(EIF_TYPE_INDEX a_id, EIF_TYPE_INDEX a_annotations);

/*
 * Type of object `obj'.
 */
#define GE_object_type(obj)	GE_new_type(((EIF_REFERENCE)(obj))->id, 0x0)
#define GE_object_encoded_type(obj) GE_encoded_type(GE_object_type(obj))

/*
 * Attachment status of `a_type'.
 */
#define GE_is_attached_type(a_type) EIF_TEST(((a_type).annotations & ATTACHED_FLAG) || GE_is_expanded_type_index((a_type).id))
#define GE_is_attached_encoded_type(a_type) GE_is_attached_type(GE_decoded_type(a_type))

/*
 * Associated detachable type of `a_type' if any,
 * otherwise `a_type'.
 */
extern EIF_TYPE GE_non_attached_type(EIF_TYPE a_type);
#define GE_non_attached_encoded_type(a_type) GE_encoded_type(GE_non_attached_type(GE_decoded_type(a_type)))

/*
 * Associated attached type of `a_type' if any,
 * otherwise `a_type'.
 */
extern EIF_TYPE GE_attached_type(EIF_TYPE a_type);
#define GE_attached_encoded_type(t) GE_encoded_type(GE_attached_type(GE_decoded_type(t)))

/*
 * Is `a_type' a SPECIAL type?
 */
#define GE_is_special_type_index(a_type) EIF_TEST(GE_type_infos[a_type].flags & GE_TYPE_FLAG_SPECIAL)
#define GE_is_special_encoded_type(a_type) GE_is_special_type_index(GE_decoded_type(a_type).id)
#define GE_is_special_object(obj) GE_is_special_type_index(((EIF_REFERENCE)(obj))->id)

/*
 * Is `a_type' a SPECIAL type of user-defined expanded type?
 */
extern EIF_BOOLEAN GE_is_special_of_expanded_type_index(EIF_TYPE_INDEX a_type);
#define GE_is_special_of_expanded_encoded_type(a_type) GE_is_special_of_expanded_type_index(GE_decoded_type(a_type).id)
#define GE_is_special_of_expanded_object(obj) GE_is_special_of_expanded_type_index(((EIF_REFERENCE)(obj))->id)

/*
 * Is `a_type' a SPECIAL type of reference type?
 */
extern EIF_BOOLEAN GE_is_special_of_reference_type_index(EIF_TYPE_INDEX a_type);
#define GE_is_special_of_reference_encoded_type(a_type) GE_is_special_of_reference_type_index(GE_decoded_type(a_type).id)
#define GE_is_special_of_reference_object(obj) GE_is_special_of_reference_type_index(((EIF_REFERENCE)(obj))->id)

/*
 * Is `a_type' a SPECIAL type of reference type or basic expanded type?
 * (Note that user-defined expanded types are excluded.)
 */
extern EIF_BOOLEAN GE_is_special_of_reference_or_basic_expanded_type_index(EIF_TYPE_INDEX a_type);
#define GE_is_special_of_reference_or_basic_expanded_encoded_type(a_type) GE_is_special_of_reference_or_basic_expanded_type_index(GE_decoded_type(a_type).id)
#define GE_is_special_of_reference_or_basic_expanded_object(obj) GE_is_special_of_reference_or_basic_expanded_type_index(((EIF_REFERENCE)(obj))->id)

/*
 * Is `a_type' a TUPLE type?
 */
#define GE_is_tuple_type_index(a_type) EIF_TEST(GE_type_infos[a_type].flags & GE_TYPE_FLAG_TUPLE)
#define GE_is_tuple_encoded_type(a_type) GE_is_tuple_type_index(GE_decoded_type(a_type).id)
#define GE_is_tuple_object(obj) GE_is_tuple_type_index(((EIF_REFERENCE)(obj))->id)

/*
 * Is `a_type' an expanded type?
 */
#define GE_is_expanded_type_index(a_type) EIF_TEST(GE_type_infos[a_type].flags & GE_TYPE_FLAG_EXPANDED)
#define GE_is_expanded_encoded_type(a_type) GE_is_expanded_type_index(GE_decoded_type(a_type).id)
#define GE_is_expanded_object(obj) GE_is_expanded_type_index(((EIF_REFERENCE)(obj))->id)

/*
 * Is `a_type' a type whose base class is deferred?
 */
#define GE_is_deferred_type_index(a_type) EIF_TEST(GE_type_infos[a_type].flags & GE_TYPE_FLAG_DEFERRED)
#define GE_is_deferred_encoded_type(a_type) GE_is_deferred_type_index(GE_decoded_type(a_type).id)

/*
 * Does `i'-th field of `a_object + a_physical_offset' (which is expected to be reference)
 * denote a reference with copy semantics?
 */
extern EIF_BOOLEAN GE_is_copy_semantics_field(EIF_INTEGER i, EIF_POINTER a_object, EIF_INTEGER a_physical_offset);

/*
 * Does `i'-th item of special `a_object' (which is expected to be reference)
 * denote a reference with copy semantics?
 */
extern EIF_BOOLEAN GE_is_special_copy_semantics_item(EIF_INTEGER i, EIF_POINTER a_object);

/*
 * Generator class name of `a_type'.
 */
extern EIF_REFERENCE GE_generator_of_type_index(EIF_TYPE_INDEX a_type);
#define GE_generator_of_encoded_type(a_type) GE_generator_of_type_index(GE_decoded_type(a_type).id)

/*
 * Full name of `a_type'.
 */
extern EIF_REFERENCE GE_generating_type_of_encoded_type(EIF_ENCODED_TYPE a_type);

/*
 * Encoded type whose name is `a_name'.
 * -1 if no such type.
 */
extern EIF_ENCODED_TYPE GE_encoded_type_from_name(EIF_POINTER a_name);

/*
 * Does `a_type_1' conform to `a_type_2'?
 */
extern EIF_BOOLEAN GE_encoded_type_conforms_to(EIF_ENCODED_TYPE a_type_1, EIF_ENCODED_TYPE a_type_2);

/*
 * Number of generic parameters.
 */
extern EIF_INTEGER GE_generic_parameter_count_of_type_index(EIF_TYPE_INDEX a_type);
#define GE_generic_parameter_count_of_encoded_type(a_type) GE_generic_parameter_count_of_type_index(GE_decoded_type(a_type).id)

/*
 * Type of `i'-th generic parameter of `a_type'.
 */
extern EIF_INTEGER GE_generic_parameter_of_type_index(EIF_TYPE_INDEX a_type, EIF_INTEGER i);
#define GE_generic_parameter_of_encoded_type(a_type,i) GE_generic_parameter_of_type_index(GE_decoded_type(a_type).id, (i))

/*
 * Number of fields of an object of dynamic type `a_type'.
 */
extern EIF_INTEGER GE_field_count_of_type_index(EIF_TYPE_INDEX a_type);
#define GE_field_count_of_encoded_type(a_type) GE_field_count_of_type_index(GE_decoded_type(a_type).id)

/*
 * Physical offset of the `i'-th field for an object of dynamic type `a_type'.
 */
extern EIF_INTEGER GE_field_offset_of_type_index(EIF_INTEGER i, EIF_TYPE_INDEX a_type);
#define GE_field_offset_of_encoded_type(i, a_type) GE_field_offset_of_type_index((i), GE_decoded_type(a_type).id)

/*
 * Name of the `i'-th field for an object of dynamic type `a_type'.
 */
extern EIF_POINTER GE_field_name_of_type_index(EIF_INTEGER i, EIF_TYPE_INDEX a_type);
#define GE_field_name_of_encoded_type(i, a_type) GE_field_name_of_type_index((i), GE_decoded_type(a_type).id)

/*
 * Static type of the `i'-th field for an object of dynamic type `a_type'.
 */
extern EIF_INTEGER GE_field_static_type_of_type_index(EIF_INTEGER i, EIF_TYPE_INDEX a_type);
#define GE_field_static_type_of_encoded_type(i, a_type) GE_field_static_type_of_type_index((i), GE_decoded_type(a_type).id)

/*
 * Kind of type of the `i'-th field for an object of dynamic type `a_type'.
 */
extern EIF_INTEGER GE_field_type_kind_of_type_index(EIF_INTEGER i, EIF_TYPE_INDEX a_type);
#define GE_field_type_kind_of_encoded_type(i, a_type) GE_field_type_kind_of_type_index((i), GE_decoded_type(a_type).id)

/*
 * Physical size of `a_object'.
 */
extern EIF_NATURAL_64 GE_object_size(EIF_POINTER a_object);

/*
 * Is `i'-th field of objects of type `a_type' a user-defined expanded attribute?
 */
extern EIF_BOOLEAN GE_is_field_expanded_of_type_index(EIF_INTEGER i, EIF_TYPE_INDEX a_type);
#define GE_is_field_expanded_of_encoded_type(i, a_type) GE_is_field_expanded_of_type_index((i), GE_decoded_type(a_type).id)

#define GE_field_address_at(a_field_offset, a_object, a_physical_offset) ((char*)(a_object) + (a_physical_offset) + (a_field_offset))
#define GE_object_at_offset(a_enclosing, a_physical_offset) (EIF_REFERENCE)(GE_field_address_at(0, (a_enclosing), (a_physical_offset)))
#define GE_raw_object_at_offset(a_enclosing, a_physical_offset) (EIF_POINTER)(GE_field_address_at(0, (a_enclosing), (a_physical_offset)))
#define GE_object_encoded_type_at_offset(a_enclosing, a_physical_offset) GE_object_encoded_type(GE_raw_object_at_offset((a_enclosing), (a_physical_offset)))
#define GE_boolean_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_BOOLEAN*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_character_8_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_CHARACTER_8*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_character_32_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_CHARACTER_32*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_integer_8_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_INTEGER_8*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_integer_16_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_INTEGER_16*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_integer_32_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_INTEGER_32*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_integer_64_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_INTEGER_64*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_natural_8_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_NATURAL_8*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_natural_16_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_NATURAL_16*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_natural_32_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_NATURAL_32*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_natural_64_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_NATURAL_64*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_pointer_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_POINTER*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_real_32_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_REAL_32*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_real_64_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_REAL_64*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_raw_reference_field_at(a_field_offset, a_object, a_physical_offset) (EIF_POINTER)*(EIF_REFERENCE*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_reference_field_at(a_field_offset, a_object, a_physical_offset) *(EIF_REFERENCE*)(GE_field_address_at((a_field_offset), (a_object), (a_physical_offset)))
#define GE_set_boolean_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_boolean_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_character_8_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_character_8_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_character_32_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_character_32_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_integer_8_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_integer_8_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_integer_16_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_integer_16_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_integer_32_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_integer_32_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_integer_64_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_integer_64_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_natural_8_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_natural_8_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_natural_16_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_natural_16_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_natural_32_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_natural_32_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_natural_64_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_natural_64_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_pointer_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_pointer_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_real_32_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_real_32_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_real_64_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_real_64_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value
#define GE_set_reference_field_at(a_field_offset, a_object, a_physical_offset, a_value) GE_reference_field_at((a_field_offset), (a_object), (a_physical_offset)) = a_value

#if defined(GE_USE_ATTRIBUTES) && defined(GE_USE_ATTRIBUTE_OFFSET)
#define GE_field_address(i, a_object, a_physical_offset) GE_field_address_at(GE_type_infos[((EIF_REFERENCE)(a_object))->id].attributes[(i) - 1]->offset, (a_object), (a_physical_offset))
#define GE_boolean_field(i, a_object, a_physical_offset) *(EIF_BOOLEAN*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_character_8_field(i, a_object, a_physical_offset) *(EIF_CHARACTER_8*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_character_32_field(i, a_object, a_physical_offset) *(EIF_CHARACTER_32*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_integer_8_field(i, a_object, a_physical_offset) *(EIF_INTEGER_8*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_integer_16_field(i, a_object, a_physical_offset) *(EIF_INTEGER_16*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_integer_32_field(i, a_object, a_physical_offset) *(EIF_INTEGER_32*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_integer_64_field(i, a_object, a_physical_offset) *(EIF_INTEGER_64*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_natural_8_field(i, a_object, a_physical_offset) *(EIF_NATURAL_8*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_natural_16_field(i, a_object, a_physical_offset) *(EIF_NATURAL_16*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_natural_32_field(i, a_object, a_physical_offset) *(EIF_NATURAL_32*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_natural_64_field(i, a_object, a_physical_offset) *(EIF_NATURAL_64*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_pointer_field(i, a_object, a_physical_offset) *(EIF_POINTER*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_real_32_field(i, a_object, a_physical_offset) *(EIF_REAL_32*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_real_64_field(i, a_object, a_physical_offset) *(EIF_REAL_64*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_reference_field(i, a_object, a_physical_offset) *(EIF_REFERENCE*)(GE_field_address((i), (a_object), (a_physical_offset)))
#define GE_set_boolean_field(i, a_object, a_physical_offset, a_value) GE_boolean_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_character_8_field(i, a_object, a_physical_offset, a_value) GE_character_8_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_character_32_field(i, a_object, a_physical_offset, a_value) GE_character_32_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_integer_8_field(i, a_object, a_physical_offset, a_value) GE_integer_8_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_integer_16_field(i, a_object, a_physical_offset, a_value) GE_integer_16_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_integer_32_field(i, a_object, a_physical_offset, a_value) GE_integer_32_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_integer_64_field(i, a_object, a_physical_offset, a_value) GE_integer_64_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_natural_8_field(i, a_object, a_physical_offset, a_value) GE_natural_8_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_natural_16_field(i, a_object, a_physical_offset, a_value) GE_natural_16_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_natural_32_field(i, a_object, a_physical_offset, a_value) GE_natural_32_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_natural_64_field(i, a_object, a_physical_offset, a_value) GE_natural_64_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_pointer_field(i, a_object, a_physical_offset, a_value) GE_pointer_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_real_32_field(i, a_object, a_physical_offset, a_value) GE_real_32_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_real_64_field(i, a_object, a_physical_offset, a_value) GE_real_64_field((i), (a_object), (a_physical_offset)) = (a_value)
#define GE_set_reference_field(i, a_object, a_physical_offset, a_value) GE_reference_field((i), (a_object), (a_physical_offset)) = (a_value)
#else
#define GE_boolean_field(i, a_object, a_physical_offset) (EIF_BOOLEAN)0
#define GE_character_8_field(i, a_object, a_physical_offset) (EIF_CHARACTER_8)0
#define GE_character_32_field(i, a_object, a_physical_offset) (EIF_CHARACTER_32)0
#define GE_integer_8_field(i, a_object, a_physical_offset) (EIF_INTEGER_8)0
#define GE_integer_16_field(i, a_object, a_physical_offset) (EIF_INTEGER_16)0
#define GE_integer_32_field(i, a_object, a_physical_offset) (EIF_INTEGER_32)0
#define GE_integer_64_field(i, a_object, a_physical_offset) (EIF_INTEGER_64)0
#define GE_natural_8_field(i, a_object, a_physical_offset) (EIF_NATURAL_8)0
#define GE_natural_16_field(i, a_object, a_physical_offset) (EIF_NATURAL_16)0
#define GE_natural_32_field(i, a_object, a_physical_offset) (EIF_NATURAL_32)0
#define GE_natural_64_field(i, a_object, a_physical_offset) (EIF_NATURAL_64)0
#define GE_pointer_field(i, a_object, a_physical_offset) (EIF_POINTER)0
#define GE_real_32_field(i, a_object, a_physical_offset) (EIF_REAL_32)0
#define GE_real_64_field(i, a_object, a_physical_offset) (EIF_REAL_64)0
#define GE_reference_field(i, a_object, a_physical_offset) (EIF_REFERENCE)0
#define GE_set_boolean_field(i, a_object, a_physical_offset, a_value)
#define GE_set_character_8_field(i, a_object, a_physical_offset, a_value)
#define GE_set_character_32_field(i, a_object, a_physical_offset, a_value)
#define GE_set_integer_8_field(i, a_object, a_physical_offset, a_value)
#define GE_set_integer_16_field(i, a_object, a_physical_offset, a_value)
#define GE_set_integer_32_field(i, a_object, a_physical_offset, a_value)
#define GE_set_integer_64_field(i, a_object, a_physical_offset, a_value)
#define GE_set_natural_8_field(i, a_object, a_physical_offset, a_value)
#define GE_set_natural_16_field(i, a_object, a_physical_offset, a_value)
#define GE_set_natural_32_field(i, a_object, a_physical_offset, a_value)
#define GE_set_natural_64_field(i, a_object, a_physical_offset, a_value)
#define GE_set_pointer_field(i, a_object, a_physical_offset, a_value)
#define GE_set_real_32_field(i, a_object, a_physical_offset, a_value)
#define GE_set_real_64_field(i, a_object, a_physical_offset, a_value)
#define GE_set_reference_field(i, a_object, a_physical_offset, a_value)
#endif

/*
 * Number of non-transient fields of an object of dynamic type `a_type'.
 * TODO: storable not implemented yet.
 */
#define GE_persistent_field_count_of_type_index(a_type) GE_field_count_of_type_index(a_type)
#define GE_persistent_field_count_of_encoded_type(a_type) GE_persistent_field_count_of_type_index(GE_decoded_type(a_type).id)

/*
 * Is `i'-th field of objects of type `a_type' a transient field?
 * TODO: storable not implemented yet.
 */
#define GE_is_field_transient_of_type_index(i, a_type) EIF_FALSE
#define GE_is_field_transient_of_encoded_type(i, a_type) GE_is_field_transient_of_type_index((i), GE_decoded_type(a_type).id)

/*
 * Storable version of objects of type `a_type'.
 * TODO: storable not implemented yet.
 */
#define GE_storable_version_of_type_index(a_type) EIF_VOID
#define GE_storable_version_of_encoded_type(a_type) GE_storable_version_of_type_index(GE_decoded_type(a_type).id)

/*
 * Get a lock on `GE_mark_object' and `GE_unmark_object' routines so that
 * 2 threads cannot `GE_mark_object' and `GE_unmark_object' at the same time.
 */
extern void GE_lock_marking(void);

/*
 * Release a lock on `GE_mark_object' and `GE_unmark_object', so that another
 * thread can use `GE_mark_object' and `GE_unmark_object'.
 */
extern void GE_unlock_marking(void);

/*
 * Is `obj' marked?
 */
extern EIF_BOOLEAN GE_is_object_marked(EIF_POINTER obj);

/*
 * Mark `obj'.
 */
extern void GE_mark_object(EIF_POINTER obj);

/*
 * Unmark `obj'.
 */
extern void GE_unmark_object(EIF_POINTER obj);

/*
 * New instance of dynamic `a_type'.
 * Note: returned object is not initialized and may
 * hence violate its invariant.
 * `a_type' cannot represent a SPECIAL type, use
 * `GE_new_special_of_reference_instance_of_type_index' instead.
 */
extern EIF_REFERENCE GE_new_instance_of_type_index(EIF_TYPE_INDEX a_type);
#define GE_new_instance_of_encoded_type(a_type) GE_new_instance_of_type_index(GE_decoded_type(a_type).id)

/*
 * New instance of dynamic `a_type' that represents
 * a SPECIAL with can contain `a_capacity' elements of reference type.
 * To create a SPECIAL of basic type, use class SPECIAL directly.
 */
extern EIF_REFERENCE GE_new_special_of_reference_instance_of_type_index(EIF_TYPE_INDEX a_type, EIF_INTEGER a_capacity);
#define GE_new_special_of_reference_instance_of_encoded_type(a_type, a_capacity) GE_new_special_of_reference_instance_of_type_index(GE_decoded_type(a_type).id, (a_capacity))

/*
 * New instance of tuple of type `a_type'.
 * Note: returned object is not initialized and may
 * hence violate its invariant.
 */
#define GE_new_tuple_instance_of_type_index(a_type) GE_new_instance_of_type_index(a_type)
#define GE_new_tuple_instance_of_encoded_type(a_type) GE_new_tuple_instance_of_type_index(GE_decoded_type(a_type).id)

/*
 * New instance of TYPE for object of type `a_type'.
 */
extern EIF_REFERENCE GE_new_type_instance_of_encoded_type(EIF_ENCODED_TYPE a_type);

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to manipulate strings"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2016-2019, Eric Bezault and others"
	license: "MIT License"
	date: "$Date$"
	revision: "$Revision$"
*/

#ifndef GE_STRING_H
#define GE_STRING_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*
 * New Eiffel empty string of type "STRING_8" with can
 * contain `c' characters.
 * Note: The implementation of this function is generated
 * by the Eiffel compiler.
 */
extern EIF_REFERENCE GE_new_str8(EIF_INTEGER c);

/*
 * New Eiffel empty string of type "IMMUTABLE_STRING_8" with can
 * contain `c' characters.
 * Note: The implementation of this function is generated
 * by the Eiffel compiler.
 */
extern EIF_REFERENCE GE_new_istr8(EIF_INTEGER c);

/*
 * New Eiffel empty string of type "STRING_32" with can
 * contain `c' characters.
 * Note: The implementation of this function is generated
 * by the Eiffel compiler.
 */
extern EIF_REFERENCE GE_new_str32(EIF_INTEGER c);

/*
 * New Eiffel empty string of type "IMMUTABLE_STRING_32" with can
 * contain `c' characters.
 * Note: The implementation of this function is generated
 * by the Eiffel compiler.
 */
extern EIF_REFERENCE GE_new_istr32(EIF_INTEGER c);

/*
 * New Eiffel string of type "STRING_8" containing the
 * first `c' characters found in ISO 8859-1 string `s'.
 */
extern EIF_REFERENCE GE_ms8(const char* s, EIF_INTEGER c);

/*
 * New Eiffel string of type "STRING_8" containing all
 * characters found in the null-terminated ISO 8859-1 string `s'.
 */
extern EIF_REFERENCE GE_str8(const char* s);

/*
 * New Eiffel string of type "IMMUTABLE_STRING_8" containing the
 * first `c' characters found in ISO 8859-1 string `s'.
 */
extern EIF_REFERENCE GE_ims8(const char* s, EIF_INTEGER c);

/*
 * New Eiffel string of type "STRING_32" containing the
 * first `c' characters found in ISO 8859-1 string `s'.
 */
extern EIF_REFERENCE GE_ms32(const char* s, EIF_INTEGER c);

/*
 * New Eiffel string of type "STRING_32" containing the
 * first `c' 32-bit characters built from `s' by reading
 * groups of four bytes with little-endian byte order.
 */
extern EIF_REFERENCE GE_ms32_from_utf32le(const char* s, EIF_INTEGER c);

/*
 * New Eiffel string of type "STRING_32" containing all
 * characters found in the null-terminated ISO 8859-1 string `s'.
 */
extern EIF_REFERENCE GE_str32(const char* s);

/*
 * New Eiffel string of type "IMMUTABLE_STRING_32" containing
 * the first `c' characters found in ISO 8859-1 string `s'.
 */
extern EIF_REFERENCE GE_ims32(const char* s, EIF_INTEGER c);

/*
 * New Eiffel string of type "IMMUTABLE_STRING_32" containing the
 * first `c' 32-bit characters built from `s' by reading
 * groups of four bytes with little-endian byte order.
 */
extern EIF_REFERENCE GE_ms32_from_utf32le(const char* s, EIF_INTEGER c);

/*
 * New Eiffel string of type "IMMUTABLE_STRING_32" containing all
 * characters found in the null-terminated ISO 8859-1 string `s'.
 */
extern EIF_REFERENCE GE_istr32(const char* s);

/*
 * New Eiffel string of type "IMMUTABLE_STRING_32" containing the
 * first `n' native characters found in native string `s'.
 * Invalid native characters are escaped.
 */
extern EIF_REFERENCE GE_ims32_from_nstr(EIF_NATIVE_CHAR* s, EIF_INTEGER n);

/*
 * New Eiffel string of type "IMMUTABLE_STRING_32" containing all
 * characters found in the null-terminated native string `s'.
 * Invalid native characters are escaped.
 */
extern EIF_REFERENCE GE_istr32_from_nstr(EIF_NATIVE_CHAR* s);

/*
 * New Eiffel string of type "STRING" containing all
 * characters found in the null-terminated ISO 8859-1 string `s'
 */
extern EIF_REFERENCE GE_str(const char* s);

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to implement class CONSOLE"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2007-2017, Eric Bezault and others"
	license: "MIT License"
	date: "$Date$"
	revision: "$Revision$"
*/

#ifndef GE_CONSOLE_H
#define GE_CONSOLE_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Initialize mutex to determine whether a new
 * console needs to be created.
 */
#ifdef EIF_WINDOWS
extern void GE_init_console(void);
#else
#define GE_init_console()
#endif

/*
 * Create a new DOS console if needed (i.e. in case of a Windows application).
 */
#ifdef EIF_WINDOWS
extern void GE_show_console(void);
#else
#define GE_show_console()
#endif

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to implement the program initialization"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2007-2017, Eric Bezault and others"
	license: "MIT License"
	date: "$Date$"
	revision: "$Revision$"
*/

#ifndef GE_MAIN_H
#define GE_MAIN_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

extern int GE_main(int argc, EIF_NATIVE_CHAR** argv);

/*
 * System name.
 */
extern char* GE_system_name;

/*
 * Root class name.
 */
extern char* GE_root_class_name;

#ifdef EIF_WINDOWS

#include <windows.h>

/*
 * Used in WEL.
 */
extern HINSTANCE eif_hInstance;
extern HINSTANCE eif_hPrevInstance;
extern LPWSTR eif_lpCmdLine;
extern int eif_nCmdShow;

/*
 * Main entry point when compiling a Windows application.
 * See:
 *    http://en.wikipedia.org/wiki/WinMain
 *    http://msdn2.microsoft.com/en-us/library/ms633559.aspx
 */
extern int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow);

#endif

#ifdef __cplusplus
}
#endif

#endif

/*
	description:

		"C functions used to access garbage collector facilities"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2007-2017, Eric Bezault and others"
	license: "MIT License"
	date: "$Date$"
	revision: "$Revision$"
*/

#ifndef GE_GC_H
#define GE_GC_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EXCEPTION_H
#include "ge_exception.h"
#endif

#ifdef GE_USE_BOEHM_GC
/*
 *	Use the Boehm garbage collector.
 *	See:
 *		http://en.wikipedia.org/wiki/Boehm_GC
 *		http://www.hpl.hp.com/personal/Hans_Boehm/gc/
 */
#define GC_IGNORE_WARN
#ifdef GE_USE_THREADS
#define GC_THREADS
#undef GC_NO_THREAD_REDIRECTS
#endif
#include "gc.h"
#endif

/*
 * GC initialization.
 */

#ifdef GE_USE_BOEHM_GC
#ifdef GE_USE_THREADS
#define GE_init_gc() \
	GC_INIT(); \
	GC_allow_register_threads(); \
	GC_enable_incremental()
#else
#define GE_init_gc() \
	GC_INIT(); \
	GC_enable_incremental()
#endif
#else /* No GC */
#define GE_init_gc() /* do nothing */
#endif

/*
 * Memory allocation.
 */

/*
 * Allocate memory that can contain pointers to collectable objects.
 * The allocated memory is not necessarily zeroed (unless `GE_malloc_cleared' is_defined).
 * The allocated object is itself collectable.
 * Raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_malloc(size) GE_null(GC_MALLOC(size))
#else /* No GC */
#define GE_malloc(size) GE_null(malloc(size))
#endif

/*
 * Allocate memory that can contain pointers to collectable objects.
 * The allocated memory is not necessarily zeroed (unless `GE_malloc_cleared' is_defined).
 * The allocated object is itself collectable.
 * Do not raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_unprotected_malloc(size) GC_MALLOC(size)
#else /* No GC */
#define GE_unprotected_malloc(size) malloc(size)
#endif

/*
 * Allocate memory that does not contain pointers to collectable objects.
 * The allocated memory is not necessarily zeroed (unless `GE_malloc_atomic_cleared' is_defined).
 * The allocated object is itself collectable.
 * Raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_malloc_atomic(size) GE_null(GC_MALLOC_ATOMIC(size))
#else /* No GC */
#define GE_malloc_atomic(size) GE_null(malloc(size))
#endif

/*
 * Allocate memory that does not contain pointers to collectable objects.
 * The allocated memory is not necessarily zeroed (unless `GE_malloc_atomic_cleared' is_defined).
 * The allocated object is itself collectable.
 * Do not raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_unprotected_malloc_atomic(size) GC_MALLOC_ATOMIC(size)
#else /* No GC */
#define GE_unprotected_malloc_atomic(size) malloc(size)
#endif

/*
 * Allocate memory that can contain pointers to collectable objects.
 * The allocated memory is zeroed.
 * The allocated object is itself collectable.
 * Raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_calloc(nelem, elsize) GE_null(GC_MALLOC((nelem) * (elsize)))
#else /* No GC */
#define GE_calloc(nelem, elsize) GE_null(calloc((nelem), (elsize)))
#endif

/*
 * Allocate memory that can contain pointers to collectable objects.
 * The allocated memory is zeroed.
 * The allocated object is itself collectable.
 * Do not raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_unprotected_calloc(nelem, elsize) GC_MALLOC((nelem) * (elsize))
#else /* No GC */
#define GE_unprotected_calloc(nelem, elsize) calloc((nelem), (elsize))
#endif

/*
 * Allocate memory that does not contain pointers to collectable objects.
 * The allocated memory is zeroed.
 * The allocated object is itself collectable.
 * Raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_calloc_atomic(nelem, elsize) memset(GE_null(GC_MALLOC_ATOMIC((nelem) * (elsize))), 0, (nelem) * (elsize))
#else /* No GC */
#define GE_calloc(nelem, elsize) GE_null(calloc((nelem), (elsize)))
#endif

/*
 * Allocate memory that can contain pointers to collectable objects.
 * The allocated memory is not necessarily zeroed (unless `GE_malloc_cleared' is_defined).
 * The allocated object is itself not collectable.
 * Raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_malloc_uncollectable(size) GE_null(GC_MALLOC_UNCOLLECTABLE(size))
#else /* No GC */
#define GE_malloc_uncollectable(size) GE_null(malloc(size))
#endif

/*
 * Allocate memory that can contain pointers to collectable objects.
 * The allocated memory is not necessarily zeroed (unless `GE_malloc_cleared' is_defined).
 * The allocated object is itself not collectable.
 * Do not raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_unprotected_malloc_uncollectable(size) GC_MALLOC_UNCOLLECTABLE(size)
#else /* No GC */
#define GE_unprotected_malloc_uncollectable(size) malloc(size)
#endif

/*
 * Allocate memory that does not contain pointers to collectable objects.
 * The allocated memory is not necessarily zeroed (unless `GE_malloc_atomic_cleared' is_defined).
 * The allocated object is itself not collectable.
 * Raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_malloc_atomic_uncollectable(size) GE_null(GC_malloc_atomic_uncollectable(size))
#else /* No GC */
#define GE_malloc_atomic_uncollectable(size) GE_null(malloc(size))
#endif

/*
 * Allocate memory that does not contain pointers to collectable objects.
 * The allocated memory is not necessarily zeroed (unless `GE_malloc_atomic_cleared' is_defined).
 * The allocated object is itself not collectable.
 * Do not raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_unprotected_malloc_atomic_uncollectable(size) GC_malloc_atomic_uncollectable(size)
#else /* No GC */
#define GE_unprotected_malloc_atomic_uncollectable(size) malloc(size)
#endif

/*
 * Allocate memory that can contain pointers to collectable objects.
 * The allocated memory is zeroed.
 * The allocated object is itself not collectable.
 * Raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_calloc_uncollectable(nelem, elsize) GE_null(GC_MALLOC_UNCOLLECTABLE((nelem) * (elsize)))
#else /* No GC */
#define GE_calloc_uncollectable(nelem, elsize) GE_null(calloc((nelem), (elsize)))
#endif

/*
 * Allocate memory that can contain pointers to collectable objects.
 * The allocated memory is zeroed.
 * The allocated object is itself not collectable.
 * Do not raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_unprotected_calloc_uncollectable(nelem, elsize) GC_MALLOC_UNCOLLECTABLE((nelem) * (elsize))
#else /* No GC */
#define GE_unprotected_calloc_uncollectable(nelem, elsize) calloc((nelem), (elsize))
#endif

/*
 * Allocate memory that does not contain pointers to collectable objects.
 * The allocated memory is zeroed.
 * The allocated object is itself not collectable.
 * Raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_calloc_atomic_uncollectable(nelem, elsize) memset(GE_null(GC_malloc_atomic_uncollectable((nelem) * (elsize))), 0, (nelem) * (elsize))
#else /* No GC */
#define GE_calloc_atomic_uncollectable(nelem, elsize) GE_null(calloc((nelem), (elsize)))
#endif

/*
 * Allocate memory that does not contain pointers to collectable objects.
 * The allocated memory is zeroed.
 * The allocated object is itself not collectable.
 * Do not raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
extern void* GE_unprotected_calloc_atomic_uncollectable(size_t nelem, size_t elsize);
#else /* No GC */
#define GE_unprotected_calloc_atomic_uncollectable(nelem, elsize) calloc((nelem), (elsize))
#endif

/*
 * Allocate more memory for the given pointer.
 * The reallocated pointer keeps the same properties (e.g. atomic or not, collectable or not).
 * The extra allocated memory is not necessarily zeroed.
 * Raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_realloc(p, size) GE_null(GC_REALLOC((p), (size)))
#else /* No GC */
#define GE_realloc(p, size) GE_null(realloc((p), (size)))
#endif

/*
 * Allocate more memory for the given pointer.
 * The reallocated pointer keeps the same properties (e.g. atomic or not, collectable or not).
 * The extra allocated memory is not necessarily zeroed.
 * Do not raise an exception when no-more-memory.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_unprotected_realloc(p, size) GC_REALLOC((p), (size))
#else /* No GC */
#define GE_unprotected_realloc(p, size) realloc((p), (size))
#endif

/*
 * Allocate more memory for the given pointer.
 * The reallocated pointer keeps the same properties (e.g. atomic or not, collectable or not).
 * The extra allocated memory is zeroed.
 * Raise an exception when no-more-memory.
 */
extern void* GE_recalloc(void* p, size_t old_nelem, size_t new_nelem, size_t elsize);

/*
 * Allocate more memory for the given pointer.
 * The reallocated pointer keeps the same properties (e.g. atomic or not, collectable or not).
 * The extra allocated memory is zeroed.
 * Do not raise an exception when no-more-memory.
 */
extern void* GE_unprotected_recalloc(void* p, size_t old_nelem, size_t new_nelem, size_t elsize);

/*
 * Explicitly deallocate an object.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_free(p) GC_FREE(p)
#else /* No GC */
#define GE_free(p) free(p)
#endif

/*
 * When defined, `GE_malloc_cleared' means that `GE_malloc' and
 * `GE_malloc_uncollectable' make sure that the allocated memory
 * is zeroed.
 */
#ifdef GE_USE_BOEHM_GC
#define GE_malloc_cleared
#else /* No GC */
#endif

/*
 * When defined, `GE_malloc_atomic_cleared' means that `GE_malloc_atomic'
 * and `GE_malloc_atomic_uncollectable' make sure that the allocated memory
 * is zeroed.
 */
#ifdef GE_USE_BOEHM_GC
/* #define GE_malloc_atomic_cleared */
#else /* No GC */
#endif

/*
 * Dispose
 */

/*
 * Register dispose routine `disp' to be called on object `obj' when it will be collected.
 */
#ifdef GE_USE_BOEHM_GC
extern void GE_boehm_dispose(void* C, void* disp); /* Call dispose routine `disp' on object `C'. */
#define GE_register_dispose(obj, disp) GC_REGISTER_FINALIZER_NO_ORDER((void*)(obj), (void (*) (void*, void*)) &GE_boehm_dispose, NULL, NULL, NULL)
#else /* No GC */
#define GE_register_dispose(obj, disp) /* do nothing */
#endif

/*
 * Access to objects, useful with GCs which move objects in memory.
 * This is not the case here, since the Boehm GC is not a moving GC.
 */

/* Access object through hector. */
#define eif_access(obj) (obj)
/* Freeze memory address. */
#define eif_freeze(obj) (obj)
/* The C side adopts an object. */
#define eif_adopt(obj) (obj)
/* The C side protects an object. */
#define eif_protect(obj) (obj)
/* The C side weans adopted object. */
#define eif_wean(obj) (obj)
/* Forget a frozen memory address. */
#define eif_unfreeze(obj)
/* Always frozen since they do not move. */
#define eif_frozen(obj) 1
/* Always frozen since they do not move. */
#define spfrozen(obj) 1

#endif

/*
	description:

		"C functions used to implement class IDENTIFIED"

	system: "Gobo Eiffel Compiler"
	copyright: "Copyright (c) 2007-2017, Eric Bezault and others"
	license: "MIT License"
	date: "$Date$"
	revision: "$Revision$"
*/

#ifndef GE_IDENTIFIED_H
#define GE_IDENTIFIED_H
#if defined(_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#ifndef GE_EIFFEL_H
#include "ge_eiffel.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Initialize data to keep track of object ids.
 */
extern void GE_init_identified(void);

/*
 * Get a new id for `object', assuming it is NOT in the stack.
 */
extern EIF_INTEGER_32 GE_object_id(EIF_OBJECT object);

/*
 * Return the object associated with `id'.
 */
extern EIF_REFERENCE GE_id_object(EIF_INTEGER_32 id);

/*
 * Remove the object associated with `id' from the stack.
 */
extern void GE_object_id_free(EIF_INTEGER_32 id);

#ifdef __cplusplus
}
#endif

#endif

#ifdef __cplusplus
extern "C" {
#endif

#define T0 EIF_ANY

/* CHARACTER */
#define EIF_CHARACTER EIF_CHARACTER_8

/* WIDE_CHARACTER */
#define EIF_WIDE_CHAR EIF_CHARACTER_32

/* INTEGER */
#define EIF_INTEGER EIF_INTEGER_32

/* NATURAL */
#define EIF_NATURAL EIF_NATURAL_32

/* REAL */
#define EIF_REAL EIF_REAL_32

/* DOUBLE */
#define EIF_DOUBLE EIF_REAL_64

/* BOOLEAN */
#define T1 EIF_BOOLEAN
extern T0* GE_boxed1(T1 a1);
typedef struct Sb1 Tb1;

/* CHARACTER_8 */
#define T2 EIF_CHARACTER_8
extern T0* GE_boxed2(T2 a1);
typedef struct Sb2 Tb2;

/* CHARACTER_32 */
#define T3 EIF_CHARACTER_32
extern T0* GE_boxed3(T3 a1);
typedef struct Sb3 Tb3;

/* INTEGER_8 */
#define T4 EIF_INTEGER_8
extern T0* GE_boxed4(T4 a1);
typedef struct Sb4 Tb4;

/* INTEGER_16 */
#define T5 EIF_INTEGER_16
extern T0* GE_boxed5(T5 a1);
typedef struct Sb5 Tb5;

/* INTEGER_32 */
#define T6 EIF_INTEGER_32
extern T0* GE_boxed6(T6 a1);
typedef struct Sb6 Tb6;

/* INTEGER_64 */
#define T7 EIF_INTEGER_64
extern T0* GE_boxed7(T7 a1);
typedef struct Sb7 Tb7;

/* NATURAL_8 */
#define T8 EIF_NATURAL_8
extern T0* GE_boxed8(T8 a1);
typedef struct Sb8 Tb8;

/* NATURAL_16 */
#define T9 EIF_NATURAL_16
extern T0* GE_boxed9(T9 a1);
typedef struct Sb9 Tb9;

/* NATURAL_32 */
#define T10 EIF_NATURAL_32
extern T0* GE_boxed10(T10 a1);
typedef struct Sb10 Tb10;

/* NATURAL_64 */
#define T11 EIF_NATURAL_64
extern T0* GE_boxed11(T11 a1);
typedef struct Sb11 Tb11;

/* REAL_32 */
#define T12 EIF_REAL_32
extern T0* GE_boxed12(T12 a1);
typedef struct Sb12 Tb12;

/* REAL_64 */
#define T13 EIF_REAL_64
extern T0* GE_boxed13(T13 a1);
typedef struct Sb13 Tb13;

/* POINTER */
#define T14 EIF_POINTER
extern T0* GE_boxed14(T14 a1);
typedef struct Sb14 Tb14;

/* [detachable] SPECIAL [CHARACTER_8] */
typedef struct S15 T15;

/* [detachable] SPECIAL [CHARACTER_32] */
typedef struct S16 T16;

/* [detachable] STRING_8 */
typedef struct S17 T17;

/* [detachable] IMMUTABLE_STRING_32 */
typedef struct S20 T20;

/* [detachable] ISE_EXCEPTION_MANAGER */
typedef struct S21 T21;

/* [detachable] XPLAIN2SQL */
typedef struct S25 T25;

/* [detachable] CELL [detachable EXCEPTION] */
typedef struct S26 T26;

/* [detachable] HASH_TABLE [[attached] INTEGER_32, [attached] INTEGER_32] */
typedef struct S27 T27;

/* [detachable] CELL [detachable TUPLE [[attached] INTEGER_32, [attached] INTEGER_32, [attached] INTEGER_32, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] INTEGER_32, [attached] BOOLEAN]] */
typedef struct S28 T28;

/* [detachable] CELL [[attached] NO_MORE_MEMORY] */
typedef struct S29 T29;

/* detachable C_STRING */
typedef struct S30 T30;

/* [detachable] TUPLE [[attached] INTEGER_32, [attached] INTEGER_32, [attached] INTEGER_32, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] INTEGER_32, [attached] BOOLEAN] */
typedef struct S31 T31;

/* detachable KL_TEXT_INPUT_FILE */
typedef struct S32 T32;

/* [detachable] KL_ARGUMENTS */
typedef struct S35 T35;

/* [detachable] KL_STANDARD_FILES */
typedef struct S36 T36;

/* [detachable] KL_STDOUT_FILE */
typedef struct S37 T37;

/* [detachable] ARRAY [[attached] STRING_8] */
typedef struct S38 T38;

/* [detachable] SPECIAL [[attached] STRING_8] */
typedef struct S39 T39;

/* [detachable] ARRAY [[attached] ANY] */
typedef struct S40 T40;

/* [detachable] SPECIAL [[attached] ANY] */
typedef struct S41 T41;

/* [detachable] KL_STDERR_FILE */
typedef struct S42 T42;

/* [detachable] SQL_GENERATOR_ANSI */
typedef struct S43 T43;

/* [detachable] SQL_GENERATOR_BASIC */
typedef struct S44 T44;

/* [detachable] SQL_GENERATOR_DB2_6 */
typedef struct S45 T45;

/* [detachable] SQL_GENERATOR_DB2_71 */
typedef struct S46 T46;

/* [detachable] SQL_GENERATOR_FOXPRO */
typedef struct S47 T47;

/* [detachable] SQL_GENERATOR_INTERBASE6 */
typedef struct S48 T48;

/* [detachable] SQL_GENERATOR_FIREBIRD_21 */
typedef struct S49 T49;

/* [detachable] SQL_GENERATOR_MSACCESS97 */
typedef struct S50 T50;

/* [detachable] SQL_GENERATOR_MSACCESS2000 */
typedef struct S51 T51;

/* [detachable] SQL_GENERATOR_MSQL */
typedef struct S52 T52;

/* [detachable] SQL_GENERATOR_MYSQL5 */
typedef struct S53 T53;

/* [detachable] SQL_GENERATOR_MYSQL4 */
typedef struct S54 T54;

/* [detachable] SQL_GENERATOR_MYSQL322 */
typedef struct S55 T55;

/* [detachable] SQL_GENERATOR_ORACLE */
typedef struct S56 T56;

/* [detachable] SQL_GENERATOR_PGSQL_95 */
typedef struct S57 T57;

/* [detachable] SQL_GENERATOR_PGSQL_82 */
typedef struct S58 T58;

/* [detachable] SQL_GENERATOR_PGSQL_81 */
typedef struct S59 T59;

/* [detachable] SQL_GENERATOR_PGSQL_73 */
typedef struct S60 T60;

/* [detachable] SQL_GENERATOR_PGSQL_72 */
typedef struct S61 T61;

/* [detachable] SQL_GENERATOR_SQLITE_3 */
typedef struct S62 T62;

/* [detachable] SQL_GENERATOR_TSQL2016 */
typedef struct S63 T63;

/* [detachable] SQL_GENERATOR_TSQL2000 */
typedef struct S64 T64;

/* [detachable] SQL_GENERATOR_TSQL70 */
typedef struct S65 T65;

/* [detachable] SQL_GENERATOR_TSQL65 */
typedef struct S66 T66;

/* [detachable] XPLAIN_PARSER */
typedef struct S67 T67;

/* [detachable] XML_GENERATOR */
typedef struct S69 T69;

/* [detachable] XPLAIN_STATEMENTS */
typedef struct S71 T71;

/* [detachable] VOID_TARGET */
typedef struct S74 T74;

/* [detachable] TYPE [[attached] VOID_TARGET] */
#define T75 EIF_TYPE_OBJ

/* [detachable] ROUTINE_FAILURE */
typedef struct S76 T76;

/* [detachable] TYPE [[attached] ROUTINE_FAILURE] */
#define T77 EIF_TYPE_OBJ

/* [detachable] OLD_VIOLATION */
typedef struct S78 T78;

/* [detachable] TYPE [[attached] OLD_VIOLATION] */
#define T79 EIF_TYPE_OBJ

/* [detachable] NO_MORE_MEMORY */
typedef struct S80 T80;

/* [detachable] INVARIANT_VIOLATION */
typedef struct S81 T81;

/* [detachable] OPERATING_SYSTEM_SIGNAL_FAILURE */
typedef struct S82 T82;

/* [detachable] IO_FAILURE */
typedef struct S83 T83;

/* [detachable] OPERATING_SYSTEM_FAILURE */
typedef struct S84 T84;

/* [detachable] COM_FAILURE */
typedef struct S85 T85;

/* [detachable] EIFFEL_RUNTIME_PANIC */
typedef struct S86 T86;

/* [detachable] PRECONDITION_VIOLATION */
typedef struct S88 T88;

/* [detachable] POSTCONDITION_VIOLATION */
typedef struct S89 T89;

/* [detachable] FLOATING_POINT_FAILURE */
typedef struct S90 T90;

/* [detachable] CHECK_VIOLATION */
typedef struct S91 T91;

/* [detachable] BAD_INSPECT_VALUE */
typedef struct S92 T92;

/* [detachable] VARIANT_VIOLATION */
typedef struct S93 T93;

/* [detachable] LOOP_INVARIANT_VIOLATION */
typedef struct S94 T94;

/* [detachable] RESCUE_FAILURE */
typedef struct S95 T95;

/* [detachable] RESUMPTION_FAILURE */
typedef struct S96 T96;

/* [detachable] CREATE_ON_DEFERRED */
typedef struct S97 T97;

/* [detachable] EXTERNAL_FAILURE */
typedef struct S98 T98;

/* [detachable] VOID_ASSIGNED_TO_EXPANDED */
typedef struct S99 T99;

/* [detachable] EXCEPTION_IN_SIGNAL_HANDLER_FAILURE */
typedef struct S100 T100;

/* [detachable] MISMATCH_FAILURE */
typedef struct S101 T101;

/* [detachable] DEVELOPER_EXCEPTION */
typedef struct S102 T102;

/* [detachable] ADDRESS_APPLIED_TO_MELTED_FEATURE */
typedef struct S103 T103;

/* [detachable] SERIALIZATION_FAILURE */
typedef struct S104 T104;

/* [detachable] ST_SCIENTIFIC_FORMATTER */
typedef struct S105 T105;

/* [detachable] PRIMES */
typedef struct S106 T106;

/* [detachable] SPECIAL [[attached] INTEGER_32] */
typedef struct S107 T107;

/* [detachable] SPECIAL [[attached] BOOLEAN] */
typedef struct S108 T108;

/* [detachable] KL_STRING_ROUTINES */
typedef struct S110 T110;

/* [detachable] MANAGED_POINTER */
typedef struct S113 T113;

/* [detachable] FILE_INFO */
typedef struct S114 T114;

/* [detachable] ARGUMENTS_32 */
typedef struct S115 T115;

/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8] */
typedef struct S116 T116;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] INTEGER_32] */
typedef struct S118 T118;

/* [detachable] XPLAIN_SCANNER_STATE */
typedef struct S119 T119;

/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_SCANNER_STATE] */
typedef struct S121 T121;

/* [detachable] XPLAIN_REQUIRED */
typedef struct S123 T123;

/* [detachable] SPECIAL [detachable ANY] */
typedef struct S126 T126;

/* [detachable] KL_SPECIAL_ROUTINES [detachable ANY] */
typedef struct S127 T127;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] STRING_8] */
typedef struct S128 T128;

/* detachable XPLAIN_CONSTANT_ASSIGNMENT_STATEMENT */
typedef struct S130 T130;

/* detachable XPLAIN_BASE */
typedef struct S131 T131;

/* detachable XPLAIN_TYPE */
typedef struct S132 T132;

/* detachable XPLAIN_CONSTANT_STATEMENT */
typedef struct S133 T133;

/* [detachable] XPLAIN_A_NODE */
typedef struct S134 T134;

/* [detachable] XPLAIN_I_NODE */
typedef struct S135 T135;

/* [detachable] XPLAIN_LOGICAL_EXPRESSION */
typedef struct S136 T136;

/* [detachable] XPLAIN_LOGICAL_VALUE_EXPRESSION */
typedef struct S138 T138;

/* [detachable] XPLAIN_ATTRIBUTE_NAME_NODE */
typedef struct S140 T140;

/* [detachable] XPLAIN_ATTRIBUTE_NAME */
typedef struct S141 T141;

/* detachable XPLAIN_ATTRIBUTE_NODE */
typedef struct S144 T144;

/* [detachable] XPLAIN_ATTRIBUTE */
typedef struct S145 T145;

/* detachable XPLAIN_ASSERTION */
typedef struct S146 T146;

/* detachable XPLAIN_ASSIGNMENT_NODE */
typedef struct S147 T147;

/* [detachable] XPLAIN_ASSIGNMENT */
typedef struct S148 T148;

/* detachable XPLAIN_CASCADE_STATEMENT */
typedef struct S149 T149;

/* detachable XPLAIN_CASCADE_FUNCTION_EXPRESSION */
typedef struct S150 T150;

/* detachable XPLAIN_DELETE_STATEMENT */
typedef struct S151 T151;

/* detachable XPLAIN_SUBJECT */
typedef struct S152 T152;

/* detachable XPLAIN_UPDATE_STATEMENT */
typedef struct S153 T153;

/* [detachable] XPLAIN_VALUE_STATEMENT */
typedef struct S154 T154;

/* detachable XPLAIN_VALUE_SELECTION_STATEMENT */
typedef struct S155 T155;

/* detachable XPLAIN_SELECTION_FUNCTION */
typedef struct S157 T157;

/* detachable XPLAIN_GET_STATEMENT */
typedef struct S158 T158;

/* detachable XPLAIN_EXTEND_STATEMENT */
typedef struct S159 T159;

/* detachable XPLAIN_EXTENSION */
typedef struct S160 T160;

/* [detachable] XPLAIN_EXTEND_ATTRIBUTE */
typedef struct S161 T161;

/* detachable XPLAIN_EXTENSION_EXPRESSION */
typedef struct S162 T162;

/* detachable XPLAIN_SELECTION_LIST */
typedef struct S164 T164;

/* detachable XPLAIN_SORT_NODE */
typedef struct S165 T165;

/* detachable XPLAIN_EXPRESSION_NODE */
typedef struct S167 T167;

/* detachable XPLAIN_PROCEDURE_STATEMENT */
typedef struct S168 T168;

/* detachable XPLAIN_PARAMETER_NODE */
typedef struct S169 T169;

/* [detachable] XPLAIN_PARAMETER */
typedef struct S170 T170;

/* [detachable] XPLAIN_SQL_STATEMENT */
typedef struct S171 T171;

/* detachable XPLAIN_STATEMENT_NODE */
typedef struct S172 T172;

/* detachable XPLAIN_IF_STATEMENT */
typedef struct S173 T173;

/* [detachable] SPECIAL [detachable XPLAIN_STATEMENT] */
typedef struct S174 T174;

/* [detachable] SPECIAL [[attached] XPLAIN_SQL_STATEMENT] */
typedef struct S175 T175;

/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_STATEMENT] */
typedef struct S176 T176;

/* [detachable] SPECIAL [detachable XPLAIN_PROCEDURE_STATEMENT] */
typedef struct S177 T177;

/* [detachable] SPECIAL [detachable XPLAIN_CONSTANT_ASSIGNMENT_STATEMENT] */
typedef struct S178 T178;

/* [detachable] SPECIAL [detachable XPLAIN_TYPE] */
typedef struct S179 T179;

/* [detachable] XPLAIN_TYPE_STATEMENT */
typedef struct S180 T180;

/* [detachable] XPLAIN_UNIVERSE */
typedef struct S181 T181;

/* [detachable] SPECIAL [detachable XPLAIN_BASE] */
typedef struct S182 T182;

/* [detachable] XPLAIN_BASE_STATEMENT */
typedef struct S183 T183;

/* [detachable] SPECIAL [detachable XPLAIN_CONSTANT_STATEMENT] */
typedef struct S184 T184;

/* detachable XPLAIN_VARIABLE */
typedef struct S185 T185;

/* [detachable] SPECIAL [[attached] XPLAIN_EXPRESSION] */
typedef struct S186 T186;

/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_CONSTANT_ASSIGNMENT_STATEMENT] */
typedef struct S187 T187;

/* [detachable] SPECIAL [[attached] XPLAIN_REPRESENTATION] */
typedef struct S188 T188;

/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_BASE] */
typedef struct S189 T189;

/* [detachable] SPECIAL [[attached] XPLAIN_DOMAIN_RESTRICTION] */
typedef struct S190 T190;

/* [detachable] SPECIAL [detachable XPLAIN_ATTRIBUTE_NODE] */
typedef struct S191 T191;

/* [detachable] SPECIAL [[attached] XPLAIN_PK_REPRESENTATION] */
typedef struct S192 T192;

/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_TYPE] */
typedef struct S193 T193;

/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_CONSTANT_STATEMENT] */
typedef struct S194 T194;

/* [detachable] SPECIAL [[attached] XPLAIN_ATTRIBUTE_NAME] */
typedef struct S196 T196;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_DOMAIN_RESTRICTION] */
typedef struct S197 T197;

/* [detachable] XPLAIN_A_PATTERN */
typedef struct S198 T198;

/* [detachable] SPECIAL [[attached] XPLAIN_A_NODE] */
typedef struct S199 T199;

/* [detachable] XPLAIN_A_ENUMERATION */
typedef struct S200 T200;

/* [detachable] SPECIAL [[attached] XPLAIN_I_NODE] */
typedef struct S201 T201;

/* [detachable] XPLAIN_I_ENUMERATION */
typedef struct S202 T202;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_A_NODE] */
typedef struct S203 T203;

/* [detachable] XPLAIN_A_REPRESENTATION */
typedef struct S204 T204;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_I_NODE] */
typedef struct S206 T206;

/* [detachable] XPLAIN_TRAJECTORY */
typedef struct S207 T207;

/* [detachable] SPECIAL [[attached] XPLAIN_LOGICAL_EXPRESSION] */
typedef struct S208 T208;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_LOGICAL_EXPRESSION] */
typedef struct S209 T209;

/* [detachable] XPLAIN_LOGICAL_INFIX_EXPRESSION */
typedef struct S210 T210;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_EXPRESSION] */
typedef struct S211 T211;

/* [detachable] SPECIAL [[attached] XPLAIN_ATTRIBUTE_NAME_NODE] */
typedef struct S212 T212;

/* [detachable] XPLAIN_NOTNOT_EXPRESSION */
typedef struct S213 T213;

/* [detachable] XPLAIN_NOT_EXPRESSION */
typedef struct S214 T214;

/* [detachable] XPLAIN_PARAMETER_EXPRESSION */
typedef struct S215 T215;

/* [detachable] XPLAIN_EQ_NULL_EXPRESSION */
typedef struct S216 T216;

/* [detachable] XPLAIN_NE_NULL_EXPRESSION */
typedef struct S217 T217;

/* [detachable] XPLAIN_INFIX_EXPRESSION */
typedef struct S218 T218;

/* [detachable] XPLAIN_PARENTHESIS_EXPRESSION */
typedef struct S219 T219;

/* [detachable] SPECIAL [[attached] XPLAIN_LOGICAL_VALUE_EXPRESSION] */
typedef struct S220 T220;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_LOGICAL_VALUE_EXPRESSION] */
typedef struct S221 T221;

/* [detachable] XPLAIN_SYSTEMDATE_EXPRESSION */
typedef struct S222 T222;

/* [detachable] SPECIAL [[attached] XPLAIN_SYSTEM_EXPRESSION] */
typedef struct S223 T223;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_SYSTEM_EXPRESSION] */
typedef struct S224 T224;

/* [detachable] XPLAIN_LOGINNAME_EXPRESSION */
typedef struct S225 T225;

/* [detachable] XPLAIN_IF_EXPRESSION */
typedef struct S226 T226;

/* [detachable] XPLAIN_PREFIX_EXPRESSION */
typedef struct S227 T227;

/* [detachable] SPECIAL [detachable XPLAIN_EXPRESSION_NODE] */
typedef struct S228 T228;

/* [detachable] XPLAIN_COMBINE_FUNCTION */
typedef struct S229 T229;

/* [detachable] XPLAIN_DATEF_FUNCTION */
typedef struct S230 T230;

/* [detachable] XPLAIN_INTEGER_FUNCTION */
typedef struct S231 T231;

/* [detachable] XPLAIN_REAL_FUNCTION */
typedef struct S232 T232;

/* [detachable] XPLAIN_STRING_FUNCTION */
typedef struct S233 T233;

/* [detachable] XPLAIN_STRING_EXPRESSION */
typedef struct S234 T234;

/* [detachable] XPLAIN_NEWDATE_FUNCTION */
typedef struct S235 T235;

/* [detachable] XPLAIN_ID_FUNCTION */
typedef struct S236 T236;

/* [detachable] XPLAIN_NUMBER_EXPRESSION */
typedef struct S237 T237;

/* [detachable] XPLAIN_UNMANAGED_PARAMETER_EXPRESSION */
typedef struct S238 T238;

/* [detachable] XPLAIN_USER_FUNCTION */
typedef struct S239 T239;

/* [detachable] XPLAIN_SQL_EXPRESSION */
typedef struct S240 T240;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ATTRIBUTE_NAME_NODE] */
typedef struct S241 T241;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ATTRIBUTE_NAME] */
typedef struct S242 T242;

/* [detachable] SPECIAL [detachable XPLAIN_EXPRESSION] */
typedef struct S243 T243;

/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXPRESSION] */
typedef struct S244 T244;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_REPRESENTATION] */
typedef struct S245 T245;

/* [detachable] XPLAIN_B_REPRESENTATION */
typedef struct S246 T246;

/* [detachable] XPLAIN_C_REPRESENTATION */
typedef struct S247 T247;

/* [detachable] XPLAIN_D_REPRESENTATION */
typedef struct S248 T248;

/* [detachable] XPLAIN_I_REPRESENTATION */
typedef struct S249 T249;

/* [detachable] XPLAIN_M_REPRESENTATION */
typedef struct S250 T250;

/* [detachable] XPLAIN_P_REPRESENTATION */
typedef struct S251 T251;

/* [detachable] XPLAIN_R_REPRESENTATION */
typedef struct S252 T252;

/* [detachable] XPLAIN_T_REPRESENTATION */
typedef struct S253 T253;

/* [detachable] XPLAIN_PK_A_REPRESENTATION */
typedef struct S254 T254;

/* detachable XPLAIN_A_REFERENCES */
typedef struct S255 T255;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_PK_REPRESENTATION] */
typedef struct S256 T256;

/* [detachable] XPLAIN_PK_I_REPRESENTATION */
typedef struct S257 T257;

/* detachable XPLAIN_I_REFERENCES */
typedef struct S258 T258;

/* [detachable] SPECIAL [[attached] XPLAIN_ATTRIBUTE] */
typedef struct S259 T259;

/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_ATTRIBUTE_NODE] */
typedef struct S260 T260;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ATTRIBUTE] */
typedef struct S262 T262;

/* [detachable] XPLAIN_VARIABLE_EXPRESSION */
typedef struct S263 T263;

/* detachable XPLAIN_VALUE */
typedef struct S264 T264;

/* [detachable] XPLAIN_VALUE_EXPRESSION */
typedef struct S265 T265;

/* [detachable] SPECIAL [detachable XPLAIN_DOMAIN_RESTRICTION] */
typedef struct S266 T266;

/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_DOMAIN_RESTRICTION] */
typedef struct S267 T267;

/* [detachable] SPECIAL [detachable XPLAIN_EXTENSION_EXPRESSION] */
typedef struct S268 T268;

/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_ASSERTION] */
typedef struct S269 T269;

/* [detachable] SPECIAL [detachable XPLAIN_ASSERTION] */
typedef struct S270 T270;

/* [detachable] SPECIAL [detachable XPLAIN_ATTRIBUTE_NAME_NODE] */
typedef struct S271 T271;

/* detachable XPLAIN_INDEX */
typedef struct S272 T272;

/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_ATTRIBUTE_NAME_NODE] */
typedef struct S273 T273;

/* [detachable] SPECIAL [detachable XPLAIN_UPDATE_STATEMENT] */
typedef struct S274 T274;

/* [detachable] SPECIAL [detachable XPLAIN_DELETE_STATEMENT] */
typedef struct S275 T275;

/* [detachable] SPECIAL [detachable XPLAIN_CASCADE_STATEMENT] */
typedef struct S276 T276;

/* [detachable] SPECIAL [detachable XPLAIN_SUBJECT] */
typedef struct S277 T277;

/* [detachable] SPECIAL [detachable XPLAIN_ASSIGNMENT_NODE] */
typedef struct S278 T278;

/* [detachable] XPLAIN_INSERT_STATEMENT */
typedef struct S279 T279;

/* [detachable] SPECIAL [detachable XPLAIN_SELECTION_LIST] */
typedef struct S280 T280;

/* [detachable] XPLAIN_GET_INSERT_STATEMENT */
typedef struct S281 T281;

/* [detachable] SPECIAL [[attached] XPLAIN_ASSIGNMENT] */
typedef struct S283 T283;

/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_ASSIGNMENT_NODE] */
typedef struct S284 T284;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ASSIGNMENT] */
typedef struct S285 T285;

/* [detachable] SPECIAL [detachable XPLAIN_CASCADE_FUNCTION_EXPRESSION] */
typedef struct S286 T286;

/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_CASCADE_STATEMENT] */
typedef struct S287 T287;

/* [detachable] SPECIAL [detachable XPLAIN_SELECTION_FUNCTION] */
typedef struct S288 T288;

/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_CASCADE_FUNCTION_EXPRESSION] */
typedef struct S289 T289;

/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_DELETE_STATEMENT] */
typedef struct S290 T290;

/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SUBJECT] */
typedef struct S291 T291;

/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_UPDATE_STATEMENT] */
typedef struct S292 T292;

/* [detachable] SPECIAL [[attached] XPLAIN_VALUE_STATEMENT] */
typedef struct S293 T293;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_VALUE_STATEMENT] */
typedef struct S294 T294;

/* [detachable] XPLAIN_SELECTION_EXPRESSION */
typedef struct S295 T295;

/* [detachable] XPLAIN_LAST_AUTO_PK_EXPRESSION */
typedef struct S297 T297;

/* [detachable] SPECIAL [detachable XPLAIN_VALUE_SELECTION_STATEMENT] */
typedef struct S298 T298;

/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_VALUE_SELECTION_STATEMENT] */
typedef struct S299 T299;

/* [detachable] SPECIAL [detachable XPLAIN_NON_ATTRIBUTE_EXPRESSION] */
typedef struct S300 T300;

/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_NON_ATTRIBUTE_EXPRESSION] */
typedef struct S301 T301;

/* [detachable] SPECIAL [[attached] XPLAIN_FUNCTION] */
typedef struct S302 T302;

/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SELECTION_FUNCTION] */
typedef struct S303 T303;

/* [detachable] SPECIAL [detachable XPLAIN_SELECTION] */
typedef struct S304 T304;

/* [detachable] SPECIAL [detachable XPLAIN_GET_STATEMENT] */
typedef struct S305 T305;

/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_GET_STATEMENT] */
typedef struct S306 T306;

/* [detachable] SPECIAL [detachable XPLAIN_EXTENSION] */
typedef struct S307 T307;

/* [detachable] SPECIAL [detachable XPLAIN_EXTEND_STATEMENT] */
typedef struct S308 T308;

/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXTEND_STATEMENT] */
typedef struct S309 T309;

/* [detachable] SPECIAL [[attached] XPLAIN_EXTEND_ATTRIBUTE] */
typedef struct S310 T310;

/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXTENSION] */
typedef struct S311 T311;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_EXTEND_ATTRIBUTE] */
typedef struct S312 T312;

/* [detachable] XPLAIN_EXTENSION_EXPRESSION_EXPRESSION */
typedef struct S313 T313;

/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXTENSION_EXPRESSION] */
typedef struct S314 T314;

/* [detachable] XPLAIN_EXTENSION_FUNCTION_EXPRESSION */
typedef struct S315 T315;

/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SELECTION] */
typedef struct S316 T316;

/* [detachable] SPECIAL [detachable XPLAIN_SORT_NODE] */
typedef struct S317 T317;

/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SELECTION_LIST] */
typedef struct S318 T318;

/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SORT_NODE] */
typedef struct S319 T319;

/* [detachable] XPLAIN_MAX_FUNCTION */
typedef struct S320 T320;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_FUNCTION] */
typedef struct S321 T321;

/* [detachable] XPLAIN_MIN_FUNCTION */
typedef struct S322 T322;

/* [detachable] XPLAIN_TOTAL_FUNCTION */
typedef struct S323 T323;

/* [detachable] XPLAIN_COUNT_FUNCTION */
typedef struct S324 T324;

/* [detachable] XPLAIN_ANY_FUNCTION */
typedef struct S325 T325;

/* [detachable] XPLAIN_NIL_FUNCTION */
typedef struct S326 T326;

/* [detachable] XPLAIN_SOME_FUNCTION */
typedef struct S327 T327;

/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXPRESSION_NODE] */
typedef struct S328 T328;

/* [detachable] SPECIAL [detachable STRING_8] */
typedef struct S329 T329;

/* [detachable] KL_SPECIAL_ROUTINES [detachable STRING_8] */
typedef struct S330 T330;

/* [detachable] SPECIAL [detachable XPLAIN_PARAMETER_NODE] */
typedef struct S331 T331;

/* [detachable] SPECIAL [detachable XPLAIN_STATEMENT_NODE] */
typedef struct S332 T332;

/* [detachable] XPLAIN_PROCEDURE */
typedef struct S333 T333;

/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_PROCEDURE_STATEMENT] */
typedef struct S334 T334;

/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_PARAMETER_NODE] */
typedef struct S335 T335;

/* [detachable] SPECIAL [[attached] XPLAIN_PARAMETER_NODE] */
typedef struct S336 T336;

/* [detachable] SPECIAL [[attached] XPLAIN_PARAMETER] */
typedef struct S337 T337;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_PARAMETER_NODE] */
typedef struct S338 T338;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_PARAMETER] */
typedef struct S339 T339;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_SQL_STATEMENT] */
typedef struct S340 T340;

/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_STATEMENT_NODE] */
typedef struct S341 T341;

/* [detachable] SPECIAL [detachable XPLAIN_IF_STATEMENT] */
typedef struct S342 T342;

/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_IF_STATEMENT] */
typedef struct S343 T343;

/* [detachable] YY_FILE_BUFFER */
typedef struct S344 T344;

/* [detachable] KL_STDIN_FILE */
typedef struct S345 T345;

/* [detachable] XML_WRITER */
typedef struct S347 T347;

/* [detachable] KL_TEXT_OUTPUT_FILE */
typedef struct S348 T348;

/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_STATEMENT] */
typedef struct S349 T349;

/* [detachable] STD_FILES */
typedef struct S351 T351;

/* [detachable] JOIN_LIST */
typedef struct S352 T352;

/* detachable JOIN_NODE */
typedef struct S353 T353;

/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_PARAMETER] */
typedef struct S358 T358;

/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_PARAMETER] */
typedef struct S359 T359;

/* [detachable] DS_BILINKED_LIST [[attached] XPLAIN_STATEMENT] */
typedef struct S360 T360;

/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE] */
typedef struct S361 T361;

/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_ATTRIBUTE] */
typedef struct S362 T362;

/* [detachable] XPLAIN_B_RESTRICTION */
typedef struct S365 T365;

/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_SCANNER_STATE] */
typedef struct S367 T367;

/* [detachable] XPLAIN_LIKE_EXPRESSION */
typedef struct S368 T368;

/* [detachable] XPLAIN_NOT_LIKE_EXPRESSION */
typedef struct S369 T369;

/* [detachable] ARRAY [[attached] INTEGER_32] */
typedef struct S370 T370;

/* [detachable] KL_ARRAY_ROUTINES [[attached] INTEGER_32] */
typedef struct S371 T371;

/* UTF_CONVERTER */
typedef struct S372 T372;
extern T0* GE_boxed372(T372 a1);
typedef struct Sb372 Tb372;

/* [detachable] CELL [[attached] INTEGER_32] */
typedef struct S373 T373;

/* [detachable] KL_NULL_TEXT_OUTPUT_STREAM */
typedef struct S374 T374;

/* [detachable] KL_STRING_OUTPUT_STREAM */
typedef struct S376 T376;

/* [detachable] DS_CELL [[attached] INTEGER_32] */
typedef struct S378 T378;

/* [detachable] ST_BOOLEAN_FORMATTER */
typedef struct S380 T380;

/* [detachable] ST_CHARACTER_FORMATTER */
typedef struct S381 T381;

/* [detachable] ST_FLOATING_POINT_FORMATTER */
typedef struct S382 T382;

/* [detachable] ST_FIXED_POINT_FORMATTER */
typedef struct S383 T383;

/* [detachable] ST_FIXED_OR_FLOATING_POINT_FORMATTER */
typedef struct S384 T384;

/* [detachable] ST_DECIMAL_FORMATTER */
typedef struct S385 T385;

/* [detachable] ST_UNSIGNED_OCTAL_FORMATTER */
typedef struct S386 T386;

/* [detachable] ST_UNSIGNED_HEXADECIMAL_FORMATTER */
typedef struct S387 T387;

/* [detachable] ST_UNSIGNED_DECIMAL_FORMATTER */
typedef struct S388 T388;

/* [detachable] ST_POINTER_FORMATTER */
typedef struct S389 T389;

/* [detachable] ST_STRING_FORMATTER */
typedef struct S390 T390;

/* [detachable] KL_ANY_ROUTINES */
typedef struct S391 T391;

/* [detachable] UC_STRING */
typedef struct S392 T392;

/* [detachable] SPECIAL [[attached] NATURAL_8] */
typedef struct S394 T394;

/* TYPED_POINTER [[attached] SPECIAL [[attached] NATURAL_8]] */
typedef struct S395 T395;
extern T0* GE_boxed395(T395 a1);
typedef struct Sb395 Tb395;

/* [detachable] ARRAY [[attached] IMMUTABLE_STRING_32] */
typedef struct S396 T396;

/* [detachable] SPECIAL [[attached] IMMUTABLE_STRING_32] */
typedef struct S397 T397;

/* detachable KL_EQUALITY_TESTER [[attached] XPLAIN_VALUE] */
typedef struct S398 T398;

/* detachable KL_EQUALITY_TESTER [[attached] STRING_8] */
typedef struct S399 T399;

/* [detachable] TYPE [[attached] XPLAIN_VALUE] */
#define T400 EIF_TYPE_OBJ

/* [detachable] TYPE [[attached] STRING_8] */
#define T401 EIF_TYPE_OBJ

/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_VALUE] */
typedef struct S402 T402;

/* [detachable] SPECIAL [[attached] XPLAIN_VALUE] */
typedef struct S403 T403;

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] XPLAIN_VALUE, [attached] STRING_8] */
typedef struct S404 T404;

/* detachable DS_LINKABLE [[attached] XPLAIN_SCANNER_STATE] */
typedef struct S406 T406;

/* [detachable] XPLAIN_ALL_ATTRIBUTES_CURSOR [[attached] XPLAIN_ATTRIBUTE] */
typedef struct S407 T407;

/* [detachable] XPLAIN_DATA_ATTRIBUTES_CURSOR */
typedef struct S408 T408;

/* [detachable] XPLAIN_ATTRIBUTE_EXPRESSION */
typedef struct S410 T410;

/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8] */
typedef struct S412 T412;

/* [detachable] DS_BILINKED_LIST_CURSOR [[attached] XPLAIN_STATEMENT] */
typedef struct S414 T414;

/* [detachable] DS_LINKED_STACK [[attached] STRING_8] */
typedef struct S415 T415;

/* [detachable] ARRAY [detachable STRING_8] */
typedef struct S416 T416;

/* TYPED_POINTER [[attached] ANY] */
typedef struct S417 T417;
extern T0* GE_boxed417(T417 a1);
typedef struct Sb417 Tb417;

/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_STATEMENT] */
typedef struct S418 T418;

/* detachable DS_LINKABLE [[attached] XPLAIN_STATEMENT] */
typedef struct S419 T419;

/* [detachable] STRING_TO_INTEGER_CONVERTOR */
typedef struct S420 T420;

/* [detachable] JOIN */
typedef struct S423 T423;

/* [detachable] DS_HASH_SET [[attached] STRING_8] */
typedef struct S424 T424;

/* [detachable] XPLAIN_F_REPRESENTATION */
typedef struct S425 T425;

/* [detachable] EXCEPTIONS */
typedef struct S427 T427;

/* TYPED_POINTER [[attached] NATURAL_8] */
typedef struct S428 T428;
extern T0* GE_boxed428(T428 a1);
typedef struct Sb428 Tb428;

/* [detachable] XPLAIN_INIT_ATTRIBUTES_CURSOR [[attached] XPLAIN_ATTRIBUTE] */
typedef struct S430 T430;

/* [detachable] KL_INTEGER_ROUTINES */
typedef struct S432 T432;

/* [detachable] ARRAY [[attached] DS_CELL [[attached] INTEGER_32]] */
typedef struct S433 T433;

/* [detachable] SPECIAL [[attached] DS_CELL [[attached] INTEGER_32]] */
typedef struct S434 T434;

/* [detachable] DS_CELL [[attached] REAL_64] */
typedef struct S435 T435;

/* [detachable] KL_CHARACTER_BUFFER */
typedef struct S436 T436;

/* [detachable] CONSOLE */
typedef struct S437 T437;

/* [detachable] JOIN_TREE_NODE */
typedef struct S438 T438;

/* [detachable] DS_LINKED_LIST [[attached] JOIN_TREE_NODE] */
typedef struct S439 T439;

/* detachable DS_LINKABLE [[attached] XPLAIN_PARAMETER] */
typedef struct S440 T440;

/* detachable DS_BILINKABLE [[attached] XPLAIN_STATEMENT] */
typedef struct S441 T441;

/* detachable DS_LINKABLE [[attached] XPLAIN_ATTRIBUTE] */
typedef struct S442 T442;

/* [detachable] DS_ARRAYED_LIST [[attached] INTEGER_32] */
typedef struct S445 T445;

/* [detachable] KL_DOUBLE_ROUTINES */
typedef struct S446 T446;

/* [detachable] KL_PLATFORM */
typedef struct S447 T447;

/* [detachable] SPECIAL [[attached] XPLAIN_ABSTRACT_OBJECT] */
typedef struct S450 T450;

/* detachable KL_EQUALITY_TESTER [[attached] XPLAIN_ABSTRACT_OBJECT] */
typedef struct S451 T451;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ABSTRACT_OBJECT] */
typedef struct S452 T452;

/* [detachable] TYPE [[attached] XPLAIN_ABSTRACT_OBJECT] */
#define T453 EIF_TYPE_OBJ

/* [detachable] DS_HASH_TABLE_CURSOR [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8] */
typedef struct S454 T454;

/* detachable DS_LINKABLE [[attached] STRING_8] */
typedef struct S456 T456;

/* TYPED_POINTER [[attached] NATURAL_16] */
typedef struct S458 T458;
extern T0* GE_boxed458(T458 a1);
typedef struct Sb458 Tb458;

/* [detachable] HEXADECIMAL_STRING_TO_INTEGER_CONVERTER */
typedef struct S459 T459;

/* [detachable] INTEGER_OVERFLOW_CHECKER */
typedef struct S460 T460;

/* [detachable] DS_HASH_SET_CURSOR [[attached] STRING_8] */
typedef struct S461 T461;

/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE] */
typedef struct S462 T462;

/* detachable DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE_NAME_NODE] */
typedef struct S463 T463;

/* detachable DS_LINKABLE [[attached] JOIN_TREE_NODE] */
typedef struct S464 T464;

/* [detachable] DS_LINKED_LIST_CURSOR [[attached] JOIN_TREE_NODE] */
typedef struct S465 T465;

/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] INTEGER_32] */
typedef struct S466 T466;

/* [detachable] XPLAIN_VIRTUAL_ATTRIBUTE */
typedef struct S468 T468;

/* [detachable] XPLAIN_EXTENSION_ATTRIBUTE */
typedef struct S469 T469;

/* [detachable] XPLAIN_ASSERTION_EXPRESSION */
typedef struct S471 T471;

/* [detachable] UC_UTF8_ROUTINES */
typedef struct S473 T473;

/* [detachable] SPECIAL [[attached] NATURAL_64] */
typedef struct S475 T475;

/* detachable KL_EQUALITY_TESTER [[attached] XPLAIN_TYPE] */
typedef struct S476 T476;

/* detachable DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] XPLAIN_TYPE] */
typedef struct S478 T478;

/* [detachable] SPECIAL [[attached] XPLAIN_TYPE] */
typedef struct S479 T479;

/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_TYPE] */
typedef struct S480 T480;

/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_ATTRIBUTE_NAME_NODE] */
typedef struct S481 T481;

/* detachable DS_LINKABLE [[attached] XPLAIN_ATTRIBUTE_NAME_NODE] */
typedef struct S482 T482;

/* [detachable] TYPE [[detachable] DEVELOPER_EXCEPTION] */
#define T484 EIF_TYPE_OBJ

/* Struct for boxed version of type BOOLEAN */
struct Sb1 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 z1; /* item */
};

/* Struct for boxed version of type CHARACTER_8 */
struct Sb2 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T2 z1; /* item */
};

/* Struct for boxed version of type CHARACTER_32 */
struct Sb3 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T3 z1; /* item */
};

/* Struct for boxed version of type INTEGER_8 */
struct Sb4 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T4 z1; /* item */
};

/* Struct for boxed version of type INTEGER_16 */
struct Sb5 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T5 z1; /* item */
};

/* Struct for boxed version of type INTEGER_32 */
struct Sb6 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 z1; /* item */
};

/* Struct for boxed version of type INTEGER_64 */
struct Sb7 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T7 z1; /* item */
};

/* Struct for boxed version of type NATURAL_8 */
struct Sb8 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T8 z1; /* item */
};

/* Struct for boxed version of type NATURAL_16 */
struct Sb9 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T9 z1; /* item */
};

/* Struct for boxed version of type NATURAL_32 */
struct Sb10 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T10 z1; /* item */
};

/* Struct for boxed version of type NATURAL_64 */
struct Sb11 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T11 z1; /* item */
};

/* Struct for boxed version of type REAL_32 */
struct Sb12 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T12 z1; /* item */
};

/* Struct for boxed version of type REAL_64 */
struct Sb13 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T13 z1; /* item */
};

/* Struct for boxed version of type POINTER */
struct Sb14 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 z1; /* item */
};

/* Struct for type UTF_CONVERTER */
struct S372 {
	char dummy;
};

/* Struct for boxed version of type UTF_CONVERTER */
struct Sb372 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T372 z1; /* item */
};

/* Struct for type TYPED_POINTER [[attached] NATURAL_16] */
struct S458 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* to_pointer */
};

/* Struct for boxed version of type TYPED_POINTER [[attached] NATURAL_16] */
struct Sb458 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T458 z1; /* item */
};

/* Struct for type TYPED_POINTER [[attached] NATURAL_8] */
struct S428 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* to_pointer */
};

/* Struct for boxed version of type TYPED_POINTER [[attached] NATURAL_8] */
struct Sb428 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T428 z1; /* item */
};

/* Struct for type TYPED_POINTER [[attached] ANY] */
struct S417 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* to_pointer */
};

/* Struct for boxed version of type TYPED_POINTER [[attached] ANY] */
struct Sb417 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T417 z1; /* item */
};

/* Struct for type TYPED_POINTER [[attached] SPECIAL [[attached] NATURAL_8]] */
struct S395 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* to_pointer */
};

/* Struct for boxed version of type TYPED_POINTER [[attached] SPECIAL [[attached] NATURAL_8]] */
struct Sb395 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T395 z1; /* item */
};

/* Struct for type [detachable] SPECIAL [CHARACTER_8] */
struct S15 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T2 z2[1]; /* item */
};

/* Struct for type [detachable] SPECIAL [CHARACTER_32] */
struct S16 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T3 z2[1]; /* item */
};

/* Struct for type [detachable] STRING_8 */
struct S17 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* count */
	T6 a3; /* internal_hash_code */
	T6 a4; /* internal_case_insensitive_hash_code */
};

/* Struct for type [detachable] IMMUTABLE_STRING_32 */
struct S20 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* count */
	T6 a3; /* internal_hash_code */
	T6 a4; /* area_lower */
};

/* Struct for type [detachable] ISE_EXCEPTION_MANAGER */
struct S21 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] XPLAIN2SQL */
struct S25 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parser */
};

/* Struct for type [detachable] CELL [detachable EXCEPTION] */
struct S26 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
};

/* Struct for type [detachable] HASH_TABLE [[attached] INTEGER_32, [attached] INTEGER_32] */
struct S27 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* capacity */
	T0* a2; /* content */
	T0* a3; /* keys */
	T0* a4; /* deleted_marks */
	T0* a5; /* indexes_map */
	T6 a6; /* iteration_position */
	T6 a7; /* count */
	T6 a8; /* deleted_item_position */
	T6 a9; /* control */
	T6 a10; /* found_item */
	T1 a11; /* has_default */
	T6 a12; /* item_position */
	T6 a13; /* ht_lowest_deleted_position */
	T6 a14; /* ht_deleted_item */
	T6 a15; /* ht_deleted_key */
	T1 a16; /* object_comparison */
};

/* Struct for type [detachable] CELL [detachable TUPLE [[attached] INTEGER_32, [attached] INTEGER_32, [attached] INTEGER_32, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] INTEGER_32, [attached] BOOLEAN]] */
struct S28 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
};

/* Struct for type [detachable] CELL [[attached] NO_MORE_MEMORY] */
struct S29 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
};

/* Struct for type detachable C_STRING */
struct S30 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* managed_data */
	T6 a2; /* count */
};

/* Struct for type [detachable] TUPLE [[attached] INTEGER_32, [attached] INTEGER_32, [attached] INTEGER_32, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] INTEGER_32, [attached] BOOLEAN] */
struct S31 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 z1;
	T6 z2;
	T6 z3;
	T0* z4;
	T0* z5;
	T0* z6;
	T0* z7;
	T0* z8;
	T0* z9;
	T6 z10;
	T1 z11;
};

/* Struct for type detachable KL_TEXT_INPUT_FILE */
struct S32 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* mode */
	T0* a2; /* last_string */
	T0* a3; /* name */
	T0* a4; /* character_buffer */
	T1 a5; /* end_of_file */
	T14 a6; /* file_pointer */
	T1 a7; /* descriptor_available */
	T0* a8; /* internal_name */
	T0* a9; /* internal_detachable_name_pointer */
	T2 a10; /* last_character */
};

/* Struct for type [detachable] KL_ARGUMENTS */
struct S35 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* program_name */
};

/* Struct for type [detachable] KL_STANDARD_FILES */
struct S36 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_STDOUT_FILE */
struct S37 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] ARRAY [[attached] STRING_8] */
struct S38 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [detachable] SPECIAL [[attached] STRING_8] */
struct S39 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] ARRAY [[attached] ANY] */
struct S40 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [detachable] SPECIAL [[attached] ANY] */
struct S41 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_STDERR_FILE */
struct S42 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SQL_GENERATOR_ANSI */
struct S43 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* DomainsEnabled */
	T1 a2; /* AttributeRequiredEnabled */
	T1 a3; /* AutoPrimaryKeyEnabled */
	T1 a4; /* SetDatabaseEnabled */
	T1 a5; /* StoredProcedureEnabled */
	T1 a6; /* IdentifierWithSpacesEnabled */
	T1 a7; /* TimestampEnabled */
	T1 a8; /* ViewsEnabled */
	T1 a9; /* ExtendIndex */
	T1 a10; /* ExtendView */
	T0* a11; /* declared_values */
	T1 a12; /* AssertEnabled */
	T1 a13; /* Iso8601Dates */
	T1 a14; /* OldConstraintNames */
	T1 a15; /* TimeZoneEnabled */
	T1 a16; /* sequence_name_format_one_parameter */
	T1 a17; /* ConstantTableCreated */
	T0* a18; /* update_type */
	T0* a19; /* updated_extension */
	T1 a20; /* InUpdateStatement */
	T1 a21; /* ValueTableCreated */
	T1 a22; /* WhereWritten */
};

/* Struct for type [detachable] SQL_GENERATOR_BASIC */
struct S44 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* DomainsEnabled */
	T1 a2; /* AttributeRequiredEnabled */
	T1 a3; /* AutoPrimaryKeyEnabled */
	T1 a4; /* SetDatabaseEnabled */
	T1 a5; /* StoredProcedureEnabled */
	T1 a6; /* IdentifierWithSpacesEnabled */
	T1 a7; /* TimestampEnabled */
	T1 a8; /* ViewsEnabled */
	T1 a9; /* ExtendIndex */
	T1 a10; /* ExtendView */
	T0* a11; /* declared_values */
	T1 a12; /* AssertEnabled */
	T1 a13; /* Iso8601Dates */
	T1 a14; /* OldConstraintNames */
	T1 a15; /* TimeZoneEnabled */
	T1 a16; /* sequence_name_format_one_parameter */
	T1 a17; /* ConstantTableCreated */
	T0* a18; /* update_type */
	T0* a19; /* updated_extension */
	T1 a20; /* InUpdateStatement */
	T1 a21; /* ValueTableCreated */
	T1 a22; /* WhereWritten */
};

/* Struct for type [detachable] SQL_GENERATOR_DB2_6 */
struct S45 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* DomainsEnabled */
	T1 a2; /* AttributeRequiredEnabled */
	T1 a3; /* AutoPrimaryKeyEnabled */
	T1 a4; /* SetDatabaseEnabled */
	T1 a5; /* StoredProcedureEnabled */
	T1 a6; /* IdentifierWithSpacesEnabled */
	T1 a7; /* TimestampEnabled */
	T1 a8; /* ViewsEnabled */
	T1 a9; /* ExtendIndex */
	T1 a10; /* ExtendView */
	T0* a11; /* declared_values */
	T1 a12; /* AssertEnabled */
	T1 a13; /* Iso8601Dates */
	T1 a14; /* OldConstraintNames */
	T1 a15; /* TimeZoneEnabled */
	T1 a16; /* sequence_name_format_one_parameter */
	T1 a17; /* ConstantTableCreated */
	T0* a18; /* update_type */
	T0* a19; /* updated_extension */
	T1 a20; /* InUpdateStatement */
	T1 a21; /* ValueTableCreated */
	T1 a22; /* WhereWritten */
};

/* Struct for type [detachable] SQL_GENERATOR_DB2_71 */
struct S46 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* DomainsEnabled */
	T1 a2; /* AttributeRequiredEnabled */
	T1 a3; /* AutoPrimaryKeyEnabled */
	T1 a4; /* SetDatabaseEnabled */
	T1 a5; /* StoredProcedureEnabled */
	T1 a6; /* IdentifierWithSpacesEnabled */
	T1 a7; /* TimestampEnabled */
	T1 a8; /* ViewsEnabled */
	T1 a9; /* ExtendIndex */
	T1 a10; /* ExtendView */
	T0* a11; /* declared_values */
	T1 a12; /* AssertEnabled */
	T1 a13; /* Iso8601Dates */
	T1 a14; /* OldConstraintNames */
	T1 a15; /* TimeZoneEnabled */
	T1 a16; /* sequence_name_format_one_parameter */
	T1 a17; /* ConstantTableCreated */
	T0* a18; /* update_type */
	T0* a19; /* updated_extension */
	T1 a20; /* InUpdateStatement */
	T1 a21; /* is_stored_procedure */
	T1 a22; /* ValueTableCreated */
	T1 a23; /* WhereWritten */
};

/* Struct for type [detachable] SQL_GENERATOR_FOXPRO */
struct S47 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* DomainsEnabled */
	T1 a2; /* AttributeRequiredEnabled */
	T1 a3; /* AutoPrimaryKeyEnabled */
	T1 a4; /* SetDatabaseEnabled */
	T1 a5; /* StoredProcedureEnabled */
	T1 a6; /* IdentifierWithSpacesEnabled */
	T1 a7; /* TimestampEnabled */
	T1 a8; /* ViewsEnabled */
	T1 a9; /* ExtendIndex */
	T1 a10; /* ExtendView */
	T0* a11; /* declared_values */
	T1 a12; /* AssertEnabled */
	T1 a13; /* Iso8601Dates */
	T1 a14; /* OldConstraintNames */
	T1 a15; /* TimeZoneEnabled */
	T1 a16; /* sequence_name_format_one_parameter */
	T1 a17; /* ConstantTableCreated */
	T0* a18; /* update_type */
	T0* a19; /* updated_extension */
	T1 a20; /* InUpdateStatement */
	T1 a21; /* ValueTableCreated */
	T1 a22; /* WhereWritten */
};

/* Struct for type [detachable] SQL_GENERATOR_INTERBASE6 */
struct S48 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* init_error */
	T1 a2; /* DomainsEnabled */
	T1 a3; /* AttributeRequiredEnabled */
	T1 a4; /* AutoPrimaryKeyEnabled */
	T1 a5; /* SetDatabaseEnabled */
	T1 a6; /* StoredProcedureEnabled */
	T1 a7; /* IdentifierWithSpacesEnabled */
	T1 a8; /* TimestampEnabled */
	T1 a9; /* ViewsEnabled */
	T1 a10; /* ExtendIndex */
	T1 a11; /* ExtendView */
	T0* a12; /* declared_values */
	T1 a13; /* AssertEnabled */
	T1 a14; /* Iso8601Dates */
	T1 a15; /* OldConstraintNames */
	T1 a16; /* TimeZoneEnabled */
	T1 a17; /* sequence_name_format_one_parameter */
	T1 a18; /* ConstantTableCreated */
	T0* a19; /* update_type */
	T0* a20; /* updated_extension */
	T1 a21; /* InUpdateStatement */
	T1 a22; /* is_stored_procedure */
	T1 a23; /* ValueTableCreated */
	T1 a24; /* WhereWritten */
};

/* Struct for type [detachable] SQL_GENERATOR_FIREBIRD_21 */
struct S49 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* init_error */
	T1 a2; /* DomainsEnabled */
	T1 a3; /* AttributeRequiredEnabled */
	T1 a4; /* AutoPrimaryKeyEnabled */
	T1 a5; /* SetDatabaseEnabled */
	T1 a6; /* StoredProcedureEnabled */
	T1 a7; /* IdentifierWithSpacesEnabled */
	T1 a8; /* TimestampEnabled */
	T1 a9; /* ViewsEnabled */
	T1 a10; /* ExtendIndex */
	T1 a11; /* ExtendView */
	T0* a12; /* declared_values */
	T1 a13; /* AssertEnabled */
	T1 a14; /* Iso8601Dates */
	T1 a15; /* OldConstraintNames */
	T1 a16; /* TimeZoneEnabled */
	T1 a17; /* sequence_name_format_one_parameter */
	T1 a18; /* ConstantTableCreated */
	T0* a19; /* update_type */
	T0* a20; /* updated_extension */
	T1 a21; /* InUpdateStatement */
	T1 a22; /* is_stored_procedure */
	T1 a23; /* ValueTableCreated */
	T1 a24; /* WhereWritten */
};

/* Struct for type [detachable] SQL_GENERATOR_MSACCESS97 */
struct S50 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* DomainsEnabled */
	T1 a2; /* AttributeRequiredEnabled */
	T1 a3; /* AutoPrimaryKeyEnabled */
	T1 a4; /* SetDatabaseEnabled */
	T1 a5; /* StoredProcedureEnabled */
	T1 a6; /* IdentifierWithSpacesEnabled */
	T1 a7; /* TimestampEnabled */
	T1 a8; /* ViewsEnabled */
	T1 a9; /* ExtendIndex */
	T1 a10; /* ExtendView */
	T0* a11; /* declared_values */
	T1 a12; /* AssertEnabled */
	T1 a13; /* Iso8601Dates */
	T1 a14; /* OldConstraintNames */
	T1 a15; /* TimeZoneEnabled */
	T1 a16; /* sequence_name_format_one_parameter */
	T1 a17; /* ConstantTableCreated */
	T0* a18; /* update_type */
	T0* a19; /* updated_extension */
	T1 a20; /* InUpdateStatement */
	T1 a21; /* ValueTableCreated */
	T1 a22; /* WhereWritten */
};

/* Struct for type [detachable] SQL_GENERATOR_MSACCESS2000 */
struct S51 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* DomainsEnabled */
	T1 a2; /* AttributeRequiredEnabled */
	T1 a3; /* AutoPrimaryKeyEnabled */
	T1 a4; /* SetDatabaseEnabled */
	T1 a5; /* StoredProcedureEnabled */
	T1 a6; /* IdentifierWithSpacesEnabled */
	T1 a7; /* TimestampEnabled */
	T1 a8; /* ViewsEnabled */
	T1 a9; /* ExtendIndex */
	T1 a10; /* ExtendView */
	T0* a11; /* declared_values */
	T1 a12; /* AssertEnabled */
	T1 a13; /* Iso8601Dates */
	T1 a14; /* OldConstraintNames */
	T1 a15; /* TimeZoneEnabled */
	T1 a16; /* sequence_name_format_one_parameter */
	T1 a17; /* ConstantTableCreated */
	T0* a18; /* update_type */
	T0* a19; /* updated_extension */
	T1 a20; /* InUpdateStatement */
	T1 a21; /* ValueTableCreated */
	T1 a22; /* WhereWritten */
};

/* Struct for type [detachable] SQL_GENERATOR_MSQL */
struct S52 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* DomainsEnabled */
	T1 a2; /* AttributeRequiredEnabled */
	T1 a3; /* AutoPrimaryKeyEnabled */
	T1 a4; /* SetDatabaseEnabled */
	T1 a5; /* StoredProcedureEnabled */
	T1 a6; /* IdentifierWithSpacesEnabled */
	T1 a7; /* TimestampEnabled */
	T1 a8; /* ViewsEnabled */
	T1 a9; /* ExtendIndex */
	T1 a10; /* ExtendView */
	T0* a11; /* declared_values */
	T1 a12; /* AssertEnabled */
	T1 a13; /* Iso8601Dates */
	T1 a14; /* OldConstraintNames */
	T1 a15; /* TimeZoneEnabled */
	T1 a16; /* sequence_name_format_one_parameter */
	T1 a17; /* ConstantTableCreated */
	T0* a18; /* update_type */
	T0* a19; /* updated_extension */
	T1 a20; /* InUpdateStatement */
	T1 a21; /* ValueTableCreated */
	T1 a22; /* WhereWritten */
};

/* Struct for type [detachable] SQL_GENERATOR_MYSQL5 */
struct S53 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* DomainsEnabled */
	T1 a2; /* AttributeRequiredEnabled */
	T1 a3; /* AutoPrimaryKeyEnabled */
	T1 a4; /* SetDatabaseEnabled */
	T1 a5; /* StoredProcedureEnabled */
	T1 a6; /* IdentifierWithSpacesEnabled */
	T1 a7; /* TimestampEnabled */
	T1 a8; /* ViewsEnabled */
	T1 a9; /* ExtendIndex */
	T1 a10; /* ExtendView */
	T0* a11; /* declared_values */
	T1 a12; /* AssertEnabled */
	T1 a13; /* Iso8601Dates */
	T1 a14; /* OldConstraintNames */
	T1 a15; /* TimeZoneEnabled */
	T1 a16; /* sequence_name_format_one_parameter */
	T1 a17; /* ConstantTableCreated */
	T0* a18; /* update_type */
	T0* a19; /* updated_extension */
	T1 a20; /* InUpdateStatement */
	T1 a21; /* is_stored_procedure */
	T1 a22; /* WhereWritten */
};

/* Struct for type [detachable] SQL_GENERATOR_MYSQL4 */
struct S54 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* DomainsEnabled */
	T1 a2; /* AttributeRequiredEnabled */
	T1 a3; /* AutoPrimaryKeyEnabled */
	T1 a4; /* SetDatabaseEnabled */
	T1 a5; /* StoredProcedureEnabled */
	T1 a6; /* IdentifierWithSpacesEnabled */
	T1 a7; /* TimestampEnabled */
	T1 a8; /* ViewsEnabled */
	T1 a9; /* ExtendIndex */
	T1 a10; /* ExtendView */
	T0* a11; /* declared_values */
	T1 a12; /* AssertEnabled */
	T1 a13; /* Iso8601Dates */
	T1 a14; /* OldConstraintNames */
	T1 a15; /* TimeZoneEnabled */
	T1 a16; /* sequence_name_format_one_parameter */
	T1 a17; /* ConstantTableCreated */
	T0* a18; /* update_type */
	T0* a19; /* updated_extension */
	T1 a20; /* InUpdateStatement */
	T1 a21; /* ValueTableCreated */
	T1 a22; /* WhereWritten */
};

/* Struct for type [detachable] SQL_GENERATOR_MYSQL322 */
struct S55 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* DomainsEnabled */
	T1 a2; /* AttributeRequiredEnabled */
	T1 a3; /* AutoPrimaryKeyEnabled */
	T1 a4; /* SetDatabaseEnabled */
	T1 a5; /* StoredProcedureEnabled */
	T1 a6; /* IdentifierWithSpacesEnabled */
	T1 a7; /* TimestampEnabled */
	T1 a8; /* ViewsEnabled */
	T1 a9; /* ExtendIndex */
	T1 a10; /* ExtendView */
	T0* a11; /* declared_values */
	T1 a12; /* AssertEnabled */
	T1 a13; /* Iso8601Dates */
	T1 a14; /* OldConstraintNames */
	T1 a15; /* TimeZoneEnabled */
	T1 a16; /* sequence_name_format_one_parameter */
	T1 a17; /* ConstantTableCreated */
	T0* a18; /* update_type */
	T0* a19; /* updated_extension */
	T1 a20; /* InUpdateStatement */
	T1 a21; /* ValueTableCreated */
	T1 a22; /* WhereWritten */
};

/* Struct for type [detachable] SQL_GENERATOR_ORACLE */
struct S56 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* DomainsEnabled */
	T1 a2; /* AttributeRequiredEnabled */
	T1 a3; /* AutoPrimaryKeyEnabled */
	T1 a4; /* SetDatabaseEnabled */
	T1 a5; /* StoredProcedureEnabled */
	T1 a6; /* IdentifierWithSpacesEnabled */
	T1 a7; /* TimestampEnabled */
	T1 a8; /* ViewsEnabled */
	T1 a9; /* ExtendIndex */
	T1 a10; /* ExtendView */
	T0* a11; /* declared_values */
	T1 a12; /* AssertEnabled */
	T1 a13; /* Iso8601Dates */
	T1 a14; /* OldConstraintNames */
	T1 a15; /* TimeZoneEnabled */
	T1 a16; /* sequence_name_format_one_parameter */
	T1 a17; /* ConstantTableCreated */
	T0* a18; /* update_type */
	T0* a19; /* updated_extension */
	T1 a20; /* InUpdateStatement */
	T1 a21; /* is_stored_procedure */
	T1 a22; /* ValueTableCreated */
	T1 a23; /* WhereWritten */
};

/* Struct for type [detachable] SQL_GENERATOR_PGSQL_95 */
struct S57 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* DomainsEnabled */
	T1 a2; /* AttributeRequiredEnabled */
	T1 a3; /* AutoPrimaryKeyEnabled */
	T1 a4; /* SetDatabaseEnabled */
	T1 a5; /* StoredProcedureEnabled */
	T1 a6; /* IdentifierWithSpacesEnabled */
	T1 a7; /* TimestampEnabled */
	T1 a8; /* ViewsEnabled */
	T1 a9; /* ExtendIndex */
	T1 a10; /* ExtendView */
	T0* a11; /* declared_values */
	T1 a12; /* AssertEnabled */
	T1 a13; /* Iso8601Dates */
	T1 a14; /* OldConstraintNames */
	T1 a15; /* TimeZoneEnabled */
	T1 a16; /* sequence_name_format_one_parameter */
	T1 a17; /* ConstantTableCreated */
	T0* a18; /* update_type */
	T0* a19; /* updated_extension */
	T1 a20; /* InUpdateStatement */
	T1 a21; /* is_stored_procedure */
	T1 a22; /* in_user_declaration */
	T1 a23; /* in_sp_type */
	T1 a24; /* ValueTableCreated */
	T1 a25; /* WhereWritten */
};

/* Struct for type [detachable] SQL_GENERATOR_PGSQL_82 */
struct S58 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* DomainsEnabled */
	T1 a2; /* AttributeRequiredEnabled */
	T1 a3; /* AutoPrimaryKeyEnabled */
	T1 a4; /* SetDatabaseEnabled */
	T1 a5; /* StoredProcedureEnabled */
	T1 a6; /* IdentifierWithSpacesEnabled */
	T1 a7; /* TimestampEnabled */
	T1 a8; /* ViewsEnabled */
	T1 a9; /* ExtendIndex */
	T1 a10; /* ExtendView */
	T0* a11; /* declared_values */
	T1 a12; /* AssertEnabled */
	T1 a13; /* Iso8601Dates */
	T1 a14; /* OldConstraintNames */
	T1 a15; /* TimeZoneEnabled */
	T1 a16; /* sequence_name_format_one_parameter */
	T1 a17; /* ConstantTableCreated */
	T0* a18; /* update_type */
	T0* a19; /* updated_extension */
	T1 a20; /* InUpdateStatement */
	T1 a21; /* is_stored_procedure */
	T1 a22; /* in_user_declaration */
	T1 a23; /* in_sp_type */
	T1 a24; /* ValueTableCreated */
	T1 a25; /* WhereWritten */
};

/* Struct for type [detachable] SQL_GENERATOR_PGSQL_81 */
struct S59 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* DomainsEnabled */
	T1 a2; /* AttributeRequiredEnabled */
	T1 a3; /* AutoPrimaryKeyEnabled */
	T1 a4; /* SetDatabaseEnabled */
	T1 a5; /* StoredProcedureEnabled */
	T1 a6; /* IdentifierWithSpacesEnabled */
	T1 a7; /* TimestampEnabled */
	T1 a8; /* ViewsEnabled */
	T1 a9; /* ExtendIndex */
	T1 a10; /* ExtendView */
	T0* a11; /* declared_values */
	T1 a12; /* AssertEnabled */
	T1 a13; /* Iso8601Dates */
	T1 a14; /* OldConstraintNames */
	T1 a15; /* TimeZoneEnabled */
	T1 a16; /* sequence_name_format_one_parameter */
	T1 a17; /* ConstantTableCreated */
	T0* a18; /* update_type */
	T0* a19; /* updated_extension */
	T1 a20; /* InUpdateStatement */
	T1 a21; /* is_stored_procedure */
	T1 a22; /* in_user_declaration */
	T1 a23; /* in_sp_type */
	T1 a24; /* ValueTableCreated */
	T1 a25; /* WhereWritten */
};

/* Struct for type [detachable] SQL_GENERATOR_PGSQL_73 */
struct S60 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* DomainsEnabled */
	T1 a2; /* AttributeRequiredEnabled */
	T1 a3; /* AutoPrimaryKeyEnabled */
	T1 a4; /* SetDatabaseEnabled */
	T1 a5; /* StoredProcedureEnabled */
	T1 a6; /* IdentifierWithSpacesEnabled */
	T1 a7; /* TimestampEnabled */
	T1 a8; /* ViewsEnabled */
	T1 a9; /* ExtendIndex */
	T1 a10; /* ExtendView */
	T0* a11; /* declared_values */
	T1 a12; /* AssertEnabled */
	T1 a13; /* Iso8601Dates */
	T1 a14; /* OldConstraintNames */
	T1 a15; /* TimeZoneEnabled */
	T1 a16; /* sequence_name_format_one_parameter */
	T1 a17; /* ConstantTableCreated */
	T0* a18; /* update_type */
	T0* a19; /* updated_extension */
	T1 a20; /* InUpdateStatement */
	T1 a21; /* is_stored_procedure */
	T1 a22; /* in_user_declaration */
	T1 a23; /* in_sp_type */
	T1 a24; /* ValueTableCreated */
	T1 a25; /* WhereWritten */
};

/* Struct for type [detachable] SQL_GENERATOR_PGSQL_72 */
struct S61 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* DomainsEnabled */
	T1 a2; /* AttributeRequiredEnabled */
	T1 a3; /* AutoPrimaryKeyEnabled */
	T1 a4; /* SetDatabaseEnabled */
	T1 a5; /* StoredProcedureEnabled */
	T1 a6; /* IdentifierWithSpacesEnabled */
	T1 a7; /* TimestampEnabled */
	T1 a8; /* ViewsEnabled */
	T1 a9; /* ExtendIndex */
	T1 a10; /* ExtendView */
	T0* a11; /* declared_values */
	T1 a12; /* AssertEnabled */
	T1 a13; /* Iso8601Dates */
	T1 a14; /* OldConstraintNames */
	T1 a15; /* TimeZoneEnabled */
	T1 a16; /* sequence_name_format_one_parameter */
	T1 a17; /* ConstantTableCreated */
	T0* a18; /* update_type */
	T0* a19; /* updated_extension */
	T1 a20; /* InUpdateStatement */
	T1 a21; /* is_stored_procedure */
	T1 a22; /* in_user_declaration */
	T1 a23; /* ValueTableCreated */
	T1 a24; /* WhereWritten */
};

/* Struct for type [detachable] SQL_GENERATOR_SQLITE_3 */
struct S62 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* DomainsEnabled */
	T1 a2; /* AttributeRequiredEnabled */
	T1 a3; /* AutoPrimaryKeyEnabled */
	T1 a4; /* SetDatabaseEnabled */
	T1 a5; /* StoredProcedureEnabled */
	T1 a6; /* IdentifierWithSpacesEnabled */
	T1 a7; /* TimestampEnabled */
	T1 a8; /* ViewsEnabled */
	T1 a9; /* ExtendIndex */
	T1 a10; /* ExtendView */
	T0* a11; /* declared_values */
	T1 a12; /* AssertEnabled */
	T1 a13; /* Iso8601Dates */
	T1 a14; /* OldConstraintNames */
	T1 a15; /* TimeZoneEnabled */
	T1 a16; /* sequence_name_format_one_parameter */
	T1 a17; /* ConstantTableCreated */
	T0* a18; /* update_type */
	T0* a19; /* updated_extension */
	T1 a20; /* InUpdateStatement */
	T1 a21; /* ValueTableCreated */
	T1 a22; /* WhereWritten */
};

/* Struct for type [detachable] SQL_GENERATOR_TSQL2016 */
struct S63 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* DomainsEnabled */
	T1 a2; /* AttributeRequiredEnabled */
	T1 a3; /* AutoPrimaryKeyEnabled */
	T1 a4; /* SetDatabaseEnabled */
	T1 a5; /* StoredProcedureEnabled */
	T1 a6; /* IdentifierWithSpacesEnabled */
	T1 a7; /* TimestampEnabled */
	T1 a8; /* ViewsEnabled */
	T1 a9; /* ExtendIndex */
	T1 a10; /* ExtendView */
	T0* a11; /* declared_values */
	T1 a12; /* AssertEnabled */
	T1 a13; /* Iso8601Dates */
	T1 a14; /* OldConstraintNames */
	T1 a15; /* TimeZoneEnabled */
	T1 a16; /* sequence_name_format_one_parameter */
	T1 a17; /* ConstantTableCreated */
	T1 a18; /* is_stored_procedure */
	T0* a19; /* update_type */
	T0* a20; /* updated_extension */
	T1 a21; /* InUpdateStatement */
	T1 a22; /* WhereWritten */
};

/* Struct for type [detachable] SQL_GENERATOR_TSQL2000 */
struct S64 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* DomainsEnabled */
	T1 a2; /* AttributeRequiredEnabled */
	T1 a3; /* AutoPrimaryKeyEnabled */
	T1 a4; /* SetDatabaseEnabled */
	T1 a5; /* StoredProcedureEnabled */
	T1 a6; /* IdentifierWithSpacesEnabled */
	T1 a7; /* TimestampEnabled */
	T1 a8; /* ViewsEnabled */
	T1 a9; /* ExtendIndex */
	T1 a10; /* ExtendView */
	T0* a11; /* declared_values */
	T1 a12; /* AssertEnabled */
	T1 a13; /* Iso8601Dates */
	T1 a14; /* OldConstraintNames */
	T1 a15; /* TimeZoneEnabled */
	T1 a16; /* sequence_name_format_one_parameter */
	T1 a17; /* ConstantTableCreated */
	T1 a18; /* is_stored_procedure */
	T0* a19; /* update_type */
	T0* a20; /* updated_extension */
	T1 a21; /* InUpdateStatement */
	T1 a22; /* WhereWritten */
};

/* Struct for type [detachable] SQL_GENERATOR_TSQL70 */
struct S65 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* DomainsEnabled */
	T1 a2; /* AttributeRequiredEnabled */
	T1 a3; /* AutoPrimaryKeyEnabled */
	T1 a4; /* SetDatabaseEnabled */
	T1 a5; /* StoredProcedureEnabled */
	T1 a6; /* IdentifierWithSpacesEnabled */
	T1 a7; /* TimestampEnabled */
	T1 a8; /* ViewsEnabled */
	T1 a9; /* ExtendIndex */
	T1 a10; /* ExtendView */
	T0* a11; /* declared_values */
	T1 a12; /* AssertEnabled */
	T1 a13; /* Iso8601Dates */
	T1 a14; /* OldConstraintNames */
	T1 a15; /* TimeZoneEnabled */
	T1 a16; /* sequence_name_format_one_parameter */
	T1 a17; /* ConstantTableCreated */
	T1 a18; /* is_stored_procedure */
	T0* a19; /* update_type */
	T0* a20; /* updated_extension */
	T1 a21; /* InUpdateStatement */
	T1 a22; /* WhereWritten */
};

/* Struct for type [detachable] SQL_GENERATOR_TSQL65 */
struct S66 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* DomainsEnabled */
	T1 a2; /* AttributeRequiredEnabled */
	T1 a3; /* AutoPrimaryKeyEnabled */
	T1 a4; /* SetDatabaseEnabled */
	T1 a5; /* StoredProcedureEnabled */
	T1 a6; /* IdentifierWithSpacesEnabled */
	T1 a7; /* TimestampEnabled */
	T1 a8; /* ViewsEnabled */
	T1 a9; /* ExtendIndex */
	T1 a10; /* ExtendView */
	T0* a11; /* declared_values */
	T1 a12; /* AssertEnabled */
	T1 a13; /* Iso8601Dates */
	T1 a14; /* OldConstraintNames */
	T1 a15; /* TimeZoneEnabled */
	T1 a16; /* sequence_name_format_one_parameter */
	T1 a17; /* ConstantTableCreated */
	T1 a18; /* is_stored_procedure */
	T0* a19; /* update_type */
	T0* a20; /* updated_extension */
	T1 a21; /* InUpdateStatement */
	T1 a22; /* WhereWritten */
};

/* Struct for type [detachable] XPLAIN_PARSER */
struct S67 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* statements */
	T6 a2; /* yy_parsing_status */
	T0* a3; /* mwgenerator */
	T6 a4; /* yy_suspended_yystacksize */
	T6 a5; /* yy_suspended_yystate */
	T6 a6; /* yy_suspended_yyn */
	T6 a7; /* yy_suspended_yychar1 */
	T6 a8; /* yy_suspended_index */
	T6 a9; /* yy_suspended_yyss_top */
	T6 a10; /* yy_suspended_yy_goto */
	T0* a11; /* yyr1 */
	T0* a12; /* yyss */
	T6 a13; /* yyssp */
	T0* a14; /* yypgoto */
	T0* a15; /* yycheck */
	T0* a16; /* yytable */
	T0* a17; /* yydefgoto */
	T6 a18; /* error_count */
	T1 a19; /* yy_lookahead_needed */
	T6 a20; /* yyerrstatus */
	T0* a21; /* yypact */
	T6 a22; /* last_token */
	T0* a23; /* yytranslate */
	T0* a24; /* yydefact */
	T0* a25; /* input_buffer */
	T0* a26; /* saved_state */
	T0* a27; /* dummyrestriction */
	T6 a28; /* yyvsp1 */
	T6 a29; /* yyvsp2 */
	T6 a30; /* yyvsp3 */
	T6 a31; /* yyvsp4 */
	T6 a32; /* yyvsp5 */
	T6 a33; /* yyvsp6 */
	T6 a34; /* yyvsp7 */
	T6 a35; /* yyvsp8 */
	T6 a36; /* yyvsp9 */
	T6 a37; /* yyvsp10 */
	T6 a38; /* yyvsp11 */
	T6 a39; /* yyvsp12 */
	T6 a40; /* yyvsp13 */
	T6 a41; /* yyvsp14 */
	T6 a42; /* yyvsp15 */
	T6 a43; /* yyvsp16 */
	T6 a44; /* yyvsp17 */
	T6 a45; /* yyvsp18 */
	T6 a46; /* yyvsp19 */
	T6 a47; /* yyvsp20 */
	T6 a48; /* yyvsp21 */
	T6 a49; /* yyvsp22 */
	T6 a50; /* yyvsp23 */
	T6 a51; /* yyvsp24 */
	T6 a52; /* yyvsp25 */
	T6 a53; /* yyvsp26 */
	T6 a54; /* yyvsp27 */
	T6 a55; /* yyvsp28 */
	T6 a56; /* yyvsp29 */
	T6 a57; /* yyvsp30 */
	T6 a58; /* yyvsp31 */
	T6 a59; /* yyvsp32 */
	T6 a60; /* yyvsp33 */
	T6 a61; /* yyvsp34 */
	T6 a62; /* yyvsp35 */
	T6 a63; /* yyvsp36 */
	T6 a64; /* yyvsp37 */
	T6 a65; /* yyvsp38 */
	T6 a66; /* yyvsp39 */
	T6 a67; /* yyvsp40 */
	T6 a68; /* yyvsp41 */
	T6 a69; /* yyvsp42 */
	T6 a70; /* yyvsp43 */
	T6 a71; /* yyvsp44 */
	T6 a72; /* yyvsp45 */
	T6 a73; /* yyvsp46 */
	T6 a74; /* yyvsp47 */
	T6 a75; /* yyvsp48 */
	T6 a76; /* yyvsp49 */
	T6 a77; /* yyvsp50 */
	T6 a78; /* yyvsp51 */
	T6 a79; /* yyvsp52 */
	T6 a80; /* yyvsp53 */
	T6 a81; /* yyvsp54 */
	T1 a82; /* yy_more_flag */
	T6 a83; /* yy_more_len */
	T6 a84; /* yy_end */
	T6 a85; /* yy_start */
	T6 a86; /* line */
	T6 a87; /* yy_line */
	T6 a88; /* column */
	T6 a89; /* yy_column */
	T6 a90; /* position */
	T6 a91; /* yy_position */
	T6 a92; /* yy_start_state */
	T0* a93; /* yy_state_stack */
	T6 a94; /* yy_state_count */
	T0* a95; /* yy_content_area */
	T0* a96; /* yy_unicode_content_area */
	T0* a97; /* yy_accept */
	T6 a98; /* yy_last_accepting_state */
	T6 a99; /* yy_last_accepting_cpos */
	T0* a100; /* yy_content */
	T0* a101; /* yy_ec */
	T0* a102; /* yy_chk */
	T0* a103; /* yy_base */
	T0* a104; /* yy_def */
	T0* a105; /* yy_meta */
	T0* a106; /* yy_nxt */
	T6 a107; /* yy_lp */
	T0* a108; /* yy_acclist */
	T6 a109; /* yy_looking_for_trail_begin */
	T6 a110; /* yy_full_match */
	T6 a111; /* yy_full_state */
	T6 a112; /* yy_full_lp */
	T1 a113; /* yy_rejected */
	T0* a114; /* yytypes2 */
	T6 a115; /* yyvsc1 */
	T0* a116; /* yyvs1 */
	T0* a117; /* yyspecial_routines1 */
	T0* a118; /* last_detachable_any_value */
	T6 a119; /* yyvsc2 */
	T0* a120; /* yyvs2 */
	T0* a121; /* yyspecial_routines2 */
	T0* a122; /* last_string_value */
	T6 a123; /* yyvsc3 */
	T0* a124; /* yyvs3 */
	T0* a125; /* yyspecial_routines3 */
	T6 a126; /* last_integer_value */
	T0* a127; /* sqlgenerator */
	T0* a128; /* yyvs41 */
	T0* a129; /* yyvs51 */
	T0* a130; /* yyspecial_routines41 */
	T0* a131; /* yyvs50 */
	T6 a132; /* yyvsc41 */
	T0* a133; /* yyvs44 */
	T0* a134; /* yyvs15 */
	T0* a135; /* yyvs4 */
	T0* a136; /* yyvs43 */
	T0* a137; /* yyvs10 */
	T1 a138; /* constant_mode */
	T0* a139; /* yyspecial_routines44 */
	T6 a140; /* yyvsc44 */
	T0* a141; /* yyvs17 */
	T6 a142; /* yyvsc4 */
	T0* a143; /* yyspecial_routines4 */
	T0* a144; /* yyvs6 */
	T0* a145; /* last_representation */
	T0* a146; /* yyvs19 */
	T0* a147; /* yyvs18 */
	T0* a148; /* pending_type */
	T6 a149; /* yyvsc15 */
	T0* a150; /* yyspecial_routines15 */
	T0* a151; /* pending_init */
	T1 a152; /* is_init_default */
	T0* a153; /* myvariable */
	T6 a154; /* yyvsc43 */
	T0* a155; /* yyspecial_routines43 */
	T0* a156; /* subject_type */
	T0* a157; /* yyvs14 */
	T0* a158; /* yyspecial_routines6 */
	T6 a159; /* yyvsc6 */
	T0* a160; /* yyvs7 */
	T0* a161; /* yyvs8 */
	T6 a162; /* yyvsc7 */
	T0* a163; /* yyspecial_routines7 */
	T6 a164; /* yyvsc8 */
	T0* a165; /* yyspecial_routines8 */
	T6 a166; /* last_width */
	T0* a167; /* yyvs9 */
	T0* a168; /* yyspecial_routines9 */
	T6 a169; /* yyvsc9 */
	T0* a170; /* yyspecial_routines10 */
	T0* a171; /* yyvs13 */
	T0* a172; /* last_object_in_tree */
	T6 a173; /* yyvsc10 */
	T0* a174; /* yyvs11 */
	T6 a175; /* yyvsc11 */
	T0* a176; /* yyspecial_routines11 */
	T6 a177; /* yyvsc12 */
	T0* a178; /* yyvs12 */
	T0* a179; /* yyspecial_routines12 */
	T0* a180; /* yyvs35 */
	T6 a181; /* yyvsc13 */
	T0* a182; /* yyspecial_routines13 */
	T6 a183; /* yyvsc14 */
	T0* a184; /* yyspecial_routines14 */
	T0* a185; /* yyvs16 */
	T6 a186; /* yyvsc16 */
	T0* a187; /* yyspecial_routines16 */
	T6 a188; /* yyvsc17 */
	T0* a189; /* yyspecial_routines17 */
	T0* a190; /* dummya */
	T6 a191; /* yyvsc18 */
	T0* a192; /* yyspecial_routines18 */
	T0* a193; /* dummyi */
	T0* a194; /* current_type_name */
	T0* a195; /* yyvs20 */
	T6 a196; /* yyvsc19 */
	T0* a197; /* yyspecial_routines19 */
	T6 a198; /* yyvsc20 */
	T0* a199; /* yyspecial_routines20 */
	T0* a200; /* dummye */
	T6 a201; /* yyvsc5 */
	T0* a202; /* yyvs5 */
	T0* a203; /* yyspecial_routines5 */
	T0* a204; /* extend_type */
	T0* a205; /* yyvs30 */
	T0* a206; /* yyspecial_routines21 */
	T0* a207; /* yyvs21 */
	T6 a208; /* yyvsc21 */
	T0* a209; /* yyvs22 */
	T0* a210; /* myindex */
	T6 a211; /* yyvsc22 */
	T0* a212; /* yyspecial_routines22 */
	T0* a213; /* yyvs52 */
	T0* a214; /* yyvs47 */
	T0* a215; /* yyvs45 */
	T0* a216; /* yyvs25 */
	T0* a217; /* yyvs23 */
	T0* a218; /* yyvs32 */
	T0* a219; /* yyvs24 */
	T6 a220; /* yyvsc23 */
	T0* a221; /* yyspecial_routines23 */
	T6 a222; /* yyvsc24 */
	T0* a223; /* yyspecial_routines24 */
	T0* a224; /* yyvs46 */
	T0* a225; /* yyspecial_routines45 */
	T6 a226; /* yyvsc45 */
	T0* a227; /* yyvs27 */
	T6 a228; /* yyvsc46 */
	T0* a229; /* yyspecial_routines46 */
	T6 a230; /* yyvsc47 */
	T0* a231; /* yyspecial_routines47 */
	T6 a232; /* yyvsc25 */
	T0* a233; /* yyspecial_routines25 */
	T6 a234; /* yyvsc52 */
	T0* a235; /* yyspecial_routines52 */
	T6 a236; /* yyvsc53 */
	T0* a237; /* yyvs53 */
	T0* a238; /* yyspecial_routines53 */
	T6 a239; /* yyvsc54 */
	T0* a240; /* yyvs54 */
	T0* a241; /* yyspecial_routines54 */
	T6 a242; /* yyvsc26 */
	T0* a243; /* yyvs26 */
	T0* a244; /* yyspecial_routines26 */
	T0* a245; /* yyvs34 */
	T6 a246; /* yyvsc27 */
	T0* a247; /* yyspecial_routines27 */
	T0* a248; /* yyvs31 */
	T6 a249; /* yyvsc49 */
	T0* a250; /* yyvs49 */
	T0* a251; /* yyspecial_routines49 */
	T0* a252; /* yyvs28 */
	T6 a253; /* yyvsc48 */
	T0* a254; /* yyvs48 */
	T0* a255; /* yyspecial_routines48 */
	T0* a256; /* yyvs29 */
	T0* a257; /* yyspecial_routines28 */
	T6 a258; /* yyvsc28 */
	T6 a259; /* yyvsc29 */
	T0* a260; /* yyspecial_routines29 */
	T6 a261; /* yyvsc30 */
	T0* a262; /* yyspecial_routines30 */
	T6 a263; /* yyvsc31 */
	T0* a264; /* yyspecial_routines31 */
	T0* a265; /* yyvs33 */
	T6 a266; /* yyvsc32 */
	T0* a267; /* yyspecial_routines32 */
	T6 a268; /* yyvsc33 */
	T0* a269; /* yyspecial_routines33 */
	T6 a270; /* yyvsc34 */
	T0* a271; /* yyspecial_routines34 */
	T6 a272; /* yyvsc35 */
	T0* a273; /* yyspecial_routines35 */
	T0* a274; /* yyvs36 */
	T6 a275; /* yyvsc36 */
	T0* a276; /* yyspecial_routines36 */
	T1 a277; /* procedure_mode */
	T0* a278; /* yyvs37 */
	T0* a279; /* yyvs40 */
	T0* a280; /* yyspecial_routines50 */
	T6 a281; /* yyvsc50 */
	T0* a282; /* my_parameters */
	T6 a283; /* yyvsc37 */
	T0* a284; /* yyspecial_routines37 */
	T0* a285; /* yyvs38 */
	T0* a286; /* yyvs39 */
	T6 a287; /* yyvsc38 */
	T0* a288; /* yyspecial_routines38 */
	T6 a289; /* yyvsc39 */
	T0* a290; /* yyspecial_routines39 */
	T6 a291; /* yyvsc51 */
	T0* a292; /* yyspecial_routines51 */
	T6 a293; /* yyvsc40 */
	T0* a294; /* yyspecial_routines40 */
	T0* a295; /* yyvs42 */
	T6 a296; /* yyvsc42 */
	T0* a297; /* yyspecial_routines42 */
	T0* a298; /* yytypes1 */
	T0* a299; /* my_sql */
	T1 a300; /* expect_domain */
	T0* a301; /* yy_pushed_start_conditions */
	T2 a302; /* last_character */
	T3 a303; /* last_unicode_character */
};

/* Struct for type [detachable] XML_GENERATOR */
struct S69 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* sqlgenerator */
	T0* a2; /* xml */
};

/* Struct for type [detachable] XPLAIN_STATEMENTS */
struct S71 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* statements */
};

/* Struct for type [detachable] VOID_TARGET */
struct S74 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] ROUTINE_FAILURE */
struct S76 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* routine_name */
	T0* a2; /* class_name */
	T0* a3; /* throwing_exception */
	T0* a4; /* internal_trace */
	T0* a5; /* c_description */
	T0* a6; /* recipient_name */
	T0* a7; /* type_name */
};

/* Struct for type [detachable] OLD_VIOLATION */
struct S78 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] NO_MORE_MEMORY */
struct S80 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_trace */
	T6 a2; /* internal_code */
	T0* a3; /* throwing_exception */
	T0* a4; /* c_description */
	T0* a5; /* recipient_name */
	T0* a6; /* type_name */
};

/* Struct for type [detachable] INVARIANT_VIOLATION */
struct S81 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T1 a2; /* is_entry */
	T0* a3; /* internal_trace */
	T0* a4; /* c_description */
	T0* a5; /* recipient_name */
	T0* a6; /* type_name */
};

/* Struct for type [detachable] OPERATING_SYSTEM_SIGNAL_FAILURE */
struct S82 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T6 a2; /* signal_code */
	T0* a3; /* internal_trace */
	T0* a4; /* c_description */
	T0* a5; /* recipient_name */
	T0* a6; /* type_name */
};

/* Struct for type [detachable] IO_FAILURE */
struct S83 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* internal_code */
	T0* a2; /* throwing_exception */
	T6 a3; /* error_code */
	T0* a4; /* internal_trace */
	T0* a5; /* c_description */
	T0* a6; /* recipient_name */
	T0* a7; /* type_name */
};

/* Struct for type [detachable] OPERATING_SYSTEM_FAILURE */
struct S84 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T6 a2; /* error_code */
	T0* a3; /* internal_trace */
	T0* a4; /* c_description */
	T0* a5; /* recipient_name */
	T0* a6; /* type_name */
};

/* Struct for type [detachable] COM_FAILURE */
struct S85 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T6 a2; /* hresult_code */
	T0* a3; /* exception_information */
	T6 a4; /* hresult */
	T0* a5; /* internal_trace */
	T0* a6; /* c_description */
	T0* a7; /* recipient_name */
	T0* a8; /* type_name */
};

/* Struct for type [detachable] EIFFEL_RUNTIME_PANIC */
struct S86 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* internal_code */
	T0* a2; /* throwing_exception */
	T0* a3; /* internal_trace */
	T0* a4; /* c_description */
	T0* a5; /* recipient_name */
	T0* a6; /* type_name */
};

/* Struct for type [detachable] PRECONDITION_VIOLATION */
struct S88 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] POSTCONDITION_VIOLATION */
struct S89 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] FLOATING_POINT_FAILURE */
struct S90 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] CHECK_VIOLATION */
struct S91 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] BAD_INSPECT_VALUE */
struct S92 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] VARIANT_VIOLATION */
struct S93 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] LOOP_INVARIANT_VIOLATION */
struct S94 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] RESCUE_FAILURE */
struct S95 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] RESUMPTION_FAILURE */
struct S96 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] CREATE_ON_DEFERRED */
struct S97 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] EXTERNAL_FAILURE */
struct S98 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] VOID_ASSIGNED_TO_EXPANDED */
struct S99 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] EXCEPTION_IN_SIGNAL_HANDLER_FAILURE */
struct S100 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] MISMATCH_FAILURE */
struct S101 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] DEVELOPER_EXCEPTION */
struct S102 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] ADDRESS_APPLIED_TO_MELTED_FEATURE */
struct S103 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] SERIALIZATION_FAILURE */
struct S104 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* throwing_exception */
	T0* a2; /* internal_trace */
	T0* a3; /* c_description */
	T0* a4; /* recipient_name */
	T0* a5; /* type_name */
};

/* Struct for type [detachable] ST_SCIENTIFIC_FORMATTER */
struct S105 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* string_output_stream */
	T0* a2; /* error */
	T2 a3; /* escape_character */
	T0* a4; /* boolean_formatter */
	T0* a5; /* character_formatter */
	T0* a6; /* floating_point_formatter */
	T0* a7; /* fixed_point_formatter */
	T0* a8; /* fixed_or_floating_point_formatter */
	T0* a9; /* decimal_formatter */
	T0* a10; /* octal_formatter */
	T0* a11; /* hexadecimal_formatter */
	T0* a12; /* unsigned_decimal_formatter */
	T0* a13; /* pointer_formatter */
	T0* a14; /* string_formatter */
	T0* a15; /* single_parameter */
};

/* Struct for type [detachable] PRIMES */
struct S106 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] INTEGER_32] */
struct S107 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T6 z2[1]; /* item */
};

/* Struct for type [detachable] SPECIAL [[attached] BOOLEAN] */
struct S108 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T1 z2[1]; /* item */
};

/* Struct for type [detachable] KL_STRING_ROUTINES */
struct S110 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] MANAGED_POINTER */
struct S113 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T14 a1; /* item */
	T6 a2; /* count */
	T1 a3; /* is_shared */
};

/* Struct for type [detachable] FILE_INFO */
struct S114 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* buffered_file_info */
	T1 a2; /* is_following_symlinks */
	T1 a3; /* exists */
	T0* a4; /* internal_file_name */
	T0* a5; /* internal_name_pointer */
};

/* Struct for type [detachable] ARGUMENTS_32 */
struct S115 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8] */
struct S116 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* equality_tester */
	T0* a2; /* key_equality_tester */
	T6 a3; /* position */
	T6 a4; /* count */
	T6 a5; /* capacity */
	T6 a6; /* slots_position */
	T6 a7; /* free_slot */
	T6 a8; /* last_position */
	T6 a9; /* found_position */
	T6 a10; /* modulus */
	T6 a11; /* clashes_previous_position */
	T0* a12; /* special_item_routines */
	T0* a13; /* item_storage */
	T0* a14; /* clashes */
	T0* a15; /* slots */
	T0* a16; /* special_key_routines */
	T0* a17; /* key_storage */
	T0* a18; /* internal_keys */
	T0* a19; /* internal_cursor */
	T0* a20; /* hash_function */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] INTEGER_32] */
struct S118 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] XPLAIN_SCANNER_STATE */
struct S119 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* filename */
	T0* a2; /* directory */
	T1 a3; /* use_mode */
	T0* a4; /* input_buffer */
};

/* Struct for type [detachable] DS_LINKED_LIST [[attached] XPLAIN_SCANNER_STATE] */
struct S121 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* last_cell */
	T0* a3; /* first_cell */
	T0* a4; /* internal_cursor */
};

/* Struct for type [detachable] XPLAIN_REQUIRED */
struct S123 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* required */
};

/* Struct for type [detachable] SPECIAL [detachable ANY] */
struct S126 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable ANY] */
struct S127 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] STRING_8] */
struct S128 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable XPLAIN_CONSTANT_ASSIGNMENT_STATEMENT */
struct S130 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* constant */
	T0* a2; /* expression */
};

/* Struct for type detachable XPLAIN_BASE */
struct S131 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* representation */
};

/* Struct for type detachable XPLAIN_TYPE */
struct S132 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* representation */
	T0* a2; /* attributes */
	T6 a3; /* constraint_number */
	T0* a4; /* name */
	T0* a5; /* abstract_representation */
};

/* Struct for type detachable XPLAIN_CONSTANT_STATEMENT */
struct S133 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* constant */
};

/* Struct for type [detachable] XPLAIN_A_NODE */
struct S134 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* next */
};

/* Struct for type [detachable] XPLAIN_I_NODE */
struct S135 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* next */
};

/* Struct for type [detachable] XPLAIN_LOGICAL_EXPRESSION */
struct S136 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* expression */
};

/* Struct for type [detachable] XPLAIN_LOGICAL_VALUE_EXPRESSION */
struct S138 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* value */
};

/* Struct for type [detachable] XPLAIN_ATTRIBUTE_NAME_NODE */
struct S140 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* next */
	T0* a3; /* prefix_table */
};

/* Struct for type [detachable] XPLAIN_ATTRIBUTE_NAME */
struct S141 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* role */
	T0* a2; /* name */
	T0* a3; /* object */
	T0* a4; /* type_attribute */
};

/* Struct for type detachable XPLAIN_ATTRIBUTE_NODE */
struct S144 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* next */
};

/* Struct for type [detachable] XPLAIN_ATTRIBUTE */
struct S145 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* role */
	T0* a2; /* internal_abstracttype */
	T1 a3; /* overrule_required */
	T1 a4; /* is_required */
	T1 a5; /* is_specialization */
	T1 a6; /* is_unique */
	T0* a7; /* init */
	T1 a8; /* is_init_default */
};

/* Struct for type detachable XPLAIN_ASSERTION */
struct S146 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* type */
	T0* a2; /* name */
	T0* a3; /* expression */
	T0* a4; /* range */
	T0* a5; /* attributes */
	T0* a6; /* representation */
	T0* a7; /* abstract_extension_representation */
	T1 a8; /* no_update_optimization */
	T0* a9; /* abstract_representation */
	T1 a10; /* explicit_domain */
};

/* Struct for type detachable XPLAIN_ASSIGNMENT_NODE */
struct S147 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* next */
};

/* Struct for type [detachable] XPLAIN_ASSIGNMENT */
struct S148 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* attribute_name */
	T0* a2; /* expression */
};

/* Struct for type detachable XPLAIN_CASCADE_STATEMENT */
struct S149 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* subtype */
	T0* a2; /* cascade_attribute */
	T0* a3; /* definition */
	T0* a4; /* counter_statement */
	T0* a5; /* increment_value_statement */
	T0* a6; /* finished_statement */
};

/* Struct for type detachable XPLAIN_CASCADE_FUNCTION_EXPRESSION */
struct S150 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* selection */
	T0* a2; /* grouping_attributes */
};

/* Struct for type detachable XPLAIN_DELETE_STATEMENT */
struct S151 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* subject */
	T0* a2; /* predicate */
};

/* Struct for type detachable XPLAIN_SUBJECT */
struct S152 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* type */
	T0* a2; /* identification */
};

/* Struct for type detachable XPLAIN_UPDATE_STATEMENT */
struct S153 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* subject */
	T0* a2; /* assignment_list */
	T0* a3; /* predicate */
};

/* Struct for type [detachable] XPLAIN_VALUE_STATEMENT */
struct S154 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* value */
};

/* Struct for type detachable XPLAIN_VALUE_SELECTION_STATEMENT */
struct S155 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* value */
};

/* Struct for type detachable XPLAIN_SELECTION_FUNCTION */
struct S157 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* function */
	T0* a2; /* subject */
	T0* a3; /* property */
	T0* a4; /* predicate */
};

/* Struct for type detachable XPLAIN_GET_STATEMENT */
struct S158 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* selection */
};

/* Struct for type detachable XPLAIN_EXTEND_STATEMENT */
struct S159 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* extension */
};

/* Struct for type detachable XPLAIN_EXTENSION */
struct S160 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* type */
	T0* a2; /* expression */
	T0* a3; /* attributes */
	T1 a4; /* explicit_domain */
	T0* a5; /* representation */
	T0* a6; /* abstract_extension_representation */
	T0* a7; /* name */
	T0* a8; /* abstract_representation */
	T1 a9; /* no_update_optimization */
};

/* Struct for type [detachable] XPLAIN_EXTEND_ATTRIBUTE */
struct S161 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* domain */
};

/* Struct for type detachable XPLAIN_EXTENSION_EXPRESSION */
struct S162 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* extension */
	T0* a2; /* anode */
};

/* Struct for type detachable XPLAIN_SELECTION_LIST */
struct S164 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* subject */
	T0* a2; /* expression_list */
	T0* a3; /* predicate */
	T0* a4; /* sort_order */
	T0* a5; /* identification_text */
};

/* Struct for type detachable XPLAIN_SORT_NODE */
struct S165 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* ascending */
	T0* a2; /* item */
	T0* a3; /* next */
};

/* Struct for type detachable XPLAIN_EXPRESSION_NODE */
struct S167 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* new_name */
	T0* a2; /* item */
	T0* a3; /* next */
};

/* Struct for type detachable XPLAIN_PROCEDURE_STATEMENT */
struct S168 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* procedure */
};

/* Struct for type detachable XPLAIN_PARAMETER_NODE */
struct S169 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* next */
};

/* Struct for type [detachable] XPLAIN_PARAMETER */
struct S170 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* is_required */
	T0* a2; /* role */
	T0* a3; /* name */
	T0* a4; /* object */
};

/* Struct for type [detachable] XPLAIN_SQL_STATEMENT */
struct S171 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* sql */
};

/* Struct for type detachable XPLAIN_STATEMENT_NODE */
struct S172 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* next */
};

/* Struct for type detachable XPLAIN_IF_STATEMENT */
struct S173 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* logical_expression */
	T0* a2; /* then_statements */
	T0* a3; /* else_statements */
};

/* Struct for type [detachable] SPECIAL [detachable XPLAIN_STATEMENT] */
struct S174 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] SPECIAL [[attached] XPLAIN_SQL_STATEMENT] */
struct S175 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_STATEMENT] */
struct S176 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable XPLAIN_PROCEDURE_STATEMENT] */
struct S177 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] SPECIAL [detachable XPLAIN_CONSTANT_ASSIGNMENT_STATEMENT] */
struct S178 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] SPECIAL [detachable XPLAIN_TYPE] */
struct S179 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] XPLAIN_TYPE_STATEMENT */
struct S180 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* type */
};

/* Struct for type [detachable] XPLAIN_UNIVERSE */
struct S181 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* objects */
};

/* Struct for type [detachable] SPECIAL [detachable XPLAIN_BASE] */
struct S182 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] XPLAIN_BASE_STATEMENT */
struct S183 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* base */
};

/* Struct for type [detachable] SPECIAL [detachable XPLAIN_CONSTANT_STATEMENT] */
struct S184 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type detachable XPLAIN_VARIABLE */
struct S185 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* representation */
	T0* a2; /* name */
	T0* a3; /* value */
};

/* Struct for type [detachable] SPECIAL [[attached] XPLAIN_EXPRESSION] */
struct S186 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_CONSTANT_ASSIGNMENT_STATEMENT] */
struct S187 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] XPLAIN_REPRESENTATION] */
struct S188 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_BASE] */
struct S189 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] XPLAIN_DOMAIN_RESTRICTION] */
struct S190 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] SPECIAL [detachable XPLAIN_ATTRIBUTE_NODE] */
struct S191 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] SPECIAL [[attached] XPLAIN_PK_REPRESENTATION] */
struct S192 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_TYPE] */
struct S193 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_CONSTANT_STATEMENT] */
struct S194 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] XPLAIN_ATTRIBUTE_NAME] */
struct S196 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_DOMAIN_RESTRICTION] */
struct S197 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] XPLAIN_A_PATTERN */
struct S198 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* pattern */
	T1 a2; /* required */
};

/* Struct for type [detachable] SPECIAL [[attached] XPLAIN_A_NODE] */
struct S199 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] XPLAIN_A_ENUMERATION */
struct S200 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* first */
	T1 a2; /* required */
};

/* Struct for type [detachable] SPECIAL [[attached] XPLAIN_I_NODE] */
struct S201 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] XPLAIN_I_ENUMERATION */
struct S202 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* first */
	T1 a2; /* required */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_A_NODE] */
struct S203 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] XPLAIN_A_REPRESENTATION */
struct S204 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* length */
	T0* a2; /* domain_restriction */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_I_NODE] */
struct S206 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] XPLAIN_TRAJECTORY */
struct S207 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* min */
	T0* a2; /* max */
	T1 a3; /* required */
};

/* Struct for type [detachable] SPECIAL [[attached] XPLAIN_LOGICAL_EXPRESSION] */
struct S208 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_LOGICAL_EXPRESSION] */
struct S209 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] XPLAIN_LOGICAL_INFIX_EXPRESSION */
struct S210 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* operator */
	T0* a2; /* left */
	T0* a3; /* right */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_EXPRESSION] */
struct S211 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] XPLAIN_ATTRIBUTE_NAME_NODE] */
struct S212 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] XPLAIN_NOTNOT_EXPRESSION */
struct S213 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* expression */
};

/* Struct for type [detachable] XPLAIN_NOT_EXPRESSION */
struct S214 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* operand */
	T0* a2; /* operator */
};

/* Struct for type [detachable] XPLAIN_PARAMETER_EXPRESSION */
struct S215 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
};

/* Struct for type [detachable] XPLAIN_EQ_NULL_EXPRESSION */
struct S216 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* expression */
};

/* Struct for type [detachable] XPLAIN_NE_NULL_EXPRESSION */
struct S217 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* expression */
};

/* Struct for type [detachable] XPLAIN_INFIX_EXPRESSION */
struct S218 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* left */
	T0* a2; /* right */
	T0* a3; /* operator */
};

/* Struct for type [detachable] XPLAIN_PARENTHESIS_EXPRESSION */
struct S219 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* expression */
};

/* Struct for type [detachable] SPECIAL [[attached] XPLAIN_LOGICAL_VALUE_EXPRESSION] */
struct S220 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_LOGICAL_VALUE_EXPRESSION] */
struct S221 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] XPLAIN_SYSTEMDATE_EXPRESSION */
struct S222 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] XPLAIN_SYSTEM_EXPRESSION] */
struct S223 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_SYSTEM_EXPRESSION] */
struct S224 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] XPLAIN_LOGINNAME_EXPRESSION */
struct S225 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] XPLAIN_IF_EXPRESSION */
struct S226 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* condition */
	T0* a2; /* then_specification */
	T0* a3; /* else_specification */
};

/* Struct for type [detachable] XPLAIN_PREFIX_EXPRESSION */
struct S227 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* operand */
	T0* a2; /* operator */
};

/* Struct for type [detachable] SPECIAL [detachable XPLAIN_EXPRESSION_NODE] */
struct S228 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] XPLAIN_COMBINE_FUNCTION */
struct S229 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* operands */
};

/* Struct for type [detachable] XPLAIN_DATEF_FUNCTION */
struct S230 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* operand */
};

/* Struct for type [detachable] XPLAIN_INTEGER_FUNCTION */
struct S231 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* operand */
};

/* Struct for type [detachable] XPLAIN_REAL_FUNCTION */
struct S232 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* operand */
};

/* Struct for type [detachable] XPLAIN_STRING_FUNCTION */
struct S233 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* operand */
};

/* Struct for type [detachable] XPLAIN_STRING_EXPRESSION */
struct S234 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* value */
};

/* Struct for type [detachable] XPLAIN_NEWDATE_FUNCTION */
struct S235 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* date */
	T0* a2; /* number */
	T0* a3; /* part */
};

/* Struct for type [detachable] XPLAIN_ID_FUNCTION */
struct S236 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* type */
};

/* Struct for type [detachable] XPLAIN_NUMBER_EXPRESSION */
struct S237 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* value */
};

/* Struct for type [detachable] XPLAIN_UNMANAGED_PARAMETER_EXPRESSION */
struct S238 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
};

/* Struct for type [detachable] XPLAIN_USER_FUNCTION */
struct S239 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* parameters */
	T0* a2; /* name */
};

/* Struct for type [detachable] XPLAIN_SQL_EXPRESSION */
struct S240 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* value */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ATTRIBUTE_NAME_NODE] */
struct S241 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ATTRIBUTE_NAME] */
struct S242 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable XPLAIN_EXPRESSION] */
struct S243 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXPRESSION] */
struct S244 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_REPRESENTATION] */
struct S245 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] XPLAIN_B_REPRESENTATION */
struct S246 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* domain_restriction */
};

/* Struct for type [detachable] XPLAIN_C_REPRESENTATION */
struct S247 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* domain_restriction */
	T6 a2; /* length */
};

/* Struct for type [detachable] XPLAIN_D_REPRESENTATION */
struct S248 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* domain_restriction */
};

/* Struct for type [detachable] XPLAIN_I_REPRESENTATION */
struct S249 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* domain_restriction */
	T6 a2; /* length */
};

/* Struct for type [detachable] XPLAIN_M_REPRESENTATION */
struct S250 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* domain_restriction */
};

/* Struct for type [detachable] XPLAIN_P_REPRESENTATION */
struct S251 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* domain_restriction */
};

/* Struct for type [detachable] XPLAIN_R_REPRESENTATION */
struct S252 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* domain_restriction */
	T6 a2; /* before */
	T6 a3; /* after */
};

/* Struct for type [detachable] XPLAIN_T_REPRESENTATION */
struct S253 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* domain_restriction */
};

/* Struct for type [detachable] XPLAIN_PK_A_REPRESENTATION */
struct S254 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* length */
	T0* a2; /* domain_restriction */
};

/* Struct for type detachable XPLAIN_A_REFERENCES */
struct S255 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* required */
	T0* a2; /* owner */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_PK_REPRESENTATION] */
struct S256 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] XPLAIN_PK_I_REPRESENTATION */
struct S257 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* length */
	T0* a2; /* domain_restriction */
};

/* Struct for type detachable XPLAIN_I_REFERENCES */
struct S258 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* owner */
	T1 a2; /* required */
};

/* Struct for type [detachable] SPECIAL [[attached] XPLAIN_ATTRIBUTE] */
struct S259 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_ATTRIBUTE_NODE] */
struct S260 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ATTRIBUTE] */
struct S262 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] XPLAIN_VARIABLE_EXPRESSION */
struct S263 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* variable */
};

/* Struct for type detachable XPLAIN_VALUE */
struct S264 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
	T0* a2; /* representation */
	T0* a3; /* expression */
};

/* Struct for type [detachable] XPLAIN_VALUE_EXPRESSION */
struct S265 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* value */
};

/* Struct for type [detachable] SPECIAL [detachable XPLAIN_DOMAIN_RESTRICTION] */
struct S266 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_DOMAIN_RESTRICTION] */
struct S267 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable XPLAIN_EXTENSION_EXPRESSION] */
struct S268 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_ASSERTION] */
struct S269 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable XPLAIN_ASSERTION] */
struct S270 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] SPECIAL [detachable XPLAIN_ATTRIBUTE_NAME_NODE] */
struct S271 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type detachable XPLAIN_INDEX */
struct S272 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* is_unique */
	T1 a2; /* is_clustered */
	T0* a3; /* type */
	T0* a4; /* first_attribute */
	T0* a5; /* name */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_ATTRIBUTE_NAME_NODE] */
struct S273 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable XPLAIN_UPDATE_STATEMENT] */
struct S274 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] SPECIAL [detachable XPLAIN_DELETE_STATEMENT] */
struct S275 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] SPECIAL [detachable XPLAIN_CASCADE_STATEMENT] */
struct S276 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] SPECIAL [detachable XPLAIN_SUBJECT] */
struct S277 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] SPECIAL [detachable XPLAIN_ASSIGNMENT_NODE] */
struct S278 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] XPLAIN_INSERT_STATEMENT */
struct S279 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* type */
	T0* a2; /* id */
	T0* a3; /* assignment_list */
};

/* Struct for type [detachable] SPECIAL [detachable XPLAIN_SELECTION_LIST] */
struct S280 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] XPLAIN_GET_INSERT_STATEMENT */
struct S281 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* selection */
	T0* a2; /* insert_type */
	T1 a3; /* auto_primary_key */
	T0* a4; /* assignment_list */
};

/* Struct for type [detachable] SPECIAL [[attached] XPLAIN_ASSIGNMENT] */
struct S283 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_ASSIGNMENT_NODE] */
struct S284 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ASSIGNMENT] */
struct S285 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable XPLAIN_CASCADE_FUNCTION_EXPRESSION] */
struct S286 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_CASCADE_STATEMENT] */
struct S287 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable XPLAIN_SELECTION_FUNCTION] */
struct S288 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_CASCADE_FUNCTION_EXPRESSION] */
struct S289 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_DELETE_STATEMENT] */
struct S290 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SUBJECT] */
struct S291 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_UPDATE_STATEMENT] */
struct S292 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] XPLAIN_VALUE_STATEMENT] */
struct S293 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_VALUE_STATEMENT] */
struct S294 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] XPLAIN_SELECTION_EXPRESSION */
struct S295 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* select_value */
};

/* Struct for type [detachable] XPLAIN_LAST_AUTO_PK_EXPRESSION */
struct S297 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* type */
};

/* Struct for type [detachable] SPECIAL [detachable XPLAIN_VALUE_SELECTION_STATEMENT] */
struct S298 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_VALUE_SELECTION_STATEMENT] */
struct S299 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable XPLAIN_NON_ATTRIBUTE_EXPRESSION] */
struct S300 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_NON_ATTRIBUTE_EXPRESSION] */
struct S301 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] XPLAIN_FUNCTION] */
struct S302 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SELECTION_FUNCTION] */
struct S303 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable XPLAIN_SELECTION] */
struct S304 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] SPECIAL [detachable XPLAIN_GET_STATEMENT] */
struct S305 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_GET_STATEMENT] */
struct S306 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable XPLAIN_EXTENSION] */
struct S307 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] SPECIAL [detachable XPLAIN_EXTEND_STATEMENT] */
struct S308 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXTEND_STATEMENT] */
struct S309 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] XPLAIN_EXTEND_ATTRIBUTE] */
struct S310 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXTENSION] */
struct S311 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_EXTEND_ATTRIBUTE] */
struct S312 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] XPLAIN_EXTENSION_EXPRESSION_EXPRESSION */
struct S313 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* expression */
	T0* a2; /* extension */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXTENSION_EXPRESSION] */
struct S314 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] XPLAIN_EXTENSION_FUNCTION_EXPRESSION */
struct S315 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* selection */
	T0* a2; /* sqlfromaliasname */
	T0* a3; /* per_property */
	T0* a4; /* extension */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SELECTION] */
struct S316 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable XPLAIN_SORT_NODE] */
struct S317 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SELECTION_LIST] */
struct S318 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SORT_NODE] */
struct S319 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] XPLAIN_MAX_FUNCTION */
struct S320 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_FUNCTION] */
struct S321 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] XPLAIN_MIN_FUNCTION */
struct S322 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] XPLAIN_TOTAL_FUNCTION */
struct S323 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] XPLAIN_COUNT_FUNCTION */
struct S324 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] XPLAIN_ANY_FUNCTION */
struct S325 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] XPLAIN_NIL_FUNCTION */
struct S326 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] XPLAIN_SOME_FUNCTION */
struct S327 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXPRESSION_NODE] */
struct S328 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable STRING_8] */
struct S329 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable STRING_8] */
struct S330 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable XPLAIN_PARAMETER_NODE] */
struct S331 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] SPECIAL [detachable XPLAIN_STATEMENT_NODE] */
struct S332 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] XPLAIN_PROCEDURE */
struct S333 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* sql_declare */
	T0* a2; /* name */
	T0* a3; /* parameters */
	T0* a4; /* statements */
	T1 a5; /* is_path_procedure */
	T1 a6; /* is_postgresql_trigger */
	T1 a7; /* recompile */
	T0* a8; /* representation */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_PROCEDURE_STATEMENT] */
struct S334 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_PARAMETER_NODE] */
struct S335 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] XPLAIN_PARAMETER_NODE] */
struct S336 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] SPECIAL [[attached] XPLAIN_PARAMETER] */
struct S337 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_PARAMETER_NODE] */
struct S338 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_PARAMETER] */
struct S339 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_SQL_STATEMENT] */
struct S340 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_STATEMENT_NODE] */
struct S341 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [detachable XPLAIN_IF_STATEMENT] */
struct S342 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_IF_STATEMENT] */
struct S343 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] YY_FILE_BUFFER */
struct S344 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T1 a2; /* filled */
	T0* a3; /* content */
	T6 a4; /* index */
	T6 a5; /* line */
	T6 a6; /* column */
	T6 a7; /* position */
	T1 a8; /* beginning_of_line */
	T1 a9; /* end_of_file */
	T1 a10; /* interactive */
	T0* a11; /* file */
	T6 a12; /* capacity */
};

/* Struct for type [detachable] KL_STDIN_FILE */
struct S345 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* end_of_file */
	T2 a2; /* last_character */
	T0* a3; /* last_string */
	T0* a4; /* character_buffer */
};

/* Struct for type [detachable] XML_WRITER */
struct S347 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* my_xml */
	T0* a2; /* tags */
	T0* a3; /* attributes */
	T0* a4; /* values */
	T1 a5; /* tag_written */
	T1 a6; /* may_close_tag */
	T1 a7; /* tag_closed */
	T1 a8; /* header_written */
	T6 a9; /* attribute_count */
	T6 a10; /* current_attribute */
};

/* Struct for type [detachable] KL_TEXT_OUTPUT_FILE */
struct S348 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* mode */
	T0* a2; /* name */
	T14 a3; /* file_pointer */
	T1 a4; /* descriptor_available */
	T0* a5; /* last_string */
	T0* a6; /* internal_name */
	T0* a7; /* internal_detachable_name_pointer */
};

/* Struct for type [detachable] DS_LINKED_LIST [[attached] XPLAIN_STATEMENT] */
struct S349 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T0* a2; /* first_cell */
	T6 a3; /* count */
	T0* a4; /* last_cell */
};

/* Struct for type [detachable] STD_FILES */
struct S351 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* default_output */
};

/* Struct for type [detachable] JOIN_LIST */
struct S352 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* per_self_join */
	T0* a2; /* my_first */
	T0* a3; /* base_aggregate */
	T0* a4; /* root */
	T1 a5; /* is_finalized */
	T1 a6; /* is_upward_join */
	T1 a7; /* force_left_outer_join */
	T6 a8; /* existential_join_optimisation_count */
};

/* Struct for type detachable JOIN_NODE */
struct S353 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* next */
};

/* Struct for type [detachable] DS_LINKED_LIST [[attached] XPLAIN_PARAMETER] */
struct S358 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* internal_cursor */
	T0* a3; /* last_cell */
	T0* a4; /* first_cell */
};

/* Struct for type [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_PARAMETER] */
struct S359 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* after */
	T0* a2; /* current_cell */
	T0* a3; /* next_cursor */
	T0* a4; /* container */
	T1 a5; /* before */
};

/* Struct for type [detachable] DS_BILINKED_LIST [[attached] XPLAIN_STATEMENT] */
struct S360 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T0* a2; /* last_cell */
	T6 a3; /* count */
	T0* a4; /* first_cell */
};

/* Struct for type [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE] */
struct S361 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* internal_cursor */
	T0* a3; /* first_cell */
	T0* a4; /* last_cell */
	T0* a5; /* equality_tester */
};

/* Struct for type [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_ATTRIBUTE] */
struct S362 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* after */
	T0* a2; /* current_cell */
	T0* a3; /* next_cursor */
	T0* a4; /* container */
	T1 a5; /* before */
};

/* Struct for type [detachable] XPLAIN_B_RESTRICTION */
struct S365 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* required */
};

/* Struct for type [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_SCANNER_STATE] */
struct S367 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* before */
	T0* a2; /* current_cell */
	T0* a3; /* next_cursor */
	T0* a4; /* container */
	T1 a5; /* after */
};

/* Struct for type [detachable] XPLAIN_LIKE_EXPRESSION */
struct S368 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* left */
	T0* a2; /* right */
	T0* a3; /* operator */
};

/* Struct for type [detachable] XPLAIN_NOT_LIKE_EXPRESSION */
struct S369 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* left */
	T0* a2; /* right */
	T0* a3; /* operator */
};

/* Struct for type [detachable] ARRAY [[attached] INTEGER_32] */
struct S370 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [detachable] KL_ARRAY_ROUTINES [[attached] INTEGER_32] */
struct S371 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] CELL [[attached] INTEGER_32] */
struct S373 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* item */
};

/* Struct for type [detachable] KL_NULL_TEXT_OUTPUT_STREAM */
struct S374 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* name */
};

/* Struct for type [detachable] KL_STRING_OUTPUT_STREAM */
struct S376 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* string */
};

/* Struct for type [detachable] DS_CELL [[attached] INTEGER_32] */
struct S378 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* item */
};

/* Struct for type [detachable] ST_BOOLEAN_FORMATTER */
struct S380 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* alignment */
	T2 a2; /* padding_character */
	T1 a3; /* space_sign_enabled */
	T1 a4; /* plus_sign_enabled */
	T6 a5; /* width */
	T6 a6; /* precision */
	T1 a7; /* is_lowercase */
};

/* Struct for type [detachable] ST_CHARACTER_FORMATTER */
struct S381 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* alignment */
	T2 a2; /* padding_character */
	T1 a3; /* space_sign_enabled */
	T1 a4; /* plus_sign_enabled */
	T6 a5; /* width */
	T6 a6; /* precision */
	T1 a7; /* is_lowercase */
};

/* Struct for type [detachable] ST_FLOATING_POINT_FORMATTER */
struct S382 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* double_buffer */
	T0* a2; /* string_output_stream */
	T0* a3; /* unsigned_decimal_formatter */
	T0* a4; /* decimal_formatter */
	T0* a5; /* fractional_part */
	T0* a6; /* fractional_list */
	T6 a7; /* alignment */
	T2 a8; /* padding_character */
	T1 a9; /* space_sign_enabled */
	T1 a10; /* plus_sign_enabled */
	T6 a11; /* width */
	T6 a12; /* precision */
	T1 a13; /* is_lowercase */
	T6 a14; /* fsign */
	T6 a15; /* integer_part */
};

/* Struct for type [detachable] ST_FIXED_POINT_FORMATTER */
struct S383 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* double_buffer */
	T0* a2; /* string_output_stream */
	T0* a3; /* unsigned_decimal_formatter */
	T0* a4; /* decimal_formatter */
	T0* a5; /* fractional_part */
	T0* a6; /* fractional_list */
	T6 a7; /* alignment */
	T2 a8; /* padding_character */
	T1 a9; /* space_sign_enabled */
	T1 a10; /* plus_sign_enabled */
	T6 a11; /* width */
	T6 a12; /* precision */
	T1 a13; /* is_lowercase */
	T6 a14; /* fsign */
	T6 a15; /* integer_part */
};

/* Struct for type [detachable] ST_FIXED_OR_FLOATING_POINT_FORMATTER */
struct S384 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* double_buffer */
	T0* a2; /* string_output_stream */
	T0* a3; /* unsigned_decimal_formatter */
	T0* a4; /* decimal_formatter */
	T0* a5; /* fractional_part */
	T0* a6; /* fractional_list */
	T6 a7; /* alignment */
	T2 a8; /* padding_character */
	T1 a9; /* space_sign_enabled */
	T1 a10; /* plus_sign_enabled */
	T6 a11; /* width */
	T6 a12; /* precision */
	T1 a13; /* is_lowercase */
	T6 a14; /* fsign */
	T6 a15; /* integer_part */
};

/* Struct for type [detachable] ST_DECIMAL_FORMATTER */
struct S385 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* integer_buffer */
	T6 a2; /* alignment */
	T2 a3; /* padding_character */
	T1 a4; /* space_sign_enabled */
	T1 a5; /* plus_sign_enabled */
	T6 a6; /* width */
	T6 a7; /* precision */
	T1 a8; /* is_lowercase */
};

/* Struct for type [detachable] ST_UNSIGNED_OCTAL_FORMATTER */
struct S386 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* integer_buffer */
	T6 a2; /* alignment */
	T2 a3; /* padding_character */
	T1 a4; /* space_sign_enabled */
	T1 a5; /* plus_sign_enabled */
	T6 a6; /* width */
	T6 a7; /* precision */
	T1 a8; /* is_lowercase */
};

/* Struct for type [detachable] ST_UNSIGNED_HEXADECIMAL_FORMATTER */
struct S387 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* integer_buffer */
	T6 a2; /* alignment */
	T2 a3; /* padding_character */
	T1 a4; /* space_sign_enabled */
	T1 a5; /* plus_sign_enabled */
	T6 a6; /* width */
	T6 a7; /* precision */
	T1 a8; /* is_lowercase */
};

/* Struct for type [detachable] ST_UNSIGNED_DECIMAL_FORMATTER */
struct S388 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* integer_buffer */
	T6 a2; /* alignment */
	T2 a3; /* padding_character */
	T1 a4; /* space_sign_enabled */
	T1 a5; /* plus_sign_enabled */
	T6 a6; /* width */
	T6 a7; /* precision */
	T1 a8; /* is_lowercase */
};

/* Struct for type [detachable] ST_POINTER_FORMATTER */
struct S389 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* alignment */
	T2 a2; /* padding_character */
	T1 a3; /* space_sign_enabled */
	T1 a4; /* plus_sign_enabled */
	T6 a5; /* width */
	T6 a6; /* precision */
	T1 a7; /* is_lowercase */
};

/* Struct for type [detachable] ST_STRING_FORMATTER */
struct S390 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* alignment */
	T2 a2; /* padding_character */
	T1 a3; /* space_sign_enabled */
	T1 a4; /* plus_sign_enabled */
	T6 a5; /* width */
	T6 a6; /* precision */
	T1 a7; /* is_lowercase */
};

/* Struct for type [detachable] KL_ANY_ROUTINES */
struct S391 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] UC_STRING */
struct S392 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T6 a2; /* byte_count */
	T6 a3; /* last_byte_index_input */
	T6 a4; /* last_byte_index_result */
	T0* a5; /* area */
	T6 a6; /* internal_hash_code */
	T6 a7; /* internal_case_insensitive_hash_code */
};

/* Struct for type [detachable] SPECIAL [[attached] NATURAL_8] */
struct S394 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T8 z2[1]; /* item */
};

/* Struct for type [detachable] ARRAY [[attached] IMMUTABLE_STRING_32] */
struct S396 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
	T1 a4; /* object_comparison */
};

/* Struct for type [detachable] SPECIAL [[attached] IMMUTABLE_STRING_32] */
struct S397 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type detachable KL_EQUALITY_TESTER [[attached] XPLAIN_VALUE] */
struct S398 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable KL_EQUALITY_TESTER [[attached] STRING_8] */
struct S399 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_VALUE] */
struct S402 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] XPLAIN_VALUE] */
struct S403 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] XPLAIN_VALUE, [attached] STRING_8] */
struct S404 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type detachable DS_LINKABLE [[attached] XPLAIN_SCANNER_STATE] */
struct S406 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
};

/* Struct for type [detachable] XPLAIN_ALL_ATTRIBUTES_CURSOR [[attached] XPLAIN_ATTRIBUTE] */
struct S407 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* after */
	T0* a2; /* current_cell */
	T1 a3; /* generate_assert */
	T1 a4; /* is_first */
	T0* a5; /* container */
	T0* a6; /* next_cursor */
	T1 a7; /* before */
};

/* Struct for type [detachable] XPLAIN_DATA_ATTRIBUTES_CURSOR */
struct S408 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* after */
	T1 a2; /* is_first */
	T0* a3; /* current_cell */
	T0* a4; /* container */
	T0* a5; /* next_cursor */
	T1 a6; /* before */
};

/* Struct for type [detachable] XPLAIN_ATTRIBUTE_EXPRESSION */
struct S410 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* first */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8] */
struct S412 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T6 a2; /* found_position */
	T0* a3; /* item_storage */
	T0* a4; /* equality_tester */
	T0* a5; /* key_equality_tester */
	T6 a6; /* count */
	T6 a7; /* capacity */
	T6 a8; /* slots_position */
	T6 a9; /* free_slot */
	T6 a10; /* last_position */
	T6 a11; /* modulus */
	T6 a12; /* clashes_previous_position */
	T0* a13; /* special_item_routines */
	T0* a14; /* clashes */
	T0* a15; /* slots */
	T0* a16; /* special_key_routines */
	T0* a17; /* key_storage */
	T0* a18; /* internal_keys */
	T0* a19; /* internal_cursor */
	T0* a20; /* hash_function */
};

/* Struct for type [detachable] DS_BILINKED_LIST_CURSOR [[attached] XPLAIN_STATEMENT] */
struct S414 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* before */
	T1 a2; /* after */
	T0* a3; /* current_cell */
	T0* a4; /* next_cursor */
	T0* a5; /* container */
};

/* Struct for type [detachable] DS_LINKED_STACK [[attached] STRING_8] */
struct S415 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* first_cell */
};

/* Struct for type [detachable] ARRAY [detachable STRING_8] */
struct S416 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_STATEMENT] */
struct S418 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* after */
	T0* a2; /* current_cell */
	T0* a3; /* next_cursor */
	T0* a4; /* container */
	T1 a5; /* before */
};

/* Struct for type detachable DS_LINKABLE [[attached] XPLAIN_STATEMENT] */
struct S419 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
};

/* Struct for type [detachable] STRING_TO_INTEGER_CONVERTOR */
struct S420 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* sign */
	T11 a2; /* part1 */
	T11 a3; /* part2 */
	T6 a4; /* last_state */
	T1 a5; /* internal_overflowed */
	T0* a6; /* leading_separators */
	T0* a7; /* trailing_separators */
	T1 a8; /* leading_separators_acceptable */
	T1 a9; /* trailing_separators_acceptable */
	T6 a10; /* conversion_type */
};

/* Struct for type [detachable] JOIN */
struct S423 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* aggregate_attribute */
	T0* a2; /* attribute_alias_name */
	T1 a3; /* is_upward_join */
	T0* a4; /* aggregate_fk */
	T0* a5; /* aggregate_alias_name */
	T0* a6; /* aggregate */
	T1 a7; /* is_forced_inner_join */
	T1 a8; /* is_forced_left_outer_join */
	T0* a9; /* actual_attribute */
};

/* Struct for type [detachable] DS_HASH_SET [[attached] STRING_8] */
struct S424 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* equality_tester */
	T6 a3; /* count */
	T6 a4; /* capacity */
	T6 a5; /* slots_position */
	T6 a6; /* free_slot */
	T6 a7; /* last_position */
	T6 a8; /* modulus */
	T6 a9; /* clashes_previous_position */
	T6 a10; /* found_position */
	T0* a11; /* special_item_routines */
	T0* a12; /* item_storage */
	T0* a13; /* clashes */
	T0* a14; /* slots */
	T0* a15; /* internal_cursor */
	T0* a16; /* hash_function */
};

/* Struct for type [detachable] XPLAIN_F_REPRESENTATION */
struct S425 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* domain_restriction */
};

/* Struct for type [detachable] EXCEPTIONS */
struct S427 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] XPLAIN_INIT_ATTRIBUTES_CURSOR [[attached] XPLAIN_ATTRIBUTE] */
struct S430 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* after */
	T0* a2; /* current_cell */
	T1 a3; /* is_first */
	T0* a4; /* container */
	T0* a5; /* next_cursor */
	T1 a6; /* before */
};

/* Struct for type [detachable] KL_INTEGER_ROUTINES */
struct S432 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] ARRAY [[attached] DS_CELL [[attached] INTEGER_32]] */
struct S433 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* area */
	T6 a2; /* lower */
	T6 a3; /* upper */
};

/* Struct for type [detachable] SPECIAL [[attached] DS_CELL [[attached] INTEGER_32]] */
struct S434 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] DS_CELL [[attached] REAL_64] */
struct S435 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T13 a1; /* item */
};

/* Struct for type [detachable] KL_CHARACTER_BUFFER */
struct S436 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* as_special */
	T0* a2; /* area */
};

/* Struct for type [detachable] CONSOLE */
struct S437 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T2 a1; /* last_character */
	T14 a2; /* file_pointer */
	T6 a3; /* mode */
	T0* a4; /* last_string */
	T0* a5; /* internal_name */
	T0* a6; /* internal_detachable_name_pointer */
};

/* Struct for type [detachable] JOIN_TREE_NODE */
struct S438 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* found_immediate_child */
	T0* a2; /* next */
	T0* a3; /* item */
	T1 a4; /* is_upward_join */
	T1 a5; /* is_forced_left_outer_join */
	T1 a6; /* is_forced_inner_join */
	T0* a7; /* other_items */
};

/* Struct for type [detachable] DS_LINKED_LIST [[attached] JOIN_TREE_NODE] */
struct S439 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* last_cell */
	T0* a2; /* internal_cursor */
	T6 a3; /* count */
	T0* a4; /* first_cell */
};

/* Struct for type detachable DS_LINKABLE [[attached] XPLAIN_PARAMETER] */
struct S440 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
};

/* Struct for type detachable DS_BILINKABLE [[attached] XPLAIN_STATEMENT] */
struct S441 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
	T0* a3; /* left */
};

/* Struct for type detachable DS_LINKABLE [[attached] XPLAIN_ATTRIBUTE] */
struct S442 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
};

/* Struct for type [detachable] DS_ARRAYED_LIST [[attached] INTEGER_32] */
struct S445 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* count */
	T0* a2; /* storage */
	T0* a3; /* special_routines */
	T6 a4; /* capacity */
	T0* a5; /* internal_cursor */
};

/* Struct for type [detachable] KL_DOUBLE_ROUTINES */
struct S446 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_PLATFORM */
struct S447 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] XPLAIN_ABSTRACT_OBJECT] */
struct S450 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type detachable KL_EQUALITY_TESTER [[attached] XPLAIN_ABSTRACT_OBJECT] */
struct S451 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ABSTRACT_OBJECT] */
struct S452 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_HASH_TABLE_CURSOR [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8] */
struct S454 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* position */
	T0* a2; /* next_cursor */
	T0* a3; /* container */
};

/* Struct for type detachable DS_LINKABLE [[attached] STRING_8] */
struct S456 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
};

/* Struct for type [detachable] HEXADECIMAL_STRING_TO_INTEGER_CONVERTER */
struct S459 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* last_state */
	T1 a2; /* internal_overflowed */
	T11 a3; /* part1 */
	T11 a4; /* part2 */
	T0* a5; /* leading_separators */
	T0* a6; /* trailing_separators */
	T6 a7; /* sign */
	T6 a8; /* conversion_type */
	T2 a9; /* internal_lookahead */
	T1 a10; /* leading_separators_acceptable */
	T1 a11; /* trailing_separators_acceptable */
};

/* Struct for type [detachable] INTEGER_OVERFLOW_CHECKER */
struct S460 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* integer_overflow_state1 */
	T0* a2; /* integer_overflow_state2 */
	T0* a3; /* natural_overflow_state1 */
	T0* a4; /* natural_overflow_state2 */
};

/* Struct for type [detachable] DS_HASH_SET_CURSOR [[attached] STRING_8] */
struct S461 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* container */
	T6 a2; /* position */
};

/* Struct for type [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE] */
struct S462 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T6 a1; /* found_position */
	T0* a2; /* item_storage */
	T0* a3; /* key_equality_tester */
	T6 a4; /* count */
	T6 a5; /* last_position */
	T6 a6; /* free_slot */
	T6 a7; /* position */
	T6 a8; /* slots_position */
	T0* a9; /* equality_tester */
	T0* a10; /* internal_cursor */
	T0* a11; /* key_storage */
	T0* a12; /* clashes */
	T6 a13; /* modulus */
	T0* a14; /* slots */
	T6 a15; /* clashes_previous_position */
	T0* a16; /* special_item_routines */
	T0* a17; /* special_key_routines */
	T6 a18; /* capacity */
	T0* a19; /* hash_function */
};

/* Struct for type detachable DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE_NAME_NODE] */
struct S463 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* internal_cursor */
	T0* a2; /* last_cell */
	T6 a3; /* count */
	T0* a4; /* first_cell */
};

/* Struct for type detachable DS_LINKABLE [[attached] JOIN_TREE_NODE] */
struct S464 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
};

/* Struct for type [detachable] DS_LINKED_LIST_CURSOR [[attached] JOIN_TREE_NODE] */
struct S465 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* after */
	T0* a2; /* current_cell */
	T0* a3; /* next_cursor */
	T0* a4; /* container */
	T1 a5; /* before */
};

/* Struct for type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] INTEGER_32] */
struct S466 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [detachable] XPLAIN_VIRTUAL_ATTRIBUTE */
struct S468 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* role */
	T0* a2; /* init */
	T1 a3; /* is_init_default */
	T1 a4; /* is_specialization */
	T1 a5; /* is_unique */
	T1 a6; /* is_required */
	T1 a7; /* overrule_required */
	T0* a8; /* internal_abstracttype */
};

/* Struct for type [detachable] XPLAIN_EXTENSION_ATTRIBUTE */
struct S469 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* role */
	T0* a2; /* init */
	T1 a3; /* is_init_default */
	T1 a4; /* is_specialization */
	T1 a5; /* is_unique */
	T1 a6; /* is_required */
	T1 a7; /* overrule_required */
	T0* a8; /* internal_extension */
};

/* Struct for type [detachable] XPLAIN_ASSERTION_EXPRESSION */
struct S471 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* anode */
	T0* a2; /* assertion */
};

/* Struct for type [detachable] UC_UTF8_ROUTINES */
struct S473 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] SPECIAL [[attached] NATURAL_64] */
struct S475 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T11 z2[1]; /* item */
};

/* Struct for type detachable KL_EQUALITY_TESTER [[attached] XPLAIN_TYPE] */
struct S476 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type detachable DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] XPLAIN_TYPE] */
struct S478 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* next_cursor */
	T0* a2; /* container */
	T6 a3; /* position */
};

/* Struct for type [detachable] SPECIAL [[attached] XPLAIN_TYPE] */
struct S479 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	uint32_t offset;
	T6 a1; /* count */
	T6 a2; /* capacity */
	T0* z2[1]; /* item */
};

/* Struct for type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_TYPE] */
struct S480 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
};

/* Struct for type [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_ATTRIBUTE_NAME_NODE] */
struct S481 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T1 a1; /* after */
	T0* a2; /* current_cell */
	T0* a3; /* next_cursor */
	T0* a4; /* container */
	T1 a5; /* before */
};

/* Struct for type detachable DS_LINKABLE [[attached] XPLAIN_ATTRIBUTE_NAME_NODE] */
struct S482 {
	EIF_TYPE_INDEX id;
	uint16_t flags;
	T0* a1; /* item */
	T0* a2; /* right */
};


extern T1 GE_default1;
extern T2 GE_default2;
extern T3 GE_default3;
extern T4 GE_default4;
extern T5 GE_default5;
extern T6 GE_default6;
extern T7 GE_default7;
extern T8 GE_default8;
extern T9 GE_default9;
extern T10 GE_default10;
extern T11 GE_default11;
extern T12 GE_default12;
extern T13 GE_default13;
extern T14 GE_default14;
extern T15 GE_default15;
extern T16 GE_default16;
extern T17 GE_default17;
extern T20 GE_default20;
extern T21 GE_default21;
extern T25 GE_default25;
extern T26 GE_default26;
extern T27 GE_default27;
extern T28 GE_default28;
extern T29 GE_default29;
extern T30 GE_default30;
extern T31 GE_default31;
extern T32 GE_default32;
extern T35 GE_default35;
extern T36 GE_default36;
extern T37 GE_default37;
extern T38 GE_default38;
extern T39 GE_default39;
extern T40 GE_default40;
extern T41 GE_default41;
extern T42 GE_default42;
extern T43 GE_default43;
extern T44 GE_default44;
extern T45 GE_default45;
extern T46 GE_default46;
extern T47 GE_default47;
extern T48 GE_default48;
extern T49 GE_default49;
extern T50 GE_default50;
extern T51 GE_default51;
extern T52 GE_default52;
extern T53 GE_default53;
extern T54 GE_default54;
extern T55 GE_default55;
extern T56 GE_default56;
extern T57 GE_default57;
extern T58 GE_default58;
extern T59 GE_default59;
extern T60 GE_default60;
extern T61 GE_default61;
extern T62 GE_default62;
extern T63 GE_default63;
extern T64 GE_default64;
extern T65 GE_default65;
extern T66 GE_default66;
extern T67 GE_default67;
extern T69 GE_default69;
extern T71 GE_default71;
extern T74 GE_default74;
extern T76 GE_default76;
extern T78 GE_default78;
extern T80 GE_default80;
extern T81 GE_default81;
extern T82 GE_default82;
extern T83 GE_default83;
extern T84 GE_default84;
extern T85 GE_default85;
extern T86 GE_default86;
extern T88 GE_default88;
extern T89 GE_default89;
extern T90 GE_default90;
extern T91 GE_default91;
extern T92 GE_default92;
extern T93 GE_default93;
extern T94 GE_default94;
extern T95 GE_default95;
extern T96 GE_default96;
extern T97 GE_default97;
extern T98 GE_default98;
extern T99 GE_default99;
extern T100 GE_default100;
extern T101 GE_default101;
extern T102 GE_default102;
extern T103 GE_default103;
extern T104 GE_default104;
extern T105 GE_default105;
extern T106 GE_default106;
extern T107 GE_default107;
extern T108 GE_default108;
extern T110 GE_default110;
extern T113 GE_default113;
extern T114 GE_default114;
extern T115 GE_default115;
extern T116 GE_default116;
extern T118 GE_default118;
extern T119 GE_default119;
extern T121 GE_default121;
extern T123 GE_default123;
extern T126 GE_default126;
extern T127 GE_default127;
extern T128 GE_default128;
extern T130 GE_default130;
extern T131 GE_default131;
extern T132 GE_default132;
extern T133 GE_default133;
extern T134 GE_default134;
extern T135 GE_default135;
extern T136 GE_default136;
extern T138 GE_default138;
extern T140 GE_default140;
extern T141 GE_default141;
extern T144 GE_default144;
extern T145 GE_default145;
extern T146 GE_default146;
extern T147 GE_default147;
extern T148 GE_default148;
extern T149 GE_default149;
extern T150 GE_default150;
extern T151 GE_default151;
extern T152 GE_default152;
extern T153 GE_default153;
extern T154 GE_default154;
extern T155 GE_default155;
extern T157 GE_default157;
extern T158 GE_default158;
extern T159 GE_default159;
extern T160 GE_default160;
extern T161 GE_default161;
extern T162 GE_default162;
extern T164 GE_default164;
extern T165 GE_default165;
extern T167 GE_default167;
extern T168 GE_default168;
extern T169 GE_default169;
extern T170 GE_default170;
extern T171 GE_default171;
extern T172 GE_default172;
extern T173 GE_default173;
extern T174 GE_default174;
extern T175 GE_default175;
extern T176 GE_default176;
extern T177 GE_default177;
extern T178 GE_default178;
extern T179 GE_default179;
extern T180 GE_default180;
extern T181 GE_default181;
extern T182 GE_default182;
extern T183 GE_default183;
extern T184 GE_default184;
extern T185 GE_default185;
extern T186 GE_default186;
extern T187 GE_default187;
extern T188 GE_default188;
extern T189 GE_default189;
extern T190 GE_default190;
extern T191 GE_default191;
extern T192 GE_default192;
extern T193 GE_default193;
extern T194 GE_default194;
extern T196 GE_default196;
extern T197 GE_default197;
extern T198 GE_default198;
extern T199 GE_default199;
extern T200 GE_default200;
extern T201 GE_default201;
extern T202 GE_default202;
extern T203 GE_default203;
extern T204 GE_default204;
extern T206 GE_default206;
extern T207 GE_default207;
extern T208 GE_default208;
extern T209 GE_default209;
extern T210 GE_default210;
extern T211 GE_default211;
extern T212 GE_default212;
extern T213 GE_default213;
extern T214 GE_default214;
extern T215 GE_default215;
extern T216 GE_default216;
extern T217 GE_default217;
extern T218 GE_default218;
extern T219 GE_default219;
extern T220 GE_default220;
extern T221 GE_default221;
extern T222 GE_default222;
extern T223 GE_default223;
extern T224 GE_default224;
extern T225 GE_default225;
extern T226 GE_default226;
extern T227 GE_default227;
extern T228 GE_default228;
extern T229 GE_default229;
extern T230 GE_default230;
extern T231 GE_default231;
extern T232 GE_default232;
extern T233 GE_default233;
extern T234 GE_default234;
extern T235 GE_default235;
extern T236 GE_default236;
extern T237 GE_default237;
extern T238 GE_default238;
extern T239 GE_default239;
extern T240 GE_default240;
extern T241 GE_default241;
extern T242 GE_default242;
extern T243 GE_default243;
extern T244 GE_default244;
extern T245 GE_default245;
extern T246 GE_default246;
extern T247 GE_default247;
extern T248 GE_default248;
extern T249 GE_default249;
extern T250 GE_default250;
extern T251 GE_default251;
extern T252 GE_default252;
extern T253 GE_default253;
extern T254 GE_default254;
extern T255 GE_default255;
extern T256 GE_default256;
extern T257 GE_default257;
extern T258 GE_default258;
extern T259 GE_default259;
extern T260 GE_default260;
extern T262 GE_default262;
extern T263 GE_default263;
extern T264 GE_default264;
extern T265 GE_default265;
extern T266 GE_default266;
extern T267 GE_default267;
extern T268 GE_default268;
extern T269 GE_default269;
extern T270 GE_default270;
extern T271 GE_default271;
extern T272 GE_default272;
extern T273 GE_default273;
extern T274 GE_default274;
extern T275 GE_default275;
extern T276 GE_default276;
extern T277 GE_default277;
extern T278 GE_default278;
extern T279 GE_default279;
extern T280 GE_default280;
extern T281 GE_default281;
extern T283 GE_default283;
extern T284 GE_default284;
extern T285 GE_default285;
extern T286 GE_default286;
extern T287 GE_default287;
extern T288 GE_default288;
extern T289 GE_default289;
extern T290 GE_default290;
extern T291 GE_default291;
extern T292 GE_default292;
extern T293 GE_default293;
extern T294 GE_default294;
extern T295 GE_default295;
extern T297 GE_default297;
extern T298 GE_default298;
extern T299 GE_default299;
extern T300 GE_default300;
extern T301 GE_default301;
extern T302 GE_default302;
extern T303 GE_default303;
extern T304 GE_default304;
extern T305 GE_default305;
extern T306 GE_default306;
extern T307 GE_default307;
extern T308 GE_default308;
extern T309 GE_default309;
extern T310 GE_default310;
extern T311 GE_default311;
extern T312 GE_default312;
extern T313 GE_default313;
extern T314 GE_default314;
extern T315 GE_default315;
extern T316 GE_default316;
extern T317 GE_default317;
extern T318 GE_default318;
extern T319 GE_default319;
extern T320 GE_default320;
extern T321 GE_default321;
extern T322 GE_default322;
extern T323 GE_default323;
extern T324 GE_default324;
extern T325 GE_default325;
extern T326 GE_default326;
extern T327 GE_default327;
extern T328 GE_default328;
extern T329 GE_default329;
extern T330 GE_default330;
extern T331 GE_default331;
extern T332 GE_default332;
extern T333 GE_default333;
extern T334 GE_default334;
extern T335 GE_default335;
extern T336 GE_default336;
extern T337 GE_default337;
extern T338 GE_default338;
extern T339 GE_default339;
extern T340 GE_default340;
extern T341 GE_default341;
extern T342 GE_default342;
extern T343 GE_default343;
extern T344 GE_default344;
extern T345 GE_default345;
extern T347 GE_default347;
extern T348 GE_default348;
extern T349 GE_default349;
extern T351 GE_default351;
extern T352 GE_default352;
extern T353 GE_default353;
extern T358 GE_default358;
extern T359 GE_default359;
extern T360 GE_default360;
extern T361 GE_default361;
extern T362 GE_default362;
extern T365 GE_default365;
extern T367 GE_default367;
extern T368 GE_default368;
extern T369 GE_default369;
extern T370 GE_default370;
extern T371 GE_default371;
extern T372 GE_default372;
extern T373 GE_default373;
extern T374 GE_default374;
extern T376 GE_default376;
extern T378 GE_default378;
extern T380 GE_default380;
extern T381 GE_default381;
extern T382 GE_default382;
extern T383 GE_default383;
extern T384 GE_default384;
extern T385 GE_default385;
extern T386 GE_default386;
extern T387 GE_default387;
extern T388 GE_default388;
extern T389 GE_default389;
extern T390 GE_default390;
extern T391 GE_default391;
extern T392 GE_default392;
extern T394 GE_default394;
extern T395 GE_default395;
extern T396 GE_default396;
extern T397 GE_default397;
extern T398 GE_default398;
extern T399 GE_default399;
extern T402 GE_default402;
extern T403 GE_default403;
extern T404 GE_default404;
extern T406 GE_default406;
extern T407 GE_default407;
extern T408 GE_default408;
extern T410 GE_default410;
extern T412 GE_default412;
extern T414 GE_default414;
extern T415 GE_default415;
extern T416 GE_default416;
extern T417 GE_default417;
extern T418 GE_default418;
extern T419 GE_default419;
extern T420 GE_default420;
extern T423 GE_default423;
extern T424 GE_default424;
extern T425 GE_default425;
extern T427 GE_default427;
extern T428 GE_default428;
extern T430 GE_default430;
extern T432 GE_default432;
extern T433 GE_default433;
extern T434 GE_default434;
extern T435 GE_default435;
extern T436 GE_default436;
extern T437 GE_default437;
extern T438 GE_default438;
extern T439 GE_default439;
extern T440 GE_default440;
extern T441 GE_default441;
extern T442 GE_default442;
extern T445 GE_default445;
extern T446 GE_default446;
extern T447 GE_default447;
extern T450 GE_default450;
extern T451 GE_default451;
extern T452 GE_default452;
extern T454 GE_default454;
extern T456 GE_default456;
extern T458 GE_default458;
extern T459 GE_default459;
extern T460 GE_default460;
extern T461 GE_default461;
extern T462 GE_default462;
extern T463 GE_default463;
extern T464 GE_default464;
extern T465 GE_default465;
extern T466 GE_default466;
extern T468 GE_default468;
extern T469 GE_default469;
extern T471 GE_default471;
extern T473 GE_default473;
extern T475 GE_default475;
extern T476 GE_default476;
extern T478 GE_default478;
extern T479 GE_default479;
extern T480 GE_default480;
extern T481 GE_default481;
extern T482 GE_default482;

extern T0* GE_new_str8(T6 c);
extern T0* GE_new_str32(T6 c);
extern T0* GE_new_istr8(T6 c);
extern T0* GE_new_istr32(T6 c);
/* Call to [detachable] ANY.out */
extern T0* T23x3555346(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.quote_identifier */
extern T0* T33x15106223T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.calculatedcolumnssupported */
extern T1 T33x16859140(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.sql_cast_to_formatted_date */
extern T0* T33x15106255T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.sqlcoalesce */
extern T0* T33x15106144(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.sqlfalse */
extern T0* T33x16859158(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.sqlcolumnconstraint_base */
extern T0* T33x15106237T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to detachable SQL_GENERATOR.sqlcolumnconstraint_type */
extern T0* T33x15106238T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to detachable SQL_GENERATOR.sql_subselect_for_extension */
extern T0* T33x15106212T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.inupdatestatement */
extern T1 T33x15106149(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.sqlcolumndefault_type */
extern T0* T33x15106175T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.sqlcolumnrequired_type */
extern T0* T33x15106177T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.sqlcolumnidentifier_type */
extern T0* T33x15106170T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to detachable SQL_GENERATOR.sqlcolumnrequired_base */
extern T0* T33x15106176T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.sqlcolumndefault_base */
extern T0* T33x15106174T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.sqlcolumnidentifier_base */
extern T0* T33x15106169T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to detachable SQL_GENERATOR.make_valid_identifier */
extern T0* T33x15106221T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.sqlminint */
extern T0* T33x15106141(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.sqlmaxint */
extern T0* T33x15106142(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.sql_infix_expression_as_wildcard */
extern T0* T33x15106192T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* Call to detachable SQL_GENERATOR.value_representation_integer */
extern T0* T33x15106250T6(GE_context* ac, T0* C, T6 a1);
/* Call to detachable SQL_GENERATOR.value_representation_char */
extern T0* T33x15106246T6(GE_context* ac, T0* C, T6 a1);
/* Call to detachable SQL_GENERATOR.value_representation_text */
extern T0* T33x15106253(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.value_representation_float */
extern T0* T33x15106249(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.value_representation_money */
extern T0* T33x15106251(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.value_representation_date */
extern T0* T33x15106248(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.as_string */
extern T0* T33x16859159T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.minimum_date_value */
extern T0* T33x15106136(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.value_representation_boolean */
extern T0* T33x15106245(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.sql_combine_expression_as_wildcard */
extern T0* T33x15106194T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.constant_identifier */
extern T0* T33x15106240T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.sp_define_param_name */
extern T0* T33x15106214T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.storedproceduresupported */
extern T1 T33x15106118(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.quote_valid_identifier */
extern T0* T33x15106224T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.supportsdomainsinstoredprocedures */
extern T1 T33x15106117(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.sp_define_in_param */
extern T0* T33x15106213T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.namedparameterssupported */
extern T1 T33x15106116(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.returnvaluesupported */
extern T1 T33x15106119(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.sp_update_name */
extern T0* T33x15106219T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.sp_insert_name */
extern T0* T33x15106216T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.sp_delete_name */
extern T0* T33x15106215T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.createstoredprocedure */
extern T1 T33x15106115(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.sql_subselect_for_attribute */
extern T0* T33x15106211T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.updated_extension */
extern T0* T33x15106267(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.sql_like_operator */
extern T0* T33x15106140(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.sqlcheck_boolean */
extern T0* T33x15106162T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to detachable SQL_GENERATOR.sqlfunction_some */
extern T0* T33x15106185(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.sqlfunction_nil */
extern T0* T33x15106184(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.existentialfromneeded */
extern T1 T33x15106125(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.sqlfunction_any */
extern T0* T33x15106180(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.sqlfunction_count */
extern T0* T33x15106181(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.sqlfunction_total */
extern T0* T33x15106186(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.sqlfunction_min */
extern T0* T33x15106183(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.sqlfunction_max */
extern T0* T33x15106182(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.sql_select_for_extension_expression */
extern T0* T33x15106199T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.value_representation_system_user */
extern T0* T33x15106252(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.value_representation_current_timestamp */
extern T0* T33x15106247(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.sqlsysfunction_current_timestamp */
extern T0* T33x15106187(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.sql_infix_expression */
extern T0* T33x15106191T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* Call to detachable SQL_GENERATOR.sqlcheck_between */
extern T0* T33x15106161T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to detachable SQL_GENERATOR.sqlcheck_in */
extern T0* T33x15106163T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to detachable SQL_GENERATOR.sqlcheck_like */
extern T0* T33x15106164T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to detachable SQL_GENERATOR.create_table_name */
extern T0* T33x15106230T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.temporarytableprefix */
extern T0* T33x15106100(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.extension_name */
extern T0* T33x15106266T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.sql_select_function_as_subselect */
extern T0* T33x15106203T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.assert_implemented_as_view */
extern T1 T33x15106268T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.sqltrue */
extern T0* T33x16859157(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.sql_expression_as_boolean_value */
extern T0* T33x15106189T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.sp_name */
extern T0* T33x15106217T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.sql_select_for_extension_function */
extern T0* T33x15106200T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to detachable SQL_GENERATOR.sql_extension_function */
extern T0* T33x15106190T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.sql_last_auto_generated_primary_key */
extern T0* T33x15106198T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.sqlgetvalue */
extern T0* T33x15106166T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.quoted_value_identifier */
extern T0* T33x15106244T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.value_identifier */
extern T0* T33x15106243T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.sqlgetconstant */
extern T0* T33x16859160T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.sp_use_param */
extern T0* T33x15106220T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.sql_newdate */
extern T0* T33x15106264T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* Call to detachable SQL_GENERATOR.sql_cast_to_string */
extern T0* T33x15106259T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.sql_cast_to_real */
extern T0* T33x15106258T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.sql_cast_to_integer */
extern T0* T33x15106257T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.sql_cast_to_date */
extern T0* T33x15106254T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.sql_combine_expression */
extern T0* T33x15106193T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.sqlsysfunction_system_user */
extern T0* T33x15106188(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.sql_not_expression */
extern T0* T33x15106195T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.sql_notnot_expression */
extern T0* T33x15106196T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.assertion_name */
extern T0* T33x15106269T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.table_name */
extern T0* T33x15106231T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.table_has_auto_pk */
extern T1 T33x15106233T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.table_pk_name */
extern T0* T33x15106232T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.createautoprimarykey */
extern T1 T33x15106073(GE_context* ac, T0* C);
/* Call to detachable SQL_GENERATOR.check_if_required */
extern T1 T33x15106228T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.target_name */
extern T0* T33x15106049(GE_context* ac, T0* C);
/* Call to [detachable] ARRAY [[attached] ANY].item */
extern T0* T40x245761T6(GE_context* ac, T0* C, T6 a1);
/* Call to [detachable] ARRAY [[attached] ANY].upper */
extern T6 T40x229380(GE_context* ac, T0* C);
/* Call to [detachable] ARRAY [[attached] ANY].lower */
extern T6 T40x229379(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_DOMAIN_RESTRICTION.required */
extern T1 T122x14696449(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_DOMAIN_RESTRICTION.sqldomainconstraint */
extern T0* T122x14696450T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to [detachable] XPLAIN_DOMAIN_RESTRICTION.sqlcolumnconstraint */
extern T0* T122x14696451T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to detachable XPLAIN_STATEMENT.uses_parameter */
extern T1 T129x15089665T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable XPLAIN_STATEMENT.updates_attribute */
extern T1 T129x15089666T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to detachable XPLAIN_TYPE.hash_code */
extern T6 T132x3047425(GE_context* ac, T0* C);
/* Call to detachable XPLAIN_TYPE.sqlpkname */
extern T0* T132x14761992T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable XPLAIN_TYPE.quoted_name */
extern T0* T132x14942215T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable XPLAIN_TYPE.sqlname */
extern T0* T132x14942214T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable XPLAIN_TYPE.sqlcolumnidentifier */
extern T0* T132x14958595T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to detachable XPLAIN_TYPE.name */
extern T0* T132x14942210(GE_context* ac, T0* C);
/* Call to detachable XPLAIN_TYPE.q_sqlpkname */
extern T0* T132x14761993T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable XPLAIN_TYPE.find_attribute */
extern T0* T132x14762003T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] XPLAIN_EXPRESSION.column_name */
extern T0* T137x17317903(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_EXPRESSION.path_name */
extern T0* T137x17317904(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_EXPRESSION.uses_parameter */
extern T1 T137x17317901T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] XPLAIN_EXPRESSION.sqlinitvalue */
extern T0* T137x17317906T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] XPLAIN_EXPRESSION.is_literal */
extern T1 T137x17317892(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_EXPRESSION.is_constant */
extern T1 T137x17317891(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_EXPRESSION.sqlvalue */
extern T0* T137x17317910T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] XPLAIN_EXPRESSION.exact_representation */
extern T0* T137x17317902T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] XPLAIN_EXPRESSION.sqlname */
extern T0* T137x17317909T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] XPLAIN_EXPRESSION.representation */
extern T0* T137x17317905T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] XPLAIN_EXPRESSION.can_be_null */
extern T1 T137x17317889(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_EXPRESSION.is_specialization */
extern T1 T137x17317895(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_EXPRESSION.sqlvalue_as_wildcard */
extern T0* T137x17317911T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] XPLAIN_EXPRESSION.is_logical_constant */
extern T1 T137x17317893(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_EXPRESSION.uses_its */
extern T1 T137x17317900(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_EXPRESSION.is_using_other_attributes */
extern T1 T137x17317898T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] XPLAIN_EXPRESSION.has_wild_card_characters */
extern T1 T137x17317890(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_EXPRESSION.sqlminvalue */
extern T0* T137x17317908T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] XPLAIN_EXPRESSION.selects_max_one_row */
extern T1 T137x17317899(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_EXPRESSION.is_logical_expression */
extern T1 T137x17317894(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_EXPRESSION.outer_sqlvalue */
extern T0* T137x17317912T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] XPLAIN_EXPRESSION.sql_alias */
extern T0* T137x17317913T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] XPLAIN_EXPRESSION.is_string */
extern T1 T137x17317897(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_EXPRESSION.sqlmaxvalue */
extern T0* T137x17317907T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] XPLAIN_REPRESENTATION.max_value */
extern T0* T142x16695307T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] XPLAIN_REPRESENTATION.min_value */
extern T0* T142x16695308T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] XPLAIN_REPRESENTATION.is_date */
extern T1 T142x16695299(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_REPRESENTATION.domain_restriction */
extern T0* T142x16695302(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_REPRESENTATION.datatype */
extern T0* T142x16695305T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] XPLAIN_REPRESENTATION.xml_schema_data_type */
extern T0* T142x16695304(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_REPRESENTATION.domain */
extern T0* T142x16695301(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_REPRESENTATION.value_representation */
extern T0* T142x16695303T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] XPLAIN_REPRESENTATION.default_value */
extern T0* T142x16695306(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_REPRESENTATION.is_blob */
extern T1 T142x16695298T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] XPLAIN_ATTRIBUTE.abstracttype */
extern T0* T145x15843338(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_ATTRIBUTE.is_extension */
extern T1 T145x15843332(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_ATTRIBUTE.is_init_default */
extern T1 T145x15843337(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_ATTRIBUTE.init */
extern T0* T145x15843340(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_ATTRIBUTE.is_required */
extern T1 T145x15843334(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_ATTRIBUTE.overrule_required */
extern T1 T145x15843343(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_ATTRIBUTE.q_sql_select_name */
extern T0* T145x15843344T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] XPLAIN_ATTRIBUTE.full_name */
extern T0* T145x15843339(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_ATTRIBUTE.is_assertion */
extern T1 T145x15843330(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_ATTRIBUTE.is_specialization */
extern T1 T145x15843335(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_ATTRIBUTE.role */
extern T0* T145x15843342(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_ATTRIBUTE.name */
extern T0* T145x15843341(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_ATTRIBUTE.is_unique */
extern T1 T145x15843336(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_ATTRIBUTE.sql_select_name */
extern T0* T145x15843345T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] XPLAIN_ATTRIBUTE.is_logical_extension */
extern T1 T145x15843333(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_ATTRIBUTE.is_date */
extern T1 T145x15843331(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_FUNCTION.is_nil */
extern T1 T166x14974977(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_FUNCTION.sqlextenddefault */
extern T0* T166x14974985T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to [detachable] XPLAIN_FUNCTION.name */
extern T0* T166x14974980(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_FUNCTION.sp_function_type */
extern T0* T166x14974986T0T0T1(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3);
/* Call to [detachable] XPLAIN_FUNCTION.representation */
extern T0* T166x14974984T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* Call to [detachable] XPLAIN_FUNCTION.is_some */
extern T1 T166x14974978(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_FUNCTION.needs_limit */
extern T1 T166x14974983T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] XPLAIN_FUNCTION.is_existential */
extern T1 T166x14974988(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_FUNCTION.needs_distinct */
extern T1 T166x14974982(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_FUNCTION.sqlfunction */
extern T0* T166x14974987T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] XPLAIN_FUNCTION.needs_coalesce */
extern T1 T166x14974981(GE_context* ac, T0* C);
/* Call to [detachable] XPLAIN_FUNCTION.property_required */
extern T6 T166x14974979(GE_context* ac, T0* C);
/* Call to detachable XPLAIN_ABSTRACT_OBJECT.representation */
extern T0* T195x14942211(GE_context* ac, T0* C);
/* Call to detachable XPLAIN_ABSTRACT_OBJECT.name */
extern T0* T195x14942210(GE_context* ac, T0* C);
/* Call to detachable XPLAIN_ABSTRACT_OBJECT.may_be_redefined */
extern T1 T195x14942212(GE_context* ac, T0* C);
/* Call to detachable XPLAIN_ABSTRACT_OBJECT.create_expression */
extern T0* T195x14942213T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable XPLAIN_ABSTRACT_TYPE.representation */
extern T0* T261x14942211(GE_context* ac, T0* C);
/* Call to detachable XPLAIN_ABSTRACT_TYPE.sqlcolumnidentifier */
extern T0* T261x14958595T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to detachable XPLAIN_ABSTRACT_TYPE.q_sql_select_name */
extern T0* T261x14958599T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to detachable XPLAIN_ABSTRACT_TYPE.sql_select_name */
extern T0* T261x14958601T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to detachable XPLAIN_ABSTRACT_TYPE.name */
extern T0* T261x14942210(GE_context* ac, T0* C);
/* Call to detachable XPLAIN_ABSTRACT_TYPE.columndatatype */
extern T0* T261x14958594T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable XPLAIN_ABSTRACT_TYPE.sqlcolumnrequired */
extern T0* T261x14958597T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to detachable XPLAIN_ABSTRACT_TYPE.sqlcolumndefault */
extern T0* T261x14958596T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to [detachable] ABSTRACT_GENERATOR.columndatatype_assertion */
extern T0* T282x16859156T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] ABSTRACT_GENERATOR.columndatatype_type */
extern T0* T282x16859155T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] ABSTRACT_GENERATOR.columndatatype_base */
extern T0* T282x16859154T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] ABSTRACT_GENERATOR.datatype_float */
extern T0* T282x16859144T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] ABSTRACT_GENERATOR.datatype_varchar */
extern T0* T282x16859153T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] ABSTRACT_GENERATOR.datatype_pk_char */
extern T0* T282x16859149T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] ABSTRACT_GENERATOR.datatype_ref_int */
extern T0* T282x16859151T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] ABSTRACT_GENERATOR.datatype_pk_int */
extern T0* T282x16859150T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] ABSTRACT_GENERATOR.datatype_text */
extern T0* T282x16859152T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] ABSTRACT_GENERATOR.datatype_numeric */
extern T0* T282x16859147T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] ABSTRACT_GENERATOR.datatype_picture */
extern T0* T282x16859148T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] ABSTRACT_GENERATOR.datatype_money */
extern T0* T282x16859146T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] ABSTRACT_GENERATOR.datatype_int */
extern T0* T282x16859145T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] ABSTRACT_GENERATOR.datatype_datetime */
extern T0* T282x16859143T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] ABSTRACT_GENERATOR.datatype_char */
extern T0* T282x16859142T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] ABSTRACT_GENERATOR.datatype_boolean */
extern T0* T282x16859141T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_ATTRIBUTE].next_cursor */
extern T0* T362x10158087(GE_context* ac, T0* C);
/* Call to [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_ATTRIBUTE].current_cell */
extern T0* T362x9650183(GE_context* ac, T0* C);
/* Call to [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_ATTRIBUTE].off */
extern T1 T362x10158083(GE_context* ac, T0* C);
/* Call to detachable ST_PARAMETER_FORMATTER.valid_parameter */
extern T1 T377x8814602T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] SQL_GENERATOR_WITH_SP.sp_function_type_for_selection_value */
extern T0* T411x16023566T0T0T1(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3);
/* Call to [detachable] SQL_GENERATOR_WITH_SP.value_representation_boolean */
extern T0* T411x15106245(GE_context* ac, T0* C);
/* Call to [detachable] SQL_GENERATOR_WITH_SP.sp_function_type_for_selection_list */
extern T0* T411x16023565T0T1(GE_context* ac, T0* C, T0* a1, T1 a2);
/* Call to [detachable] SQL_GENERATOR_WITH_SP.is_value_declared */
extern T1 T411x15106242T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] SQL_GENERATOR_WITH_SP.storedproceduresupportstruefunction */
extern T1 T411x16023558(GE_context* ac, T0* C);
/* Call to [detachable] SQL_GENERATOR_WITH_SP.sp_function_type_for_selection */
extern T0* T411x16023564T0T1(GE_context* ac, T0* C, T0* a1, T1 a2);
/* Call to [detachable] SQL_GENERATOR_WITH_TRIGGERS.sqlinitvalue_attribute */
extern T0* T470x17367046T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] SQL_GENERATOR_WITH_TRIGGERS.sql_init_infix_expression */
extern T0* T470x17367045T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* Call to [detachable] SQL_GENERATOR_WITH_TRIGGERS.sql_init_cast_to_string */
extern T0* T470x17367053T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] SQL_GENERATOR_WITH_TRIGGERS.sql_init_cast_to_real */
extern T0* T470x17367052T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] SQL_GENERATOR_WITH_TRIGGERS.sql_init_cast_to_integer */
extern T0* T470x17367051T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] SQL_GENERATOR_WITH_TRIGGERS.sql_init_cast_to_date */
extern T0* T470x17367050T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] SQL_GENERATOR_WITH_TRIGGERS.sql_init_combine_expression */
extern T0* T470x17367048T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] SQL_GENERATOR_WITH_TRIGGERS.sqlinitvalue_if */
extern T0* T470x17367047T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable EXCEPTION.set_type_name */
extern void T22x4423704T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable EXCEPTION.set_recipient_name */
extern void T22x4423707T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable EXCEPTION.set_description */
extern void T22x4423709T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable EXCEPTION.set_exception_trace */
extern void T22x4423703T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable EXCEPTION.set_throwing_exception */
extern void T22x4423708T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.write_drop_assertion */
extern void T33x15106351T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.write_drop_extension */
extern void T33x15106349T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.drop_procedure */
extern void T33x15106284T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.drop_value */
extern void T33x15106280T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.drop_constant */
extern void T33x15106287T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.write_drop_column */
extern void T33x15106350T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to detachable SQL_GENERATOR.drop_table */
extern void T33x15106283T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.drop_domain */
extern void T33x15106286T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.write_type */
extern void T33x16859163T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.write_init */
extern void T33x15106342T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.write_one_line_comment */
extern void T33x15106358T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.write_sql */
extern void T33x16859164T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.write_procedure */
extern void T33x16859169T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.write_echo */
extern void T33x15106348T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.echo */
extern void T33x15106359T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.write_extend */
extern void T33x16859173T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.write_select */
extern void T33x16859168T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.write_select_value */
extern void T33x16859165T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.write_value */
extern void T33x16859161T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.write_update */
extern void T33x16859162T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* Call to detachable SQL_GENERATOR.write_delete */
extern void T33x16859174T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to detachable SQL_GENERATOR.write_insert */
extern void T33x16859170T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* Call to detachable SQL_GENERATOR.write_index */
extern void T33x15106343T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.write_assertion */
extern void T33x15106357T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.write_drop */
extern void T33x15106352T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.write_constant */
extern void T33x16859176T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.write_constant_assignment */
extern void T33x16859175T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to detachable SQL_GENERATOR.write_base */
extern void T33x15106356T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.write_use_database */
extern void T33x15106331T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.write_end */
extern void T33x15106347T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.set_sequence_name_format */
extern void T33x15106272T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.set_primary_key_format */
extern void T33x15106273T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable SQL_GENERATOR.set_options */
extern void T33x15106274T1T1T1T1T1T1T1T1T1T1T1T1T1T1(GE_context* ac, T0* C, T1 a1, T1 a2, T1 a3, T1 a4, T1 a5, T1 a6, T1 a7, T1 a8, T1 a9, T1 a10, T1 a11, T1 a12, T1 a13, T1 a14);
/* Call to [detachable] XPLAIN_DOMAIN_RESTRICTION.check_attachment */
extern void T122x14696452T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to detachable XPLAIN_STATEMENT.write */
extern void T129x15089668T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable XPLAIN_STATEMENT.cleanup */
extern void T129x15089670(GE_context* ac, T0* C);
/* Call to detachable XPLAIN_STATEMENT.optimize_for_procedure */
extern void T129x15089669T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable XPLAIN_STATEMENT.write_value_declare_inside_sp */
extern void T129x15089667T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] XPLAIN_EXPRESSION.add_to_join */
extern void T137x17317914T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to [detachable] XPLAIN_REPRESENTATION.set_domain_restriction */
extern void T142x16695313T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to [detachable] XPLAIN_ATTRIBUTE.set_required */
extern void T145x15843350T1(GE_context* ac, T0* C, T1 a1);
/* Call to [detachable] XPLAIN_ATTRIBUTE.write_drop */
extern void T145x15843347T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to [detachable] XPLAIN_ATTRIBUTE.set_init */
extern void T145x15843348T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] XPLAIN_ATTRIBUTE.set_init_default */
extern void T145x15843349T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable XPLAIN_ABSTRACT_OBJECT.write_drop */
extern void T195x14942216T0(GE_context* ac, T0* C, T0* a1);
/* Call to detachable XPLAIN_ABSTRACT_TYPE.hack_anode */
extern void T261x14958603T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] ABSTRACT_GENERATOR.write_sql */
extern void T282x16859164T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] ABSTRACT_GENERATOR.write_if */
extern void T282x16859171T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] ABSTRACT_GENERATOR.write_procedure */
extern void T282x16859169T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] ABSTRACT_GENERATOR.write_select_list */
extern void T282x16859166T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] ABSTRACT_GENERATOR.write_extend */
extern void T282x16859173T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] ABSTRACT_GENERATOR.write_select */
extern void T282x16859168T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] ABSTRACT_GENERATOR.write_select_function */
extern void T282x16859167T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] ABSTRACT_GENERATOR.write_select_value */
extern void T282x16859165T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] ABSTRACT_GENERATOR.write_update */
extern void T282x16859162T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* Call to [detachable] ABSTRACT_GENERATOR.write_delete */
extern void T282x16859174T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to [detachable] ABSTRACT_GENERATOR.write_constant */
extern void T282x16859176T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] ABSTRACT_GENERATOR.write_constant_assignment */
extern void T282x16859175T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to [detachable] ABSTRACT_GENERATOR.write_get_insert */
extern void T282x16859172T0T0T1T0(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* Call to [detachable] ABSTRACT_GENERATOR.write_insert */
extern void T282x16859170T0T0T0(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* Call to [detachable] ABSTRACT_GENERATOR.write_type */
extern void T282x16859163T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] ABSTRACT_GENERATOR.write_value */
extern void T282x16859161T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_ATTRIBUTE].set_next_cursor */
extern void T362x10158088T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_ATTRIBUTE].set_after */
extern void T362x9650187(GE_context* ac, T0* C);
/* Call to [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_ATTRIBUTE].set_current_cell */
extern void T362x9650188T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_ATTRIBUTE].set */
extern void T362x9650185T0T1T1(GE_context* ac, T0* C, T0* a1, T1 a2, T1 a3);
/* Call to detachable ST_PARAMETER_FORMATTER.set_lowercase */
extern void T377x8814612T1(GE_context* ac, T0* C, T1 a1);
/* Call to detachable ST_PARAMETER_FORMATTER.reset_options */
extern void T377x8814621(GE_context* ac, T0* C);
/* Call to detachable ST_PARAMETER_FORMATTER.format_to */
extern void T377x8814611T0T0(GE_context* ac, T0* C, T0* a1, T0* a2);
/* Call to detachable ST_PARAMETER_FORMATTER.set_precision */
extern void T377x8814620T6(GE_context* ac, T0* C, T6 a1);
/* Call to detachable ST_PARAMETER_FORMATTER.set_width */
extern void T377x8814619T6(GE_context* ac, T0* C, T6 a1);
/* Call to detachable ST_PARAMETER_FORMATTER.set_plus_sign_enabled */
extern void T377x8814614T1(GE_context* ac, T0* C, T1 a1);
/* Call to detachable ST_PARAMETER_FORMATTER.set_space_sign_enabled */
extern void T377x8814613T1(GE_context* ac, T0* C, T1 a1);
/* Call to detachable ST_PARAMETER_FORMATTER.set_padding_character */
extern void T377x8814615T2(GE_context* ac, T0* C, T2 a1);
/* Call to detachable ST_PARAMETER_FORMATTER.set_align_center */
extern void T377x8814616(GE_context* ac, T0* C);
/* Call to detachable ST_PARAMETER_FORMATTER.set_align_left */
extern void T377x8814618(GE_context* ac, T0* C);
/* Call to [detachable] SQL_GENERATOR_WITH_SP.sp_value_declarations */
extern void T411x16023600T0(GE_context* ac, T0* C, T0* a1);
/* Call to [detachable] SQL_GENERATOR_WITH_SP.create_value_declare */
extern void T411x15106296T0(GE_context* ac, T0* C, T0* a1);
/* New instance of type [detachable] SPECIAL [CHARACTER_8] */
extern T0* GE_new15(T6 a1, T1 initialize);
/* New instance of type [detachable] SPECIAL [CHARACTER_32] */
extern T0* GE_new16(T6 a1, T1 initialize);
/* New instance of type [detachable] STRING_8 */
extern T0* GE_new17(T1 initialize);
/* New instance of type [detachable] IMMUTABLE_STRING_32 */
extern T0* GE_new20(T1 initialize);
/* New instance of type [detachable] ISE_EXCEPTION_MANAGER */
extern T0* GE_new21(T1 initialize);
/* New instance of type [detachable] XPLAIN2SQL */
extern T0* GE_new25(T1 initialize);
/* New instance of type [detachable] CELL [detachable EXCEPTION] */
extern T0* GE_new26(T1 initialize);
/* New instance of type [detachable] HASH_TABLE [[attached] INTEGER_32, [attached] INTEGER_32] */
extern T0* GE_new27(T1 initialize);
/* New instance of type [detachable] CELL [detachable TUPLE [[attached] INTEGER_32, [attached] INTEGER_32, [attached] INTEGER_32, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] INTEGER_32, [attached] BOOLEAN]] */
extern T0* GE_new28(T1 initialize);
/* New instance of type [detachable] CELL [[attached] NO_MORE_MEMORY] */
extern T0* GE_new29(T1 initialize);
/* New instance of type detachable C_STRING */
extern T0* GE_new30(T1 initialize);
/* New instance of type [detachable] TUPLE [[attached] INTEGER_32, [attached] INTEGER_32, [attached] INTEGER_32, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] INTEGER_32, [attached] BOOLEAN] */
extern T0* GE_new31(T1 initialize);
/* New instance of type detachable KL_TEXT_INPUT_FILE */
extern T0* GE_new32(T1 initialize);
/* New instance of type [detachable] KL_ARGUMENTS */
extern T0* GE_new35(T1 initialize);
/* New instance of type [detachable] KL_STANDARD_FILES */
extern T0* GE_new36(T1 initialize);
/* New instance of type [detachable] KL_STDOUT_FILE */
extern T0* GE_new37(T1 initialize);
/* New instance of type [detachable] ARRAY [[attached] STRING_8] */
extern T0* GE_new38(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] STRING_8] */
extern T0* GE_new39(T6 a1, T1 initialize);
/* New instance of type [detachable] ARRAY [[attached] ANY] */
extern T0* GE_new40(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] ANY] */
extern T0* GE_new41(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_STDERR_FILE */
extern T0* GE_new42(T1 initialize);
/* New instance of type [detachable] SQL_GENERATOR_ANSI */
extern T0* GE_new43(T1 initialize);
/* New instance of type [detachable] SQL_GENERATOR_BASIC */
extern T0* GE_new44(T1 initialize);
/* New instance of type [detachable] SQL_GENERATOR_DB2_6 */
extern T0* GE_new45(T1 initialize);
/* New instance of type [detachable] SQL_GENERATOR_DB2_71 */
extern T0* GE_new46(T1 initialize);
/* New instance of type [detachable] SQL_GENERATOR_FOXPRO */
extern T0* GE_new47(T1 initialize);
/* New instance of type [detachable] SQL_GENERATOR_INTERBASE6 */
extern T0* GE_new48(T1 initialize);
/* New instance of type [detachable] SQL_GENERATOR_FIREBIRD_21 */
extern T0* GE_new49(T1 initialize);
/* New instance of type [detachable] SQL_GENERATOR_MSACCESS97 */
extern T0* GE_new50(T1 initialize);
/* New instance of type [detachable] SQL_GENERATOR_MSACCESS2000 */
extern T0* GE_new51(T1 initialize);
/* New instance of type [detachable] SQL_GENERATOR_MSQL */
extern T0* GE_new52(T1 initialize);
/* New instance of type [detachable] SQL_GENERATOR_MYSQL5 */
extern T0* GE_new53(T1 initialize);
/* New instance of type [detachable] SQL_GENERATOR_MYSQL4 */
extern T0* GE_new54(T1 initialize);
/* New instance of type [detachable] SQL_GENERATOR_MYSQL322 */
extern T0* GE_new55(T1 initialize);
/* New instance of type [detachable] SQL_GENERATOR_ORACLE */
extern T0* GE_new56(T1 initialize);
/* New instance of type [detachable] SQL_GENERATOR_PGSQL_95 */
extern T0* GE_new57(T1 initialize);
/* New instance of type [detachable] SQL_GENERATOR_PGSQL_82 */
extern T0* GE_new58(T1 initialize);
/* New instance of type [detachable] SQL_GENERATOR_PGSQL_81 */
extern T0* GE_new59(T1 initialize);
/* New instance of type [detachable] SQL_GENERATOR_PGSQL_73 */
extern T0* GE_new60(T1 initialize);
/* New instance of type [detachable] SQL_GENERATOR_PGSQL_72 */
extern T0* GE_new61(T1 initialize);
/* New instance of type [detachable] SQL_GENERATOR_SQLITE_3 */
extern T0* GE_new62(T1 initialize);
/* New instance of type [detachable] SQL_GENERATOR_TSQL2016 */
extern T0* GE_new63(T1 initialize);
/* New instance of type [detachable] SQL_GENERATOR_TSQL2000 */
extern T0* GE_new64(T1 initialize);
/* New instance of type [detachable] SQL_GENERATOR_TSQL70 */
extern T0* GE_new65(T1 initialize);
/* New instance of type [detachable] SQL_GENERATOR_TSQL65 */
extern T0* GE_new66(T1 initialize);
/* New instance of type [detachable] XPLAIN_PARSER */
extern T0* GE_new67(T1 initialize);
/* New instance of type [detachable] XML_GENERATOR */
extern T0* GE_new69(T1 initialize);
/* New instance of type [detachable] XPLAIN_STATEMENTS */
extern T0* GE_new71(T1 initialize);
/* New instance of type [detachable] VOID_TARGET */
extern T0* GE_new74(T1 initialize);
/* New instance of type [detachable] ROUTINE_FAILURE */
extern T0* GE_new76(T1 initialize);
/* New instance of type [detachable] OLD_VIOLATION */
extern T0* GE_new78(T1 initialize);
/* New instance of type [detachable] NO_MORE_MEMORY */
extern T0* GE_new80(T1 initialize);
/* New instance of type [detachable] INVARIANT_VIOLATION */
extern T0* GE_new81(T1 initialize);
/* New instance of type [detachable] OPERATING_SYSTEM_SIGNAL_FAILURE */
extern T0* GE_new82(T1 initialize);
/* New instance of type [detachable] IO_FAILURE */
extern T0* GE_new83(T1 initialize);
/* New instance of type [detachable] OPERATING_SYSTEM_FAILURE */
extern T0* GE_new84(T1 initialize);
/* New instance of type [detachable] COM_FAILURE */
extern T0* GE_new85(T1 initialize);
/* New instance of type [detachable] EIFFEL_RUNTIME_PANIC */
extern T0* GE_new86(T1 initialize);
/* New instance of type [detachable] PRECONDITION_VIOLATION */
extern T0* GE_new88(T1 initialize);
/* New instance of type [detachable] POSTCONDITION_VIOLATION */
extern T0* GE_new89(T1 initialize);
/* New instance of type [detachable] FLOATING_POINT_FAILURE */
extern T0* GE_new90(T1 initialize);
/* New instance of type [detachable] CHECK_VIOLATION */
extern T0* GE_new91(T1 initialize);
/* New instance of type [detachable] BAD_INSPECT_VALUE */
extern T0* GE_new92(T1 initialize);
/* New instance of type [detachable] VARIANT_VIOLATION */
extern T0* GE_new93(T1 initialize);
/* New instance of type [detachable] LOOP_INVARIANT_VIOLATION */
extern T0* GE_new94(T1 initialize);
/* New instance of type [detachable] RESCUE_FAILURE */
extern T0* GE_new95(T1 initialize);
/* New instance of type [detachable] RESUMPTION_FAILURE */
extern T0* GE_new96(T1 initialize);
/* New instance of type [detachable] CREATE_ON_DEFERRED */
extern T0* GE_new97(T1 initialize);
/* New instance of type [detachable] EXTERNAL_FAILURE */
extern T0* GE_new98(T1 initialize);
/* New instance of type [detachable] VOID_ASSIGNED_TO_EXPANDED */
extern T0* GE_new99(T1 initialize);
/* New instance of type [detachable] EXCEPTION_IN_SIGNAL_HANDLER_FAILURE */
extern T0* GE_new100(T1 initialize);
/* New instance of type [detachable] MISMATCH_FAILURE */
extern T0* GE_new101(T1 initialize);
/* New instance of type [detachable] DEVELOPER_EXCEPTION */
extern T0* GE_new102(T1 initialize);
/* New instance of type [detachable] ADDRESS_APPLIED_TO_MELTED_FEATURE */
extern T0* GE_new103(T1 initialize);
/* New instance of type [detachable] SERIALIZATION_FAILURE */
extern T0* GE_new104(T1 initialize);
/* New instance of type [detachable] ST_SCIENTIFIC_FORMATTER */
extern T0* GE_new105(T1 initialize);
/* New instance of type [detachable] PRIMES */
extern T0* GE_new106(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] INTEGER_32] */
extern T0* GE_new107(T6 a1, T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] BOOLEAN] */
extern T0* GE_new108(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_STRING_ROUTINES */
extern T0* GE_new110(T1 initialize);
/* New instance of type [detachable] MANAGED_POINTER */
extern T0* GE_new113(T1 initialize);
/* New instance of type [detachable] FILE_INFO */
extern T0* GE_new114(T1 initialize);
/* New instance of type [detachable] ARGUMENTS_32 */
extern T0* GE_new115(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8] */
extern T0* GE_new116(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] INTEGER_32] */
extern T0* GE_new118(T1 initialize);
/* New instance of type [detachable] XPLAIN_SCANNER_STATE */
extern T0* GE_new119(T1 initialize);
/* New instance of type [detachable] DS_LINKED_LIST [[attached] XPLAIN_SCANNER_STATE] */
extern T0* GE_new121(T1 initialize);
/* New instance of type [detachable] XPLAIN_REQUIRED */
extern T0* GE_new123(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable ANY] */
extern T0* GE_new126(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable ANY] */
extern T0* GE_new127(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] STRING_8] */
extern T0* GE_new128(T1 initialize);
/* New instance of type detachable XPLAIN_CONSTANT_ASSIGNMENT_STATEMENT */
extern T0* GE_new130(T1 initialize);
/* New instance of type detachable XPLAIN_BASE */
extern T0* GE_new131(T1 initialize);
/* New instance of type detachable XPLAIN_TYPE */
extern T0* GE_new132(T1 initialize);
/* New instance of type detachable XPLAIN_CONSTANT_STATEMENT */
extern T0* GE_new133(T1 initialize);
/* New instance of type [detachable] XPLAIN_A_NODE */
extern T0* GE_new134(T1 initialize);
/* New instance of type [detachable] XPLAIN_I_NODE */
extern T0* GE_new135(T1 initialize);
/* New instance of type [detachable] XPLAIN_LOGICAL_EXPRESSION */
extern T0* GE_new136(T1 initialize);
/* New instance of type [detachable] XPLAIN_LOGICAL_VALUE_EXPRESSION */
extern T0* GE_new138(T1 initialize);
/* New instance of type [detachable] XPLAIN_ATTRIBUTE_NAME_NODE */
extern T0* GE_new140(T1 initialize);
/* New instance of type [detachable] XPLAIN_ATTRIBUTE_NAME */
extern T0* GE_new141(T1 initialize);
/* New instance of type detachable XPLAIN_ATTRIBUTE_NODE */
extern T0* GE_new144(T1 initialize);
/* New instance of type [detachable] XPLAIN_ATTRIBUTE */
extern T0* GE_new145(T1 initialize);
/* New instance of type detachable XPLAIN_ASSERTION */
extern T0* GE_new146(T1 initialize);
/* New instance of type detachable XPLAIN_ASSIGNMENT_NODE */
extern T0* GE_new147(T1 initialize);
/* New instance of type [detachable] XPLAIN_ASSIGNMENT */
extern T0* GE_new148(T1 initialize);
/* New instance of type detachable XPLAIN_CASCADE_STATEMENT */
extern T0* GE_new149(T1 initialize);
/* New instance of type detachable XPLAIN_CASCADE_FUNCTION_EXPRESSION */
extern T0* GE_new150(T1 initialize);
/* New instance of type detachable XPLAIN_DELETE_STATEMENT */
extern T0* GE_new151(T1 initialize);
/* New instance of type detachable XPLAIN_SUBJECT */
extern T0* GE_new152(T1 initialize);
/* New instance of type detachable XPLAIN_UPDATE_STATEMENT */
extern T0* GE_new153(T1 initialize);
/* New instance of type [detachable] XPLAIN_VALUE_STATEMENT */
extern T0* GE_new154(T1 initialize);
/* New instance of type detachable XPLAIN_VALUE_SELECTION_STATEMENT */
extern T0* GE_new155(T1 initialize);
/* New instance of type detachable XPLAIN_SELECTION_FUNCTION */
extern T0* GE_new157(T1 initialize);
/* New instance of type detachable XPLAIN_GET_STATEMENT */
extern T0* GE_new158(T1 initialize);
/* New instance of type detachable XPLAIN_EXTEND_STATEMENT */
extern T0* GE_new159(T1 initialize);
/* New instance of type detachable XPLAIN_EXTENSION */
extern T0* GE_new160(T1 initialize);
/* New instance of type [detachable] XPLAIN_EXTEND_ATTRIBUTE */
extern T0* GE_new161(T1 initialize);
/* New instance of type detachable XPLAIN_EXTENSION_EXPRESSION */
extern T0* GE_new162(T1 initialize);
/* New instance of type detachable XPLAIN_SELECTION_LIST */
extern T0* GE_new164(T1 initialize);
/* New instance of type detachable XPLAIN_SORT_NODE */
extern T0* GE_new165(T1 initialize);
/* New instance of type detachable XPLAIN_EXPRESSION_NODE */
extern T0* GE_new167(T1 initialize);
/* New instance of type detachable XPLAIN_PROCEDURE_STATEMENT */
extern T0* GE_new168(T1 initialize);
/* New instance of type detachable XPLAIN_PARAMETER_NODE */
extern T0* GE_new169(T1 initialize);
/* New instance of type [detachable] XPLAIN_PARAMETER */
extern T0* GE_new170(T1 initialize);
/* New instance of type [detachable] XPLAIN_SQL_STATEMENT */
extern T0* GE_new171(T1 initialize);
/* New instance of type detachable XPLAIN_STATEMENT_NODE */
extern T0* GE_new172(T1 initialize);
/* New instance of type detachable XPLAIN_IF_STATEMENT */
extern T0* GE_new173(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable XPLAIN_STATEMENT] */
extern T0* GE_new174(T6 a1, T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] XPLAIN_SQL_STATEMENT] */
extern T0* GE_new175(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_STATEMENT] */
extern T0* GE_new176(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable XPLAIN_PROCEDURE_STATEMENT] */
extern T0* GE_new177(T6 a1, T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable XPLAIN_CONSTANT_ASSIGNMENT_STATEMENT] */
extern T0* GE_new178(T6 a1, T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable XPLAIN_TYPE] */
extern T0* GE_new179(T6 a1, T1 initialize);
/* New instance of type [detachable] XPLAIN_TYPE_STATEMENT */
extern T0* GE_new180(T1 initialize);
/* New instance of type [detachable] XPLAIN_UNIVERSE */
extern T0* GE_new181(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable XPLAIN_BASE] */
extern T0* GE_new182(T6 a1, T1 initialize);
/* New instance of type [detachable] XPLAIN_BASE_STATEMENT */
extern T0* GE_new183(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable XPLAIN_CONSTANT_STATEMENT] */
extern T0* GE_new184(T6 a1, T1 initialize);
/* New instance of type detachable XPLAIN_VARIABLE */
extern T0* GE_new185(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] XPLAIN_EXPRESSION] */
extern T0* GE_new186(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_CONSTANT_ASSIGNMENT_STATEMENT] */
extern T0* GE_new187(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] XPLAIN_REPRESENTATION] */
extern T0* GE_new188(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_BASE] */
extern T0* GE_new189(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] XPLAIN_DOMAIN_RESTRICTION] */
extern T0* GE_new190(T6 a1, T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable XPLAIN_ATTRIBUTE_NODE] */
extern T0* GE_new191(T6 a1, T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] XPLAIN_PK_REPRESENTATION] */
extern T0* GE_new192(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_TYPE] */
extern T0* GE_new193(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_CONSTANT_STATEMENT] */
extern T0* GE_new194(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] XPLAIN_ATTRIBUTE_NAME] */
extern T0* GE_new196(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_DOMAIN_RESTRICTION] */
extern T0* GE_new197(T1 initialize);
/* New instance of type [detachable] XPLAIN_A_PATTERN */
extern T0* GE_new198(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] XPLAIN_A_NODE] */
extern T0* GE_new199(T6 a1, T1 initialize);
/* New instance of type [detachable] XPLAIN_A_ENUMERATION */
extern T0* GE_new200(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] XPLAIN_I_NODE] */
extern T0* GE_new201(T6 a1, T1 initialize);
/* New instance of type [detachable] XPLAIN_I_ENUMERATION */
extern T0* GE_new202(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_A_NODE] */
extern T0* GE_new203(T1 initialize);
/* New instance of type [detachable] XPLAIN_A_REPRESENTATION */
extern T0* GE_new204(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_I_NODE] */
extern T0* GE_new206(T1 initialize);
/* New instance of type [detachable] XPLAIN_TRAJECTORY */
extern T0* GE_new207(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] XPLAIN_LOGICAL_EXPRESSION] */
extern T0* GE_new208(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_LOGICAL_EXPRESSION] */
extern T0* GE_new209(T1 initialize);
/* New instance of type [detachable] XPLAIN_LOGICAL_INFIX_EXPRESSION */
extern T0* GE_new210(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_EXPRESSION] */
extern T0* GE_new211(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] XPLAIN_ATTRIBUTE_NAME_NODE] */
extern T0* GE_new212(T6 a1, T1 initialize);
/* New instance of type [detachable] XPLAIN_NOTNOT_EXPRESSION */
extern T0* GE_new213(T1 initialize);
/* New instance of type [detachable] XPLAIN_NOT_EXPRESSION */
extern T0* GE_new214(T1 initialize);
/* New instance of type [detachable] XPLAIN_PARAMETER_EXPRESSION */
extern T0* GE_new215(T1 initialize);
/* New instance of type [detachable] XPLAIN_EQ_NULL_EXPRESSION */
extern T0* GE_new216(T1 initialize);
/* New instance of type [detachable] XPLAIN_NE_NULL_EXPRESSION */
extern T0* GE_new217(T1 initialize);
/* New instance of type [detachable] XPLAIN_INFIX_EXPRESSION */
extern T0* GE_new218(T1 initialize);
/* New instance of type [detachable] XPLAIN_PARENTHESIS_EXPRESSION */
extern T0* GE_new219(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] XPLAIN_LOGICAL_VALUE_EXPRESSION] */
extern T0* GE_new220(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_LOGICAL_VALUE_EXPRESSION] */
extern T0* GE_new221(T1 initialize);
/* New instance of type [detachable] XPLAIN_SYSTEMDATE_EXPRESSION */
extern T0* GE_new222(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] XPLAIN_SYSTEM_EXPRESSION] */
extern T0* GE_new223(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_SYSTEM_EXPRESSION] */
extern T0* GE_new224(T1 initialize);
/* New instance of type [detachable] XPLAIN_LOGINNAME_EXPRESSION */
extern T0* GE_new225(T1 initialize);
/* New instance of type [detachable] XPLAIN_IF_EXPRESSION */
extern T0* GE_new226(T1 initialize);
/* New instance of type [detachable] XPLAIN_PREFIX_EXPRESSION */
extern T0* GE_new227(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable XPLAIN_EXPRESSION_NODE] */
extern T0* GE_new228(T6 a1, T1 initialize);
/* New instance of type [detachable] XPLAIN_COMBINE_FUNCTION */
extern T0* GE_new229(T1 initialize);
/* New instance of type [detachable] XPLAIN_DATEF_FUNCTION */
extern T0* GE_new230(T1 initialize);
/* New instance of type [detachable] XPLAIN_INTEGER_FUNCTION */
extern T0* GE_new231(T1 initialize);
/* New instance of type [detachable] XPLAIN_REAL_FUNCTION */
extern T0* GE_new232(T1 initialize);
/* New instance of type [detachable] XPLAIN_STRING_FUNCTION */
extern T0* GE_new233(T1 initialize);
/* New instance of type [detachable] XPLAIN_STRING_EXPRESSION */
extern T0* GE_new234(T1 initialize);
/* New instance of type [detachable] XPLAIN_NEWDATE_FUNCTION */
extern T0* GE_new235(T1 initialize);
/* New instance of type [detachable] XPLAIN_ID_FUNCTION */
extern T0* GE_new236(T1 initialize);
/* New instance of type [detachable] XPLAIN_NUMBER_EXPRESSION */
extern T0* GE_new237(T1 initialize);
/* New instance of type [detachable] XPLAIN_UNMANAGED_PARAMETER_EXPRESSION */
extern T0* GE_new238(T1 initialize);
/* New instance of type [detachable] XPLAIN_USER_FUNCTION */
extern T0* GE_new239(T1 initialize);
/* New instance of type [detachable] XPLAIN_SQL_EXPRESSION */
extern T0* GE_new240(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ATTRIBUTE_NAME_NODE] */
extern T0* GE_new241(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ATTRIBUTE_NAME] */
extern T0* GE_new242(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable XPLAIN_EXPRESSION] */
extern T0* GE_new243(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXPRESSION] */
extern T0* GE_new244(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_REPRESENTATION] */
extern T0* GE_new245(T1 initialize);
/* New instance of type [detachable] XPLAIN_B_REPRESENTATION */
extern T0* GE_new246(T1 initialize);
/* New instance of type [detachable] XPLAIN_C_REPRESENTATION */
extern T0* GE_new247(T1 initialize);
/* New instance of type [detachable] XPLAIN_D_REPRESENTATION */
extern T0* GE_new248(T1 initialize);
/* New instance of type [detachable] XPLAIN_I_REPRESENTATION */
extern T0* GE_new249(T1 initialize);
/* New instance of type [detachable] XPLAIN_M_REPRESENTATION */
extern T0* GE_new250(T1 initialize);
/* New instance of type [detachable] XPLAIN_P_REPRESENTATION */
extern T0* GE_new251(T1 initialize);
/* New instance of type [detachable] XPLAIN_R_REPRESENTATION */
extern T0* GE_new252(T1 initialize);
/* New instance of type [detachable] XPLAIN_T_REPRESENTATION */
extern T0* GE_new253(T1 initialize);
/* New instance of type [detachable] XPLAIN_PK_A_REPRESENTATION */
extern T0* GE_new254(T1 initialize);
/* New instance of type detachable XPLAIN_A_REFERENCES */
extern T0* GE_new255(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_PK_REPRESENTATION] */
extern T0* GE_new256(T1 initialize);
/* New instance of type [detachable] XPLAIN_PK_I_REPRESENTATION */
extern T0* GE_new257(T1 initialize);
/* New instance of type detachable XPLAIN_I_REFERENCES */
extern T0* GE_new258(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] XPLAIN_ATTRIBUTE] */
extern T0* GE_new259(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_ATTRIBUTE_NODE] */
extern T0* GE_new260(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ATTRIBUTE] */
extern T0* GE_new262(T1 initialize);
/* New instance of type [detachable] XPLAIN_VARIABLE_EXPRESSION */
extern T0* GE_new263(T1 initialize);
/* New instance of type detachable XPLAIN_VALUE */
extern T0* GE_new264(T1 initialize);
/* New instance of type [detachable] XPLAIN_VALUE_EXPRESSION */
extern T0* GE_new265(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable XPLAIN_DOMAIN_RESTRICTION] */
extern T0* GE_new266(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_DOMAIN_RESTRICTION] */
extern T0* GE_new267(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable XPLAIN_EXTENSION_EXPRESSION] */
extern T0* GE_new268(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_ASSERTION] */
extern T0* GE_new269(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable XPLAIN_ASSERTION] */
extern T0* GE_new270(T6 a1, T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable XPLAIN_ATTRIBUTE_NAME_NODE] */
extern T0* GE_new271(T6 a1, T1 initialize);
/* New instance of type detachable XPLAIN_INDEX */
extern T0* GE_new272(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_ATTRIBUTE_NAME_NODE] */
extern T0* GE_new273(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable XPLAIN_UPDATE_STATEMENT] */
extern T0* GE_new274(T6 a1, T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable XPLAIN_DELETE_STATEMENT] */
extern T0* GE_new275(T6 a1, T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable XPLAIN_CASCADE_STATEMENT] */
extern T0* GE_new276(T6 a1, T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable XPLAIN_SUBJECT] */
extern T0* GE_new277(T6 a1, T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable XPLAIN_ASSIGNMENT_NODE] */
extern T0* GE_new278(T6 a1, T1 initialize);
/* New instance of type [detachable] XPLAIN_INSERT_STATEMENT */
extern T0* GE_new279(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable XPLAIN_SELECTION_LIST] */
extern T0* GE_new280(T6 a1, T1 initialize);
/* New instance of type [detachable] XPLAIN_GET_INSERT_STATEMENT */
extern T0* GE_new281(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] XPLAIN_ASSIGNMENT] */
extern T0* GE_new283(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_ASSIGNMENT_NODE] */
extern T0* GE_new284(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ASSIGNMENT] */
extern T0* GE_new285(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable XPLAIN_CASCADE_FUNCTION_EXPRESSION] */
extern T0* GE_new286(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_CASCADE_STATEMENT] */
extern T0* GE_new287(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable XPLAIN_SELECTION_FUNCTION] */
extern T0* GE_new288(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_CASCADE_FUNCTION_EXPRESSION] */
extern T0* GE_new289(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_DELETE_STATEMENT] */
extern T0* GE_new290(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SUBJECT] */
extern T0* GE_new291(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_UPDATE_STATEMENT] */
extern T0* GE_new292(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] XPLAIN_VALUE_STATEMENT] */
extern T0* GE_new293(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_VALUE_STATEMENT] */
extern T0* GE_new294(T1 initialize);
/* New instance of type [detachable] XPLAIN_SELECTION_EXPRESSION */
extern T0* GE_new295(T1 initialize);
/* New instance of type [detachable] XPLAIN_LAST_AUTO_PK_EXPRESSION */
extern T0* GE_new297(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable XPLAIN_VALUE_SELECTION_STATEMENT] */
extern T0* GE_new298(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_VALUE_SELECTION_STATEMENT] */
extern T0* GE_new299(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable XPLAIN_NON_ATTRIBUTE_EXPRESSION] */
extern T0* GE_new300(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_NON_ATTRIBUTE_EXPRESSION] */
extern T0* GE_new301(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] XPLAIN_FUNCTION] */
extern T0* GE_new302(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SELECTION_FUNCTION] */
extern T0* GE_new303(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable XPLAIN_SELECTION] */
extern T0* GE_new304(T6 a1, T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable XPLAIN_GET_STATEMENT] */
extern T0* GE_new305(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_GET_STATEMENT] */
extern T0* GE_new306(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable XPLAIN_EXTENSION] */
extern T0* GE_new307(T6 a1, T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable XPLAIN_EXTEND_STATEMENT] */
extern T0* GE_new308(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXTEND_STATEMENT] */
extern T0* GE_new309(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] XPLAIN_EXTEND_ATTRIBUTE] */
extern T0* GE_new310(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXTENSION] */
extern T0* GE_new311(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_EXTEND_ATTRIBUTE] */
extern T0* GE_new312(T1 initialize);
/* New instance of type [detachable] XPLAIN_EXTENSION_EXPRESSION_EXPRESSION */
extern T0* GE_new313(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXTENSION_EXPRESSION] */
extern T0* GE_new314(T1 initialize);
/* New instance of type [detachable] XPLAIN_EXTENSION_FUNCTION_EXPRESSION */
extern T0* GE_new315(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SELECTION] */
extern T0* GE_new316(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable XPLAIN_SORT_NODE] */
extern T0* GE_new317(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SELECTION_LIST] */
extern T0* GE_new318(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SORT_NODE] */
extern T0* GE_new319(T1 initialize);
/* New instance of type [detachable] XPLAIN_MAX_FUNCTION */
extern T0* GE_new320(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_FUNCTION] */
extern T0* GE_new321(T1 initialize);
/* New instance of type [detachable] XPLAIN_MIN_FUNCTION */
extern T0* GE_new322(T1 initialize);
/* New instance of type [detachable] XPLAIN_TOTAL_FUNCTION */
extern T0* GE_new323(T1 initialize);
/* New instance of type [detachable] XPLAIN_COUNT_FUNCTION */
extern T0* GE_new324(T1 initialize);
/* New instance of type [detachable] XPLAIN_ANY_FUNCTION */
extern T0* GE_new325(T1 initialize);
/* New instance of type [detachable] XPLAIN_NIL_FUNCTION */
extern T0* GE_new326(T1 initialize);
/* New instance of type [detachable] XPLAIN_SOME_FUNCTION */
extern T0* GE_new327(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXPRESSION_NODE] */
extern T0* GE_new328(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable STRING_8] */
extern T0* GE_new329(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable STRING_8] */
extern T0* GE_new330(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable XPLAIN_PARAMETER_NODE] */
extern T0* GE_new331(T6 a1, T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable XPLAIN_STATEMENT_NODE] */
extern T0* GE_new332(T6 a1, T1 initialize);
/* New instance of type [detachable] XPLAIN_PROCEDURE */
extern T0* GE_new333(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_PROCEDURE_STATEMENT] */
extern T0* GE_new334(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_PARAMETER_NODE] */
extern T0* GE_new335(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] XPLAIN_PARAMETER_NODE] */
extern T0* GE_new336(T6 a1, T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] XPLAIN_PARAMETER] */
extern T0* GE_new337(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_PARAMETER_NODE] */
extern T0* GE_new338(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_PARAMETER] */
extern T0* GE_new339(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_SQL_STATEMENT] */
extern T0* GE_new340(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_STATEMENT_NODE] */
extern T0* GE_new341(T1 initialize);
/* New instance of type [detachable] SPECIAL [detachable XPLAIN_IF_STATEMENT] */
extern T0* GE_new342(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_IF_STATEMENT] */
extern T0* GE_new343(T1 initialize);
/* New instance of type [detachable] YY_FILE_BUFFER */
extern T0* GE_new344(T1 initialize);
/* New instance of type [detachable] KL_STDIN_FILE */
extern T0* GE_new345(T1 initialize);
/* New instance of type [detachable] XML_WRITER */
extern T0* GE_new347(T1 initialize);
/* New instance of type [detachable] KL_TEXT_OUTPUT_FILE */
extern T0* GE_new348(T1 initialize);
/* New instance of type [detachable] DS_LINKED_LIST [[attached] XPLAIN_STATEMENT] */
extern T0* GE_new349(T1 initialize);
/* New instance of type [detachable] STD_FILES */
extern T0* GE_new351(T1 initialize);
/* New instance of type [detachable] JOIN_LIST */
extern T0* GE_new352(T1 initialize);
/* New instance of type detachable JOIN_NODE */
extern T0* GE_new353(T1 initialize);
/* New instance of type [detachable] DS_LINKED_LIST [[attached] XPLAIN_PARAMETER] */
extern T0* GE_new358(T1 initialize);
/* New instance of type [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_PARAMETER] */
extern T0* GE_new359(T1 initialize);
/* New instance of type [detachable] DS_BILINKED_LIST [[attached] XPLAIN_STATEMENT] */
extern T0* GE_new360(T1 initialize);
/* New instance of type [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE] */
extern T0* GE_new361(T1 initialize);
/* New instance of type [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_ATTRIBUTE] */
extern T0* GE_new362(T1 initialize);
/* New instance of type [detachable] XPLAIN_B_RESTRICTION */
extern T0* GE_new365(T1 initialize);
/* New instance of type [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_SCANNER_STATE] */
extern T0* GE_new367(T1 initialize);
/* New instance of type [detachable] XPLAIN_LIKE_EXPRESSION */
extern T0* GE_new368(T1 initialize);
/* New instance of type [detachable] XPLAIN_NOT_LIKE_EXPRESSION */
extern T0* GE_new369(T1 initialize);
/* New instance of type [detachable] ARRAY [[attached] INTEGER_32] */
extern T0* GE_new370(T1 initialize);
/* New instance of type [detachable] KL_ARRAY_ROUTINES [[attached] INTEGER_32] */
extern T0* GE_new371(T1 initialize);
/* New instance of type [detachable] CELL [[attached] INTEGER_32] */
extern T0* GE_new373(T1 initialize);
/* New instance of type [detachable] KL_NULL_TEXT_OUTPUT_STREAM */
extern T0* GE_new374(T1 initialize);
/* New instance of type [detachable] KL_STRING_OUTPUT_STREAM */
extern T0* GE_new376(T1 initialize);
/* New instance of type [detachable] DS_CELL [[attached] INTEGER_32] */
extern T0* GE_new378(T1 initialize);
/* New instance of type [detachable] ST_BOOLEAN_FORMATTER */
extern T0* GE_new380(T1 initialize);
/* New instance of type [detachable] ST_CHARACTER_FORMATTER */
extern T0* GE_new381(T1 initialize);
/* New instance of type [detachable] ST_FLOATING_POINT_FORMATTER */
extern T0* GE_new382(T1 initialize);
/* New instance of type [detachable] ST_FIXED_POINT_FORMATTER */
extern T0* GE_new383(T1 initialize);
/* New instance of type [detachable] ST_FIXED_OR_FLOATING_POINT_FORMATTER */
extern T0* GE_new384(T1 initialize);
/* New instance of type [detachable] ST_DECIMAL_FORMATTER */
extern T0* GE_new385(T1 initialize);
/* New instance of type [detachable] ST_UNSIGNED_OCTAL_FORMATTER */
extern T0* GE_new386(T1 initialize);
/* New instance of type [detachable] ST_UNSIGNED_HEXADECIMAL_FORMATTER */
extern T0* GE_new387(T1 initialize);
/* New instance of type [detachable] ST_UNSIGNED_DECIMAL_FORMATTER */
extern T0* GE_new388(T1 initialize);
/* New instance of type [detachable] ST_POINTER_FORMATTER */
extern T0* GE_new389(T1 initialize);
/* New instance of type [detachable] ST_STRING_FORMATTER */
extern T0* GE_new390(T1 initialize);
/* New instance of type [detachable] KL_ANY_ROUTINES */
extern T0* GE_new391(T1 initialize);
/* New instance of type [detachable] UC_STRING */
extern T0* GE_new392(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] NATURAL_8] */
extern T0* GE_new394(T6 a1, T1 initialize);
/* New instance of type [detachable] ARRAY [[attached] IMMUTABLE_STRING_32] */
extern T0* GE_new396(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] IMMUTABLE_STRING_32] */
extern T0* GE_new397(T6 a1, T1 initialize);
/* New instance of type detachable KL_EQUALITY_TESTER [[attached] XPLAIN_VALUE] */
extern T0* GE_new398(T1 initialize);
/* New instance of type detachable KL_EQUALITY_TESTER [[attached] STRING_8] */
extern T0* GE_new399(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_VALUE] */
extern T0* GE_new402(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] XPLAIN_VALUE] */
extern T0* GE_new403(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] XPLAIN_VALUE, [attached] STRING_8] */
extern T0* GE_new404(T1 initialize);
/* New instance of type detachable DS_LINKABLE [[attached] XPLAIN_SCANNER_STATE] */
extern T0* GE_new406(T1 initialize);
/* New instance of type [detachable] XPLAIN_ALL_ATTRIBUTES_CURSOR [[attached] XPLAIN_ATTRIBUTE] */
extern T0* GE_new407(T1 initialize);
/* New instance of type [detachable] XPLAIN_DATA_ATTRIBUTES_CURSOR */
extern T0* GE_new408(T1 initialize);
/* New instance of type [detachable] XPLAIN_ATTRIBUTE_EXPRESSION */
extern T0* GE_new410(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8] */
extern T0* GE_new412(T1 initialize);
/* New instance of type [detachable] DS_BILINKED_LIST_CURSOR [[attached] XPLAIN_STATEMENT] */
extern T0* GE_new414(T1 initialize);
/* New instance of type [detachable] DS_LINKED_STACK [[attached] STRING_8] */
extern T0* GE_new415(T1 initialize);
/* New instance of type [detachable] ARRAY [detachable STRING_8] */
extern T0* GE_new416(T1 initialize);
/* New instance of type [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_STATEMENT] */
extern T0* GE_new418(T1 initialize);
/* New instance of type detachable DS_LINKABLE [[attached] XPLAIN_STATEMENT] */
extern T0* GE_new419(T1 initialize);
/* New instance of type [detachable] STRING_TO_INTEGER_CONVERTOR */
extern T0* GE_new420(T1 initialize);
/* New instance of type [detachable] JOIN */
extern T0* GE_new423(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET [[attached] STRING_8] */
extern T0* GE_new424(T1 initialize);
/* New instance of type [detachable] XPLAIN_F_REPRESENTATION */
extern T0* GE_new425(T1 initialize);
/* New instance of type [detachable] EXCEPTIONS */
extern T0* GE_new427(T1 initialize);
/* New instance of type [detachable] XPLAIN_INIT_ATTRIBUTES_CURSOR [[attached] XPLAIN_ATTRIBUTE] */
extern T0* GE_new430(T1 initialize);
/* New instance of type [detachable] KL_INTEGER_ROUTINES */
extern T0* GE_new432(T1 initialize);
/* New instance of type [detachable] ARRAY [[attached] DS_CELL [[attached] INTEGER_32]] */
extern T0* GE_new433(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] DS_CELL [[attached] INTEGER_32]] */
extern T0* GE_new434(T6 a1, T1 initialize);
/* New instance of type [detachable] DS_CELL [[attached] REAL_64] */
extern T0* GE_new435(T1 initialize);
/* New instance of type [detachable] KL_CHARACTER_BUFFER */
extern T0* GE_new436(T1 initialize);
/* New instance of type [detachable] CONSOLE */
extern T0* GE_new437(T1 initialize);
/* New instance of type [detachable] JOIN_TREE_NODE */
extern T0* GE_new438(T1 initialize);
/* New instance of type [detachable] DS_LINKED_LIST [[attached] JOIN_TREE_NODE] */
extern T0* GE_new439(T1 initialize);
/* New instance of type detachable DS_LINKABLE [[attached] XPLAIN_PARAMETER] */
extern T0* GE_new440(T1 initialize);
/* New instance of type detachable DS_BILINKABLE [[attached] XPLAIN_STATEMENT] */
extern T0* GE_new441(T1 initialize);
/* New instance of type detachable DS_LINKABLE [[attached] XPLAIN_ATTRIBUTE] */
extern T0* GE_new442(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST [[attached] INTEGER_32] */
extern T0* GE_new445(T1 initialize);
/* New instance of type [detachable] KL_DOUBLE_ROUTINES */
extern T0* GE_new446(T1 initialize);
/* New instance of type [detachable] KL_PLATFORM */
extern T0* GE_new447(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] XPLAIN_ABSTRACT_OBJECT] */
extern T0* GE_new450(T6 a1, T1 initialize);
/* New instance of type detachable KL_EQUALITY_TESTER [[attached] XPLAIN_ABSTRACT_OBJECT] */
extern T0* GE_new451(T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ABSTRACT_OBJECT] */
extern T0* GE_new452(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE_CURSOR [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8] */
extern T0* GE_new454(T1 initialize);
/* New instance of type detachable DS_LINKABLE [[attached] STRING_8] */
extern T0* GE_new456(T1 initialize);
/* New instance of type [detachable] HEXADECIMAL_STRING_TO_INTEGER_CONVERTER */
extern T0* GE_new459(T1 initialize);
/* New instance of type [detachable] INTEGER_OVERFLOW_CHECKER */
extern T0* GE_new460(T1 initialize);
/* New instance of type [detachable] DS_HASH_SET_CURSOR [[attached] STRING_8] */
extern T0* GE_new461(T1 initialize);
/* New instance of type [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE] */
extern T0* GE_new462(T1 initialize);
/* New instance of type detachable DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE_NAME_NODE] */
extern T0* GE_new463(T1 initialize);
/* New instance of type detachable DS_LINKABLE [[attached] JOIN_TREE_NODE] */
extern T0* GE_new464(T1 initialize);
/* New instance of type [detachable] DS_LINKED_LIST_CURSOR [[attached] JOIN_TREE_NODE] */
extern T0* GE_new465(T1 initialize);
/* New instance of type [detachable] DS_ARRAYED_LIST_CURSOR [[attached] INTEGER_32] */
extern T0* GE_new466(T1 initialize);
/* New instance of type [detachable] XPLAIN_VIRTUAL_ATTRIBUTE */
extern T0* GE_new468(T1 initialize);
/* New instance of type [detachable] XPLAIN_EXTENSION_ATTRIBUTE */
extern T0* GE_new469(T1 initialize);
/* New instance of type [detachable] XPLAIN_ASSERTION_EXPRESSION */
extern T0* GE_new471(T1 initialize);
/* New instance of type [detachable] UC_UTF8_ROUTINES */
extern T0* GE_new473(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] NATURAL_64] */
extern T0* GE_new475(T6 a1, T1 initialize);
/* New instance of type detachable KL_EQUALITY_TESTER [[attached] XPLAIN_TYPE] */
extern T0* GE_new476(T1 initialize);
/* New instance of type detachable DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] XPLAIN_TYPE] */
extern T0* GE_new478(T1 initialize);
/* New instance of type [detachable] SPECIAL [[attached] XPLAIN_TYPE] */
extern T0* GE_new479(T6 a1, T1 initialize);
/* New instance of type [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_TYPE] */
extern T0* GE_new480(T1 initialize);
/* New instance of type [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_ATTRIBUTE_NAME_NODE] */
extern T0* GE_new481(T1 initialize);
/* New instance of type detachable DS_LINKABLE [[attached] XPLAIN_ATTRIBUTE_NAME_NODE] */
extern T0* GE_new482(T1 initialize);
/* [detachable] XPLAIN2SQL.make */
extern T0* T25c9(GE_context* ac);
/* [detachable] XML_GENERATOR.dump_statements */
extern void T69f19(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] KL_TEXT_OUTPUT_FILE.close */
extern void T348f28(GE_context* ac, T0* C);
/* [detachable] KL_TEXT_OUTPUT_FILE.old_close */
extern void T348f29(GE_context* ac, T0* C);
/* [detachable] KL_TEXT_OUTPUT_FILE.default_pointer */
extern T14 T348f16(GE_context* ac, T0* C);
/* [detachable] KL_TEXT_OUTPUT_FILE.file_close */
extern void T348f33(GE_context* ac, T0* C, T14 a1);
/* [detachable] KL_TEXT_OUTPUT_FILE.put_string */
extern void T348f27(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_TEXT_OUTPUT_FILE.old_put_string */
extern void T348f32(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_TEXT_OUTPUT_FILE.file_ps */
extern void T348f35(GE_context* ac, T0* C, T14 a1, T14 a2, T6 a3);
/* [detachable] KL_STRING_ROUTINES.as_string */
extern T0* T110f1(GE_context* ac, T0* C, T0* a1);
/* [detachable] STRING_8.string */
extern T0* T17f19(GE_context* ac, T0* C);
/* [detachable] STRING_8.make_from_string */
extern T0* T17c52(GE_context* ac, T0* a1);
/* [detachable] SPECIAL [CHARACTER_8].copy_data */
extern void T15f12(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3, T6 a4);
/* [detachable] SPECIAL [CHARACTER_8].force */
extern void T15f17(GE_context* ac, T0* C, T2 a1, T6 a2);
/* [detachable] SPECIAL [CHARACTER_8].extend */
extern void T15f18(GE_context* ac, T0* C, T2 a1);
/* [detachable] SPECIAL [CHARACTER_8].move_data */
extern void T15f16(GE_context* ac, T0* C, T6 a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [CHARACTER_8].overlapping_move */
extern void T15f20(GE_context* ac, T0* C, T6 a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [CHARACTER_8].fill_with */
extern void T15f14(GE_context* ac, T0* C, T2 a1, T6 a2, T6 a3);
/* INTEGER_32.min */
extern T6 T6f16(GE_context* ac, T6* C, T6 a1);
/* INTEGER_32.is_greater_equal */
extern T1 T6f9(GE_context* ac, T6* C, T6 a1);
/* [detachable] SPECIAL [CHARACTER_8].non_overlapping_move */
extern void T15f19(GE_context* ac, T0* C, T6 a1, T6 a2, T6 a3);
/* [detachable] STRING_8.area_lower */
extern T6 T17f9(GE_context* ac, T0* C);
/* [detachable] SPECIAL [CHARACTER_8].make_empty */
extern void T15f15(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [CHARACTER_8].make_empty */
extern T0* T15c15(GE_context* ac, T6 a1);
/* [detachable] KL_ANY_ROUTINES.same_types */
extern T1 T391f1(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] KL_STRING_ROUTINES.any_ */
extern T0* T110f3(GE_context* ac, T0* C);
/* [detachable] KL_ANY_ROUTINES.default_create */
extern T0* T391c2(GE_context* ac);
/* [detachable] KL_TEXT_OUTPUT_FILE.string_ */
extern T0* T348f10(GE_context* ac, T0* C);
/* [detachable] KL_STRING_ROUTINES.default_create */
extern T0* T110c8(GE_context* ac);
/* [detachable] XML_WRITER.xml */
extern T0* T347f15(GE_context* ac, T0* C);
/* [detachable] XML_GENERATOR.die */
extern void T69f21(GE_context* ac, T0* C, T6 a1);
/* [detachable] XML_GENERATOR.std */
extern T0* T69f4(GE_context* ac, T0* C);
/* [detachable] KL_STANDARD_FILES.default_create */
extern T0* T36c4(GE_context* ac);
/* [detachable] KL_TEXT_OUTPUT_FILE.is_open_write */
extern T1 T348f14(GE_context* ac, T0* C);
/* [detachable] KL_TEXT_OUTPUT_FILE.old_is_open_write */
extern T1 T348f9(GE_context* ac, T0* C);
/* [detachable] KL_TEXT_OUTPUT_FILE.open_write */
extern void T348f26(GE_context* ac, T0* C);
/* [detachable] KL_TEXT_OUTPUT_FILE.is_closed */
extern T1 T348f13(GE_context* ac, T0* C);
/* [detachable] KL_TEXT_OUTPUT_FILE.old_is_closed */
extern T1 T348f8(GE_context* ac, T0* C);
/* [detachable] KL_TEXT_OUTPUT_FILE.old_open_write */
extern void T348f31(GE_context* ac, T0* C);
/* [detachable] KL_TEXT_OUTPUT_FILE.open_write */
extern void T348f31p1(GE_context* ac, T0* C);
/* [detachable] KL_TEXT_OUTPUT_FILE.file_open */
extern T14 T348f19(GE_context* ac, T0* C, T14 a1, T6 a2);
/* [detachable] KL_TEXT_OUTPUT_FILE.internal_name_pointer */
extern T0* T348f18(GE_context* ac, T0* C);
/* [detachable] KL_TEXT_OUTPUT_FILE.string_name */
extern T0* T348f12(GE_context* ac, T0* C);
/* [detachable] STRING_8.as_string_8 */
extern T0* T17f13(GE_context* ac, T0* C);
/* [detachable] STRING_8.put_code */
extern void T17f40(GE_context* ac, T0* C, T10 a1, T6 a2);
/* [detachable] STRING_8.reset_hash_codes */
extern void T17f37(GE_context* ac, T0* C);
/* [detachable] STRING_8.valid_code */
extern T1 T17f24(GE_context* ac, T0* C, T10 a1);
/* NATURAL_32.is_less_equal */
extern T1 T10f4(GE_context* ac, T10* C, T10 a1);
/* INTEGER_32.to_natural_32 */
extern T10 T6f13(GE_context* ac, T6* C);
/* [detachable] STRING_8.code */
extern T10 T17f10(GE_context* ac, T0* C, T6 a1);
/* [detachable] STRING_8.set_count */
extern void T17f39(GE_context* ac, T0* C, T6 a1);
/* [detachable] STRING_8.make */
extern void T17f33(GE_context* ac, T0* C, T6 a1);
/* [detachable] STRING_8.make */
extern T0* T17c33(GE_context* ac, T6 a1);
/* [detachable] SPECIAL [CHARACTER_8].make_filled */
extern T0* T15c11(GE_context* ac, T2 a1, T6 a2);
/* [detachable] KL_TEXT_OUTPUT_FILE.make */
extern T0* T348c25(GE_context* ac, T0* a1);
/* [detachable] KL_TEXT_OUTPUT_FILE.old_make */
extern void T348f30(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_TEXT_OUTPUT_FILE.make_with_name */
extern void T348f34(GE_context* ac, T0* C, T0* a1);
/* [detachable] STRING_8.make_empty */
extern T0* T17c41(GE_context* ac);
/* [detachable] KL_TEXT_OUTPUT_FILE.set_name */
extern void T348f36(GE_context* ac, T0* C, T0* a1);
/* [detachable] FILE_INFO.file_name_to_pointer */
extern T0* T114f11(GE_context* ac, T0* C, T0* a1, T0* a2);
/* detachable C_STRING.set_string */
extern void T30f8(GE_context* ac, T0* C, T0* a1);
/* detachable C_STRING.set_substring */
extern void T30f9(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] MANAGED_POINTER.put_natural_8 */
extern void T113f12(GE_context* ac, T0* C, T8 a1, T6 a2);
/* POINTER.memory_copy */
extern void T14f11(GE_context* ac, T14* C, T14 a1, T6 a2);
/* POINTER.c_memcpy */
extern void T14f15(GE_context* ac, T14* C, T14 a1, T14 a2, T6 a3);
/* NATURAL_32.to_natural_8 */
extern T8 T10f12(GE_context* ac, T10* C);
/* detachable C_STRING.make_shared_from_pointer_and_count */
extern T0* T30c7(GE_context* ac, T14 a1, T6 a2);
/* [detachable] MANAGED_POINTER.share_from_pointer */
extern T0* T113c11(GE_context* ac, T14 a1, T6 a2);
/* [detachable] MANAGED_POINTER.increment_counter */
extern void T113f13(GE_context* ac, T0* C);
/* UTF_CONVERTER.escaped_utf_32_substring_into_utf_8_0_pointer */
extern void T372f9(GE_context* ac, T372* C, T0* a1, T6 a2, T6 a3, T0* a4, T6 a5, T0* a6);
/* UTF_CONVERTER.to_natural_32 */
extern T10 T372f5(GE_context* ac, T372* C, T0* a1);
/* [detachable] HEXADECIMAL_STRING_TO_INTEGER_CONVERTER.parsed_natural_32 */
extern T10 T459f12(GE_context* ac, T0* C);
/* [detachable] HEXADECIMAL_STRING_TO_INTEGER_CONVERTER.parse_string_with_type */
extern void T459f19(GE_context* ac, T0* C, T0* a1, T6 a2);
/* NATURAL_32.is_valid_character_8_code */
extern T1 T10f19(GE_context* ac, T10* C);
/* CHARACTER_32.is_character_8 */
extern T1 T3f2(GE_context* ac, T3* C);
/* [detachable] HEXADECIMAL_STRING_TO_INTEGER_CONVERTER.parse_character */
extern void T459f22(GE_context* ac, T0* C, T2 a1);
/* [detachable] HEXADECIMAL_STRING_TO_INTEGER_CONVERTER.overflowed */
extern T1 T459f16(GE_context* ac, T0* C);
/* [detachable] INTEGER_OVERFLOW_CHECKER.will_overflow */
extern T1 T460f7(GE_context* ac, T0* C, T11 a1, T11 a2, T6 a3, T6 a4);
/* NATURAL_64.is_greater */
extern T1 T11f5(GE_context* ac, T11* C, T11 a1);
/* [detachable] HEXADECIMAL_STRING_TO_INTEGER_CONVERTER.overflow_checker */
extern T0* T459f15(GE_context* ac, T0* C);
/* [detachable] INTEGER_OVERFLOW_CHECKER.make */
extern T0* T460c13(GE_context* ac);
/* NATURAL_64.to_natural_64 */
extern T11 T11f8(GE_context* ac, T11* C);
/* NATURAL_32.to_natural_64 */
extern T11 T10f22(GE_context* ac, T10* C);
/* NATURAL_16.to_natural_64 */
extern T11 T9f3(GE_context* ac, T9* C);
/* NATURAL_8.to_natural_64 */
extern T11 T8f7(GE_context* ac, T8* C);
/* INTEGER_64.to_natural_64 */
extern T11 T7f7(GE_context* ac, T7* C);
/* INTEGER_16.to_natural_64 */
extern T11 T5f3(GE_context* ac, T5* C);
/* INTEGER_8.to_natural_64 */
extern T11 T4f3(GE_context* ac, T4* C);
/* [detachable] SPECIAL [[attached] NATURAL_64].extend */
extern void T475f5(GE_context* ac, T0* C, T11 a1);
/* [detachable] SPECIAL [[attached] NATURAL_64].make_empty */
extern T0* T475c4(GE_context* ac, T6 a1);
/* [detachable] STRING_8.has */
extern T1 T17f23(GE_context* ac, T0* C, T2 a1);
/* INTEGER_32.to_natural_64 */
extern T11 T6f30(GE_context* ac, T6* C);
/* CHARACTER_8.is_less_equal */
extern T1 T2f21(GE_context* ac, T2* C, T2 a1);
/* CHARACTER_8.is_less */
extern T1 T2f24(GE_context* ac, T2* C, T2 a1);
/* CHARACTER_8.natural_32_code */
extern T10 T2f20(GE_context* ac, T2* C);
/* CHARACTER_8.is_greater_equal */
extern T1 T2f23(GE_context* ac, T2* C, T2 a1);
/* CHARACTER_8.is_hexa_digit */
extern T1 T2f22(GE_context* ac, T2* C);
/* NATURAL_8.is_greater */
extern T1 T8f2(GE_context* ac, T8* C, T8 a1);
/* CHARACTER_8.character_types */
extern T8 T2f9(GE_context* ac, T2* C, T6 a1);
/* CHARACTER_8.internal_character_types */
extern T0* T2f11(GE_context* ac, T2* C);
/* [detachable] SPECIAL [[attached] NATURAL_8].fill_with */
extern void T394f7(GE_context* ac, T0* C, T8 a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] NATURAL_8].extend */
extern void T394f6(GE_context* ac, T0* C, T8 a1);
/* [detachable] SPECIAL [[attached] NATURAL_8].make_empty */
extern void T394f5(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] NATURAL_8].make_empty */
extern T0* T394c5(GE_context* ac, T6 a1);
/* [detachable] HEXADECIMAL_STRING_TO_INTEGER_CONVERTER.reset */
extern void T459f18(GE_context* ac, T0* C, T6 a1);
/* UTF_CONVERTER.ctoi_convertor */
extern T0* T372f6(GE_context* ac, T372* C);
/* [detachable] HEXADECIMAL_STRING_TO_INTEGER_CONVERTER.set_trailing_separators_acceptable */
extern void T459f21(GE_context* ac, T0* C, T1 a1);
/* [detachable] HEXADECIMAL_STRING_TO_INTEGER_CONVERTER.set_leading_separators_acceptable */
extern void T459f20(GE_context* ac, T0* C, T1 a1);
/* [detachable] HEXADECIMAL_STRING_TO_INTEGER_CONVERTER.make */
extern T0* T459c17(GE_context* ac);
/* UTF_CONVERTER.is_hexa_decimal */
extern T1 T372f4(GE_context* ac, T372* C, T0* a1);
/* [detachable] HEXADECIMAL_STRING_TO_INTEGER_CONVERTER.is_integral_integer */
extern T1 T459f14(GE_context* ac, T0* C);
/* [detachable] STRING_8.character_32_item */
extern T3 T17f31(GE_context* ac, T0* C, T6 a1);
/* UTF_CONVERTER.utf_8_bytes_count */
extern T6 T372f1(GE_context* ac, T372* C, T0* a1, T6 a2, T6 a3);
/* INTEGER_32.do_nothing */
extern void T6f33(GE_context* ac, T6* C);
/* [detachable] FILE_INFO.multi_byte_to_utf_16 */
extern T6 T114f10(GE_context* ac, T0* C, T14 a1, T14 a2, T6 a3);
/* [detachable] FILE_INFO.default_pointer */
extern T14 T114f9(GE_context* ac, T0* C);
/* detachable C_STRING.item */
extern T14 T30f3(GE_context* ac, T0* C);
/* detachable C_STRING.make */
extern T0* T30c6(GE_context* ac, T0* a1);
/* detachable C_STRING.make_empty */
extern void T30f4(GE_context* ac, T0* C, T6 a1);
/* detachable C_STRING.make_empty */
extern T0* T30c4(GE_context* ac, T6 a1);
/* UTF_CONVERTER.escaped_utf_32_substring_into_utf_16_0_pointer */
extern void T372f8(GE_context* ac, T372* C, T0* a1, T6 a2, T6 a3, T0* a4, T6 a5, T0* a6);
/* [detachable] MANAGED_POINTER.put_natural_16 */
extern void T113f14(GE_context* ac, T0* C, T9 a1, T6 a2);
/* NATURAL_32.to_natural_16 */
extern T9 T10f14(GE_context* ac, T10* C);
/* NATURAL_32.is_greater */
extern T1 T10f5(GE_context* ac, T10* C, T10 a1);
/* UTF_CONVERTER.utf_16_bytes_count */
extern T6 T372f2(GE_context* ac, T372* C, T0* a1, T6 a2, T6 a3);
/* [detachable] MANAGED_POINTER.make */
extern T0* T113c9(GE_context* ac, T6 a1);
/* [detachable] EXCEPTIONS.raise */
extern void T427f2(GE_context* ac, T0* C, T0* a1);
/* [detachable] DEVELOPER_EXCEPTION.raise */
extern void T102f16(GE_context* ac, T0* C);
/* [detachable] ISE_EXCEPTION_MANAGER.raise */
extern void T21f19(GE_context* ac, T0* C, T0* a1);
/* [detachable] DEVELOPER_EXCEPTION.code */
extern T6 T102f8(GE_context* ac, T0* C);
/* [detachable] ISE_EXCEPTION_MANAGER.default_pointer */
extern T14 T21f9(GE_context* ac, T0* C);
/* [detachable] ISE_EXCEPTION_MANAGER.set_last_exception */
extern void T21f17(GE_context* ac, T0* C, T0* a1);
/* [detachable] CELL [detachable EXCEPTION].put */
extern void T26f2(GE_context* ac, T0* C, T0* a1);
/* [detachable] CELL [detachable EXCEPTION].put */
extern T0* T26c2(GE_context* ac, T0* a1);
/* [detachable] ISE_EXCEPTION_MANAGER.last_exception_cell */
extern T0* T21f2(GE_context* ac, T0* C);
/* [detachable] ISE_EXCEPTION_MANAGER.in_rescue */
extern T1 T21f8(GE_context* ac, T0* C);
/* [detachable] DEVELOPER_EXCEPTION.is_ignored */
extern T1 T102f7(GE_context* ac, T0* C);
/* [detachable] ISE_EXCEPTION_MANAGER.is_ignored */
extern T1 T21f13(GE_context* ac, T0* C, T0* a1);
/* [detachable] HASH_TABLE [[attached] INTEGER_32, [attached] INTEGER_32].has */
extern T1 T27f34(GE_context* ac, T0* C, T6 a1);
/* [detachable] HASH_TABLE [[attached] INTEGER_32, [attached] INTEGER_32].same_keys */
extern T1 T27f26(GE_context* ac, T0* C, T6 a1, T6 a2);
/* [detachable] HASH_TABLE [[attached] INTEGER_32, [attached] INTEGER_32].hash_code_of */
extern T6 T27f25(GE_context* ac, T0* C, T6 a1);
/* INTEGER_32.hash_code */
extern T6 T6f19(GE_context* ac, T6* C);
/* [detachable] ISE_EXCEPTION_MANAGER.ignored_exceptions */
extern T0* T21f3(GE_context* ac, T0* C);
/* [detachable] HASH_TABLE [[attached] INTEGER_32, [attached] INTEGER_32].make */
extern T0* T27c35(GE_context* ac, T6 a1);
/* [detachable] SPECIAL [[attached] INTEGER_32].make_filled */
extern T0* T107c7(GE_context* ac, T6 a1, T6 a2);
/* [detachable] SPECIAL [[attached] INTEGER_32].fill_with */
extern void T107f10(GE_context* ac, T0* C, T6 a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] INTEGER_32].extend */
extern void T107f11(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] BOOLEAN].make_filled */
extern T0* T108c4(GE_context* ac, T1 a1, T6 a2);
/* [detachable] SPECIAL [[attached] BOOLEAN].fill_with */
extern void T108f7(GE_context* ac, T0* C, T1 a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] BOOLEAN].extend */
extern void T108f9(GE_context* ac, T0* C, T1 a1);
/* [detachable] SPECIAL [[attached] BOOLEAN].make_empty */
extern void T108f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] INTEGER_32].make_empty */
extern void T107f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] INTEGER_32].make_empty */
extern T0* T107c6(GE_context* ac, T6 a1);
/* [detachable] PRIMES.higher_prime */
extern T6 T106f1(GE_context* ac, T0* C, T6 a1);
/* [detachable] PRIMES.is_prime */
extern T1 T106f3(GE_context* ac, T0* C, T6 a1);
/* [detachable] PRIMES.default_create */
extern T0* T106c5(GE_context* ac);
/* [detachable] DEVELOPER_EXCEPTION.exception_manager */
extern T0* T102f6(GE_context* ac, T0* C);
/* [detachable] DEVELOPER_EXCEPTION.default_create */
extern T0* T102c10(GE_context* ac);
/* [detachable] EXCEPTIONS.default_create */
extern T0* T427c1(GE_context* ac);
/* [detachable] MANAGED_POINTER.default_pointer */
extern T14 T113f4(GE_context* ac, T0* C);
/* POINTER.memory_calloc */
extern T14 T14f6(GE_context* ac, T14* C, T6 a1, T6 a2);
/* POINTER.c_calloc */
extern T14 T14f5(GE_context* ac, T14* C, T6 a1, T6 a2);
/* INTEGER_32.max */
extern T6 T6f6(GE_context* ac, T6* C, T6 a1);
/* [detachable] MANAGED_POINTER.resize */
extern void T113f10(GE_context* ac, T0* C, T6 a1);
/* POINTER.memory_set */
extern void T14f10(GE_context* ac, T14* C, T6 a1, T6 a2);
/* POINTER.c_memset */
extern void T14f14(GE_context* ac, T14* C, T14 a1, T6 a2, T6 a3);
/* POINTER.memory_realloc */
extern T14 T14f2(GE_context* ac, T14* C, T6 a1);
/* POINTER.c_realloc */
extern T14 T14f7(GE_context* ac, T14* C, T14 a1, T6 a2);
/* [detachable] KL_TEXT_OUTPUT_FILE.buffered_file_info */
extern T0* T348f23(GE_context* ac, T0* C);
/* [detachable] FILE_INFO.make */
extern T0* T114c13(GE_context* ac);
/* [detachable] FILE_INFO.make_filled_area */
extern void T114f15(GE_context* ac, T0* C, T8 a1, T6 a2);
/* [detachable] SPECIAL [[attached] NATURAL_8].make_filled */
extern T0* T394c4(GE_context* ac, T8 a1, T6 a2);
/* [detachable] FILE_INFO.stat_size */
extern T6 T114f7(GE_context* ac, T0* C);
/* [detachable] XML_WRITER.stop_tag */
extern void T347f26(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_STACK [[attached] STRING_8].remove */
extern void T415f6(GE_context* ac, T0* C);
/* [detachable] XML_WRITER.new_line_after_closing_tag */
extern void T347f32(GE_context* ac, T0* C, T0* a1);
/* [detachable] XML_WRITER.new_line */
extern void T347f34(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_STACK [[attached] STRING_8].item */
extern T0* T415f3(GE_context* ac, T0* C);
/* [detachable] XML_WRITER.extend */
extern void T347f31(GE_context* ac, T0* C, T0* a1);
/* [detachable] STRING_8.append_string */
extern void T17f43(GE_context* ac, T0* C, T0* a1);
/* [detachable] STRING_8.append */
extern void T17f34(GE_context* ac, T0* C, T0* a1);
/* [detachable] STRING_8.resize */
extern void T17f36(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [CHARACTER_8].aliased_resized_area_with_default */
extern T0* T15f3(GE_context* ac, T0* C, T2 a1, T6 a2);
/* [detachable] SPECIAL [CHARACTER_8].aliased_resized_area */
extern T0* T15f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] STRING_8.additional_space */
extern T6 T17f8(GE_context* ac, T0* C);
/* [detachable] STRING_8.capacity */
extern T6 T17f7(GE_context* ac, T0* C);
/* [detachable] STRING_8.is_empty */
extern T1 T17f14(GE_context* ac, T0* C);
/* [detachable] XML_WRITER.indent */
extern void T347f30(GE_context* ac, T0* C);
/* [detachable] STRING_8.make_filled */
extern T0* T17c44(GE_context* ac, T2 a1, T6 a2);
/* [detachable] STRING_8.fill_character */
extern void T17f50(GE_context* ac, T0* C, T2 a1);
/* [detachable] XML_WRITER.on_new_line */
extern T1 T347f11(GE_context* ac, T0* C);
/* [detachable] XML_WRITER.assure_last_tag_written */
extern void T347f29(GE_context* ac, T0* C);
/* [detachable] XML_WRITER.empty_tag_closing_chars */
extern T0* T347f12(GE_context* ac, T0* C);
/* [detachable] XML_WRITER.clear_attributes */
extern void T347f33(GE_context* ac, T0* C);
/* [detachable] ARRAY [detachable STRING_8].make_filled */
extern void T416f6(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] ARRAY [detachable STRING_8].make_filled */
extern T0* T416c6(GE_context* ac, T0* a1, T6 a2, T6 a3);
/* [detachable] ARRAY [detachable STRING_8].make_filled_area */
extern void T416f7(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable STRING_8].make_filled */
extern T0* T329c12(GE_context* ac, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable STRING_8].fill_with */
extern void T329f9(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable STRING_8].extend */
extern void T329f11(GE_context* ac, T0* C, T0* a1);
/* [detachable] SPECIAL [detachable STRING_8].make_empty */
extern void T329f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable STRING_8].make_empty */
extern T0* T329c6(GE_context* ac, T6 a1);
/* [detachable] ARRAY [[attached] STRING_8].make_filled */
extern void T38f7(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] ARRAY [[attached] STRING_8].make_filled */
extern T0* T38c7(GE_context* ac, T0* a1, T6 a2, T6 a3);
/* [detachable] ARRAY [[attached] STRING_8].make_filled_area */
extern void T38f8(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [[attached] STRING_8].make_filled */
extern T0* T39c10(GE_context* ac, T0* a1, T6 a2);
/* [detachable] SPECIAL [[attached] STRING_8].fill_with */
extern void T39f9(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] STRING_8].extend */
extern void T39f12(GE_context* ac, T0* C, T0* a1);
/* [detachable] SPECIAL [[attached] STRING_8].make_empty */
extern void T39f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] STRING_8].make_empty */
extern T0* T39c6(GE_context* ac, T6 a1);
/* [detachable] ARRAY [detachable STRING_8].item */
extern T0* T416f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] XML_WRITER.new_line_before_starting_tag */
extern void T347f35(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_STATEMENT].forth */
extern void T349f13(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_STATEMENT].cursor_forth */
extern void T349f17(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_STATEMENT].add_traversing_cursor */
extern void T349f18(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_STATEMENT].set_next_cursor */
extern void T418f9(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_STATEMENT].remove_traversing_cursor */
extern void T349f19(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_STATEMENT].set */
extern void T418f8(GE_context* ac, T0* C, T0* a1, T1 a2, T1 a3);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_STATEMENT].item_for_iteration */
extern T0* T349f7(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_STATEMENT].cursor_item */
extern T0* T349f5(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_STATEMENT].after */
extern T1 T349f8(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_STATEMENT].cursor_after */
extern T1 T349f9(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_STATEMENT].start */
extern void T349f12(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_STATEMENT].cursor_start */
extern void T349f16(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_STATEMENT].cursor_off */
extern T1 T349f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_STATEMENT].off */
extern T1 T418f6(GE_context* ac, T0* C);
/* [detachable] XML_WRITER.start_tag */
extern void T347f25(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_STACK [[attached] STRING_8].put */
extern void T415f5(GE_context* ac, T0* C, T0* a1);
/* detachable DS_LINKABLE [[attached] STRING_8].put_right */
extern void T456f4(GE_context* ac, T0* C, T0* a1);
/* detachable DS_LINKABLE [[attached] STRING_8].make */
extern T0* T456c3(GE_context* ac, T0* a1);
/* [detachable] XML_WRITER.add_header_iso_8859_1_encoding */
extern void T347f24(GE_context* ac, T0* C);
/* [detachable] XML_WRITER.add_header */
extern void T347f28(GE_context* ac, T0* C, T0* a1);
/* [detachable] XML_WRITER.make_with_capacity */
extern T0* T347c23(GE_context* ac, T6 a1);
/* [detachable] XML_WRITER.clear */
extern void T347f27(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_STACK [[attached] STRING_8].wipe_out */
extern void T415f7(GE_context* ac, T0* C);
/* [detachable] STRING_8.wipe_out */
extern void T17f42(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_STACK [[attached] STRING_8].make */
extern T0* T415c4(GE_context* ac);
/* [detachable] XPLAIN_PARSER.syntax_error */
extern T1 T67f313(GE_context* ac, T0* C);
/* detachable KL_TEXT_INPUT_FILE.close */
extern void T32f42(GE_context* ac, T0* C);
/* detachable KL_TEXT_INPUT_FILE.old_close */
extern void T32f43(GE_context* ac, T0* C);
/* detachable KL_TEXT_INPUT_FILE.default_pointer */
extern T14 T32f20(GE_context* ac, T0* C);
/* detachable KL_TEXT_INPUT_FILE.file_close */
extern void T32f46(GE_context* ac, T0* C, T14 a1);
/* [detachable] XPLAIN_PARSER.parse */
extern void T67f533(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.yy_pop_last_value */
extern void T67f545(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_PARSER.yy_push_error_value */
extern void T67f544(GE_context* ac, T0* C);
/* [detachable] KL_SPECIAL_ROUTINES [detachable ANY].force */
extern void T127f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [detachable ANY].fill_with */
extern void T126f7(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable ANY].extend */
extern void T126f9(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable ANY].aliased_resized_area */
extern T0* T127f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable ANY].aliased_resized_area */
extern T0* T126f3(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_PARSER.yy_do_action */
extern void T67f543(GE_context* ac, T0* C, T6 a1);
extern T1 T67f543ot1(T0* a1, T0** a2);
extern T1 T67f543ot2(T0* a1, T0** a2);
extern T1 T67f543ot3(T0* a1, T0** a2);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_IF_STATEMENT].force */
extern void T343f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_IF_STATEMENT].fill_with */
extern void T342f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_IF_STATEMENT].extend */
extern void T342f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_IF_STATEMENT].aliased_resized_area */
extern T0* T343f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable XPLAIN_IF_STATEMENT].aliased_resized_area */
extern T0* T342f4(GE_context* ac, T0* C, T6 a1);
/* detachable XPLAIN_IF_STATEMENT.make */
extern T0* T173c8(GE_context* ac, T0* a1, T0* a2, T0* a3);
/* detachable XPLAIN_IF_STATEMENT.new_statements */
extern T0* T173f4(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_BILINKED_LIST [[attached] XPLAIN_STATEMENT].put_last */
extern void T360f16(GE_context* ac, T0* C, T0* a1);
/* detachable DS_BILINKABLE [[attached] XPLAIN_STATEMENT].put_right */
extern void T441f5(GE_context* ac, T0* C, T0* a1);
/* detachable DS_BILINKABLE [[attached] XPLAIN_STATEMENT].attach_left */
extern void T441f6(GE_context* ac, T0* C, T0* a1);
/* detachable DS_BILINKABLE [[attached] XPLAIN_STATEMENT].make */
extern T0* T441c4(GE_context* ac, T0* a1);
/* [detachable] DS_BILINKED_LIST [[attached] XPLAIN_STATEMENT].make */
extern T0* T360c13(GE_context* ac);
/* [detachable] DS_BILINKED_LIST [[attached] XPLAIN_STATEMENT].set_internal_cursor */
extern void T360f17(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_BILINKED_LIST [[attached] XPLAIN_STATEMENT].new_cursor */
extern T0* T360f8(GE_context* ac, T0* C);
/* [detachable] DS_BILINKED_LIST_CURSOR [[attached] XPLAIN_STATEMENT].make */
extern T0* T414c8(GE_context* ac, T0* a1);
/* detachable XPLAIN_STATEMENT_NODE.make */
extern T0* T172c3(GE_context* ac, T0* a1, T0* a2);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_STATEMENT_NODE].force */
extern void T341f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_STATEMENT_NODE].fill_with */
extern void T332f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_STATEMENT_NODE].extend */
extern void T332f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_STATEMENT_NODE].aliased_resized_area */
extern T0* T341f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable XPLAIN_STATEMENT_NODE].aliased_resized_area */
extern T0* T332f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_SQL_STATEMENT].force */
extern void T340f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_SQL_STATEMENT].fill_with */
extern void T175f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_SQL_STATEMENT].extend */
extern void T175f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_SQL_STATEMENT].aliased_resized_area */
extern T0* T340f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [[attached] XPLAIN_SQL_STATEMENT].aliased_resized_area */
extern T0* T175f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_SQL_STATEMENT.make */
extern T0* T171c4(GE_context* ac, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_PARAMETER].force */
extern void T339f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_PARAMETER].fill_with */
extern void T337f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_PARAMETER].extend */
extern void T337f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_PARAMETER].aliased_resized_area */
extern T0* T339f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [[attached] XPLAIN_PARAMETER].aliased_resized_area */
extern T0* T337f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_PARAMETER.make */
extern T0* T170c8(GE_context* ac, T0* a1, T1 a2);
/* [detachable] XPLAIN_PARAMETER.set_object */
extern void T170f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARAMETER.old_make */
extern void T170f9(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_PARAMETER_NODE].force */
extern void T338f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_PARAMETER_NODE].fill_with */
extern void T336f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_PARAMETER_NODE].extend */
extern void T336f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_PARAMETER_NODE].aliased_resized_area */
extern T0* T338f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [[attached] XPLAIN_PARAMETER_NODE].aliased_resized_area */
extern T0* T336f4(GE_context* ac, T0* C, T6 a1);
/* detachable XPLAIN_PARAMETER_NODE.make */
extern T0* T169c4(GE_context* ac, T0* a1, T0* a2);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_PARAMETER_NODE].force */
extern void T335f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_PARAMETER_NODE].fill_with */
extern void T331f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_PARAMETER_NODE].extend */
extern void T331f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_PARAMETER_NODE].aliased_resized_area */
extern T0* T335f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable XPLAIN_PARAMETER_NODE].aliased_resized_area */
extern T0* T331f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_PROCEDURE_STATEMENT].aliased_resized_area */
extern T0* T334f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable XPLAIN_PROCEDURE_STATEMENT].aliased_resized_area */
extern T0* T177f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_PROCEDURE_STATEMENT].force */
extern void T334f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_PROCEDURE_STATEMENT].fill_with */
extern void T177f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_PROCEDURE_STATEMENT].extend */
extern void T177f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.new_procedure_statement */
extern T0* T67f343(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3, T0* a4);
/* detachable XPLAIN_PROCEDURE_STATEMENT.make */
extern T0* T168c4(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_PROCEDURE.make */
extern T0* T333c19(GE_context* ac, T0* a1, T0* a2, T6 a3, T0* a4);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_PARAMETER].put_last */
extern void T358f14(GE_context* ac, T0* C, T0* a1);
/* detachable DS_LINKABLE [[attached] XPLAIN_PARAMETER].put_right */
extern void T440f4(GE_context* ac, T0* C, T0* a1);
/* detachable DS_LINKABLE [[attached] XPLAIN_PARAMETER].make */
extern T0* T440c3(GE_context* ac, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_PARAMETER].make */
extern T0* T358c11(GE_context* ac);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_PARAMETER].set_internal_cursor */
extern void T358f15(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_PARAMETER].new_cursor */
extern T0* T358f5(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_PARAMETER].make */
extern T0* T359c8(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_PROCEDURE.abstract_make */
extern void T333f23(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] KL_SPECIAL_ROUTINES [detachable STRING_8].force */
extern void T330f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] KL_SPECIAL_ROUTINES [detachable STRING_8].aliased_resized_area */
extern T0* T330f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable STRING_8].aliased_resized_area */
extern T0* T329f4(GE_context* ac, T0* C, T6 a1);
/* detachable XPLAIN_EXPRESSION_NODE.make */
extern T0* T167c6(GE_context* ac, T0* a1, T0* a2, T0* a3);
/* detachable XPLAIN_EXPRESSION_NODE.make_node */
extern void T167f7(GE_context* ac, T0* C, T0* a1, T0* a2);
/* detachable XPLAIN_EXPRESSION_NODE.give_column_names */
extern void T167f8(GE_context* ac, T0* C, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXPRESSION_NODE].force */
extern void T328f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_EXPRESSION_NODE].fill_with */
extern void T228f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_EXPRESSION_NODE].extend */
extern void T228f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXPRESSION_NODE].aliased_resized_area */
extern T0* T328f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable XPLAIN_EXPRESSION_NODE].aliased_resized_area */
extern T0* T228f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_SOME_FUNCTION.default_create */
extern T0* T327c13(GE_context* ac);
/* [detachable] XPLAIN_NIL_FUNCTION.default_create */
extern T0* T326c13(GE_context* ac);
/* [detachable] XPLAIN_ANY_FUNCTION.default_create */
extern T0* T325c13(GE_context* ac);
/* [detachable] XPLAIN_COUNT_FUNCTION.default_create */
extern T0* T324c13(GE_context* ac);
/* [detachable] XPLAIN_TOTAL_FUNCTION.default_create */
extern T0* T323c13(GE_context* ac);
/* [detachable] XPLAIN_MIN_FUNCTION.default_create */
extern T0* T322c13(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_FUNCTION].force */
extern void T321f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_FUNCTION].fill_with */
extern void T302f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_FUNCTION].extend */
extern void T302f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_FUNCTION].aliased_resized_area */
extern T0* T321f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [[attached] XPLAIN_FUNCTION].aliased_resized_area */
extern T0* T302f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_MAX_FUNCTION.default_create */
extern T0* T320c13(GE_context* ac);
/* detachable XPLAIN_SORT_NODE.make */
extern T0* T165c4(GE_context* ac, T0* a1, T1 a2, T0* a3);
/* [detachable] XPLAIN_ATTRIBUTE_NAME.abstracttype */
extern T0* T141f6(GE_context* ac, T0* C);
extern T1 T141f6ot1(T0* a1, T0** a2);
/* detachable XPLAIN_SORT_NODE.make_node */
extern void T165f5(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SORT_NODE].force */
extern void T319f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_SORT_NODE].fill_with */
extern void T317f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_SORT_NODE].extend */
extern void T317f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SORT_NODE].aliased_resized_area */
extern T0* T319f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable XPLAIN_SORT_NODE].aliased_resized_area */
extern T0* T317f4(GE_context* ac, T0* C, T6 a1);
/* detachable XPLAIN_SELECTION_LIST.set_identification_text */
extern void T164f11(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SELECTION_LIST].force */
extern void T318f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_SELECTION_LIST].fill_with */
extern void T280f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_SELECTION_LIST].extend */
extern void T280f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SELECTION_LIST].aliased_resized_area */
extern T0* T318f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable XPLAIN_SELECTION_LIST].aliased_resized_area */
extern T0* T280f4(GE_context* ac, T0* C, T6 a1);
/* detachable XPLAIN_SELECTION_LIST.make */
extern T0* T164c10(GE_context* ac, T0* a1, T0* a2, T0* a3, T0* a4);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SELECTION].force */
extern void T316f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_SELECTION].fill_with */
extern void T304f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_SELECTION].extend */
extern void T304f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SELECTION].aliased_resized_area */
extern T0* T316f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable XPLAIN_SELECTION].aliased_resized_area */
extern T0* T304f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_EXTENSION_FUNCTION_EXPRESSION.make */
extern T0* T315c22(GE_context* ac, T0* a1, T0* a2);
/* [detachable] XPLAIN_ATTRIBUTE_NAME_NODE.last */
extern T0* T140f4(GE_context* ac, T0* C);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXTENSION_EXPRESSION].force */
extern void T314f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_EXTENSION_EXPRESSION].fill_with */
extern void T268f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_EXTENSION_EXPRESSION].extend */
extern void T268f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXTENSION_EXPRESSION].aliased_resized_area */
extern T0* T314f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable XPLAIN_EXTENSION_EXPRESSION].aliased_resized_area */
extern T0* T268f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_EXTENSION_EXPRESSION_EXPRESSION.make */
extern T0* T313c14(GE_context* ac, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_EXTEND_ATTRIBUTE].force */
extern void T312f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_EXTEND_ATTRIBUTE].fill_with */
extern void T310f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_EXTEND_ATTRIBUTE].extend */
extern void T310f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_EXTEND_ATTRIBUTE].aliased_resized_area */
extern T0* T312f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [[attached] XPLAIN_EXTEND_ATTRIBUTE].aliased_resized_area */
extern T0* T310f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_EXTEND_ATTRIBUTE.make */
extern T0* T161c3(GE_context* ac, T0* a1, T0* a2);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXTENSION].aliased_resized_area */
extern T0* T311f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable XPLAIN_EXTENSION].aliased_resized_area */
extern T0* T307f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXTENSION].force */
extern void T311f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_EXTENSION].fill_with */
extern void T307f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_EXTENSION].extend */
extern void T307f10(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_EXTENSION.make */
extern T0* T160c32(GE_context* ac, T0* a1, T0* a2, T0* a3, T0* a4);
/* detachable XPLAIN_TYPE.add_extension */
extern void T132f38(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE].put_last */
extern void T361f18(GE_context* ac, T0* C, T0* a1);
/* detachable DS_LINKABLE [[attached] XPLAIN_ATTRIBUTE].put_right */
extern void T442f4(GE_context* ac, T0* C, T0* a1);
/* detachable DS_LINKABLE [[attached] XPLAIN_ATTRIBUTE].make */
extern T0* T442c3(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_EXTENSION_ATTRIBUTE.make */
extern T0* T469c18(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_EXTENSION_ATTRIBUTE.inherited_make */
extern void T469f23(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T1 a4, T1 a5, T1 a6);
/* [detachable] XPLAIN_EXTENSION_FUNCTION_EXPRESSION.set_extension */
extern void T315f23(GE_context* ac, T0* C, T0* a1, T0* a2);
/* detachable XPLAIN_SELECTION_FUNCTION.set_predicate */
extern void T157f11(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_ATTRIBUTE_NAME_NODE.set_prefix_table */
extern void T140f7(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_ATTRIBUTE_EXPRESSION.make */
extern T0* T410c28(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_EXTENSION_FUNCTION_EXPRESSION.supports_left_outer_join */
extern T1 T315f14(GE_context* ac, T0* C);
/* [detachable] XPLAIN_EXTENSION_FUNCTION_EXPRESSION.set_extension */
extern void T315f23p1(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_EXTENSION_EXPRESSION_EXPRESSION.set_extension */
extern void T313f15(GE_context* ac, T0* C, T0* a1, T0* a2);
/* detachable XPLAIN_EXTENSION.make_abstract_type */
extern void T160f33(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_EXTENSION_EXPRESSION_EXPRESSION.representation */
extern T0* T313f4(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE].make */
extern T0* T361c17(GE_context* ac);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE].set_internal_cursor */
extern void T361f21(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE].new_cursor */
extern T0* T361f6(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_ATTRIBUTE].make */
extern T0* T362c10(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_UNIVERSE.has */
extern T1 T181f3(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].has */
extern T1 T412f29(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].search_position */
extern void T412f40(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].hash_position */
extern T6 T412f27(GE_context* ac, T0* C, T0* a1);
/* [detachable] STRING_8.hash_code */
extern T6 T17f30(GE_context* ac, T0* C);
/* detachable KL_EQUALITY_TESTER [[attached] STRING_8].test */
extern T1 T399f1(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].key_storage_item */
extern T0* T412f24(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].clashes_item */
extern T6 T412f34(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].slots_item */
extern T6 T412f35(GE_context* ac, T0* C, T6 a1);
/* detachable XPLAIN_TYPE.has_attribute */
extern T1 T132f25(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXTEND_STATEMENT].force */
extern void T309f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_EXTEND_STATEMENT].fill_with */
extern void T308f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_EXTEND_STATEMENT].extend */
extern void T308f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXTEND_STATEMENT].aliased_resized_area */
extern T0* T309f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable XPLAIN_EXTEND_STATEMENT].aliased_resized_area */
extern T0* T308f4(GE_context* ac, T0* C, T6 a1);
/* detachable XPLAIN_EXTEND_STATEMENT.make */
extern T0* T159c4(GE_context* ac, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_GET_STATEMENT].force */
extern void T306f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_GET_STATEMENT].fill_with */
extern void T305f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_GET_STATEMENT].extend */
extern void T305f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_GET_STATEMENT].aliased_resized_area */
extern T0* T306f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable XPLAIN_GET_STATEMENT].aliased_resized_area */
extern T0* T305f4(GE_context* ac, T0* C, T6 a1);
/* detachable XPLAIN_GET_STATEMENT.make */
extern T0* T158c4(GE_context* ac, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SELECTION_FUNCTION].force */
extern void T303f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_SELECTION_FUNCTION].fill_with */
extern void T288f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_SELECTION_FUNCTION].extend */
extern void T288f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SELECTION_FUNCTION].aliased_resized_area */
extern T0* T303f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable XPLAIN_SELECTION_FUNCTION].aliased_resized_area */
extern T0* T288f4(GE_context* ac, T0* C, T6 a1);
/* detachable XPLAIN_SELECTION_FUNCTION.make */
extern T0* T157c10(GE_context* ac, T0* a1, T0* a2, T0* a3, T0* a4);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_NON_ATTRIBUTE_EXPRESSION].force */
extern void T301f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_NON_ATTRIBUTE_EXPRESSION].fill_with */
extern void T300f7(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_NON_ATTRIBUTE_EXPRESSION].extend */
extern void T300f9(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_NON_ATTRIBUTE_EXPRESSION].aliased_resized_area */
extern T0* T301f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable XPLAIN_NON_ATTRIBUTE_EXPRESSION].aliased_resized_area */
extern T0* T300f3(GE_context* ac, T0* C, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_VALUE_SELECTION_STATEMENT].force */
extern void T299f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_VALUE_SELECTION_STATEMENT].fill_with */
extern void T298f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_VALUE_SELECTION_STATEMENT].extend */
extern void T298f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_VALUE_SELECTION_STATEMENT].aliased_resized_area */
extern T0* T299f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable XPLAIN_VALUE_SELECTION_STATEMENT].aliased_resized_area */
extern T0* T298f4(GE_context* ac, T0* C, T6 a1);
/* detachable XPLAIN_VALUE_SELECTION_STATEMENT.make */
extern T0* T155c4(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_LAST_AUTO_PK_EXPRESSION.make */
extern T0* T297c27(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_SELECTION_EXPRESSION.make */
extern T0* T295c27(GE_context* ac, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_VALUE_STATEMENT].force */
extern void T294f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_VALUE_STATEMENT].fill_with */
extern void T293f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_VALUE_STATEMENT].extend */
extern void T293f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_VALUE_STATEMENT].aliased_resized_area */
extern T0* T294f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [[attached] XPLAIN_VALUE_STATEMENT].aliased_resized_area */
extern T0* T293f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_PARSER.new_value_statement */
extern T0* T67f362(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] XPLAIN_VALUE_STATEMENT.make */
extern T0* T154c5(GE_context* ac, T0* a1);
/* detachable XPLAIN_VALUE.make */
extern T0* T264c8(GE_context* ac, T0* a1, T0* a2, T0* a3);
/* detachable XPLAIN_VALUE.inherited_make */
extern void T264f9(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_ATTRIBUTE_NAME.set_attribute */
extern void T141f18(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_UPDATE_STATEMENT].force */
extern void T292f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_UPDATE_STATEMENT].fill_with */
extern void T274f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_UPDATE_STATEMENT].extend */
extern void T274f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_UPDATE_STATEMENT].aliased_resized_area */
extern T0* T292f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable XPLAIN_UPDATE_STATEMENT].aliased_resized_area */
extern T0* T274f4(GE_context* ac, T0* C, T6 a1);
/* detachable XPLAIN_UPDATE_STATEMENT.make */
extern T0* T153c6(GE_context* ac, T0* a1, T0* a2, T0* a3);
/* [detachable] XPLAIN_PARSER.get_known_value */
extern T0* T67f359(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SUBJECT].force */
extern void T291f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_SUBJECT].fill_with */
extern void T277f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_SUBJECT].extend */
extern void T277f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SUBJECT].aliased_resized_area */
extern T0* T291f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable XPLAIN_SUBJECT].aliased_resized_area */
extern T0* T277f4(GE_context* ac, T0* C, T6 a1);
/* detachable XPLAIN_SUBJECT.make */
extern T0* T152c3(GE_context* ac, T0* a1, T0* a2);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_DELETE_STATEMENT].force */
extern void T290f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_DELETE_STATEMENT].fill_with */
extern void T275f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_DELETE_STATEMENT].extend */
extern void T275f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_DELETE_STATEMENT].aliased_resized_area */
extern T0* T290f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable XPLAIN_DELETE_STATEMENT].aliased_resized_area */
extern T0* T275f4(GE_context* ac, T0* C, T6 a1);
/* detachable XPLAIN_DELETE_STATEMENT.make */
extern T0* T151c5(GE_context* ac, T0* a1, T0* a2);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_CASCADE_FUNCTION_EXPRESSION].force */
extern void T289f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_CASCADE_FUNCTION_EXPRESSION].fill_with */
extern void T286f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_CASCADE_FUNCTION_EXPRESSION].extend */
extern void T286f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_CASCADE_FUNCTION_EXPRESSION].aliased_resized_area */
extern T0* T289f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable XPLAIN_CASCADE_FUNCTION_EXPRESSION].aliased_resized_area */
extern T0* T286f4(GE_context* ac, T0* C, T6 a1);
/* detachable XPLAIN_CASCADE_FUNCTION_EXPRESSION.make */
extern T0* T150c4(GE_context* ac, T0* a1, T0* a2);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_CASCADE_STATEMENT].aliased_resized_area */
extern T0* T287f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable XPLAIN_CASCADE_STATEMENT].aliased_resized_area */
extern T0* T276f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_CASCADE_STATEMENT].force */
extern void T287f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_CASCADE_STATEMENT].fill_with */
extern void T276f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_CASCADE_STATEMENT].extend */
extern void T276f10(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_CASCADE_STATEMENT.make */
extern T0* T149c14(GE_context* ac, T0* a1, T0* a2, T0* a3, T0* a4);
/* [detachable] XPLAIN_ATTRIBUTE_NAME.make_from_attribute */
extern T0* T141c19(GE_context* ac, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ASSIGNMENT].force */
extern void T285f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_ASSIGNMENT].fill_with */
extern void T283f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_ASSIGNMENT].extend */
extern void T283f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ASSIGNMENT].aliased_resized_area */
extern T0* T285f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [[attached] XPLAIN_ASSIGNMENT].aliased_resized_area */
extern T0* T283f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_ASSIGNMENT.make */
extern T0* T148c3(GE_context* ac, T0* a1, T0* a2);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_ASSIGNMENT_NODE].force */
extern void T284f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_ASSIGNMENT_NODE].fill_with */
extern void T278f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_ASSIGNMENT_NODE].extend */
extern void T278f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_ASSIGNMENT_NODE].aliased_resized_area */
extern T0* T284f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable XPLAIN_ASSIGNMENT_NODE].aliased_resized_area */
extern T0* T278f4(GE_context* ac, T0* C, T6 a1);
/* detachable XPLAIN_ASSIGNMENT_NODE.make */
extern T0* T147c3(GE_context* ac, T0* a1, T0* a2);
/* [detachable] XPLAIN_GET_INSERT_STATEMENT.make */
extern T0* T281c7(GE_context* ac, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] XPLAIN_INSERT_STATEMENT.make */
extern T0* T279c6(GE_context* ac, T0* a1, T0* a2, T0* a3);
/* [detachable] KL_STDERR_FILE.put_line */
extern void T42f8(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_STDERR_FILE.put_new_line */
extern void T42f10(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.std */
extern T0* T67f341(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.warn_attributes_with_inits */
extern void T67f562(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_ATTRIBUTE_NAME_NODE].force */
extern void T273f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_ATTRIBUTE_NAME_NODE].fill_with */
extern void T271f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_ATTRIBUTE_NAME_NODE].extend */
extern void T271f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_ATTRIBUTE_NAME_NODE].aliased_resized_area */
extern T0* T273f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable XPLAIN_ATTRIBUTE_NAME_NODE].aliased_resized_area */
extern T0* T271f4(GE_context* ac, T0* C, T6 a1);
/* detachable XPLAIN_INDEX.make */
extern T0* T272c6(GE_context* ac, T0* a1, T0* a2, T1 a3, T1 a4, T0* a5);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_ASSERTION].aliased_resized_area */
extern T0* T269f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable XPLAIN_ASSERTION].aliased_resized_area */
extern T0* T270f3(GE_context* ac, T0* C, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_ASSERTION].force */
extern void T269f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_ASSERTION].fill_with */
extern void T270f7(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_ASSERTION].extend */
extern void T270f9(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_ASSERTION.is_complex */
extern T1 T146f11(GE_context* ac, T0* C);
/* [detachable] XPLAIN_EXTENSION_FUNCTION_EXPRESSION.uses_non_data_attributes */
extern T1 T315f6(GE_context* ac, T0* C);
/* [detachable] XPLAIN_EXTENSION_EXPRESSION_EXPRESSION.uses_non_data_attributes */
extern T1 T313f3(GE_context* ac, T0* C);
/* detachable XPLAIN_ASSERTION.is_function */
extern T1 T146f14(GE_context* ac, T0* C);
extern T1 T146f14ot1(T0* a1);
/* detachable XPLAIN_ASSERTION.make */
extern T0* T146c32(GE_context* ac, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5);
/* detachable XPLAIN_TYPE.add_assertion */
extern void T132f37(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_VIRTUAL_ATTRIBUTE.make */
extern T0* T468c18(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_VIRTUAL_ATTRIBUTE.inherited_make */
extern void T468f23(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T1 a4, T1 a5, T1 a6);
/* detachable XPLAIN_ASSERTION.make_abstract_type */
extern void T146f33(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_DOMAIN_RESTRICTION].force */
extern void T267f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_DOMAIN_RESTRICTION].fill_with */
extern void T266f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_DOMAIN_RESTRICTION].extend */
extern void T266f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_DOMAIN_RESTRICTION].aliased_resized_area */
extern T0* T267f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable XPLAIN_DOMAIN_RESTRICTION].aliased_resized_area */
extern T0* T266f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_VALUE_EXPRESSION.make */
extern T0* T265c27(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_UNIVERSE.find_value */
extern T0* T181f2(GE_context* ac, T0* C, T0* a1);
extern T1 T181f2ot1(T0* a1, T0** a2);
/* [detachable] XPLAIN_UNIVERSE.find_object */
extern T0* T181f5(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].found_item */
extern T0* T412f28(GE_context* ac, T0* C);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].item_storage_item */
extern T0* T412f22(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].found */
extern T1 T412f21(GE_context* ac, T0* C);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].search */
extern void T412f37(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_VARIABLE_EXPRESSION.make */
extern T0* T263c27(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_PARSER.silence_compiler_logical_expression */
extern T0* T67f322(GE_context* ac, T0* C);
/* [detachable] XPLAIN_ATTRIBUTE_NAME.set_object */
extern void T141f17(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.get_known_base_or_type */
extern T0* T67f321(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_UNIVERSE.find_base_or_type */
extern T0* T181f4(GE_context* ac, T0* C, T0* a1);
extern T1 T181f4ot1(T0* a1, T0** a2);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ATTRIBUTE].force */
extern void T262f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_ATTRIBUTE].fill_with */
extern void T259f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_ATTRIBUTE].extend */
extern void T259f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ATTRIBUTE].aliased_resized_area */
extern T0* T262f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [[attached] XPLAIN_ATTRIBUTE].aliased_resized_area */
extern T0* T259f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_ATTRIBUTE.make */
extern T0* T145c19(GE_context* ac, T0* a1, T0* a2, T1 a3, T1 a4, T1 a5, T1 a6);
/* [detachable] XPLAIN_ATTRIBUTE_NAME.abstracttype_if_known */
extern T0* T141f7(GE_context* ac, T0* C);
extern T1 T141f7ot1(T0* a1, T0** a2);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_ATTRIBUTE_NODE].force */
extern void T260f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_ATTRIBUTE_NODE].fill_with */
extern void T191f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_ATTRIBUTE_NODE].extend */
extern void T191f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_ATTRIBUTE_NODE].aliased_resized_area */
extern T0* T260f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable XPLAIN_ATTRIBUTE_NODE].aliased_resized_area */
extern T0* T191f4(GE_context* ac, T0* C, T6 a1);
/* detachable XPLAIN_ATTRIBUTE_NODE.make */
extern T0* T144c3(GE_context* ac, T0* a1, T0* a2);
/* detachable XPLAIN_I_REFERENCES.make */
extern T0* T258c4(GE_context* ac);
/* detachable XPLAIN_I_REFERENCES.make_xplain_domain_restriction */
extern void T258f7(GE_context* ac, T0* C, T1 a1);
/* [detachable] XPLAIN_PK_I_REPRESENTATION.make */
extern T0* T257c15(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_PK_REPRESENTATION].force */
extern void T256f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_PK_REPRESENTATION].fill_with */
extern void T192f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_PK_REPRESENTATION].extend */
extern void T192f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_PK_REPRESENTATION].aliased_resized_area */
extern T0* T256f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [[attached] XPLAIN_PK_REPRESENTATION].aliased_resized_area */
extern T0* T192f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_PK_I_REPRESENTATION.set_domain_restriction */
extern void T257f16(GE_context* ac, T0* C, T0* a1, T0* a2);
/* detachable XPLAIN_I_REFERENCES.check_attachment */
extern void T258f5(GE_context* ac, T0* C, T0* a1, T0* a2);
/* detachable XPLAIN_A_REFERENCES.check_attachment */
extern void T255f5(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_PK_A_REPRESENTATION.set_domain_restriction */
extern void T254f17(GE_context* ac, T0* C, T0* a1, T0* a2);
/* detachable XPLAIN_A_REFERENCES.make */
extern T0* T255c4(GE_context* ac);
/* detachable XPLAIN_A_REFERENCES.make_xplain_domain_restriction */
extern void T255f6(GE_context* ac, T0* C, T1 a1);
/* [detachable] XPLAIN_PK_A_REPRESENTATION.make */
extern T0* T254c16(GE_context* ac, T6 a1);
/* [detachable] XPLAIN_T_REPRESENTATION.default_create */
extern T0* T253c12(GE_context* ac);
/* [detachable] XPLAIN_R_REPRESENTATION.make */
extern T0* T252c13(GE_context* ac, T6 a1, T6 a2);
/* [detachable] XPLAIN_P_REPRESENTATION.default_create */
extern T0* T251c12(GE_context* ac);
/* [detachable] XPLAIN_M_REPRESENTATION.default_create */
extern T0* T250c11(GE_context* ac);
/* [detachable] XPLAIN_I_REPRESENTATION.make */
extern T0* T249c12(GE_context* ac, T6 a1);
/* [detachable] XPLAIN_D_REPRESENTATION.default_create */
extern T0* T248c11(GE_context* ac);
/* [detachable] XPLAIN_C_REPRESENTATION.make */
extern T0* T247c13(GE_context* ac, T6 a1);
/* [detachable] XPLAIN_B_REPRESENTATION.default_create */
extern T0* T246c12(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_REPRESENTATION].force */
extern void T245f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_REPRESENTATION].fill_with */
extern void T188f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_REPRESENTATION].extend */
extern void T188f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_REPRESENTATION].aliased_resized_area */
extern T0* T245f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [[attached] XPLAIN_REPRESENTATION].aliased_resized_area */
extern T0* T188f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_A_REPRESENTATION.make */
extern T0* T204c13(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXPRESSION].force */
extern void T244f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_EXPRESSION].fill_with */
extern void T243f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_EXPRESSION].extend */
extern void T243f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXPRESSION].aliased_resized_area */
extern T0* T244f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable XPLAIN_EXPRESSION].aliased_resized_area */
extern T0* T243f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_PARSER.format */
extern T0* T67f305(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_SCIENTIFIC_FORMATTER.format */
extern T0* T105f21(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] KL_STRING_OUTPUT_STREAM.set_string */
extern void T376f4(GE_context* ac, T0* C, T0* a1);
/* [detachable] ST_SCIENTIFIC_FORMATTER.do_format_to */
extern void T105f30(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T105f30ot1(T0* a1, T0** a2);
extern T1 T105f30ot2(T0* a1, T0** a2);
/* [detachable] KL_STRING_OUTPUT_STREAM.put_string */
extern void T376f7(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_STRING_OUTPUT_STREAM.string_ */
extern T0* T376f2(GE_context* ac, T0* C);
/* [detachable] KL_NULL_TEXT_OUTPUT_STREAM.put_string */
extern void T374f5(GE_context* ac, T0* C, T0* a1);
/* CHARACTER_8.out */
extern T0* T2f7(GE_context* ac, T2* C);
/* [detachable] STRING_8.append_character */
extern void T17f46(GE_context* ac, T0* C, T2 a1);
/* [detachable] ST_SCIENTIFIC_FORMATTER.internal_parameter_formatter */
extern T0* T105f16(GE_context* ac, T0* C, T2 a1);
/* [detachable] KL_STRING_OUTPUT_STREAM.put_character */
extern void T376f6(GE_context* ac, T0* C, T2 a1);
/* [detachable] KL_NULL_TEXT_OUTPUT_STREAM.put_character */
extern void T374f4(GE_context* ac, T0* C, T2 a1);
/* [detachable] ST_SCIENTIFIC_FORMATTER.set_error */
extern void T105f33(GE_context* ac, T0* C, T0* a1);
/* [detachable] ST_SCIENTIFIC_FORMATTER.has_error */
extern T1 T105f17(GE_context* ac, T0* C);
/* [detachable] KL_STRING_OUTPUT_STREAM.put_substring */
extern void T376f5(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] KL_NULL_TEXT_OUTPUT_STREAM.put_substring */
extern void T374f3(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] ST_SCIENTIFIC_FORMATTER.reset_error */
extern void T105f32(GE_context* ac, T0* C);
/* [detachable] KL_STRING_ROUTINES.new_empty_string */
extern T0* T110f2(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] KL_STRING_ROUTINES.cloned_string */
extern T0* T110f5(GE_context* ac, T0* C, T0* a1);
/* [detachable] STRING_8.twin */
extern T0* T17f20(GE_context* ac, T0* C);
/* [detachable] STRING_8.copy */
extern void T17f54(GE_context* ac, T0* C, T0* a1);
/* [detachable] SPECIAL [CHARACTER_8].resized_area */
extern T0* T15f7(GE_context* ac, T0* C, T6 a1);
/* [detachable] ST_SCIENTIFIC_FORMATTER.string_ */
extern T0* T105f18(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.formatter */
extern T0* T67f520(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.scientific_formatter */
extern T0* T67f528(GE_context* ac, T0* C);
/* [detachable] ST_SCIENTIFIC_FORMATTER.make */
extern T0* T105c29(GE_context* ac);
/* [detachable] ST_SCIENTIFIC_FORMATTER.make_with_escape */
extern void T105f31(GE_context* ac, T0* C, T2 a1);
/* [detachable] ST_STRING_FORMATTER.make */
extern T0* T390c12(GE_context* ac);
/* [detachable] ST_POINTER_FORMATTER.make */
extern T0* T389c12(GE_context* ac);
/* [detachable] ST_UNSIGNED_DECIMAL_FORMATTER.make */
extern T0* T388c19(GE_context* ac);
/* [detachable] ST_UNSIGNED_DECIMAL_FORMATTER.make */
extern void T388f19p1(GE_context* ac, T0* C);
/* [detachable] ST_UNSIGNED_HEXADECIMAL_FORMATTER.make */
extern T0* T387c19(GE_context* ac);
/* [detachable] ST_UNSIGNED_HEXADECIMAL_FORMATTER.make */
extern void T387f19p1(GE_context* ac, T0* C);
/* [detachable] ST_UNSIGNED_OCTAL_FORMATTER.make */
extern T0* T386c19(GE_context* ac);
/* [detachable] ST_UNSIGNED_OCTAL_FORMATTER.make */
extern void T386f19p1(GE_context* ac, T0* C);
/* [detachable] ST_DECIMAL_FORMATTER.make */
extern T0* T385c19(GE_context* ac);
/* [detachable] ST_DECIMAL_FORMATTER.make */
extern void T385f19p1(GE_context* ac, T0* C);
/* [detachable] ST_FIXED_OR_FLOATING_POINT_FORMATTER.make */
extern T0* T384c25(GE_context* ac);
/* [detachable] DS_ARRAYED_LIST [[attached] INTEGER_32].make */
extern T0* T445c10(GE_context* ac, T6 a1);
/* [detachable] DS_ARRAYED_LIST [[attached] INTEGER_32].set_internal_cursor */
extern void T445f14(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_ARRAYED_LIST [[attached] INTEGER_32].new_cursor */
extern T0* T445f6(GE_context* ac, T0* C);
/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] INTEGER_32].make */
extern T0* T466c5(GE_context* ac, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] INTEGER_32].make */
extern T0* T118f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] INTEGER_32].default_create */
extern T0* T118c5(GE_context* ac);
/* [detachable] KL_STRING_OUTPUT_STREAM.make */
extern T0* T376c3(GE_context* ac, T0* a1);
/* [detachable] ST_FIXED_OR_FLOATING_POINT_FORMATTER.make */
extern void T384f25p1(GE_context* ac, T0* C);
/* [detachable] ST_FIXED_POINT_FORMATTER.make */
extern T0* T383c24(GE_context* ac);
/* [detachable] ST_FIXED_POINT_FORMATTER.make */
extern void T383f24p1(GE_context* ac, T0* C);
/* [detachable] ST_FLOATING_POINT_FORMATTER.make */
extern T0* T382c25(GE_context* ac);
/* [detachable] ST_FLOATING_POINT_FORMATTER.make */
extern void T382f25p1(GE_context* ac, T0* C);
/* [detachable] ST_CHARACTER_FORMATTER.make */
extern T0* T381c13(GE_context* ac);
/* [detachable] ST_BOOLEAN_FORMATTER.make */
extern T0* T380c16(GE_context* ac);
/* [detachable] ST_SCIENTIFIC_FORMATTER.make_with_escape */
extern void T105f31p1(GE_context* ac, T0* C, T2 a1);
/* [detachable] ARRAY [[attached] ANY].make_filled */
extern T0* T40c5(GE_context* ac, T0* a1, T6 a2, T6 a3);
/* [detachable] ARRAY [[attached] ANY].make_filled_area */
extern void T40f6(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [[attached] ANY].make_filled */
extern T0* T41c4(GE_context* ac, T0* a1, T6 a2);
/* [detachable] SPECIAL [[attached] ANY].fill_with */
extern void T41f6(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] ANY].extend */
extern void T41f8(GE_context* ac, T0* C, T0* a1);
/* [detachable] SPECIAL [[attached] ANY].make_empty */
extern void T41f5(GE_context* ac, T0* C, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ATTRIBUTE_NAME].force */
extern void T242f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_ATTRIBUTE_NAME].fill_with */
extern void T196f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_ATTRIBUTE_NAME].extend */
extern void T196f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ATTRIBUTE_NAME].aliased_resized_area */
extern T0* T242f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [[attached] XPLAIN_ATTRIBUTE_NAME].aliased_resized_area */
extern T0* T196f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_ATTRIBUTE_NAME.make */
extern T0* T141c16(GE_context* ac, T0* a1, T0* a2);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ATTRIBUTE_NAME_NODE].force */
extern void T241f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_ATTRIBUTE_NAME_NODE].fill_with */
extern void T212f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_ATTRIBUTE_NAME_NODE].extend */
extern void T212f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ATTRIBUTE_NAME_NODE].aliased_resized_area */
extern T0* T241f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [[attached] XPLAIN_ATTRIBUTE_NAME_NODE].aliased_resized_area */
extern T0* T212f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_ATTRIBUTE_NAME_NODE.make */
extern T0* T140c5(GE_context* ac, T0* a1, T0* a2);
/* [detachable] XPLAIN_SQL_EXPRESSION.make */
extern T0* T240c27(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_USER_FUNCTION.make */
extern T0* T239c28(GE_context* ac, T0* a1, T0* a2);
/* [detachable] XPLAIN_UNMANAGED_PARAMETER_EXPRESSION.make */
extern T0* T238c27(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_ATTRIBUTE_NAME.full_name */
extern T0* T141f5(GE_context* ac, T0* C);
/* [detachable] XPLAIN_NUMBER_EXPRESSION.make */
extern T0* T237c27(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_STRING_EXPRESSION.make */
extern T0* T234c27(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_ID_FUNCTION.make */
extern T0* T236c27(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_NEWDATE_FUNCTION.make */
extern T0* T235c29(GE_context* ac, T0* a1, T0* a2, T0* a3);
/* [detachable] XPLAIN_PARSER.newdate_default_days */
extern T0* T67f383(GE_context* ac, T0* C);
/* [detachable] XPLAIN_STRING_FUNCTION.make */
extern T0* T233c27(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_REAL_FUNCTION.make */
extern T0* T232c27(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_INTEGER_FUNCTION.make */
extern T0* T231c27(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_DATEF_FUNCTION.make */
extern T0* T230c27(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_COMBINE_FUNCTION.make */
extern T0* T229c27(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_PARSER.silence_compiler_expression */
extern T0* T67f361(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PREFIX_EXPRESSION.make */
extern T0* T227c28(GE_context* ac, T0* a1, T0* a2);
/* [detachable] XPLAIN_PREFIX_EXPRESSION.make_one_operand */
extern void T227f29(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_IF_EXPRESSION.make */
extern T0* T226c29(GE_context* ac, T0* a1, T0* a2, T0* a3);
/* [detachable] XPLAIN_INFIX_EXPRESSION.make */
extern T0* T218c29(GE_context* ac, T0* a1, T0* a2, T0* a3);
/* [detachable] XPLAIN_INFIX_EXPRESSION.make_two_operands */
extern void T218f30(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_LOGINNAME_EXPRESSION.default_create */
extern T0* T225c26(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_SYSTEM_EXPRESSION].force */
extern void T224f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_SYSTEM_EXPRESSION].fill_with */
extern void T223f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_SYSTEM_EXPRESSION].extend */
extern void T223f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_SYSTEM_EXPRESSION].aliased_resized_area */
extern T0* T224f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [[attached] XPLAIN_SYSTEM_EXPRESSION].aliased_resized_area */
extern T0* T223f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_SYSTEMDATE_EXPRESSION.default_create */
extern T0* T222c26(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_LOGICAL_VALUE_EXPRESSION].force */
extern void T221f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_LOGICAL_VALUE_EXPRESSION].fill_with */
extern void T220f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_LOGICAL_VALUE_EXPRESSION].extend */
extern void T220f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_LOGICAL_VALUE_EXPRESSION].aliased_resized_area */
extern T0* T221f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [[attached] XPLAIN_LOGICAL_VALUE_EXPRESSION].aliased_resized_area */
extern T0* T220f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_PARENTHESIS_EXPRESSION.make */
extern T0* T219c27(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_PARSER.new_logical_infix_expression */
extern T0* T67f353(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] XPLAIN_NOT_LIKE_EXPRESSION.make */
extern T0* T369c30(GE_context* ac, T0* a1, T0* a2);
/* [detachable] XPLAIN_NOT_LIKE_EXPRESSION.make_infix */
extern void T369f32(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] XPLAIN_NOT_LIKE_EXPRESSION.make_two_operands */
extern void T369f33(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_NOT_LIKE_EXPRESSION.xplain_operator */
extern T0* T369f24(GE_context* ac, T0* C);
/* [detachable] XPLAIN_LIKE_EXPRESSION.make */
extern T0* T368c30(GE_context* ac, T0* a1, T0* a2);
/* [detachable] XPLAIN_LIKE_EXPRESSION.make_infix */
extern void T368f32(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] XPLAIN_LIKE_EXPRESSION.make_two_operands */
extern void T368f33(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_LIKE_EXPRESSION.xplain_operator */
extern T0* T368f23(GE_context* ac, T0* C);
/* [detachable] STRING_8.same_string */
extern T1 T17f22(GE_context* ac, T0* C, T0* a1);
/* [detachable] STRING_8.same_characters */
extern T1 T17f27(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3, T6 a4);
/* [detachable] SPECIAL [CHARACTER_8].same_items */
extern T1 T15f6(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3, T6 a4);
/* [detachable] XPLAIN_NE_NULL_EXPRESSION.make */
extern T0* T217c28(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_EQ_NULL_EXPRESSION.make */
extern T0* T216c28(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_PARAMETER_EXPRESSION.make */
extern T0* T215c27(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_PARSER.is_parameter_declared */
extern T1 T67f352(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_PARAMETER_NODE.has_parameter */
extern T1 T169f3(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_NOT_EXPRESSION.make */
extern T0* T214c29(GE_context* ac, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_EXPRESSION].aliased_resized_area */
extern T0* T211f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [[attached] XPLAIN_EXPRESSION].aliased_resized_area */
extern T0* T186f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_NOTNOT_EXPRESSION.make */
extern T0* T213c27(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_PARSER.get_object_if_valid_tree */
extern T0* T67f349(GE_context* ac, T0* C, T0* a1);
extern T1 T67f349ot1(T0* a1, T0** a2);
extern T1 T67f349ot2(T0* a1, T0** a2);
extern T1 T67f349ot3(T0* a1, T0** a2);
/* [detachable] XPLAIN_LOGICAL_VALUE_EXPRESSION.make */
extern T0* T138c27(GE_context* ac, T1 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_EXPRESSION].force */
extern void T211f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_EXPRESSION].fill_with */
extern void T186f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_EXPRESSION].extend */
extern void T186f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_LOGICAL_INFIX_EXPRESSION.make */
extern T0* T210c30(GE_context* ac, T0* a1, T0* a2, T0* a3);
/* [detachable] XPLAIN_LOGICAL_INFIX_EXPRESSION.make_two_operands */
extern void T210f31(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_LOGICAL_EXPRESSION].aliased_resized_area */
extern T0* T209f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [[attached] XPLAIN_LOGICAL_EXPRESSION].aliased_resized_area */
extern T0* T208f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_LOGICAL_EXPRESSION.make */
extern T0* T136c26(GE_context* ac, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_LOGICAL_EXPRESSION].force */
extern void T209f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_LOGICAL_EXPRESSION].fill_with */
extern void T208f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_LOGICAL_EXPRESSION].extend */
extern void T208f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.new_logical_expression */
extern T0* T67f344(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] STRING_8].aliased_resized_area */
extern T0* T128f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [[attached] STRING_8].aliased_resized_area */
extern T0* T39f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] STRING_8.insert_character */
extern void T17f45(GE_context* ac, T0* C, T2 a1, T6 a2);
/* [detachable] XPLAIN_TRAJECTORY.make */
extern T0* T207c7(GE_context* ac, T0* a1, T0* a2);
/* [detachable] XPLAIN_TRAJECTORY.make_xplain_domain_restriction */
extern void T207f8(GE_context* ac, T0* C, T1 a1);
/* [detachable] STRING_8.to_integer */
extern T6 T17f17(GE_context* ac, T0* C);
/* [detachable] STRING_TO_INTEGER_CONVERTOR.parsed_integer */
extern T6 T420f13(GE_context* ac, T0* C);
/* [detachable] STRING_TO_INTEGER_CONVERTOR.parse_string_with_type */
extern void T420f17(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] STRING_TO_INTEGER_CONVERTOR.parse_character */
extern void T420f23(GE_context* ac, T0* C, T2 a1);
/* [detachable] STRING_TO_INTEGER_CONVERTOR.overflowed */
extern T1 T420f15(GE_context* ac, T0* C);
/* [detachable] STRING_TO_INTEGER_CONVERTOR.overflow_checker */
extern T0* T420f14(GE_context* ac, T0* C);
/* CHARACTER_8.is_digit */
extern T1 T2f19(GE_context* ac, T2* C);
/* [detachable] STRING_TO_INTEGER_CONVERTOR.reset */
extern void T420f22(GE_context* ac, T0* C, T6 a1);
/* [detachable] STRING_8.ctoi_convertor */
extern T0* T17f25(GE_context* ac, T0* C);
/* [detachable] STRING_TO_INTEGER_CONVERTOR.set_trailing_separators_acceptable */
extern void T420f21(GE_context* ac, T0* C, T1 a1);
/* [detachable] STRING_TO_INTEGER_CONVERTOR.set_leading_separators_acceptable */
extern void T420f20(GE_context* ac, T0* C, T1 a1);
/* [detachable] STRING_TO_INTEGER_CONVERTOR.set_trailing_separators */
extern void T420f19(GE_context* ac, T0* C, T0* a1);
/* [detachable] STRING_TO_INTEGER_CONVERTOR.set_leading_separators */
extern void T420f18(GE_context* ac, T0* C, T0* a1);
/* [detachable] STRING_TO_INTEGER_CONVERTOR.make */
extern T0* T420c16(GE_context* ac);
/* [detachable] STRING_8.is_integer */
extern T1 T17f21(GE_context* ac, T0* C);
/* [detachable] STRING_8.is_valid_integer_or_natural */
extern T1 T17f26(GE_context* ac, T0* C, T6 a1);
/* [detachable] STRING_TO_INTEGER_CONVERTOR.is_integral_integer */
extern T1 T420f12(GE_context* ac, T0* C);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_I_NODE].force */
extern void T206f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_I_NODE].fill_with */
extern void T201f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_I_NODE].extend */
extern void T201f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_I_NODE].aliased_resized_area */
extern T0* T206f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [[attached] XPLAIN_I_NODE].aliased_resized_area */
extern T0* T201f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_I_NODE.make */
extern T0* T135c4(GE_context* ac, T0* a1, T0* a2);
/* INTEGER_32.out */
extern T0* T6f11(GE_context* ac, T6* C);
/* [detachable] STRING_8.append_integer */
extern void T17f48(GE_context* ac, T0* C, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_A_NODE].force */
extern void T203f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_A_NODE].fill_with */
extern void T199f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_A_NODE].extend */
extern void T199f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_A_NODE].aliased_resized_area */
extern T0* T203f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [[attached] XPLAIN_A_NODE].aliased_resized_area */
extern T0* T199f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_A_NODE.make */
extern T0* T134c5(GE_context* ac, T0* a1, T0* a2);
/* [detachable] XPLAIN_I_ENUMERATION.make */
extern T0* T202c12(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_I_ENUMERATION.make_xplain_domain_restriction */
extern void T202f13(GE_context* ac, T0* C, T1 a1);
/* [detachable] XPLAIN_A_ENUMERATION.make */
extern T0* T200c6(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_A_ENUMERATION.make_xplain_domain_restriction */
extern void T200f7(GE_context* ac, T0* C, T1 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_DOMAIN_RESTRICTION].aliased_resized_area */
extern T0* T197f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [[attached] XPLAIN_DOMAIN_RESTRICTION].aliased_resized_area */
extern T0* T190f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_A_PATTERN.make */
extern T0* T198c5(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_A_PATTERN.make_xplain_domain_restriction */
extern void T198f6(GE_context* ac, T0* C, T1 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_DOMAIN_RESTRICTION].force */
extern void T197f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_DOMAIN_RESTRICTION].fill_with */
extern void T190f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_DOMAIN_RESTRICTION].extend */
extern void T190f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.report_warning */
extern void T67f561(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_STDERR_FILE.put_integer */
extern void T42f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] KL_STDERR_FILE.put_integer_64 */
extern void T42f11(GE_context* ac, T0* C, T7 a1);
/* INTEGER_32.to_integer_64 */
extern T7 T6f21(GE_context* ac, T6* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_SCANNER_STATE].last */
extern T0* T121f5(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_SCANNER_STATE].is_empty */
extern T1 T121f7(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.get_known_type */
extern T0* T67f331(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_UNIVERSE.find_base */
extern T0* T181f7(GE_context* ac, T0* C, T0* a1);
extern T1 T181f7ot1(T0* a1, T0** a2);
/* [detachable] XPLAIN_UNIVERSE.find_type */
extern T0* T181f6(GE_context* ac, T0* C, T0* a1);
extern T1 T181f6ot1(T0* a1, T0** a2);
/* [detachable] XPLAIN_PARSER.error_unknown_attribute */
extern void T67f560(GE_context* ac, T0* C, T0* a1, T0* a2);
/* detachable XPLAIN_TYPE.write_drop_attribute */
extern void T132f36(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE].delete */
extern void T361f26(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE].set_last_cell */
extern void T361f29(GE_context* ac, T0* C, T0* a1);
/* detachable DS_LINKABLE [[attached] XPLAIN_ATTRIBUTE].forget_right */
extern void T442f5(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE].set_first_cell */
extern void T361f28(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE].is_empty */
extern T1 T361f7(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE].move_all_cursors_after */
extern void T361f27(GE_context* ac, T0* C);
/* [detachable] XPLAIN_UNIVERSE.delete */
extern void T181f12(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].remove_found_item */
extern void T412f39(GE_context* ac, T0* C);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].unset_found_item */
extern void T412f42(GE_context* ac, T0* C);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].remove_position */
extern void T412f48(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].key_storage_put */
extern void T412f47(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].item_storage_put */
extern void T412f43(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ABSTRACT_OBJECT].force */
extern void T452f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_ABSTRACT_OBJECT].fill_with */
extern void T450f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_ABSTRACT_OBJECT].extend */
extern void T450f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].key_storage_keep_head */
extern void T412f57(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] STRING_8].keep_head */
extern void T39f7(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] STRING_8].set_count */
extern void T39f11(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].item_storage_keep_head */
extern void T412f56(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_ABSTRACT_OBJECT].keep_head */
extern void T450f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_ABSTRACT_OBJECT].set_count */
extern void T450f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].clashes_put */
extern void T412f45(GE_context* ac, T0* C, T6 a1, T6 a2);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].slots_put */
extern void T412f46(GE_context* ac, T0* C, T6 a1, T6 a2);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].move_cursors_forth */
extern void T412f55(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE_CURSOR [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].set_position */
extern void T454f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE_CURSOR [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].set_next_cursor */
extern void T454f8(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_HASH_TABLE_CURSOR [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].set_after */
extern void T454f7(GE_context* ac, T0* C);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].internal_set_key_equality_tester */
extern void T412f54(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_UNIVERSE.std */
extern T0* T181f9(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.immediate_output_mode */
extern T1 T67f328(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.get_known_object */
extern T0* T67f327(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_CONSTANT_STATEMENT].force */
extern void T194f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_CONSTANT_STATEMENT].fill_with */
extern void T184f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_CONSTANT_STATEMENT].extend */
extern void T184f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_CONSTANT_STATEMENT].aliased_resized_area */
extern T0* T194f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable XPLAIN_CONSTANT_STATEMENT].aliased_resized_area */
extern T0* T184f4(GE_context* ac, T0* C, T6 a1);
/* detachable XPLAIN_CONSTANT_STATEMENT.make */
extern T0* T133c4(GE_context* ac, T0* a1);
/* detachable XPLAIN_VARIABLE.make */
extern T0* T185c9(GE_context* ac, T0* a1, T0* a2);
/* [detachable] XPLAIN_REQUIRED.make */
extern T0* T123c4(GE_context* ac, T1 a1);
/* [detachable] XPLAIN_PARSER.write_pending_init */
extern void T67f559(GE_context* ac, T0* C);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_TYPE].force */
extern void T193f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_TYPE].fill_with */
extern void T179f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_TYPE].extend */
extern void T179f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_TYPE].aliased_resized_area */
extern T0* T193f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable XPLAIN_TYPE].aliased_resized_area */
extern T0* T179f4(GE_context* ac, T0* C, T6 a1);
/* detachable XPLAIN_TYPE.make */
extern T0* T132c31(GE_context* ac, T0* a1, T0* a2, T0* a3);
/* detachable XPLAIN_I_REFERENCES.set_owner */
extern void T258f6(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_A_REFERENCES.set_owner */
extern void T255f7(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_ATTRIBUTE.set_abstracttype */
extern void T145f20(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_ATTRIBUTE.is_abstracttype_attached */
extern T1 T145f9(GE_context* ac, T0* C);
/* detachable XPLAIN_TYPE.make_abstract_type */
extern void T132f33(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_BASE].force */
extern void T189f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_BASE].fill_with */
extern void T182f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_BASE].extend */
extern void T182f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_BASE].aliased_resized_area */
extern T0* T189f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable XPLAIN_BASE].aliased_resized_area */
extern T0* T182f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_PARSER.new_base */
extern T0* T67f310(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3);
/* detachable XPLAIN_BASE.make */
extern T0* T131c11(GE_context* ac, T0* a1, T0* a2);
/* [detachable] XPLAIN_B_RESTRICTION.make */
extern T0* T365c4(GE_context* ac, T1 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_CONSTANT_ASSIGNMENT_STATEMENT].aliased_resized_area */
extern T0* T187f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable XPLAIN_CONSTANT_ASSIGNMENT_STATEMENT].aliased_resized_area */
extern T0* T178f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_CONSTANT_ASSIGNMENT_STATEMENT].force */
extern void T187f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_CONSTANT_ASSIGNMENT_STATEMENT].fill_with */
extern void T178f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_CONSTANT_ASSIGNMENT_STATEMENT].extend */
extern void T178f10(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_CONSTANT_ASSIGNMENT_STATEMENT.make */
extern T0* T130c5(GE_context* ac, T0* a1, T0* a2);
/* detachable XPLAIN_VARIABLE.set_value */
extern void T185f11(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_VARIABLE.has_value */
extern T1 T185f6(GE_context* ac, T0* C);
/* [detachable] XPLAIN_UNIVERSE.find_variable */
extern T0* T181f8(GE_context* ac, T0* C, T0* a1);
extern T1 T181f8ot1(T0* a1, T0** a2);
/* [detachable] XPLAIN_BASE_STATEMENT.make */
extern T0* T183c4(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_UNIVERSE.add */
extern void T181f11(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].force */
extern void T412f38(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].resize */
extern void T412f44(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].clashes_resize */
extern void T412f53(GE_context* ac, T0* C, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] INTEGER_32].aliased_resized_area_with_default */
extern T0* T118f4(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] INTEGER_32].aliased_resized_area_with_default */
extern T0* T107f5(GE_context* ac, T0* C, T6 a1, T6 a2);
/* [detachable] SPECIAL [[attached] INTEGER_32].aliased_resized_area */
extern T0* T107f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].special_integer_ */
extern T0* T412f32(GE_context* ac, T0* C);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].key_storage_resize */
extern void T412f52(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].item_storage_resize */
extern void T412f51(GE_context* ac, T0* C, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ABSTRACT_OBJECT].aliased_resized_area */
extern T0* T452f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [[attached] XPLAIN_ABSTRACT_OBJECT].aliased_resized_area */
extern T0* T450f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].slots_resize */
extern void T412f50(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].new_modulus */
extern T6 T412f26(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].new_capacity */
extern T6 T412f25(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_PARSER.universe */
extern T0* T67f502(GE_context* ac, T0* C);
/* [detachable] XPLAIN_UNIVERSE.make */
extern T0* T181c10(GE_context* ac);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].make_equal */
extern T0* T412c36(GE_context* ac, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].make_with_equality_testers */
extern void T412f41(GE_context* ac, T0* C, T6 a1, T0* a2, T0* a3);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].make_sparse_container */
extern void T412f49(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].set_internal_cursor */
extern void T412f62(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].new_cursor */
extern T0* T412f31(GE_context* ac, T0* C);
/* [detachable] DS_HASH_TABLE_CURSOR [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].make */
extern T0* T454c6(GE_context* ac, T0* a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].make_slots */
extern void T412f61(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].make_clashes */
extern void T412f60(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].make_key_storage */
extern void T412f59(GE_context* ac, T0* C, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] STRING_8].make */
extern T0* T128f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] STRING_8].default_create */
extern T0* T128c3(GE_context* ac);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_ABSTRACT_OBJECT, [attached] STRING_8].make_item_storage */
extern void T412f58(GE_context* ac, T0* C, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ABSTRACT_OBJECT].make */
extern T0* T452f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_ABSTRACT_OBJECT].make_empty */
extern T0* T450c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ABSTRACT_OBJECT].default_create */
extern T0* T452c3(GE_context* ac);
/* detachable KL_EQUALITY_TESTER [[attached] STRING_8].default_create */
extern T0* T399c2(GE_context* ac);
/* detachable KL_EQUALITY_TESTER [[attached] XPLAIN_ABSTRACT_OBJECT].default_create */
extern T0* T451c1(GE_context* ac);
/* [detachable] XPLAIN_TYPE_STATEMENT.make */
extern T0* T180c4(GE_context* ac, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_STATEMENT].aliased_resized_area */
extern T0* T176f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable XPLAIN_STATEMENT].aliased_resized_area */
extern T0* T174f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_STATEMENT].force */
extern void T176f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_STATEMENT].fill_with */
extern void T174f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable XPLAIN_STATEMENT].extend */
extern void T174f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.equal */
extern T1 T67f355(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_STATEMENTS.add */
extern void T71f3(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_STATEMENT].put_first */
extern void T349f14(GE_context* ac, T0* C, T0* a1);
/* detachable DS_LINKABLE [[attached] XPLAIN_STATEMENT].put_right */
extern void T419f4(GE_context* ac, T0* C, T0* a1);
/* detachable DS_LINKABLE [[attached] XPLAIN_STATEMENT].make */
extern T0* T419c3(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_PARSER.use_mode */
extern T1 T67f350(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_SCANNER_STATE].back */
extern void T367f11(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_SCANNER_STATE].cursor_back */
extern void T121f19(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_SCANNER_STATE].add_traversing_cursor */
extern void T121f20(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_SCANNER_STATE].set_next_cursor */
extern void T367f13(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_SCANNER_STATE].remove_traversing_cursor */
extern void T121f21(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_SCANNER_STATE].set */
extern void T367f14(GE_context* ac, T0* C, T0* a1, T1 a2, T1 a3);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_SCANNER_STATE].is_first */
extern T1 T367f7(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_SCANNER_STATE].cursor_is_first */
extern T1 T121f9(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_SCANNER_STATE].item */
extern T0* T367f6(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_SCANNER_STATE].finish */
extern void T367f10(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_SCANNER_STATE].cursor_finish */
extern void T121f18(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_SCANNER_STATE].cursor_off */
extern T1 T121f8(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_SCANNER_STATE].off */
extern T1 T367f8(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_SCANNER_STATE].new_cursor */
extern T0* T121f6(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_SCANNER_STATE].make */
extern T0* T367c9(GE_context* ac, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] STRING_8].force */
extern void T128f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] XPLAIN_PARSER.text */
extern T0* T67f348(GE_context* ac, T0* C);
/* [detachable] KL_CHARACTER_BUFFER.substring */
extern T0* T436f4(GE_context* ac, T0* C, T6 a1, T6 a2);
/* [detachable] XPLAIN_PARSER.report_error */
extern void T67f557(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.write_pending_statement */
extern void T67f558(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.write_pending_type */
extern void T67f576(GE_context* ac, T0* C);
/* [detachable] XML_GENERATOR.dump_type */
extern void T69f20(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_PARSER.yy_push_last_value */
extern void T67f542(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_PARSER.accept */
extern void T67f541(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.yy_do_error_action */
extern void T67f539(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_PARSER.report_eof_expected_error */
extern void T67f556(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.read_token */
extern void T67f538(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.yy_execute_action */
extern void T67f555(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_PARSER.default_action */
extern void T67f575(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.echo */
extern void T67f583(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.output */
extern void T67f586(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.text_item */
extern T2 T67f491(GE_context* ac, T0* C, T6 a1);
/* [detachable] KL_CHARACTER_BUFFER.item */
extern T2 T436f3(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_PARSER.preprocessor_use */
extern void T67f574(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_SCANNER_STATE].put_last */
extern void T121f11(GE_context* ac, T0* C, T0* a1);
/* detachable DS_LINKABLE [[attached] XPLAIN_SCANNER_STATE].put_right */
extern void T406f4(GE_context* ac, T0* C, T0* a1);
/* detachable DS_LINKABLE [[attached] XPLAIN_SCANNER_STATE].make */
extern T0* T406c3(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_SCANNER_STATE.make_use */
extern T0* T119c8(GE_context* ac, T0* a1, T0* a2);
/* [detachable] XPLAIN_SCANNER_STATE.set_directory */
extern void T119f10(GE_context* ac, T0* C);
/* [detachable] XPLAIN_SCANNER_STATE.do_make */
extern void T119f9(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.set_input_buffer */
extern void T67f582(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yy_load_input_buffer */
extern void T67f579(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.yy_set_content */
extern void T67f568(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_CHARACTER_BUFFER.count */
extern T6 T436f7(GE_context* ac, T0* C);
/* [detachable] KL_CHARACTER_BUFFER.as_unicode_special */
extern T0* T436f6(GE_context* ac, T0* C);
/* [detachable] YY_FILE_BUFFER.set_position */
extern void T344f20(GE_context* ac, T0* C, T6 a1, T6 a2, T6 a3);
/* [detachable] YY_FILE_BUFFER.set_index */
extern void T344f17(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_PARSER.new_file_buffer */
extern T0* T67f364(GE_context* ac, T0* C, T0* a1);
/* [detachable] YY_FILE_BUFFER.make */
extern T0* T344c16(GE_context* ac, T0* a1);
/* [detachable] YY_FILE_BUFFER.make_with_size */
extern void T344f21(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] YY_FILE_BUFFER.set_file */
extern void T344f23(GE_context* ac, T0* C, T0* a1);
/* [detachable] YY_FILE_BUFFER.set_file_with_size */
extern void T344f25(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] KL_CHARACTER_BUFFER.resize */
extern void T436f13(GE_context* ac, T0* C, T6 a1);
/* [detachable] YY_FILE_BUFFER.flush */
extern void T344f26(GE_context* ac, T0* C);
/* [detachable] KL_CHARACTER_BUFFER.put */
extern void T436f11(GE_context* ac, T0* C, T2 a1, T6 a2);
/* detachable KL_TEXT_INPUT_FILE.end_of_input */
extern T1 T32f28(GE_context* ac, T0* C);
/* [detachable] YY_FILE_BUFFER.new_default_buffer */
extern T0* T344f15(GE_context* ac, T0* C, T6 a1);
/* [detachable] KL_CHARACTER_BUFFER.make */
extern T0* T436c10(GE_context* ac, T6 a1);
/* [detachable] YY_FILE_BUFFER.default_capacity */
extern T6 T344f13(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.make_relative_path */
extern T0* T67f507(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.preprocessor_include */
extern void T67f573(GE_context* ac, T0* C);
/* [detachable] XPLAIN_SCANNER_STATE.make_include */
extern T0* T119c7(GE_context* ac, T0* a1, T0* a2);
/* [detachable] XPLAIN_PARSER.unquote_string */
extern T0* T67f487(GE_context* ac, T0* C, T0* a1);
/* [detachable] STRING_8.remove */
extern void T17f53(GE_context* ac, T0* C, T6 a1);
/* [detachable] STRING_8.right_adjust */
extern void T17f47(GE_context* ac, T0* C);
/* CHARACTER_8.is_space */
extern T1 T2f6(GE_context* ac, T2* C);
/* [detachable] XPLAIN_PARSER.conditional_parse_domain */
extern void T67f572(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_PARSER.less */
extern void T67f581(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_PARSER.read_character */
extern void T67f580(GE_context* ac, T0* C);
/* [detachable] YY_FILE_BUFFER.set_beginning_of_line */
extern void T344f19(GE_context* ac, T0* C, T1 a1);
/* [detachable] KL_CHARACTER_BUFFER.unicode_item */
extern T3 T436f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_PARSER.set_start_condition */
extern void T67f571(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_PARSER.yy_set_line_column */
extern void T67f570(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.yy_execute_eof_action */
extern void T67f554(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_PARSER.terminate */
extern void T67f569(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.wrap */
extern T1 T67f324(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_SCANNER_STATE].remove_last */
extern void T121f12(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_SCANNER_STATE].set_last_cell */
extern void T121f16(GE_context* ac, T0* C, T0* a1);
/* detachable DS_LINKABLE [[attached] XPLAIN_SCANNER_STATE].forget_right */
extern void T406f5(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_SCANNER_STATE].move_last_cursors_after */
extern void T121f15(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_SCANNER_STATE].set_after */
extern void T367f12(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_SCANNER_STATE].wipe_out */
extern void T121f14(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_SCANNER_STATE].move_all_cursors_after */
extern void T121f17(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.yy_refill_input_buffer */
extern void T67f553(GE_context* ac, T0* C);
/* [detachable] YY_FILE_BUFFER.fill */
extern void T344f18(GE_context* ac, T0* C);
/* [detachable] KL_CHARACTER_BUFFER.fill_from_stream */
extern T6 T436f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] KL_STDIN_FILE.read_to_string */
extern T6 T345f7(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] STRING_8.subcopy */
extern void T17f60(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3, T6 a4);
/* [detachable] CONSOLE.read_stream */
extern void T437f32(GE_context* ac, T0* C, T6 a1);
/* [detachable] CONSOLE.read_stream_thread_aware */
extern void T437f33(GE_context* ac, T0* C, T6 a1);
/* detachable C_STRING.read_string_into */
extern void T30f10(GE_context* ac, T0* C, T0* a1);
/* detachable C_STRING.read_substring_into */
extern void T30f11(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* NATURAL_8.to_natural_32 */
extern T10 T8f10(GE_context* ac, T8* C);
/* [detachable] MANAGED_POINTER.read_natural_8 */
extern T8 T113f7(GE_context* ac, T0* C, T6 a1);
/* TYPED_POINTER [[attached] NATURAL_8].memory_copy */
extern void T428f2(GE_context* ac, T428* C, T14 a1, T6 a2);
/* TYPED_POINTER [[attached] NATURAL_8].c_memcpy */
extern void T428f3(GE_context* ac, T428* C, T14 a1, T14 a2, T6 a3);
/* [detachable] STRING_8.grow */
extern void T17f61(GE_context* ac, T0* C, T6 a1);
/* [detachable] CONSOLE.console_readstream */
extern T6 T437f17(GE_context* ac, T0* C, T14 a1, T14 a2, T6 a3);
/* detachable C_STRING.set_count */
extern void T30f5(GE_context* ac, T0* C, T6 a1);
/* [detachable] CONSOLE.read_data_buffer */
extern T0* T437f16(GE_context* ac, T0* C);
/* [detachable] KL_STDIN_FILE.any_ */
extern T0* T345f8(GE_context* ac, T0* C);
/* [detachable] CONSOLE.end_of_file */
extern T1 T437f7(GE_context* ac, T0* C);
/* [detachable] CONSOLE.console_eof */
extern T1 T437f10(GE_context* ac, T0* C, T14 a1);
/* [detachable] KL_STDIN_FILE.console */
extern T0* T345f5(GE_context* ac, T0* C);
/* [detachable] STD_FILES.input */
extern T0* T351f5(GE_context* ac, T0* C);
/* [detachable] CONSOLE.make_open_stdin */
extern T0* T437c24(GE_context* ac, T0* a1);
/* [detachable] CONSOLE.set_read_mode */
extern void T437f30(GE_context* ac, T0* C);
/* [detachable] CONSOLE.console_def */
extern T14 T437f13(GE_context* ac, T0* C, T6 a1);
/* [detachable] CONSOLE.make_with_name */
extern void T437f26(GE_context* ac, T0* C, T0* a1);
/* [detachable] CONSOLE.default_pointer */
extern T14 T437f9(GE_context* ac, T0* C);
/* [detachable] CONSOLE.set_name */
extern void T437f31(GE_context* ac, T0* C, T0* a1);
/* [detachable] CONSOLE.buffered_file_info */
extern T0* T437f15(GE_context* ac, T0* C);
/* [detachable] KL_STDIN_FILE.io */
extern T0* T345f6(GE_context* ac, T0* C);
/* [detachable] STD_FILES.set_output_default */
extern void T351f7(GE_context* ac, T0* C);
/* [detachable] STD_FILES.output */
extern T0* T351f3(GE_context* ac, T0* C);
/* [detachable] CONSOLE.make_open_stdout */
extern T0* T437c20(GE_context* ac, T0* a1);
/* [detachable] CONSOLE.set_write_mode */
extern void T437f27(GE_context* ac, T0* C);
/* [detachable] STD_FILES.default_create */
extern T0* T351c6(GE_context* ac);
/* [detachable] STRING_8.put */
extern void T17f49(GE_context* ac, T0* C, T2 a1, T6 a2);
/* detachable KL_TEXT_INPUT_FILE.read_to_string */
extern T6 T32f33(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* detachable KL_TEXT_INPUT_FILE.dummy_kl_character_buffer */
extern T0* T32f37(GE_context* ac, T0* C);
/* detachable KL_TEXT_INPUT_FILE.old_read_to_string */
extern T6 T32f36(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* detachable KL_TEXT_INPUT_FILE.file_gss */
extern T6 T32f38(GE_context* ac, T0* C, T14 a1, T14 a2, T6 a3);
/* [detachable] SPECIAL [CHARACTER_8].item_address */
extern T14 T15f8(GE_context* ac, T0* C, T6 a1);
/* detachable KL_TEXT_INPUT_FILE.any_ */
extern T0* T32f34(GE_context* ac, T0* C);
/* detachable KL_TEXT_INPUT_FILE.old_end_of_file */
extern T1 T32f30(GE_context* ac, T0* C);
/* detachable KL_TEXT_INPUT_FILE.file_feof */
extern T1 T32f32(GE_context* ac, T0* C, T14 a1);
/* [detachable] KL_STDIN_FILE.read_character */
extern void T345f11(GE_context* ac, T0* C);
/* [detachable] CONSOLE.read_character */
extern void T437f25(GE_context* ac, T0* C);
/* [detachable] CONSOLE.console_readchar */
extern T2 T437f14(GE_context* ac, T0* C, T14 a1);
/* detachable KL_TEXT_INPUT_FILE.read_character */
extern void T32f50(GE_context* ac, T0* C);
/* detachable KL_TEXT_INPUT_FILE.old_read_character */
extern void T32f51(GE_context* ac, T0* C);
/* detachable KL_TEXT_INPUT_FILE.file_gc */
extern T2 T32f31(GE_context* ac, T0* C, T14 a1);
/* [detachable] YY_FILE_BUFFER.compact_left */
extern void T344f22(GE_context* ac, T0* C);
/* [detachable] KL_CHARACTER_BUFFER.move_left */
extern void T436f12(GE_context* ac, T0* C, T6 a1, T6 a2, T6 a3);
/* [detachable] YY_FILE_BUFFER.resize */
extern void T344f24(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.yy_null_trans_state */
extern T6 T67f323(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_PARSER.yy_previous_state */
extern T6 T67f320(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.fatal_error */
extern void T67f552(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.report_invalid_unicode_character_error */
extern void T67f551(GE_context* ac, T0* C, T10 a1);
/* NATURAL_32.to_hex_string */
extern T0* T10f6(GE_context* ac, T10* C);
/* NATURAL_32.to_hex_character */
extern T2 T10f10(GE_context* ac, T10* C);
/* CHARACTER_8.plus */
extern T2 T2f5(GE_context* ac, T2* C, T6 a1);
/* CHARACTER_8.minus */
extern T2 T2f4(GE_context* ac, T2* C, T6 a1);
/* NATURAL_32.to_integer_32 */
extern T6 T10f3(GE_context* ac, T10* C);
/* [detachable] KL_CHARACTER_BUFFER.item_code */
extern T10 T436f5(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_PARSER.yy_at_beginning_of_line */
extern T6 T67f337(GE_context* ac, T0* C);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] INTEGER_32].force */
extern void T118f6(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] INTEGER_32].aliased_resized_area */
extern T0* T118f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] XPLAIN_PARSER.special_integer_ */
extern T0* T67f346(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.yy_init_value_stacks */
extern void T67f537(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.yy_clear_all */
extern void T67f546(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.clear_all */
extern void T67f563(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.clear_stacks */
extern void T67f577(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.yy_clear_value_stacks */
extern void T67f584(GE_context* ac, T0* C);
/* [detachable] SPECIAL [detachable XPLAIN_VALUE_SELECTION_STATEMENT].keep_head */
extern void T298f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_VALUE_SELECTION_STATEMENT].set_count */
extern void T298f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_VALUE_STATEMENT].keep_head */
extern void T293f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_VALUE_STATEMENT].set_count */
extern void T293f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_UPDATE_STATEMENT].keep_head */
extern void T274f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_UPDATE_STATEMENT].set_count */
extern void T274f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_SQL_STATEMENT].keep_head */
extern void T175f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_SQL_STATEMENT].set_count */
extern void T175f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_PROCEDURE_STATEMENT].keep_head */
extern void T177f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_PROCEDURE_STATEMENT].set_count */
extern void T177f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_GET_STATEMENT].keep_head */
extern void T305f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_GET_STATEMENT].set_count */
extern void T305f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_EXTEND_STATEMENT].keep_head */
extern void T308f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_EXTEND_STATEMENT].set_count */
extern void T308f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_DELETE_STATEMENT].keep_head */
extern void T275f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_DELETE_STATEMENT].set_count */
extern void T275f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_CASCADE_FUNCTION_EXPRESSION].keep_head */
extern void T286f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_CASCADE_FUNCTION_EXPRESSION].set_count */
extern void T286f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_CASCADE_STATEMENT].keep_head */
extern void T276f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_CASCADE_STATEMENT].set_count */
extern void T276f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_CONSTANT_ASSIGNMENT_STATEMENT].keep_head */
extern void T178f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_CONSTANT_ASSIGNMENT_STATEMENT].set_count */
extern void T178f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_CONSTANT_STATEMENT].keep_head */
extern void T184f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_CONSTANT_STATEMENT].set_count */
extern void T184f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_IF_STATEMENT].keep_head */
extern void T342f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_IF_STATEMENT].set_count */
extern void T342f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_STATEMENT].keep_head */
extern void T174f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_STATEMENT].set_count */
extern void T174f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_STATEMENT_NODE].keep_head */
extern void T332f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_STATEMENT_NODE].set_count */
extern void T332f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_PARAMETER].keep_head */
extern void T337f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_PARAMETER].set_count */
extern void T337f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_PARAMETER_NODE].keep_head */
extern void T336f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_PARAMETER_NODE].set_count */
extern void T336f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_PARAMETER_NODE].keep_head */
extern void T331f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_PARAMETER_NODE].set_count */
extern void T331f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable STRING_8].keep_head */
extern void T329f7(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable STRING_8].set_count */
extern void T329f10(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_EXPRESSION_NODE].keep_head */
extern void T228f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_EXPRESSION_NODE].set_count */
extern void T228f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_FUNCTION].keep_head */
extern void T302f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_FUNCTION].set_count */
extern void T302f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_SORT_NODE].keep_head */
extern void T317f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_SORT_NODE].set_count */
extern void T317f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_SELECTION_LIST].keep_head */
extern void T280f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_SELECTION_LIST].set_count */
extern void T280f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_SELECTION].keep_head */
extern void T304f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_SELECTION].set_count */
extern void T304f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_EXTENSION_EXPRESSION].keep_head */
extern void T268f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_EXTENSION_EXPRESSION].set_count */
extern void T268f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_EXTEND_ATTRIBUTE].keep_head */
extern void T310f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_EXTEND_ATTRIBUTE].set_count */
extern void T310f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_EXTENSION].keep_head */
extern void T307f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_EXTENSION].set_count */
extern void T307f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_SELECTION_FUNCTION].keep_head */
extern void T288f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_SELECTION_FUNCTION].set_count */
extern void T288f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_NON_ATTRIBUTE_EXPRESSION].keep_head */
extern void T300f5(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_NON_ATTRIBUTE_EXPRESSION].set_count */
extern void T300f8(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_SUBJECT].keep_head */
extern void T277f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_SUBJECT].set_count */
extern void T277f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_ASSIGNMENT].keep_head */
extern void T283f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_ASSIGNMENT].set_count */
extern void T283f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_ASSIGNMENT_NODE].keep_head */
extern void T278f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_ASSIGNMENT_NODE].set_count */
extern void T278f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_ATTRIBUTE_NAME_NODE].keep_head */
extern void T271f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_ATTRIBUTE_NAME_NODE].set_count */
extern void T271f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_ASSERTION].keep_head */
extern void T270f5(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_ASSERTION].set_count */
extern void T270f8(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_ATTRIBUTE].keep_head */
extern void T259f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_ATTRIBUTE].set_count */
extern void T259f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_ATTRIBUTE_NODE].keep_head */
extern void T191f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_ATTRIBUTE_NODE].set_count */
extern void T191f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_PK_REPRESENTATION].keep_head */
extern void T192f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_PK_REPRESENTATION].set_count */
extern void T192f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_REPRESENTATION].keep_head */
extern void T188f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_REPRESENTATION].set_count */
extern void T188f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_EXPRESSION].keep_head */
extern void T243f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_EXPRESSION].set_count */
extern void T243f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_TYPE].keep_head */
extern void T179f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_TYPE].set_count */
extern void T179f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_ATTRIBUTE_NAME].keep_head */
extern void T196f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_ATTRIBUTE_NAME].set_count */
extern void T196f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_ATTRIBUTE_NAME_NODE].keep_head */
extern void T212f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_ATTRIBUTE_NAME_NODE].set_count */
extern void T212f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_SYSTEM_EXPRESSION].keep_head */
extern void T223f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_SYSTEM_EXPRESSION].set_count */
extern void T223f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_LOGICAL_VALUE_EXPRESSION].keep_head */
extern void T220f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_LOGICAL_VALUE_EXPRESSION].set_count */
extern void T220f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_EXPRESSION].keep_head */
extern void T186f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_EXPRESSION].set_count */
extern void T186f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_LOGICAL_EXPRESSION].keep_head */
extern void T208f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_LOGICAL_EXPRESSION].set_count */
extern void T208f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_I_NODE].keep_head */
extern void T201f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_I_NODE].set_count */
extern void T201f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_A_NODE].keep_head */
extern void T199f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_A_NODE].set_count */
extern void T199f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_DOMAIN_RESTRICTION].keep_head */
extern void T190f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_DOMAIN_RESTRICTION].set_count */
extern void T190f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_DOMAIN_RESTRICTION].keep_head */
extern void T266f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_DOMAIN_RESTRICTION].set_count */
extern void T266f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_BASE].keep_head */
extern void T182f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_BASE].set_count */
extern void T182f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] INTEGER_32].keep_head */
extern void T107f12(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] INTEGER_32].set_count */
extern void T107f13(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable ANY].keep_head */
extern void T126f5(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable ANY].set_count */
extern void T126f8(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_PARSER.abort */
extern void T67f540(GE_context* ac, T0* C);
/* [detachable] STRING_8.plus */
extern T0* T17f5(GE_context* ac, T0* C, T0* a1);
/* [detachable] STRING_8.append_string_general */
extern void T17f35(GE_context* ac, T0* C, T0* a1);
/* [detachable] STRING_8.append */
extern void T17f35p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] STRING_8.append_code */
extern void T17f38(GE_context* ac, T0* C, T10 a1);
/* [detachable] STRING_8.new_string */
extern T0* T17f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_PARSER.set_mwgenerator */
extern void T67f532(GE_context* ac, T0* C, T0* a1);
/* [detachable] XML_GENERATOR.make */
extern T0* T69c18(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_PARSER.make_with_file */
extern T0* T67c531(GE_context* ac, T0* a1, T0* a2);
/* [detachable] XPLAIN_PARSER.make */
extern void T67f535(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.set_sqlgenerator */
extern void T67f549(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_STATEMENTS.make */
extern T0* T71c2(GE_context* ac);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_STATEMENT].make */
extern T0* T349c11(GE_context* ac);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_STATEMENT].set_internal_cursor */
extern void T349f15(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_STATEMENT].new_cursor */
extern T0* T349f6(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_STATEMENT].make */
extern T0* T418c7(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_PARSER.make_parser_skeleton */
extern void T67f548(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.yy_build_parser_tables */
extern void T67f566(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.yycheck_template */
extern T0* T67f501(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.yyfixed_array */
extern T0* T67f524(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] INTEGER_32].to_special */
extern T0* T118f3(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yycheck_template_10 */
extern void T67f622(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yyarray_subcopy */
extern void T67f652(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3, T6 a4, T6 a5);
/* [detachable] KL_ARRAY_ROUTINES [[attached] INTEGER_32].subcopy */
extern void T371f2(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3, T6 a4, T6 a5);
/* [detachable] ARRAY [[attached] INTEGER_32].subcopy */
extern void T370f7(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3, T6 a4);
/* [detachable] SPECIAL [[attached] INTEGER_32].copy_data */
extern void T107f14(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3, T6 a4);
/* [detachable] SPECIAL [[attached] INTEGER_32].force */
extern void T107f9(GE_context* ac, T0* C, T6 a1, T6 a2);
/* [detachable] SPECIAL [[attached] INTEGER_32].move_data */
extern void T107f15(GE_context* ac, T0* C, T6 a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] INTEGER_32].overlapping_move */
extern void T107f17(GE_context* ac, T0* C, T6 a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] INTEGER_32].non_overlapping_move */
extern void T107f16(GE_context* ac, T0* C, T6 a1, T6 a2, T6 a3);
/* [detachable] XPLAIN_PARSER.integer_array_ */
extern T0* T67f529(GE_context* ac, T0* C);
/* [detachable] KL_ARRAY_ROUTINES [[attached] INTEGER_32].default_create */
extern T0* T371c1(GE_context* ac);
/* [detachable] XPLAIN_PARSER.yycheck_template_9 */
extern void T67f621(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yycheck_template_8 */
extern void T67f620(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yycheck_template_7 */
extern void T67f619(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yycheck_template_6 */
extern void T67f618(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yycheck_template_5 */
extern void T67f617(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yycheck_template_4 */
extern void T67f616(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yycheck_template_3 */
extern void T67f615(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yycheck_template_2 */
extern void T67f614(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yycheck_template_1 */
extern void T67f613(GE_context* ac, T0* C, T0* a1);
/* [detachable] ARRAY [[attached] INTEGER_32].make_filled */
extern T0* T370c4(GE_context* ac, T6 a1, T6 a2, T6 a3);
/* [detachable] ARRAY [[attached] INTEGER_32].make_filled_area */
extern void T370f6(GE_context* ac, T0* C, T6 a1, T6 a2);
/* [detachable] XPLAIN_PARSER.yytable_template */
extern T0* T67f500(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.yytable_template_10 */
extern void T67f612(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yytable_template_9 */
extern void T67f611(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yytable_template_8 */
extern void T67f610(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yytable_template_7 */
extern void T67f609(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yytable_template_6 */
extern void T67f608(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yytable_template_5 */
extern void T67f607(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yytable_template_4 */
extern void T67f606(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yytable_template_3 */
extern void T67f605(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yytable_template_2 */
extern void T67f604(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yytable_template_1 */
extern void T67f603(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yypgoto_template */
extern T0* T67f499(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.yypact_template */
extern T0* T67f498(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.yypact_template_4 */
extern void T67f602(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yypact_template_3 */
extern void T67f601(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yypact_template_2 */
extern void T67f600(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yypact_template_1 */
extern void T67f599(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yydefgoto_template */
extern T0* T67f497(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.yydefact_template */
extern T0* T67f496(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.yydefact_template_4 */
extern void T67f598(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yydefact_template_3 */
extern void T67f597(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yydefact_template_2 */
extern void T67f596(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yydefact_template_1 */
extern void T67f595(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yytypes2_template */
extern T0* T67f495(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.yytypes1_template */
extern T0* T67f494(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.yytypes1_template_4 */
extern void T67f594(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yytypes1_template_3 */
extern void T67f593(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yytypes1_template_2 */
extern void T67f592(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yytypes1_template_1 */
extern void T67f591(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yyr1_template */
extern T0* T67f493(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.yyr1_template_2 */
extern void T67f590(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yyr1_template_1 */
extern void T67f589(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yytranslate_template */
extern T0* T67f492(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.yytranslate_template_2 */
extern void T67f588(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yytranslate_template_1 */
extern void T67f587(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yy_create_value_stacks */
extern void T67f565(GE_context* ac, T0* C);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_VALUE_SELECTION_STATEMENT].make */
extern T0* T299f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_VALUE_SELECTION_STATEMENT].make_empty */
extern T0* T298c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_VALUE_SELECTION_STATEMENT].default_create */
extern T0* T299c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_VALUE_STATEMENT].make */
extern T0* T294f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_VALUE_STATEMENT].make_empty */
extern T0* T293c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_VALUE_STATEMENT].default_create */
extern T0* T294c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_UPDATE_STATEMENT].make */
extern T0* T292f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_UPDATE_STATEMENT].make_empty */
extern T0* T274c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_UPDATE_STATEMENT].default_create */
extern T0* T292c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_SQL_STATEMENT].make */
extern T0* T340f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_SQL_STATEMENT].make_empty */
extern T0* T175c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_SQL_STATEMENT].default_create */
extern T0* T340c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_PROCEDURE_STATEMENT].make */
extern T0* T334f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_PROCEDURE_STATEMENT].make_empty */
extern T0* T177c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_PROCEDURE_STATEMENT].default_create */
extern T0* T334c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_GET_STATEMENT].make */
extern T0* T306f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_GET_STATEMENT].make_empty */
extern T0* T305c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_GET_STATEMENT].default_create */
extern T0* T306c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXTEND_STATEMENT].make */
extern T0* T309f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_EXTEND_STATEMENT].make_empty */
extern T0* T308c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXTEND_STATEMENT].default_create */
extern T0* T309c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_DELETE_STATEMENT].make */
extern T0* T290f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_DELETE_STATEMENT].make_empty */
extern T0* T275c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_DELETE_STATEMENT].default_create */
extern T0* T290c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_CASCADE_FUNCTION_EXPRESSION].make */
extern T0* T289f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_CASCADE_FUNCTION_EXPRESSION].make_empty */
extern T0* T286c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_CASCADE_FUNCTION_EXPRESSION].default_create */
extern T0* T289c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_CASCADE_STATEMENT].make */
extern T0* T287f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_CASCADE_STATEMENT].make_empty */
extern T0* T276c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_CASCADE_STATEMENT].default_create */
extern T0* T287c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_CONSTANT_ASSIGNMENT_STATEMENT].make */
extern T0* T187f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_CONSTANT_ASSIGNMENT_STATEMENT].make_empty */
extern T0* T178c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_CONSTANT_ASSIGNMENT_STATEMENT].default_create */
extern T0* T187c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_CONSTANT_STATEMENT].make */
extern T0* T194f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_CONSTANT_STATEMENT].make_empty */
extern T0* T184c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_CONSTANT_STATEMENT].default_create */
extern T0* T194c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_IF_STATEMENT].make */
extern T0* T343f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_IF_STATEMENT].make_empty */
extern T0* T342c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_IF_STATEMENT].default_create */
extern T0* T343c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_STATEMENT].make */
extern T0* T176f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_STATEMENT].make_empty */
extern T0* T174c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_STATEMENT].default_create */
extern T0* T176c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_STATEMENT_NODE].make */
extern T0* T341f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_STATEMENT_NODE].make_empty */
extern T0* T332c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_STATEMENT_NODE].default_create */
extern T0* T341c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_PARAMETER].make */
extern T0* T339f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_PARAMETER].make_empty */
extern T0* T337c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_PARAMETER].default_create */
extern T0* T339c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_PARAMETER_NODE].make */
extern T0* T338f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_PARAMETER_NODE].make_empty */
extern T0* T336c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_PARAMETER_NODE].default_create */
extern T0* T338c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_PARAMETER_NODE].make */
extern T0* T335f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_PARAMETER_NODE].make_empty */
extern T0* T331c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_PARAMETER_NODE].default_create */
extern T0* T335c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [detachable STRING_8].make */
extern T0* T330f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable STRING_8].default_create */
extern T0* T330c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXPRESSION_NODE].make */
extern T0* T328f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_EXPRESSION_NODE].make_empty */
extern T0* T228c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXPRESSION_NODE].default_create */
extern T0* T328c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_FUNCTION].make */
extern T0* T321f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_FUNCTION].make_empty */
extern T0* T302c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_FUNCTION].default_create */
extern T0* T321c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SORT_NODE].make */
extern T0* T319f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_SORT_NODE].make_empty */
extern T0* T317c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SORT_NODE].default_create */
extern T0* T319c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SELECTION_LIST].make */
extern T0* T318f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_SELECTION_LIST].make_empty */
extern T0* T280c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SELECTION_LIST].default_create */
extern T0* T318c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SELECTION].make */
extern T0* T316f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_SELECTION].make_empty */
extern T0* T304c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SELECTION].default_create */
extern T0* T316c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXTENSION_EXPRESSION].make */
extern T0* T314f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_EXTENSION_EXPRESSION].make_empty */
extern T0* T268c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXTENSION_EXPRESSION].default_create */
extern T0* T314c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_EXTEND_ATTRIBUTE].make */
extern T0* T312f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_EXTEND_ATTRIBUTE].make_empty */
extern T0* T310c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_EXTEND_ATTRIBUTE].default_create */
extern T0* T312c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXTENSION].make */
extern T0* T311f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_EXTENSION].make_empty */
extern T0* T307c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXTENSION].default_create */
extern T0* T311c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SELECTION_FUNCTION].make */
extern T0* T303f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_SELECTION_FUNCTION].make_empty */
extern T0* T288c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SELECTION_FUNCTION].default_create */
extern T0* T303c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_NON_ATTRIBUTE_EXPRESSION].make */
extern T0* T301f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_NON_ATTRIBUTE_EXPRESSION].make_empty */
extern T0* T300c4(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_NON_ATTRIBUTE_EXPRESSION].default_create */
extern T0* T301c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SUBJECT].make */
extern T0* T291f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_SUBJECT].make_empty */
extern T0* T277c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_SUBJECT].default_create */
extern T0* T291c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ASSIGNMENT].make */
extern T0* T285f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_ASSIGNMENT].make_empty */
extern T0* T283c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ASSIGNMENT].default_create */
extern T0* T285c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_ASSIGNMENT_NODE].make */
extern T0* T284f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_ASSIGNMENT_NODE].make_empty */
extern T0* T278c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_ASSIGNMENT_NODE].default_create */
extern T0* T284c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_ATTRIBUTE_NAME_NODE].make */
extern T0* T273f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_ATTRIBUTE_NAME_NODE].make_empty */
extern T0* T271c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_ATTRIBUTE_NAME_NODE].default_create */
extern T0* T273c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_ASSERTION].make */
extern T0* T269f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_ASSERTION].make_empty */
extern T0* T270c4(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_ASSERTION].default_create */
extern T0* T269c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ATTRIBUTE].make */
extern T0* T262f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_ATTRIBUTE].make_empty */
extern T0* T259c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ATTRIBUTE].default_create */
extern T0* T262c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_ATTRIBUTE_NODE].make */
extern T0* T260f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_ATTRIBUTE_NODE].make_empty */
extern T0* T191c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_ATTRIBUTE_NODE].default_create */
extern T0* T260c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_PK_REPRESENTATION].make */
extern T0* T256f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_PK_REPRESENTATION].make_empty */
extern T0* T192c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_PK_REPRESENTATION].default_create */
extern T0* T256c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_REPRESENTATION].make */
extern T0* T245f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_REPRESENTATION].make_empty */
extern T0* T188c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_REPRESENTATION].default_create */
extern T0* T245c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXPRESSION].make */
extern T0* T244f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_EXPRESSION].make_empty */
extern T0* T243c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_EXPRESSION].default_create */
extern T0* T244c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_TYPE].make */
extern T0* T193f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_TYPE].make_empty */
extern T0* T179c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_TYPE].default_create */
extern T0* T193c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ATTRIBUTE_NAME].make */
extern T0* T242f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_ATTRIBUTE_NAME].make_empty */
extern T0* T196c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ATTRIBUTE_NAME].default_create */
extern T0* T242c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ATTRIBUTE_NAME_NODE].make */
extern T0* T241f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_ATTRIBUTE_NAME_NODE].make_empty */
extern T0* T212c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_ATTRIBUTE_NAME_NODE].default_create */
extern T0* T241c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_SYSTEM_EXPRESSION].make */
extern T0* T224f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_SYSTEM_EXPRESSION].make_empty */
extern T0* T223c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_SYSTEM_EXPRESSION].default_create */
extern T0* T224c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_LOGICAL_VALUE_EXPRESSION].make */
extern T0* T221f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_LOGICAL_VALUE_EXPRESSION].make_empty */
extern T0* T220c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_LOGICAL_VALUE_EXPRESSION].default_create */
extern T0* T221c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_EXPRESSION].make */
extern T0* T211f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_EXPRESSION].make_empty */
extern T0* T186c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_EXPRESSION].default_create */
extern T0* T211c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_LOGICAL_EXPRESSION].make */
extern T0* T209f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_LOGICAL_EXPRESSION].make_empty */
extern T0* T208c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_LOGICAL_EXPRESSION].default_create */
extern T0* T209c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_I_NODE].make */
extern T0* T206f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_I_NODE].make_empty */
extern T0* T201c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_I_NODE].default_create */
extern T0* T206c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_A_NODE].make */
extern T0* T203f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_A_NODE].make_empty */
extern T0* T199c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_A_NODE].default_create */
extern T0* T203c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_DOMAIN_RESTRICTION].make */
extern T0* T197f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_DOMAIN_RESTRICTION].make_empty */
extern T0* T190c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_DOMAIN_RESTRICTION].default_create */
extern T0* T197c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_DOMAIN_RESTRICTION].make */
extern T0* T267f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_DOMAIN_RESTRICTION].make_empty */
extern T0* T266c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_DOMAIN_RESTRICTION].default_create */
extern T0* T267c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_BASE].make */
extern T0* T189f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable XPLAIN_BASE].make_empty */
extern T0* T182c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable XPLAIN_BASE].default_create */
extern T0* T189c3(GE_context* ac);
/* [detachable] KL_SPECIAL_ROUTINES [detachable ANY].make */
extern T0* T127f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [detachable ANY].make_empty */
extern T0* T126c4(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [detachable ANY].default_create */
extern T0* T127c3(GE_context* ac);
/* [detachable] XPLAIN_PARSER.make_scanner_with_file */
extern void T67f536(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_SCANNER_STATE.make_file */
extern T0* T119c6(GE_context* ac, T0* a1, T0* a2);
/* [detachable] XPLAIN_PARSER.make_compressed_scanner_skeleton_with_file */
extern void T67f550(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.make_with_buffer */
extern void T67f567(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yy_initialize */
extern void T67f578(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.yy_initialize */
extern void T67f578p1(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.yy_build_tables */
extern void T67f585(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.yy_accept_template */
extern T0* T67f519(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.yy_fixed_array */
extern T0* T67f526(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yy_accept_template_4 */
extern void T67f651(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yy_array_subcopy */
extern void T67f653(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3, T6 a4, T6 a5);
/* [detachable] XPLAIN_PARSER.yy_accept_template_3 */
extern void T67f650(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yy_accept_template_2 */
extern void T67f649(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yy_accept_template_1 */
extern void T67f648(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yy_meta_template */
extern T0* T67f518(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.yy_ec_template */
extern T0* T67f517(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.yy_ec_template_1 */
extern void T67f647(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yy_def_template */
extern T0* T67f516(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.yy_def_template_5 */
extern void T67f646(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yy_def_template_4 */
extern void T67f645(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yy_def_template_3 */
extern void T67f644(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yy_def_template_2 */
extern void T67f643(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yy_def_template_1 */
extern void T67f642(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yy_base_template */
extern T0* T67f515(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.yy_base_template_3 */
extern void T67f641(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yy_base_template_2 */
extern void T67f640(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yy_base_template_1 */
extern void T67f639(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yy_chk_template */
extern T0* T67f514(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.yy_chk_template_8 */
extern void T67f638(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yy_chk_template_7 */
extern void T67f637(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yy_chk_template_6 */
extern void T67f636(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yy_chk_template_5 */
extern void T67f635(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yy_chk_template_4 */
extern void T67f634(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yy_chk_template_3 */
extern void T67f633(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yy_chk_template_2 */
extern void T67f632(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yy_chk_template_1 */
extern void T67f631(GE_context* ac, T0* C, T0* a1);
/* [detachable] ARRAY [[attached] INTEGER_32].put */
extern void T370f5(GE_context* ac, T0* C, T6 a1, T6 a2);
/* [detachable] XPLAIN_PARSER.yy_nxt_template */
extern T0* T67f513(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARSER.yy_nxt_template_8 */
extern void T67f630(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yy_nxt_template_7 */
extern void T67f629(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yy_nxt_template_6 */
extern void T67f628(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yy_nxt_template_5 */
extern void T67f627(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yy_nxt_template_4 */
extern void T67f626(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yy_nxt_template_3 */
extern void T67f625(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yy_nxt_template_2 */
extern void T67f624(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.yy_nxt_template_1 */
extern void T67f623(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.do_make */
extern void T67f547(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_SCANNER_STATE].make */
extern T0* T121c10(GE_context* ac);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_SCANNER_STATE].set_internal_cursor */
extern void T121f13(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARSER.make_compressed_scanner_skeleton */
extern void T67f564(GE_context* ac, T0* C);
/* [detachable] KL_STANDARD_FILES.input */
extern T0* T36f3(GE_context* ac, T0* C);
/* [detachable] KL_STDIN_FILE.make */
extern T0* T345c10(GE_context* ac);
/* [detachable] XPLAIN_PARSER.make_with_stdin */
extern T0* T67c530(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_PARSER.make_scanner_with_stdin */
extern void T67f534(GE_context* ac, T0* C);
/* [detachable] XPLAIN_SCANNER_STATE.make_stdin */
extern T0* T119c5(GE_context* ac, T0* a1);
/* detachable KL_TEXT_INPUT_FILE.is_open_read */
extern T1 T32f25(GE_context* ac, T0* C);
/* detachable KL_TEXT_INPUT_FILE.old_is_open_read */
extern T1 T32f24(GE_context* ac, T0* C);
/* detachable KL_TEXT_INPUT_FILE.open_read */
extern void T32f41(GE_context* ac, T0* C);
/* detachable KL_TEXT_INPUT_FILE.is_closed */
extern T1 T32f17(GE_context* ac, T0* C);
/* detachable KL_TEXT_INPUT_FILE.old_is_closed */
extern T1 T32f21(GE_context* ac, T0* C);
/* detachable KL_TEXT_INPUT_FILE.old_open_read */
extern void T32f45(GE_context* ac, T0* C);
/* detachable KL_TEXT_INPUT_FILE.open_read */
extern void T32f45p1(GE_context* ac, T0* C);
/* detachable KL_TEXT_INPUT_FILE.file_open */
extern T14 T32f23(GE_context* ac, T0* C, T14 a1, T6 a2);
/* detachable KL_TEXT_INPUT_FILE.internal_name_pointer */
extern T0* T32f22(GE_context* ac, T0* C);
/* detachable KL_TEXT_INPUT_FILE.old_is_readable */
extern T1 T32f16(GE_context* ac, T0* C);
/* [detachable] FILE_INFO.is_readable */
extern T1 T114f8(GE_context* ac, T0* C);
/* [detachable] FILE_INFO.file_eaccess */
extern T1 T114f6(GE_context* ac, T0* C, T14 a1, T6 a2);
/* detachable KL_TEXT_INPUT_FILE.buffered_file_info */
extern T0* T32f26(GE_context* ac, T0* C);
/* detachable KL_TEXT_INPUT_FILE.set_buffer */
extern void T32f49(GE_context* ac, T0* C);
/* [detachable] FILE_INFO.fast_update */
extern void T114f14(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] FILE_INFO.eif_file_stat */
extern T6 T114f12(GE_context* ac, T0* C, T14 a1, T14 a2, T1 a3);
/* detachable KL_TEXT_INPUT_FILE.old_exists */
extern T1 T32f15(GE_context* ac, T0* C);
/* detachable KL_TEXT_INPUT_FILE.file_exists */
extern T1 T32f27(GE_context* ac, T0* C, T14 a1);
/* detachable KL_TEXT_INPUT_FILE.string_name */
extern T0* T32f12(GE_context* ac, T0* C);
/* detachable KL_TEXT_INPUT_FILE.make */
extern T0* T32c40(GE_context* ac, T0* a1);
/* detachable KL_TEXT_INPUT_FILE.make */
extern void T32f40p1(GE_context* ac, T0* C, T0* a1);
/* detachable KL_TEXT_INPUT_FILE.old_make */
extern void T32f44(GE_context* ac, T0* C, T0* a1);
/* detachable KL_TEXT_INPUT_FILE.make_with_name */
extern void T32f47(GE_context* ac, T0* C, T0* a1);
/* detachable KL_TEXT_INPUT_FILE.set_name */
extern void T32f48(GE_context* ac, T0* C, T0* a1);
/* detachable KL_TEXT_INPUT_FILE.string_ */
extern T0* T32f29(GE_context* ac, T0* C);
/* [detachable] STRING_8.item */
extern T2 T17f11(GE_context* ac, T0* C, T6 a1);
/* [detachable] SQL_GENERATOR_TSQL65.make */
extern T0* T66c250(GE_context* ac);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8].make_equal */
extern T0* T116c35(GE_context* ac, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8].make_with_equality_testers */
extern void T116f39(GE_context* ac, T0* C, T6 a1, T0* a2, T0* a3);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8].make_sparse_container */
extern void T116f48(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8].set_internal_cursor */
extern void T116f61(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8].new_cursor */
extern T0* T116f27(GE_context* ac, T0* C);
/* [detachable] DS_HASH_TABLE_CURSOR [[attached] XPLAIN_VALUE, [attached] STRING_8].make */
extern T0* T404c6(GE_context* ac, T0* a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8].unset_found_item */
extern void T116f40(GE_context* ac, T0* C);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8].make_slots */
extern void T116f60(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8].new_modulus */
extern T6 T116f22(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8].make_clashes */
extern void T116f59(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8].make_key_storage */
extern void T116f58(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8].make_item_storage */
extern void T116f57(GE_context* ac, T0* C, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_VALUE].make */
extern T0* T402f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_VALUE].make_empty */
extern T0* T403c5(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_VALUE].default_create */
extern T0* T402c3(GE_context* ac);
/* detachable KL_EQUALITY_TESTER [[attached] XPLAIN_VALUE].default_create */
extern T0* T398c1(GE_context* ac);
/* [detachable] SQL_GENERATOR_TSQL70.make */
extern T0* T65c249(GE_context* ac);
/* [detachable] SQL_GENERATOR_TSQL2000.make */
extern T0* T64c249(GE_context* ac);
/* [detachable] SQL_GENERATOR_TSQL2016.make */
extern T0* T63c249(GE_context* ac);
/* [detachable] SQL_GENERATOR_SQLITE_3.make */
extern T0* T62c248(GE_context* ac);
/* [detachable] SQL_GENERATOR_PGSQL_72.make */
extern T0* T61c272(GE_context* ac);
/* [detachable] SQL_GENERATOR_PGSQL_73.make */
extern T0* T60c275(GE_context* ac);
/* [detachable] SQL_GENERATOR_PGSQL_81.make */
extern T0* T59c276(GE_context* ac);
/* [detachable] SQL_GENERATOR_PGSQL_82.make */
extern T0* T58c276(GE_context* ac);
/* [detachable] SQL_GENERATOR_PGSQL_95.make */
extern T0* T57c276(GE_context* ac);
/* [detachable] SQL_GENERATOR_ORACLE.make */
extern T0* T56c270(GE_context* ac);
/* [detachable] SQL_GENERATOR_MYSQL322.make */
extern T0* T55c236(GE_context* ac);
/* [detachable] SQL_GENERATOR_MYSQL4.make */
extern T0* T54c236(GE_context* ac);
/* [detachable] SQL_GENERATOR_MYSQL5.make */
extern T0* T53c261(GE_context* ac);
/* [detachable] SQL_GENERATOR_MSQL.make */
extern T0* T52c235(GE_context* ac);
/* [detachable] SQL_GENERATOR_MSACCESS2000.make */
extern T0* T51c235(GE_context* ac);
/* [detachable] SQL_GENERATOR_MSACCESS97.make */
extern T0* T50c235(GE_context* ac);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.make */
extern T0* T49c265(GE_context* ac);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.make */
extern void T49f265p1(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.make */
extern T0* T48c265(GE_context* ac);
/* [detachable] SQL_GENERATOR_INTERBASE6.make */
extern void T48f265p1(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.make */
extern T0* T47c235(GE_context* ac);
/* [detachable] SQL_GENERATOR_DB2_71.make */
extern T0* T46c266(GE_context* ac);
/* [detachable] SQL_GENERATOR_DB2_6.make */
extern T0* T45c237(GE_context* ac);
/* [detachable] SQL_GENERATOR_BASIC.make */
extern T0* T44c236(GE_context* ac);
/* [detachable] SQL_GENERATOR_ANSI.make */
extern T0* T43c236(GE_context* ac);
/* [detachable] KL_STDERR_FILE.put_character */
extern void T42f7(GE_context* ac, T0* C, T2 a1);
/* [detachable] CONSOLE.put_character */
extern void T437f22(GE_context* ac, T0* C, T2 a1);
/* [detachable] CONSOLE.console_pc */
extern void T437f29(GE_context* ac, T0* C, T14 a1, T2 a2);
/* [detachable] KL_STDERR_FILE.console */
extern T0* T42f2(GE_context* ac, T0* C);
/* [detachable] STD_FILES.error */
extern T0* T351f2(GE_context* ac, T0* C);
/* [detachable] CONSOLE.make_open_stderr */
extern T0* T437c23(GE_context* ac, T0* a1);
/* [detachable] KL_STDERR_FILE.io */
extern T0* T42f3(GE_context* ac, T0* C);
/* [detachable] KL_STDERR_FILE.put_string */
extern void T42f6(GE_context* ac, T0* C, T0* a1);
/* [detachable] CONSOLE.put_string */
extern void T437f21(GE_context* ac, T0* C, T0* a1);
/* [detachable] CONSOLE.console_ps */
extern void T437f28(GE_context* ac, T0* C, T14 a1, T14 a2, T6 a3);
/* [detachable] KL_STDERR_FILE.string_ */
extern T0* T42f1(GE_context* ac, T0* C);
/* [detachable] KL_STANDARD_FILES.error */
extern T0* T36f2(GE_context* ac, T0* C);
/* [detachable] KL_STDERR_FILE.make */
extern T0* T42c5(GE_context* ac);
/* [detachable] XPLAIN2SQL.valid_format_and_parameters */
extern T1 T25f5(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_SCIENTIFIC_FORMATTER.valid_format_and_parameters */
extern T1 T105f19(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_SCIENTIFIC_FORMATTER.null_output_stream */
extern T0* T105f22(GE_context* ac, T0* C);
/* [detachable] KL_NULL_TEXT_OUTPUT_STREAM.make */
extern T0* T374c2(GE_context* ac, T0* a1);
/* [detachable] XPLAIN2SQL.formatter */
extern T0* T25f7(GE_context* ac, T0* C);
/* [detachable] XPLAIN2SQL.scientific_formatter */
extern T0* T25f8(GE_context* ac, T0* C);
/* INTEGER_32.is_less_equal */
extern T1 T6f4(GE_context* ac, T6* C, T6 a1);
/* [detachable] KL_STDOUT_FILE.put_string */
extern void T37f6(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_STDOUT_FILE.console */
extern T0* T37f2(GE_context* ac, T0* C);
/* [detachable] KL_STDOUT_FILE.io */
extern T0* T37f3(GE_context* ac, T0* C);
/* [detachable] KL_STDOUT_FILE.string_ */
extern T0* T37f1(GE_context* ac, T0* C);
/* [detachable] KL_STANDARD_FILES.output */
extern T0* T36f1(GE_context* ac, T0* C);
/* [detachable] KL_STDOUT_FILE.make */
extern T0* T37c5(GE_context* ac);
/* [detachable] XPLAIN2SQL.std */
extern T0* T25f3(GE_context* ac, T0* C);
/* [detachable] STRING_8.is_equal */
extern T1 T17f18(GE_context* ac, T0* C, T0* a1);
/* [detachable] STRING_8.keep_tail */
extern void T17f58(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN2SQL.equal */
extern T1 T25f2(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] STRING_8.substring */
extern T0* T17f16(GE_context* ac, T0* C, T6 a1, T6 a2);
/* [detachable] KL_ARGUMENTS.argument */
extern T0* T35f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] IMMUTABLE_STRING_32.as_string_8 */
extern T0* T20f6(GE_context* ac, T0* C);
/* [detachable] IMMUTABLE_STRING_32.code */
extern T10 T20f5(GE_context* ac, T0* C, T6 a1);
/* [detachable] ARGUMENTS_32.argument */
extern T0* T115f2(GE_context* ac, T0* C, T6 a1);
/* [detachable] ARRAY [[attached] IMMUTABLE_STRING_32].item */
extern T0* T396f5(GE_context* ac, T0* C, T6 a1);
/* [detachable] ARGUMENTS_32.internal_argument_array */
extern T0* T115f3(GE_context* ac, T0* C);
/* [detachable] ARRAY [[attached] IMMUTABLE_STRING_32].put */
extern void T396f8(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] ARRAY [[attached] IMMUTABLE_STRING_32].compare_objects */
extern void T396f7(GE_context* ac, T0* C);
/* [detachable] ARRAY [[attached] IMMUTABLE_STRING_32].make_filled */
extern T0* T396c6(GE_context* ac, T0* a1, T6 a2, T6 a3);
/* [detachable] ARRAY [[attached] IMMUTABLE_STRING_32].make_filled_area */
extern void T396f9(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [[attached] IMMUTABLE_STRING_32].make_filled */
extern T0* T397c4(GE_context* ac, T0* a1, T6 a2);
/* [detachable] SPECIAL [[attached] IMMUTABLE_STRING_32].fill_with */
extern void T397f7(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] IMMUTABLE_STRING_32].extend */
extern void T397f8(GE_context* ac, T0* C, T0* a1);
/* [detachable] SPECIAL [[attached] IMMUTABLE_STRING_32].make_empty */
extern void T397f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] IMMUTABLE_STRING_32.make_empty */
extern T0* T20c7(GE_context* ac);
/* [detachable] IMMUTABLE_STRING_32.make */
extern void T20f8(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [CHARACTER_32].make_filled */
extern T0* T16c4(GE_context* ac, T3 a1, T6 a2);
/* [detachable] SPECIAL [CHARACTER_32].fill_with */
extern void T16f6(GE_context* ac, T0* C, T3 a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [CHARACTER_32].extend */
extern void T16f8(GE_context* ac, T0* C, T3 a1);
/* [detachable] SPECIAL [CHARACTER_32].make_empty */
extern void T16f5(GE_context* ac, T0* C, T6 a1);
/* [detachable] KL_ARGUMENTS.internal_arguments */
extern T0* T35f3(GE_context* ac, T0* C);
/* [detachable] ARGUMENTS_32.default_create */
extern T0* T115c5(GE_context* ac);
/* INTEGER_32.is_greater */
extern T1 T6f14(GE_context* ac, T6* C, T6 a1);
/* [detachable] XPLAIN2SQL.die */
extern void T25f11(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN2SQL.show_usage */
extern void T25f10(GE_context* ac, T0* C);
/* [detachable] KL_ARGUMENTS.argument_count */
extern T6 T35f4(GE_context* ac, T0* C);
/* [detachable] XPLAIN2SQL.arguments */
extern T0* T25f6(GE_context* ac, T0* C);
/* [detachable] KL_ARGUMENTS.make */
extern T0* T35c5(GE_context* ac);
/* [detachable] ISE_EXCEPTION_MANAGER.set_exception_data */
extern void T21f16(GE_context* ac, T0* C, T6 a1, T1 a2, T6 a3, T6 a4, T0* a5, T0* a6, T0* a7, T0* a8, T0* a9, T0* a10, T6 a11, T1 a12);
/* [detachable] ISE_EXCEPTION_MANAGER.exception_from_data */
extern T0* T21f10(GE_context* ac, T0* C);
extern T1 T21f10ot1(T0* a1, T0** a2);
extern T1 T21f10ot2(T0* a1, T0** a2);
extern T1 T21f10ot3(T0* a1, T0** a2);
extern T1 T21f10ot4(T0* a1, T0** a2);
extern T1 T21f10ot5(T0* a1, T0** a2);
extern T1 T21f10ot6(T0* a1, T0** a2);
extern T1 T21f10ot7(T0* a1, T0** a2);
/* [detachable] COM_FAILURE.set_exception_information */
extern void T85f15(GE_context* ac, T0* C, T0* a1);
/* [detachable] COM_FAILURE.ccom_hresult_to_string */
extern T0* T85f11(GE_context* ac, T0* C, T6 a1);
/* INTEGER_32.to_hex_string */
extern T0* T6f22(GE_context* ac, T6* C);
/* INTEGER_32.to_hex_character */
extern T2 T6f26(GE_context* ac, T6* C);
/* [detachable] COM_FAILURE.set_hresult_code */
extern void T85f14(GE_context* ac, T0* C, T6 a1);
/* [detachable] OPERATING_SYSTEM_FAILURE.set_error_code */
extern void T84f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] IO_FAILURE.set_error_code */
extern void T83f11(GE_context* ac, T0* C, T6 a1);
/* [detachable] OPERATING_SYSTEM_SIGNAL_FAILURE.set_signal_code */
extern void T82f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] INVARIANT_VIOLATION.set_is_entry */
extern void T81f9(GE_context* ac, T0* C, T1 a1);
/* [detachable] ROUTINE_FAILURE.set_class_name */
extern void T76f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] ROUTINE_FAILURE.set_routine_name */
extern void T76f9(GE_context* ac, T0* C, T0* a1);
/* [detachable] ISE_EXCEPTION_MANAGER.exception_from_code */
extern T0* T21f12(GE_context* ac, T0* C, T6 a1);
/* [detachable] SERIALIZATION_FAILURE.default_create */
extern T0* T104c6(GE_context* ac);
/* [detachable] OLD_VIOLATION.default_create */
extern T0* T78c6(GE_context* ac);
/* [detachable] COM_FAILURE.default_create */
extern T0* T85c12(GE_context* ac);
/* [detachable] ADDRESS_APPLIED_TO_MELTED_FEATURE.default_create */
extern T0* T103c6(GE_context* ac);
/* [detachable] MISMATCH_FAILURE.default_create */
extern T0* T101c6(GE_context* ac);
/* [detachable] OPERATING_SYSTEM_FAILURE.default_create */
extern T0* T84c7(GE_context* ac);
/* [detachable] IO_FAILURE.set_code */
extern void T83f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] IO_FAILURE.default_create */
extern T0* T83c8(GE_context* ac);
/* [detachable] EXCEPTION_IN_SIGNAL_HANDLER_FAILURE.default_create */
extern T0* T100c6(GE_context* ac);
/* [detachable] VOID_ASSIGNED_TO_EXPANDED.default_create */
extern T0* T99c6(GE_context* ac);
/* [detachable] EXTERNAL_FAILURE.default_create */
extern T0* T98c6(GE_context* ac);
/* [detachable] CREATE_ON_DEFERRED.default_create */
extern T0* T97c6(GE_context* ac);
/* [detachable] RESUMPTION_FAILURE.default_create */
extern T0* T96c6(GE_context* ac);
/* [detachable] RESCUE_FAILURE.default_create */
extern T0* T95c6(GE_context* ac);
/* [detachable] EIFFEL_RUNTIME_PANIC.set_code */
extern void T86f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] EIFFEL_RUNTIME_PANIC.code */
extern T6 T86f7(GE_context* ac, T0* C);
/* [detachable] EIFFEL_RUNTIME_PANIC.default_create */
extern T0* T86c8(GE_context* ac);
/* [detachable] OPERATING_SYSTEM_SIGNAL_FAILURE.default_create */
extern T0* T82c7(GE_context* ac);
/* [detachable] LOOP_INVARIANT_VIOLATION.default_create */
extern T0* T94c6(GE_context* ac);
/* [detachable] VARIANT_VIOLATION.default_create */
extern T0* T93c6(GE_context* ac);
/* [detachable] BAD_INSPECT_VALUE.default_create */
extern T0* T92c6(GE_context* ac);
/* [detachable] ROUTINE_FAILURE.default_create */
extern T0* T76c8(GE_context* ac);
/* [detachable] CHECK_VIOLATION.default_create */
extern T0* T91c6(GE_context* ac);
/* [detachable] INVARIANT_VIOLATION.default_create */
extern T0* T81c7(GE_context* ac);
/* [detachable] FLOATING_POINT_FAILURE.default_create */
extern T0* T90c6(GE_context* ac);
/* [detachable] POSTCONDITION_VIOLATION.default_create */
extern T0* T89c6(GE_context* ac);
/* [detachable] PRECONDITION_VIOLATION.default_create */
extern T0* T88c6(GE_context* ac);
/* [detachable] NO_MORE_MEMORY.set_code */
extern void T80f10(GE_context* ac, T0* C, T6 a1);
/* [detachable] NO_MORE_MEMORY.code */
extern T6 T80f7(GE_context* ac, T0* C);
/* [detachable] ISE_EXCEPTION_MANAGER.no_memory_exception_object_cell */
extern T0* T21f7(GE_context* ac, T0* C);
/* [detachable] CELL [[attached] NO_MORE_MEMORY].put */
extern T0* T29c2(GE_context* ac, T0* a1);
/* [detachable] NO_MORE_MEMORY.default_create */
extern T0* T80c8(GE_context* ac);
/* [detachable] VOID_TARGET.default_create */
extern T0* T74c6(GE_context* ac);
/* [detachable] ISE_EXCEPTION_MANAGER.exception_data */
extern T0* T21f11(GE_context* ac, T0* C);
/* [detachable] CELL [detachable TUPLE [[attached] INTEGER_32, [attached] INTEGER_32, [attached] INTEGER_32, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] INTEGER_32, [attached] BOOLEAN]].put */
extern void T28f2(GE_context* ac, T0* C, T0* a1);
/* [detachable] CELL [detachable TUPLE [[attached] INTEGER_32, [attached] INTEGER_32, [attached] INTEGER_32, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] INTEGER_32, [attached] BOOLEAN]].put */
extern T0* T28c2(GE_context* ac, T0* a1);
/* [detachable] ISE_EXCEPTION_MANAGER.exception_data_cell */
extern T0* T21f6(GE_context* ac, T0* C);
/* [detachable] ISE_EXCEPTION_MANAGER.once_raise */
extern void T21f15(GE_context* ac, T0* C, T0* a1);
/* [detachable] ISE_EXCEPTION_MANAGER.last_exception */
extern T0* T21f1(GE_context* ac, T0* C);
/* [detachable] ISE_EXCEPTION_MANAGER.init_exception_manager */
extern void T21f14(GE_context* ac, T0* C);
/* [detachable] CELL [[attached] NO_MORE_MEMORY].do_nothing */
extern void T29f3(GE_context* ac, T0* C);
/* [detachable] CELL [detachable EXCEPTION].do_nothing */
extern void T26f3(GE_context* ac, T0* C);
/* [detachable] CELL [detachable TUPLE [[attached] INTEGER_32, [attached] INTEGER_32, [attached] INTEGER_32, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] STRING_8, [attached] INTEGER_32, [attached] BOOLEAN]].do_nothing */
extern void T28f3(GE_context* ac, T0* C);
/* [detachable] ISE_EXCEPTION_MANAGER.unraisable_exceptions */
extern T0* T21f5(GE_context* ac, T0* C);
/* [detachable] HASH_TABLE [[attached] INTEGER_32, [attached] INTEGER_32].force */
extern void T27f37(GE_context* ac, T0* C, T6 a1, T6 a2);
/* [detachable] HASH_TABLE [[attached] INTEGER_32, [attached] INTEGER_32].position */
extern T6 T27f22(GE_context* ac, T0* C);
/* [detachable] SPECIAL [[attached] BOOLEAN].force */
extern void T108f5(GE_context* ac, T0* C, T1 a1, T6 a2);
/* [detachable] HASH_TABLE [[attached] INTEGER_32, [attached] INTEGER_32].deleted_position */
extern T6 T27f21(GE_context* ac, T0* C, T6 a1);
/* [detachable] HASH_TABLE [[attached] INTEGER_32, [attached] INTEGER_32].add_space */
extern void T27f39(GE_context* ac, T0* C);
/* [detachable] HASH_TABLE [[attached] INTEGER_32, [attached] INTEGER_32].accommodate */
extern void T27f40(GE_context* ac, T0* C, T6 a1);
/* [detachable] HASH_TABLE [[attached] INTEGER_32, [attached] INTEGER_32].set_indexes_map */
extern void T27f44(GE_context* ac, T0* C, T0* a1);
/* [detachable] HASH_TABLE [[attached] INTEGER_32, [attached] INTEGER_32].set_deleted_marks */
extern void T27f43(GE_context* ac, T0* C, T0* a1);
/* [detachable] HASH_TABLE [[attached] INTEGER_32, [attached] INTEGER_32].set_keys */
extern void T27f42(GE_context* ac, T0* C, T0* a1);
/* [detachable] HASH_TABLE [[attached] INTEGER_32, [attached] INTEGER_32].set_content */
extern void T27f41(GE_context* ac, T0* C, T0* a1);
/* [detachable] HASH_TABLE [[attached] INTEGER_32, [attached] INTEGER_32].put */
extern void T27f45(GE_context* ac, T0* C, T6 a1, T6 a2);
/* [detachable] HASH_TABLE [[attached] INTEGER_32, [attached] INTEGER_32].set_conflict */
extern void T27f47(GE_context* ac, T0* C);
/* [detachable] HASH_TABLE [[attached] INTEGER_32, [attached] INTEGER_32].found */
extern T1 T27f31(GE_context* ac, T0* C);
/* [detachable] HASH_TABLE [[attached] INTEGER_32, [attached] INTEGER_32].occupied */
extern T1 T27f29(GE_context* ac, T0* C, T6 a1);
/* [detachable] HASH_TABLE [[attached] INTEGER_32, [attached] INTEGER_32].empty_duplicate */
extern T0* T27f28(GE_context* ac, T0* C, T6 a1);
/* [detachable] HASH_TABLE [[attached] INTEGER_32, [attached] INTEGER_32].compare_objects */
extern void T27f46(GE_context* ac, T0* C);
/* [detachable] HASH_TABLE [[attached] INTEGER_32, [attached] INTEGER_32].soon_full */
extern T1 T27f20(GE_context* ac, T0* C);
/* [detachable] HASH_TABLE [[attached] INTEGER_32, [attached] INTEGER_32].not_found */
extern T1 T27f19(GE_context* ac, T0* C);
/* [detachable] HASH_TABLE [[attached] INTEGER_32, [attached] INTEGER_32].internal_search */
extern void T27f38(GE_context* ac, T0* C, T6 a1);
/* [detachable] ISE_EXCEPTION_MANAGER.unignorable_exceptions */
extern T0* T21f4(GE_context* ac, T0* C);
/* [detachable] HASH_TABLE [[attached] INTEGER_32, [attached] INTEGER_32].do_nothing */
extern void T27f36(GE_context* ac, T0* C);
/* [detachable] KL_TEXT_OUTPUT_FILE.dispose */
extern void T348f24(GE_context* ac, T0* C);
/* [detachable] MANAGED_POINTER.dispose */
extern void T113f8(GE_context* ac, T0* C);
/* POINTER.memory_free */
extern void T14f9(GE_context* ac, T14* C);
/* POINTER.default_pointer */
extern T14 T14f8(GE_context* ac, T14* C);
/* POINTER.c_free */
extern void T14f12(GE_context* ac, T14* C, T14 a1);
/* detachable KL_TEXT_INPUT_FILE.dispose */
extern void T32f39(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_value_declare */
extern void T61f351(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8].force */
extern void T116f37(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8].key_storage_put */
extern void T116f47(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8].slots_put */
extern void T116f46(GE_context* ac, T0* C, T6 a1, T6 a2);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8].clashes_put */
extern void T116f45(GE_context* ac, T0* C, T6 a1, T6 a2);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8].slots_item */
extern T6 T116f23(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8].clashes_item */
extern T6 T116f21(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8].hash_position */
extern T6 T116f26(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8].resize */
extern void T116f44(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8].clashes_resize */
extern void T116f56(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8].special_integer_ */
extern T0* T116f30(GE_context* ac, T0* C);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8].key_storage_resize */
extern void T116f55(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8].item_storage_resize */
extern void T116f54(GE_context* ac, T0* C, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_VALUE].aliased_resized_area */
extern T0* T402f1(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [[attached] XPLAIN_VALUE].aliased_resized_area */
extern T0* T403f3(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8].key_storage_item */
extern T0* T116f29(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8].slots_resize */
extern void T116f53(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8].new_capacity */
extern T6 T116f25(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8].item_storage_put */
extern void T116f43(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_VALUE].force */
extern void T402f4(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_VALUE].fill_with */
extern void T403f8(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_VALUE].extend */
extern void T403f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8].search_position */
extern void T116f41(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_value_declare_outside_sp */
extern void T61f360(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_value_alter_table */
extern void T61f362(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.commandseparator */
extern T0* T61f45(GE_context* ac, T0* C);
/* detachable XPLAIN_VALUE.quoted_name */
extern T0* T264f5(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.valuetablename */
extern T0* T61f99(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.createtemporaryvaluetable */
extern T1 T61f110(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.temporarytablessupported */
extern T1 T61f176(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_value_create_table */
extern void T61f361(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.finishtemporaryvaluetablestatement */
extern T0* T61f111(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.finishtemporarytablestatement */
extern T0* T61f104(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.createtemporarytablestatement */
extern T0* T61f102(GE_context* ac, T0* C);
/* [detachable] KL_STDOUT_FILE.put_character */
extern void T37f7(GE_context* ac, T0* C, T2 a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.std */
extern T0* T61f109(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_value_declare_inside_sp */
extern void T61f359(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_value_declare */
extern void T60f354(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_value_declare_outside_sp */
extern void T60f365(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_value_alter_table */
extern void T60f367(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.commandseparator */
extern T0* T60f45(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.valuetablename */
extern T0* T60f99(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.createtemporaryvaluetable */
extern T1 T60f111(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.temporarytablessupported */
extern T1 T60f177(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_value_create_table */
extern void T60f366(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.finishtemporaryvaluetablestatement */
extern T0* T60f112(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.finishtemporarytablestatement */
extern T0* T60f104(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.createtemporarytablestatement */
extern T0* T60f102(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.std */
extern T0* T60f109(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_value_declare_inside_sp */
extern void T60f364(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_value_declare */
extern void T59f355(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_value_declare_outside_sp */
extern void T59f366(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_value_alter_table */
extern void T59f368(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.commandseparator */
extern T0* T59f45(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.valuetablename */
extern T0* T59f100(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.createtemporaryvaluetable */
extern T1 T59f112(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.temporarytablessupported */
extern T1 T59f177(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_value_create_table */
extern void T59f367(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.finishtemporaryvaluetablestatement */
extern T0* T59f113(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.finishtemporarytablestatement */
extern T0* T59f105(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.createtemporarytablestatement */
extern T0* T59f103(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.std */
extern T0* T59f110(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_value_declare_inside_sp */
extern void T59f365(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_value_declare */
extern void T58f355(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_value_declare_outside_sp */
extern void T58f366(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_value_alter_table */
extern void T58f368(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.commandseparator */
extern T0* T58f45(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.valuetablename */
extern T0* T58f100(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.createtemporaryvaluetable */
extern T1 T58f112(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.temporarytablessupported */
extern T1 T58f177(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_value_create_table */
extern void T58f367(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.finishtemporaryvaluetablestatement */
extern T0* T58f113(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.finishtemporarytablestatement */
extern T0* T58f105(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.createtemporarytablestatement */
extern T0* T58f103(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.std */
extern T0* T58f110(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_value_declare_inside_sp */
extern void T58f365(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_value_declare */
extern void T57f355(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_value_declare_outside_sp */
extern void T57f364(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_value_alter_table */
extern void T57f366(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.commandseparator */
extern T0* T57f45(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.valuetablename */
extern T0* T57f99(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.createtemporaryvaluetable */
extern T1 T57f111(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.temporarytablessupported */
extern T1 T57f174(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_value_create_table */
extern void T57f365(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.finishtemporaryvaluetablestatement */
extern T0* T57f112(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.finishtemporarytablestatement */
extern T0* T57f104(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.createtemporarytablestatement */
extern T0* T57f102(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.std */
extern T0* T57f109(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_value_declare_inside_sp */
extern void T57f363(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.sp_value_declarations */
extern void T61f358(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_BILINKED_LIST [[attached] XPLAIN_STATEMENT].forth */
extern void T360f15(GE_context* ac, T0* C);
/* [detachable] DS_BILINKED_LIST [[attached] XPLAIN_STATEMENT].cursor_forth */
extern void T360f19(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_BILINKED_LIST [[attached] XPLAIN_STATEMENT].add_traversing_cursor */
extern void T360f20(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_BILINKED_LIST_CURSOR [[attached] XPLAIN_STATEMENT].set_next_cursor */
extern void T414f14(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_BILINKED_LIST [[attached] XPLAIN_STATEMENT].remove_traversing_cursor */
extern void T360f21(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_BILINKED_LIST_CURSOR [[attached] XPLAIN_STATEMENT].set */
extern void T414f13(GE_context* ac, T0* C, T0* a1, T1 a2, T1 a3);
/* [detachable] DS_BILINKED_LIST [[attached] XPLAIN_STATEMENT].item_for_iteration */
extern T0* T360f7(GE_context* ac, T0* C);
/* [detachable] DS_BILINKED_LIST [[attached] XPLAIN_STATEMENT].cursor_item */
extern T0* T360f6(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_BILINKED_LIST [[attached] XPLAIN_STATEMENT].after */
extern T1 T360f9(GE_context* ac, T0* C);
/* [detachable] DS_BILINKED_LIST [[attached] XPLAIN_STATEMENT].cursor_after */
extern T1 T360f5(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_BILINKED_LIST [[attached] XPLAIN_STATEMENT].start */
extern void T360f14(GE_context* ac, T0* C);
/* [detachable] DS_BILINKED_LIST [[attached] XPLAIN_STATEMENT].cursor_start */
extern void T360f18(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_BILINKED_LIST [[attached] XPLAIN_STATEMENT].cursor_off */
extern T1 T360f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_BILINKED_LIST_CURSOR [[attached] XPLAIN_STATEMENT].off */
extern T1 T414f7(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.sp_value_declarations */
extern void T60f363(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.sp_value_declarations */
extern void T59f364(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.sp_value_declarations */
extern void T58f364(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.sp_value_declarations */
extern void T57f362(GE_context* ac, T0* C, T0* a1);
/* [detachable] ST_STRING_FORMATTER.set_align_left */
extern void T390f13(GE_context* ac, T0* C);
/* [detachable] ST_POINTER_FORMATTER.set_align_left */
extern void T389f13(GE_context* ac, T0* C);
/* [detachable] ST_UNSIGNED_DECIMAL_FORMATTER.set_align_left */
extern void T388f20(GE_context* ac, T0* C);
/* [detachable] ST_UNSIGNED_HEXADECIMAL_FORMATTER.set_align_left */
extern void T387f20(GE_context* ac, T0* C);
/* [detachable] ST_UNSIGNED_OCTAL_FORMATTER.set_align_left */
extern void T386f20(GE_context* ac, T0* C);
/* [detachable] ST_DECIMAL_FORMATTER.set_align_left */
extern void T385f20(GE_context* ac, T0* C);
/* [detachable] ST_FIXED_OR_FLOATING_POINT_FORMATTER.set_align_left */
extern void T384f26(GE_context* ac, T0* C);
/* [detachable] ST_FIXED_POINT_FORMATTER.set_align_left */
extern void T383f25(GE_context* ac, T0* C);
/* [detachable] ST_FLOATING_POINT_FORMATTER.set_align_left */
extern void T382f26(GE_context* ac, T0* C);
/* [detachable] ST_CHARACTER_FORMATTER.set_align_left */
extern void T381f14(GE_context* ac, T0* C);
/* [detachable] ST_BOOLEAN_FORMATTER.set_align_left */
extern void T380f17(GE_context* ac, T0* C);
/* [detachable] ST_STRING_FORMATTER.set_align_center */
extern void T390f14(GE_context* ac, T0* C);
/* [detachable] ST_POINTER_FORMATTER.set_align_center */
extern void T389f14(GE_context* ac, T0* C);
/* [detachable] ST_UNSIGNED_DECIMAL_FORMATTER.set_align_center */
extern void T388f21(GE_context* ac, T0* C);
/* [detachable] ST_UNSIGNED_HEXADECIMAL_FORMATTER.set_align_center */
extern void T387f21(GE_context* ac, T0* C);
/* [detachable] ST_UNSIGNED_OCTAL_FORMATTER.set_align_center */
extern void T386f21(GE_context* ac, T0* C);
/* [detachable] ST_DECIMAL_FORMATTER.set_align_center */
extern void T385f21(GE_context* ac, T0* C);
/* [detachable] ST_FIXED_OR_FLOATING_POINT_FORMATTER.set_align_center */
extern void T384f27(GE_context* ac, T0* C);
/* [detachable] ST_FIXED_POINT_FORMATTER.set_align_center */
extern void T383f26(GE_context* ac, T0* C);
/* [detachable] ST_FLOATING_POINT_FORMATTER.set_align_center */
extern void T382f27(GE_context* ac, T0* C);
/* [detachable] ST_CHARACTER_FORMATTER.set_align_center */
extern void T381f15(GE_context* ac, T0* C);
/* [detachable] ST_BOOLEAN_FORMATTER.set_align_center */
extern void T380f18(GE_context* ac, T0* C);
/* [detachable] ST_STRING_FORMATTER.set_padding_character */
extern void T390f15(GE_context* ac, T0* C, T2 a1);
/* [detachable] ST_POINTER_FORMATTER.set_padding_character */
extern void T389f15(GE_context* ac, T0* C, T2 a1);
/* [detachable] ST_UNSIGNED_DECIMAL_FORMATTER.set_padding_character */
extern void T388f22(GE_context* ac, T0* C, T2 a1);
/* [detachable] ST_UNSIGNED_HEXADECIMAL_FORMATTER.set_padding_character */
extern void T387f22(GE_context* ac, T0* C, T2 a1);
/* [detachable] ST_UNSIGNED_OCTAL_FORMATTER.set_padding_character */
extern void T386f22(GE_context* ac, T0* C, T2 a1);
/* [detachable] ST_DECIMAL_FORMATTER.set_padding_character */
extern void T385f22(GE_context* ac, T0* C, T2 a1);
/* [detachable] ST_FIXED_OR_FLOATING_POINT_FORMATTER.set_padding_character */
extern void T384f28(GE_context* ac, T0* C, T2 a1);
/* [detachable] ST_FIXED_POINT_FORMATTER.set_padding_character */
extern void T383f27(GE_context* ac, T0* C, T2 a1);
/* [detachable] ST_FLOATING_POINT_FORMATTER.set_padding_character */
extern void T382f28(GE_context* ac, T0* C, T2 a1);
/* [detachable] ST_CHARACTER_FORMATTER.set_padding_character */
extern void T381f16(GE_context* ac, T0* C, T2 a1);
/* [detachable] ST_BOOLEAN_FORMATTER.set_padding_character */
extern void T380f19(GE_context* ac, T0* C, T2 a1);
/* [detachable] ST_STRING_FORMATTER.set_space_sign_enabled */
extern void T390f16(GE_context* ac, T0* C, T1 a1);
/* [detachable] ST_POINTER_FORMATTER.set_space_sign_enabled */
extern void T389f16(GE_context* ac, T0* C, T1 a1);
/* [detachable] ST_UNSIGNED_DECIMAL_FORMATTER.set_space_sign_enabled */
extern void T388f23(GE_context* ac, T0* C, T1 a1);
/* [detachable] ST_UNSIGNED_HEXADECIMAL_FORMATTER.set_space_sign_enabled */
extern void T387f23(GE_context* ac, T0* C, T1 a1);
/* [detachable] ST_UNSIGNED_OCTAL_FORMATTER.set_space_sign_enabled */
extern void T386f23(GE_context* ac, T0* C, T1 a1);
/* [detachable] ST_DECIMAL_FORMATTER.set_space_sign_enabled */
extern void T385f23(GE_context* ac, T0* C, T1 a1);
/* [detachable] ST_FIXED_OR_FLOATING_POINT_FORMATTER.set_space_sign_enabled */
extern void T384f29(GE_context* ac, T0* C, T1 a1);
/* [detachable] ST_FIXED_POINT_FORMATTER.set_space_sign_enabled */
extern void T383f28(GE_context* ac, T0* C, T1 a1);
/* [detachable] ST_FLOATING_POINT_FORMATTER.set_space_sign_enabled */
extern void T382f29(GE_context* ac, T0* C, T1 a1);
/* [detachable] ST_CHARACTER_FORMATTER.set_space_sign_enabled */
extern void T381f17(GE_context* ac, T0* C, T1 a1);
/* [detachable] ST_BOOLEAN_FORMATTER.set_space_sign_enabled */
extern void T380f20(GE_context* ac, T0* C, T1 a1);
/* [detachable] ST_STRING_FORMATTER.set_plus_sign_enabled */
extern void T390f17(GE_context* ac, T0* C, T1 a1);
/* [detachable] ST_POINTER_FORMATTER.set_plus_sign_enabled */
extern void T389f17(GE_context* ac, T0* C, T1 a1);
/* [detachable] ST_UNSIGNED_DECIMAL_FORMATTER.set_plus_sign_enabled */
extern void T388f24(GE_context* ac, T0* C, T1 a1);
/* [detachable] ST_UNSIGNED_HEXADECIMAL_FORMATTER.set_plus_sign_enabled */
extern void T387f24(GE_context* ac, T0* C, T1 a1);
/* [detachable] ST_UNSIGNED_OCTAL_FORMATTER.set_plus_sign_enabled */
extern void T386f24(GE_context* ac, T0* C, T1 a1);
/* [detachable] ST_DECIMAL_FORMATTER.set_plus_sign_enabled */
extern void T385f24(GE_context* ac, T0* C, T1 a1);
/* [detachable] ST_FIXED_OR_FLOATING_POINT_FORMATTER.set_plus_sign_enabled */
extern void T384f30(GE_context* ac, T0* C, T1 a1);
/* [detachable] ST_FIXED_POINT_FORMATTER.set_plus_sign_enabled */
extern void T383f29(GE_context* ac, T0* C, T1 a1);
/* [detachable] ST_FLOATING_POINT_FORMATTER.set_plus_sign_enabled */
extern void T382f30(GE_context* ac, T0* C, T1 a1);
/* [detachable] ST_CHARACTER_FORMATTER.set_plus_sign_enabled */
extern void T381f18(GE_context* ac, T0* C, T1 a1);
/* [detachable] ST_BOOLEAN_FORMATTER.set_plus_sign_enabled */
extern void T380f21(GE_context* ac, T0* C, T1 a1);
/* [detachable] ST_STRING_FORMATTER.set_width */
extern void T390f18(GE_context* ac, T0* C, T6 a1);
/* [detachable] ST_POINTER_FORMATTER.set_width */
extern void T389f18(GE_context* ac, T0* C, T6 a1);
/* [detachable] ST_UNSIGNED_DECIMAL_FORMATTER.set_width */
extern void T388f25(GE_context* ac, T0* C, T6 a1);
/* [detachable] ST_UNSIGNED_HEXADECIMAL_FORMATTER.set_width */
extern void T387f25(GE_context* ac, T0* C, T6 a1);
/* [detachable] ST_UNSIGNED_OCTAL_FORMATTER.set_width */
extern void T386f25(GE_context* ac, T0* C, T6 a1);
/* [detachable] ST_DECIMAL_FORMATTER.set_width */
extern void T385f25(GE_context* ac, T0* C, T6 a1);
/* [detachable] ST_FIXED_OR_FLOATING_POINT_FORMATTER.set_width */
extern void T384f31(GE_context* ac, T0* C, T6 a1);
/* [detachable] ST_FIXED_POINT_FORMATTER.set_width */
extern void T383f30(GE_context* ac, T0* C, T6 a1);
/* [detachable] ST_FLOATING_POINT_FORMATTER.set_width */
extern void T382f31(GE_context* ac, T0* C, T6 a1);
/* [detachable] ST_CHARACTER_FORMATTER.set_width */
extern void T381f19(GE_context* ac, T0* C, T6 a1);
/* [detachable] ST_BOOLEAN_FORMATTER.set_width */
extern void T380f22(GE_context* ac, T0* C, T6 a1);
/* [detachable] ST_STRING_FORMATTER.set_precision */
extern void T390f19(GE_context* ac, T0* C, T6 a1);
/* [detachable] ST_POINTER_FORMATTER.set_precision */
extern void T389f19(GE_context* ac, T0* C, T6 a1);
/* [detachable] ST_UNSIGNED_DECIMAL_FORMATTER.set_precision */
extern void T388f26(GE_context* ac, T0* C, T6 a1);
/* [detachable] ST_UNSIGNED_HEXADECIMAL_FORMATTER.set_precision */
extern void T387f26(GE_context* ac, T0* C, T6 a1);
/* [detachable] ST_UNSIGNED_OCTAL_FORMATTER.set_precision */
extern void T386f26(GE_context* ac, T0* C, T6 a1);
/* [detachable] ST_DECIMAL_FORMATTER.set_precision */
extern void T385f26(GE_context* ac, T0* C, T6 a1);
/* [detachable] ST_FIXED_OR_FLOATING_POINT_FORMATTER.set_precision */
extern void T384f32(GE_context* ac, T0* C, T6 a1);
/* [detachable] ST_FIXED_POINT_FORMATTER.set_precision */
extern void T383f31(GE_context* ac, T0* C, T6 a1);
/* [detachable] ST_FLOATING_POINT_FORMATTER.set_precision */
extern void T382f32(GE_context* ac, T0* C, T6 a1);
/* [detachable] ST_CHARACTER_FORMATTER.set_precision */
extern void T381f20(GE_context* ac, T0* C, T6 a1);
/* [detachable] ST_BOOLEAN_FORMATTER.set_precision */
extern void T380f23(GE_context* ac, T0* C, T6 a1);
/* [detachable] ST_STRING_FORMATTER.format_to */
extern void T390f20(GE_context* ac, T0* C, T0* a1, T0* a2);
extern T1 T390f20ot1(T0* a1, T0** a2);
/* [detachable] ST_STRING_FORMATTER.string_format_to */
extern void T390f23(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_STRING_FORMATTER.justify_format_to */
extern void T390f25(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_STRING_FORMATTER.center_format_to */
extern void T390f28(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_STRING_FORMATTER.right_format_to */
extern void T390f27(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_STRING_FORMATTER.left_format_to */
extern void T390f26(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_POINTER_FORMATTER.format_to */
extern void T389f20(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_POINTER_FORMATTER.pointer_format_to */
extern void T389f23(GE_context* ac, T0* C, T14 a1, T0* a2);
/* [detachable] ST_POINTER_FORMATTER.justify_format_to */
extern void T389f25(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_POINTER_FORMATTER.center_format_to */
extern void T389f28(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_POINTER_FORMATTER.right_format_to */
extern void T389f27(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_POINTER_FORMATTER.left_format_to */
extern void T389f26(GE_context* ac, T0* C, T0* a1, T0* a2);
/* POINTER.out */
extern T0* T14f4(GE_context* ac, T14* C);
/* [detachable] ST_UNSIGNED_DECIMAL_FORMATTER.format_to */
extern void T388f27(GE_context* ac, T0* C, T0* a1, T0* a2);
extern T1 T388f27ot1(T0* a1, T0** a2);
/* [detachable] ST_UNSIGNED_DECIMAL_FORMATTER.integer_format_to */
extern void T388f30(GE_context* ac, T0* C, T6 a1, T0* a2);
/* [detachable] ST_UNSIGNED_DECIMAL_FORMATTER.integer_format_to */
extern void T388f30p1(GE_context* ac, T0* C, T6 a1, T0* a2);
/* [detachable] ST_UNSIGNED_DECIMAL_FORMATTER.justify_format_to */
extern void T388f37(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_UNSIGNED_DECIMAL_FORMATTER.center_format_to */
extern void T388f36(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_UNSIGNED_DECIMAL_FORMATTER.right_format_to */
extern void T388f35(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_UNSIGNED_DECIMAL_FORMATTER.left_format_to */
extern void T388f34(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] STRING_8.precede */
extern void T17f57(GE_context* ac, T0* C, T2 a1);
/* [detachable] ST_UNSIGNED_DECIMAL_FORMATTER.append_to_string */
extern void T388f33(GE_context* ac, T0* C, T6 a1, T0* a2);
/* INTEGER_32.abs */
extern T6 T6f24(GE_context* ac, T6* C);
/* INTEGER_32.abs_ref */
extern T6 T6f28(GE_context* ac, T6* C);
/* [detachable] ST_UNSIGNED_DECIMAL_FORMATTER.append_abs_to_string */
extern void T388f32(GE_context* ac, T0* C, T6 a1, T0* a2);
/* [detachable] KL_INTEGER_ROUTINES.mod */
extern T6 T432f3(GE_context* ac, T0* C, T6 a1, T6 a2);
/* [detachable] KL_INTEGER_ROUTINES.div */
extern T6 T432f2(GE_context* ac, T0* C, T6 a1, T6 a2);
/* [detachable] ST_UNSIGNED_DECIMAL_FORMATTER.integer_ */
extern T0* T388f15(GE_context* ac, T0* C);
/* [detachable] KL_INTEGER_ROUTINES.default_create */
extern T0* T432c6(GE_context* ac);
/* [detachable] KL_PLATFORM.minimum_integer */
extern T6 T447f1(GE_context* ac, T0* C);
/* [detachable] ST_UNSIGNED_DECIMAL_FORMATTER.platform */
extern T0* T388f13(GE_context* ac, T0* C);
/* [detachable] KL_PLATFORM.default_create */
extern T0* T447c2(GE_context* ac);
/* [detachable] KL_STRING_ROUTINES.wipe_out */
extern void T110f9(GE_context* ac, T0* C, T0* a1);
/* [detachable] STRING_8.keep_head */
extern void T17f55(GE_context* ac, T0* C, T6 a1);
/* [detachable] ST_UNSIGNED_DECIMAL_FORMATTER.string_ */
extern T0* T388f12(GE_context* ac, T0* C);
/* [detachable] ST_UNSIGNED_HEXADECIMAL_FORMATTER.format_to */
extern void T387f27(GE_context* ac, T0* C, T0* a1, T0* a2);
extern T1 T387f27ot1(T0* a1, T0** a2);
/* [detachable] ST_UNSIGNED_HEXADECIMAL_FORMATTER.integer_format_to */
extern void T387f30(GE_context* ac, T0* C, T6 a1, T0* a2);
/* [detachable] ST_UNSIGNED_HEXADECIMAL_FORMATTER.integer_format_to */
extern void T387f30p1(GE_context* ac, T0* C, T6 a1, T0* a2);
/* [detachable] ST_UNSIGNED_HEXADECIMAL_FORMATTER.justify_format_to */
extern void T387f37(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_UNSIGNED_HEXADECIMAL_FORMATTER.center_format_to */
extern void T387f36(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_UNSIGNED_HEXADECIMAL_FORMATTER.right_format_to */
extern void T387f35(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_UNSIGNED_HEXADECIMAL_FORMATTER.left_format_to */
extern void T387f34(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_UNSIGNED_HEXADECIMAL_FORMATTER.append_to_string */
extern void T387f33(GE_context* ac, T0* C, T6 a1, T0* a2);
/* [detachable] ST_UNSIGNED_HEXADECIMAL_FORMATTER.append_abs_to_string */
extern void T387f32(GE_context* ac, T0* C, T6 a1, T0* a2);
/* [detachable] ST_UNSIGNED_HEXADECIMAL_FORMATTER.integer_ */
extern T0* T387f15(GE_context* ac, T0* C);
/* [detachable] ST_UNSIGNED_HEXADECIMAL_FORMATTER.platform */
extern T0* T387f13(GE_context* ac, T0* C);
/* [detachable] ST_UNSIGNED_HEXADECIMAL_FORMATTER.string_ */
extern T0* T387f12(GE_context* ac, T0* C);
/* [detachable] ST_UNSIGNED_OCTAL_FORMATTER.format_to */
extern void T386f27(GE_context* ac, T0* C, T0* a1, T0* a2);
extern T1 T386f27ot1(T0* a1, T0** a2);
/* [detachable] ST_UNSIGNED_OCTAL_FORMATTER.integer_format_to */
extern void T386f30(GE_context* ac, T0* C, T6 a1, T0* a2);
/* [detachable] ST_UNSIGNED_OCTAL_FORMATTER.integer_format_to */
extern void T386f30p1(GE_context* ac, T0* C, T6 a1, T0* a2);
/* [detachable] ST_UNSIGNED_OCTAL_FORMATTER.justify_format_to */
extern void T386f37(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_UNSIGNED_OCTAL_FORMATTER.center_format_to */
extern void T386f36(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_UNSIGNED_OCTAL_FORMATTER.right_format_to */
extern void T386f35(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_UNSIGNED_OCTAL_FORMATTER.left_format_to */
extern void T386f34(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_UNSIGNED_OCTAL_FORMATTER.append_to_string */
extern void T386f33(GE_context* ac, T0* C, T6 a1, T0* a2);
/* [detachable] ST_UNSIGNED_OCTAL_FORMATTER.append_abs_to_string */
extern void T386f32(GE_context* ac, T0* C, T6 a1, T0* a2);
/* [detachable] ST_UNSIGNED_OCTAL_FORMATTER.integer_ */
extern T0* T386f15(GE_context* ac, T0* C);
/* [detachable] ST_UNSIGNED_OCTAL_FORMATTER.platform */
extern T0* T386f13(GE_context* ac, T0* C);
/* [detachable] ST_UNSIGNED_OCTAL_FORMATTER.string_ */
extern T0* T386f12(GE_context* ac, T0* C);
/* [detachable] ST_DECIMAL_FORMATTER.format_to */
extern void T385f27(GE_context* ac, T0* C, T0* a1, T0* a2);
extern T1 T385f27ot1(T0* a1, T0** a2);
/* [detachable] ST_DECIMAL_FORMATTER.integer_format_to */
extern void T385f30(GE_context* ac, T0* C, T6 a1, T0* a2);
/* [detachable] ST_DECIMAL_FORMATTER.justify_format_to */
extern void T385f37(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_DECIMAL_FORMATTER.center_format_to */
extern void T385f36(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_DECIMAL_FORMATTER.right_format_to */
extern void T385f35(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_DECIMAL_FORMATTER.left_format_to */
extern void T385f34(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_DECIMAL_FORMATTER.append_to_string */
extern void T385f33(GE_context* ac, T0* C, T6 a1, T0* a2);
/* [detachable] ST_DECIMAL_FORMATTER.append_abs_to_string */
extern void T385f32(GE_context* ac, T0* C, T6 a1, T0* a2);
/* [detachable] ST_DECIMAL_FORMATTER.integer_ */
extern T0* T385f15(GE_context* ac, T0* C);
/* [detachable] ST_DECIMAL_FORMATTER.platform */
extern T0* T385f13(GE_context* ac, T0* C);
/* [detachable] ST_DECIMAL_FORMATTER.string_ */
extern T0* T385f12(GE_context* ac, T0* C);
/* [detachable] ST_FIXED_OR_FLOATING_POINT_FORMATTER.format_to */
extern void T384f33(GE_context* ac, T0* C, T0* a1, T0* a2);
extern T1 T384f33ot1(T0* a1, T0** a2);
/* [detachable] ST_FIXED_OR_FLOATING_POINT_FORMATTER.double_format_to */
extern void T384f36(GE_context* ac, T0* C, T13 a1, T0* a2);
/* [detachable] ST_FIXED_OR_FLOATING_POINT_FORMATTER.double_format_to */
extern void T384f36p0(GE_context* ac, T0* C, T13 a1, T0* a2);
/* [detachable] ST_FIXED_OR_FLOATING_POINT_FORMATTER.center_format_to */
extern void T384f42(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_DECIMAL_FORMATTER.set_align_right */
extern void T385f31(GE_context* ac, T0* C);
/* [detachable] ST_FIXED_OR_FLOATING_POINT_FORMATTER.right_format_to */
extern void T384f41(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_FIXED_OR_FLOATING_POINT_FORMATTER.left_format_to */
extern void T384f40(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_FIXED_OR_FLOATING_POINT_FORMATTER.string_ */
extern T0* T384f21(GE_context* ac, T0* C);
/* [detachable] ST_FIXED_OR_FLOATING_POINT_FORMATTER.build_integer_and_fractional_parts */
extern void T384f38(GE_context* ac, T0* C, T13 a1);
/* [detachable] ST_FIXED_OR_FLOATING_POINT_FORMATTER.append_integer_to_string */
extern void T384f43(GE_context* ac, T0* C, T6 a1, T6 a2, T0* a3);
/* [detachable] DS_ARRAYED_LIST [[attached] INTEGER_32].replace */
extern void T445f13(GE_context* ac, T0* C, T6 a1, T6 a2);
/* [detachable] DS_ARRAYED_LIST [[attached] INTEGER_32].item */
extern T6 T445f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] KL_DOUBLE_ROUTINES.truncated_to_integer */
extern T6 T446f4(GE_context* ac, T0* C, T13 a1);
/* [detachable] KL_DOUBLE_ROUTINES.rounded_to_integer */
extern T6 T446f3(GE_context* ac, T0* C, T13 a1);
/* REAL_64.rounded */
extern T6 T13f15(GE_context* ac, T13* C);
/* REAL_64.sign */
extern T6 T13f16(GE_context* ac, T13* C);
/* REAL_64.is_greater */
extern T1 T13f4(GE_context* ac, T13* C, T13 a1);
/* REAL_64.floor */
extern T6 T13f9(GE_context* ac, T13* C);
/* [detachable] DS_ARRAYED_LIST [[attached] INTEGER_32].force_last */
extern void T445f12(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_ARRAYED_LIST [[attached] INTEGER_32].resize */
extern void T445f16(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_ARRAYED_LIST [[attached] INTEGER_32].new_capacity */
extern T6 T445f8(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_ARRAYED_LIST [[attached] INTEGER_32].extendible */
extern T1 T445f7(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_ARRAYED_LIST [[attached] INTEGER_32].wipe_out */
extern void T445f11(GE_context* ac, T0* C);
/* [detachable] DS_ARRAYED_LIST [[attached] INTEGER_32].move_all_cursors_after */
extern void T445f15(GE_context* ac, T0* C);
/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] INTEGER_32].set_next_cursor */
extern void T466f7(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_ARRAYED_LIST_CURSOR [[attached] INTEGER_32].set_after */
extern void T466f6(GE_context* ac, T0* C);
/* [detachable] ST_FIXED_OR_FLOATING_POINT_FORMATTER.double_format_to */
extern void T384f36p1(GE_context* ac, T0* C, T13 a1, T0* a2);
/* [detachable] ST_FIXED_OR_FLOATING_POINT_FORMATTER.justify_format_to */
extern void T384f39(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_FIXED_OR_FLOATING_POINT_FORMATTER.ten_power */
extern T13 T384f20(GE_context* ac, T0* C, T6 a1);
/* [detachable] KL_DOUBLE_ROUTINES.floor_to_integer */
extern T6 T446f2(GE_context* ac, T0* C, T13 a1);
/* [detachable] KL_DOUBLE_ROUTINES.log10 */
extern T13 T446f1(GE_context* ac, T0* C, T13 a1);
/* [detachable] KL_DOUBLE_ROUTINES.old_log10 */
extern T13 T446f5(GE_context* ac, T0* C, T13 a1);
/* [detachable] ST_FIXED_OR_FLOATING_POINT_FORMATTER.double_ */
extern T0* T384f18(GE_context* ac, T0* C);
/* [detachable] KL_DOUBLE_ROUTINES.default_create */
extern T0* T446c6(GE_context* ac);
/* [detachable] ST_FIXED_OR_FLOATING_POINT_FORMATTER.double_sign */
extern T6 T384f17(GE_context* ac, T0* C, T13 a1);
/* REAL_64.abs */
extern T13 T13f6(GE_context* ac, T13* C);
/* REAL_64.abs_ref */
extern T13 T13f10(GE_context* ac, T13* C);
/* REAL_64.zero */
extern T13 T13f14(GE_context* ac, T13* C);
/* REAL_64.default_create */
extern T13 T13c18(GE_context* ac);
/* [detachable] ST_FIXED_POINT_FORMATTER.format_to */
extern void T383f32(GE_context* ac, T0* C, T0* a1, T0* a2);
extern T1 T383f32ot1(T0* a1, T0** a2);
/* [detachable] ST_FIXED_POINT_FORMATTER.double_format_to */
extern void T383f35(GE_context* ac, T0* C, T13 a1, T0* a2);
/* [detachable] ST_FIXED_POINT_FORMATTER.center_format_to */
extern void T383f40(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_FIXED_POINT_FORMATTER.right_format_to */
extern void T383f39(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_FIXED_POINT_FORMATTER.left_format_to */
extern void T383f38(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_FIXED_POINT_FORMATTER.string_ */
extern T0* T383f17(GE_context* ac, T0* C);
/* [detachable] ST_FIXED_POINT_FORMATTER.build_integer_and_fractional_parts */
extern void T383f37(GE_context* ac, T0* C, T13 a1);
/* [detachable] ST_FIXED_POINT_FORMATTER.append_integer_to_string */
extern void T383f41(GE_context* ac, T0* C, T6 a1, T6 a2, T0* a3);
/* [detachable] ST_FIXED_POINT_FORMATTER.double_ */
extern T0* T383f22(GE_context* ac, T0* C);
/* [detachable] ST_FIXED_POINT_FORMATTER.double_sign */
extern T6 T383f21(GE_context* ac, T0* C, T13 a1);
/* [detachable] ST_FLOATING_POINT_FORMATTER.format_to */
extern void T382f33(GE_context* ac, T0* C, T0* a1, T0* a2);
extern T1 T382f33ot1(T0* a1, T0** a2);
/* [detachable] ST_FLOATING_POINT_FORMATTER.double_format_to */
extern void T382f36(GE_context* ac, T0* C, T13 a1, T0* a2);
/* [detachable] ST_FLOATING_POINT_FORMATTER.justify_format_to */
extern void T382f39(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_FLOATING_POINT_FORMATTER.center_format_to */
extern void T382f43(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_FLOATING_POINT_FORMATTER.right_format_to */
extern void T382f42(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_FLOATING_POINT_FORMATTER.left_format_to */
extern void T382f41(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_FLOATING_POINT_FORMATTER.string_ */
extern T0* T382f21(GE_context* ac, T0* C);
/* [detachable] ST_FLOATING_POINT_FORMATTER.build_integer_and_fractional_parts */
extern void T382f38(GE_context* ac, T0* C, T13 a1);
/* [detachable] ST_FLOATING_POINT_FORMATTER.append_integer_to_string */
extern void T382f40(GE_context* ac, T0* C, T6 a1, T6 a2, T0* a3);
/* [detachable] ST_FLOATING_POINT_FORMATTER.ten_power */
extern T13 T382f20(GE_context* ac, T0* C, T6 a1);
/* [detachable] ST_FLOATING_POINT_FORMATTER.double_ */
extern T0* T382f19(GE_context* ac, T0* C);
/* [detachable] ST_FLOATING_POINT_FORMATTER.double_sign */
extern T6 T382f18(GE_context* ac, T0* C, T13 a1);
/* [detachable] ST_CHARACTER_FORMATTER.format_to */
extern void T381f21(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_CHARACTER_FORMATTER.character_format_to */
extern void T381f24(GE_context* ac, T0* C, T2 a1, T0* a2);
/* [detachable] ST_CHARACTER_FORMATTER.justify_format_to */
extern void T381f26(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_CHARACTER_FORMATTER.center_format_to */
extern void T381f29(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_CHARACTER_FORMATTER.right_format_to */
extern void T381f28(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_CHARACTER_FORMATTER.left_format_to */
extern void T381f27(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_BOOLEAN_FORMATTER.format_to */
extern void T380f24(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_BOOLEAN_FORMATTER.boolean_format_to */
extern void T380f27(GE_context* ac, T0* C, T1 a1, T0* a2);
/* [detachable] ST_BOOLEAN_FORMATTER.justify_format_to */
extern void T380f29(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_BOOLEAN_FORMATTER.center_format_to */
extern void T380f32(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_BOOLEAN_FORMATTER.right_format_to */
extern void T380f31(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_BOOLEAN_FORMATTER.left_format_to */
extern void T380f30(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] ST_STRING_FORMATTER.reset_options */
extern void T390f21(GE_context* ac, T0* C);
/* [detachable] ST_STRING_FORMATTER.reset_options */
extern void T390f21p1(GE_context* ac, T0* C);
/* [detachable] ST_STRING_FORMATTER.set_align_right */
extern void T390f24(GE_context* ac, T0* C);
/* [detachable] ST_POINTER_FORMATTER.reset_options */
extern void T389f21(GE_context* ac, T0* C);
/* [detachable] ST_POINTER_FORMATTER.set_align_right */
extern void T389f24(GE_context* ac, T0* C);
/* [detachable] ST_UNSIGNED_DECIMAL_FORMATTER.reset_options */
extern void T388f28(GE_context* ac, T0* C);
/* [detachable] ST_UNSIGNED_DECIMAL_FORMATTER.reset_options */
extern void T388f28p1(GE_context* ac, T0* C);
/* [detachable] ST_UNSIGNED_DECIMAL_FORMATTER.set_align_right */
extern void T388f31(GE_context* ac, T0* C);
/* [detachable] ST_UNSIGNED_HEXADECIMAL_FORMATTER.reset_options */
extern void T387f28(GE_context* ac, T0* C);
/* [detachable] ST_UNSIGNED_HEXADECIMAL_FORMATTER.reset_options */
extern void T387f28p1(GE_context* ac, T0* C);
/* [detachable] ST_UNSIGNED_HEXADECIMAL_FORMATTER.set_align_right */
extern void T387f31(GE_context* ac, T0* C);
/* [detachable] ST_UNSIGNED_OCTAL_FORMATTER.reset_options */
extern void T386f28(GE_context* ac, T0* C);
/* [detachable] ST_UNSIGNED_OCTAL_FORMATTER.reset_options */
extern void T386f28p1(GE_context* ac, T0* C);
/* [detachable] ST_UNSIGNED_OCTAL_FORMATTER.set_align_right */
extern void T386f31(GE_context* ac, T0* C);
/* [detachable] ST_DECIMAL_FORMATTER.reset_options */
extern void T385f28(GE_context* ac, T0* C);
/* [detachable] ST_DECIMAL_FORMATTER.reset_options */
extern void T385f28p1(GE_context* ac, T0* C);
/* [detachable] ST_FIXED_OR_FLOATING_POINT_FORMATTER.reset_options */
extern void T384f34(GE_context* ac, T0* C);
/* [detachable] ST_FIXED_OR_FLOATING_POINT_FORMATTER.reset_options */
extern void T384f34p1(GE_context* ac, T0* C);
/* [detachable] ST_FIXED_OR_FLOATING_POINT_FORMATTER.set_align_right */
extern void T384f37(GE_context* ac, T0* C);
/* [detachable] ST_FIXED_POINT_FORMATTER.reset_options */
extern void T383f33(GE_context* ac, T0* C);
/* [detachable] ST_FIXED_POINT_FORMATTER.reset_options */
extern void T383f33p1(GE_context* ac, T0* C);
/* [detachable] ST_FIXED_POINT_FORMATTER.set_align_right */
extern void T383f36(GE_context* ac, T0* C);
/* [detachable] ST_FLOATING_POINT_FORMATTER.reset_options */
extern void T382f34(GE_context* ac, T0* C);
/* [detachable] ST_FLOATING_POINT_FORMATTER.reset_options */
extern void T382f34p1(GE_context* ac, T0* C);
/* [detachable] ST_FLOATING_POINT_FORMATTER.set_align_right */
extern void T382f37(GE_context* ac, T0* C);
/* [detachable] ST_CHARACTER_FORMATTER.reset_options */
extern void T381f22(GE_context* ac, T0* C);
/* [detachable] ST_CHARACTER_FORMATTER.set_align_right */
extern void T381f25(GE_context* ac, T0* C);
/* [detachable] ST_BOOLEAN_FORMATTER.reset_options */
extern void T380f25(GE_context* ac, T0* C);
/* [detachable] ST_BOOLEAN_FORMATTER.set_align_right */
extern void T380f28(GE_context* ac, T0* C);
/* [detachable] ST_STRING_FORMATTER.set_lowercase */
extern void T390f22(GE_context* ac, T0* C, T1 a1);
/* [detachable] ST_POINTER_FORMATTER.set_lowercase */
extern void T389f22(GE_context* ac, T0* C, T1 a1);
/* [detachable] ST_UNSIGNED_DECIMAL_FORMATTER.set_lowercase */
extern void T388f29(GE_context* ac, T0* C, T1 a1);
/* [detachable] ST_UNSIGNED_HEXADECIMAL_FORMATTER.set_lowercase */
extern void T387f29(GE_context* ac, T0* C, T1 a1);
/* [detachable] ST_UNSIGNED_OCTAL_FORMATTER.set_lowercase */
extern void T386f29(GE_context* ac, T0* C, T1 a1);
/* [detachable] ST_DECIMAL_FORMATTER.set_lowercase */
extern void T385f29(GE_context* ac, T0* C, T1 a1);
/* [detachable] ST_FIXED_OR_FLOATING_POINT_FORMATTER.set_lowercase */
extern void T384f35(GE_context* ac, T0* C, T1 a1);
/* [detachable] ST_FIXED_POINT_FORMATTER.set_lowercase */
extern void T383f34(GE_context* ac, T0* C, T1 a1);
/* [detachable] ST_FLOATING_POINT_FORMATTER.set_lowercase */
extern void T382f35(GE_context* ac, T0* C, T1 a1);
/* [detachable] ST_CHARACTER_FORMATTER.set_lowercase */
extern void T381f23(GE_context* ac, T0* C, T1 a1);
/* [detachable] ST_BOOLEAN_FORMATTER.set_lowercase */
extern void T380f26(GE_context* ac, T0* C, T1 a1);
/* [detachable] XPLAIN_INIT_ATTRIBUTES_CURSOR [[attached] XPLAIN_ATTRIBUTE].set */
extern void T430f14(GE_context* ac, T0* C, T0* a1, T1 a2, T1 a3);
/* [detachable] XPLAIN_DATA_ATTRIBUTES_CURSOR.set */
extern void T408f15(GE_context* ac, T0* C, T0* a1, T1 a2, T1 a3);
/* [detachable] XPLAIN_ALL_ATTRIBUTES_CURSOR [[attached] XPLAIN_ATTRIBUTE].set */
extern void T407f15(GE_context* ac, T0* C, T0* a1, T1 a2, T1 a3);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_ATTRIBUTE].set */
extern void T362f13(GE_context* ac, T0* C, T0* a1, T1 a2, T1 a3);
/* [detachable] XPLAIN_INIT_ATTRIBUTES_CURSOR [[attached] XPLAIN_ATTRIBUTE].set_current_cell */
extern void T430f17(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_DATA_ATTRIBUTES_CURSOR.set_current_cell */
extern void T408f18(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_ALL_ATTRIBUTES_CURSOR [[attached] XPLAIN_ATTRIBUTE].set_current_cell */
extern void T407f18(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_ATTRIBUTE].set_current_cell */
extern void T362f16(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_INIT_ATTRIBUTES_CURSOR [[attached] XPLAIN_ATTRIBUTE].set_after */
extern void T430f16(GE_context* ac, T0* C);
/* [detachable] XPLAIN_DATA_ATTRIBUTES_CURSOR.set_after */
extern void T408f17(GE_context* ac, T0* C);
/* [detachable] XPLAIN_ALL_ATTRIBUTES_CURSOR [[attached] XPLAIN_ATTRIBUTE].set_after */
extern void T407f17(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_ATTRIBUTE].set_after */
extern void T362f15(GE_context* ac, T0* C);
/* [detachable] XPLAIN_INIT_ATTRIBUTES_CURSOR [[attached] XPLAIN_ATTRIBUTE].set_next_cursor */
extern void T430f15(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_DATA_ATTRIBUTES_CURSOR.set_next_cursor */
extern void T408f16(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_ALL_ATTRIBUTES_CURSOR [[attached] XPLAIN_ATTRIBUTE].set_next_cursor */
extern void T407f16(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_ATTRIBUTE].set_next_cursor */
extern void T362f14(GE_context* ac, T0* C, T0* a1);
/* [detachable] XML_GENERATOR.write_value */
extern void T69f27(GE_context* ac, T0* C, T0* a1);
/* [detachable] XML_GENERATOR.write_type */
extern void T69f22(GE_context* ac, T0* C, T0* a1);
/* [detachable] XML_GENERATOR.dump_columns */
extern void T69f37(GE_context* ac, T0* C, T0* a1);
extern T1 T69f37ot1(T0* a1, T0** a2);
extern T1 T69f37ot2(T0* a1, T0** a2);
extern T1 T69f37ot3(T0* a1, T0** a2);
extern T1 T69f37ot4(T0* a1, T0** a2);
/* [detachable] XPLAIN_DATA_ATTRIBUTES_CURSOR.forth */
extern void T408f13(GE_context* ac, T0* C);
/* [detachable] XPLAIN_DATA_ATTRIBUTES_CURSOR.is_not_virtual_attribute */
extern T1 T408f7(GE_context* ac, T0* C);
/* [detachable] XPLAIN_DATA_ATTRIBUTES_CURSOR.forth */
extern void T408f13p1(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE].cursor_forth */
extern void T361f23(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE].add_traversing_cursor */
extern void T361f24(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE].remove_traversing_cursor */
extern void T361f25(GE_context* ac, T0* C, T0* a1);
/* [detachable] XML_WRITER.add_tag */
extern void T347f37(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XML_WRITER.add_data */
extern void T347f39(GE_context* ac, T0* C, T0* a1);
/* [detachable] XML_WRITER.replace_content_meta_characters */
extern void T347f42(GE_context* ac, T0* C, T0* a1);
/* [detachable] STRING_8.insert_string */
extern void T17f59(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] XPLAIN_DATA_ATTRIBUTES_CURSOR.item */
extern T0* T408f8(GE_context* ac, T0* C);
/* [detachable] XPLAIN_DATA_ATTRIBUTES_CURSOR.start */
extern void T408f12(GE_context* ac, T0* C);
/* [detachable] XPLAIN_DATA_ATTRIBUTES_CURSOR.start */
extern void T408f12p1(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE].cursor_start */
extern void T361f22(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE].cursor_off */
extern T1 T361f12(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_EXTENSION.new_data_attributes_cursor */
extern T0* T160f21(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_DATA_ATTRIBUTES_CURSOR.make */
extern T0* T408c11(GE_context* ac, T0* a1, T0* a2);
/* [detachable] XPLAIN_DATA_ATTRIBUTES_CURSOR.ds_make */
extern void T408f14(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_TYPE.new_data_attributes_cursor */
extern T0* T132f6(GE_context* ac, T0* C, T0* a1);
/* [detachable] XML_WRITER.set_attribute */
extern void T347f36(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XML_WRITER.make_valid_attribute_value */
extern T0* T347f13(GE_context* ac, T0* C, T0* a1);
/* [detachable] XML_WRITER.do_set_attribute */
extern void T347f38(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XML_WRITER.add_attribute */
extern void T347f41(GE_context* ac, T0* C, T0* a1);
/* [detachable] ARRAY [detachable STRING_8].put */
extern void T416f8(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] ARRAY [[attached] STRING_8].put */
extern void T38f10(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] ARRAY [detachable STRING_8].conservative_resize_with_default */
extern void T416f9(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable STRING_8].move_data */
extern void T329f13(GE_context* ac, T0* C, T6 a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable STRING_8].overlapping_move */
extern void T329f15(GE_context* ac, T0* C, T6 a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [detachable STRING_8].force */
extern void T329f16(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [detachable STRING_8].non_overlapping_move */
extern void T329f14(GE_context* ac, T0* C, T6 a1, T6 a2, T6 a3);
/* [detachable] ARRAY [detachable STRING_8].set_area */
extern void T416f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] SPECIAL [detachable STRING_8].aliased_resized_area_with_default */
extern T0* T329f5(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] ARRAY [detachable STRING_8].empty_area */
extern T1 T416f5(GE_context* ac, T0* C);
/* [detachable] ARRAY [[attached] STRING_8].conservative_resize_with_default */
extern void T38f9(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] STRING_8].move_data */
extern void T39f13(GE_context* ac, T0* C, T6 a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] STRING_8].overlapping_move */
extern void T39f15(GE_context* ac, T0* C, T6 a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] STRING_8].force */
extern void T39f16(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SPECIAL [[attached] STRING_8].non_overlapping_move */
extern void T39f14(GE_context* ac, T0* C, T6 a1, T6 a2, T6 a3);
/* [detachable] ARRAY [[attached] STRING_8].set_area */
extern void T38f11(GE_context* ac, T0* C, T0* a1);
/* [detachable] SPECIAL [[attached] STRING_8].aliased_resized_area_with_default */
extern T0* T39f5(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] ARRAY [[attached] STRING_8].empty_area */
extern T1 T38f6(GE_context* ac, T0* C);
/* [detachable] ARRAY [[attached] STRING_8].count */
extern T6 T38f5(GE_context* ac, T0* C);
/* [detachable] XML_WRITER.set_value */
extern void T347f40(GE_context* ac, T0* C, T0* a1);
/* [detachable] XML_WRITER.exist_attribute */
extern T1 T347f14(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_STRING_ROUTINES.same_string */
extern T1 T110f6(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] STRING_8.item_code */
extern T6 T17f29(GE_context* ac, T0* C, T6 a1);
/* [detachable] KL_STRING_ROUTINES.elks_same_string */
extern T1 T110f7(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XML_WRITER.string_ */
extern T0* T347f22(GE_context* ac, T0* C);
/* [detachable] XML_GENERATOR.set_sp_name */
extern void T69f36(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XML_GENERATOR.as_c_string */
extern T0* T69f8(GE_context* ac, T0* C, T0* a1);
/* [detachable] XML_GENERATOR.as_some_string */
extern T0* T69f11(GE_context* ac, T0* C, T0* a1, T2 a2);
/* [detachable] STRING_8.index_of */
extern T6 T17f28(GE_context* ac, T0* C, T2 a1, T6 a2);
/* [detachable] XML_GENERATOR.as_eiffel_string */
extern T0* T69f7(GE_context* ac, T0* C, T0* a1);
/* [detachable] XML_GENERATOR.set_names */
extern void T69f35(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5, T0* a6);
/* [detachable] XML_GENERATOR.as_ncname */
extern T0* T69f5(GE_context* ac, T0* C, T0* a1);
/* [detachable] UC_STRING.as_string */
extern T0* T392f10(GE_context* ac, T0* C);
/* [detachable] UC_STRING.to_utf8 */
extern T0* T392f8(GE_context* ac, T0* C);
/* [detachable] UC_STRING.byte_item */
extern T2 T392f13(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_STRING.old_item */
extern T2 T392f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_STRING.set_count */
extern void T392f21(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_STRING.reset_byte_index_cache */
extern void T392f29(GE_context* ac, T0* C);
/* [detachable] UC_STRING.insert_character */
extern void T392f20(GE_context* ac, T0* C, T2 a1, T6 a2);
/* [detachable] UC_STRING.put_character_at_byte_index */
extern void T392f27(GE_context* ac, T0* C, T2 a1, T6 a2, T6 a3);
/* [detachable] UC_STRING.put_code_at_byte_index */
extern void T392f34(GE_context* ac, T0* C, T6 a1, T6 a2, T6 a3);
/* [detachable] KL_INTEGER_ROUTINES.to_character */
extern T2 T432f5(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_STRING.integer_ */
extern T0* T392f17(GE_context* ac, T0* C);
/* [detachable] UC_STRING.put_byte */
extern void T392f31(GE_context* ac, T0* C, T2 a1, T6 a2);
/* [detachable] UC_STRING.old_put */
extern void T392f32(GE_context* ac, T0* C, T2 a1, T6 a2);
/* [detachable] UC_STRING.reset_hash_codes */
extern void T392f35(GE_context* ac, T0* C);
/* [detachable] UC_STRING.move_bytes_right */
extern void T392f26(GE_context* ac, T0* C, T6 a1, T6 a2);
/* [detachable] UC_STRING.resize_byte_storage */
extern void T392f25(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_STRING.old_set_count */
extern void T392f30(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_STRING.resize */
extern void T392f33(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_STRING.byte_capacity */
extern T6 T392f14(GE_context* ac, T0* C);
/* [detachable] UC_STRING.capacity */
extern T6 T392f14p1(GE_context* ac, T0* C);
/* [detachable] UC_UTF8_ROUTINES.character_byte_count */
extern T6 T473f4(GE_context* ac, T0* C, T2 a1);
/* [detachable] UC_UTF8_ROUTINES.character_8_byte_count */
extern T6 T473f9(GE_context* ac, T0* C, T2 a1);
/* [detachable] UC_STRING.utf8 */
extern T0* T392f12(GE_context* ac, T0* C);
/* [detachable] UC_UTF8_ROUTINES.default_create */
extern T0* T473c10(GE_context* ac);
/* [detachable] UC_STRING.byte_index */
extern T6 T392f15(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_UTF8_ROUTINES.encoded_byte_count */
extern T6 T473f1(GE_context* ac, T0* C, T2 a1);
/* [detachable] UC_STRING.append_character */
extern void T392f28(GE_context* ac, T0* C, T2 a1);
/* [detachable] XML_GENERATOR.is_name_first */
extern T1 T69f10(GE_context* ac, T0* C, T6 a1);
/* [detachable] XML_GENERATOR.is_letter */
extern T1 T69f15(GE_context* ac, T0* C, T6 a1);
/* [detachable] XML_GENERATOR.is_ideographic */
extern T1 T69f17(GE_context* ac, T0* C, T6 a1);
/* [detachable] XML_GENERATOR.is_base_char */
extern T1 T69f16(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_STRING.put */
extern void T392f19(GE_context* ac, T0* C, T2 a1, T6 a2);
/* [detachable] UC_STRING.move_bytes_left */
extern void T392f24(GE_context* ac, T0* C, T6 a1, T6 a2);
/* [detachable] XML_GENERATOR.is_name_char */
extern T1 T69f9(GE_context* ac, T0* C, T6 a1);
/* [detachable] XML_GENERATOR.is_extender */
extern T1 T69f14(GE_context* ac, T0* C, T6 a1);
/* [detachable] XML_GENERATOR.is_combining_char */
extern T1 T69f13(GE_context* ac, T0* C, T6 a1);
/* [detachable] XML_GENERATOR.is_digit */
extern T1 T69f12(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_STRING.item_code */
extern T6 T392f11(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_STRING.item_code_at_byte_index */
extern T6 T392f16(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_UTF8_ROUTINES.encoded_next_value */
extern T6 T473f3(GE_context* ac, T0* C, T2 a1);
/* [detachable] UC_UTF8_ROUTINES.encoded_first_value */
extern T6 T473f2(GE_context* ac, T0* C, T2 a1);
/* [detachable] UC_STRING.make_from_utf8 */
extern T0* T392c18(GE_context* ac, T0* a1);
/* [detachable] UC_STRING.append_utf8 */
extern void T392f23(GE_context* ac, T0* C, T0* a1);
/* [detachable] UC_STRING.make */
extern void T392f22(GE_context* ac, T0* C, T6 a1);
/* [detachable] UC_STRING.make */
extern void T392f22p1(GE_context* ac, T0* C, T6 a1);
/* [detachable] XML_GENERATOR.code_common_identifier */
extern T0* T69f6(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_TYPE.sqltablename */
extern T0* T132f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] XML_GENERATOR.write_insert */
extern void T69f23(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* detachable XPLAIN_EXTENSION.sqltablename */
extern T0* T160f17(GE_context* ac, T0* C, T0* a1);
/* [detachable] XML_GENERATOR.write_get_insert */
extern void T69f24(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_TSQL65.write_get_insert */
extern void T66f344(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_TSQL65.create_get_insert */
extern void T66f345(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] XPLAIN_PK_I_REPRESENTATION.write_with_quotes */
extern T1 T257f13(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PK_A_REPRESENTATION.write_with_quotes */
extern T1 T254f14(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.do_create_select_list */
extern void T66f346(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_TSQL65.commandseparator */
extern T0* T66f43(GE_context* ac, T0* C);
/* [detachable] XPLAIN_ATTRIBUTE_NAME.q_sql_select_name */
extern T0* T141f11(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.do_do_create_select_list */
extern void T66f347(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_TSQL65.create_predicate */
extern void T66f309(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL65.sql_predicate */
extern T0* T66f99(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL65.sql_select_joins */
extern T0* T66f61(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.sql_select_joins_oldstyle */
extern T0* T66f148(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.equal */
extern T1 T66f187(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] JOIN_LIST.first */
extern T0* T352f10(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.sql_select_joins_ansi92 */
extern T0* T66f147(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.string_ */
extern T0* T66f186(GE_context* ac, T0* C);
/* [detachable] JOIN.is_inner_join */
extern T1 T423f10(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.ansi92joinssupported */
extern T1 T66f146(GE_context* ac, T0* C);
/* detachable XPLAIN_SELECTION_LIST.show_only_identifier_column */
extern T1 T164f7(GE_context* ac, T0* C);
extern T1 T164f7ot1(T0* a1, T0** a2);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_ATTRIBUTE].forth */
extern void T362f12(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.sql_cast_to_iso_8601_date */
extern T0* T66f159(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_ATTRIBUTE].item */
extern T0* T362f6(GE_context* ac, T0* C);
/* [detachable] XPLAIN_DATA_ATTRIBUTES_CURSOR.new_iterator */
extern T0* T408f9(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE].new_iterator */
extern T0* T361f13(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_ATTRIBUTE].start */
extern void T362f11(GE_context* ac, T0* C);
/* detachable XPLAIN_TYPE.has_a_date_attribute */
extern T1 T132f26(GE_context* ac, T0* C);
extern T1 T132f26ot1(T0* a1);
/* detachable XPLAIN_EXTENSION.has_a_date_attribute */
extern T1 T160f24(GE_context* ac, T0* C);
extern T1 T160f24ot1(T0* a1);
/* [detachable] JOIN_LIST.finalize */
extern void T352f15(GE_context* ac, T0* C, T0* a1);
/* [detachable] JOIN_TREE_NODE.generate_join_nodes */
extern T0* T438f11(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T1 a4);
/* [detachable] DS_LINKED_LIST [[attached] JOIN_TREE_NODE].forth */
extern void T439f15(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] JOIN_TREE_NODE].cursor_forth */
extern void T439f19(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] JOIN_TREE_NODE].add_traversing_cursor */
extern void T439f20(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] JOIN_TREE_NODE].set_next_cursor */
extern void T465f9(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] JOIN_TREE_NODE].remove_traversing_cursor */
extern void T439f21(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] JOIN_TREE_NODE].set */
extern void T465f8(GE_context* ac, T0* C, T0* a1, T1 a2, T1 a3);
/* detachable JOIN_NODE.set_next */
extern void T353f5(GE_context* ac, T0* C, T0* a1);
/* detachable JOIN_NODE.last */
extern T0* T353f3(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] JOIN_TREE_NODE].item_for_iteration */
extern T0* T439f9(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] JOIN_TREE_NODE].cursor_item */
extern T0* T439f8(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] JOIN_TREE_NODE].after */
extern T1 T439f6(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] JOIN_TREE_NODE].cursor_after */
extern T1 T439f7(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] JOIN_TREE_NODE].start */
extern void T439f14(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] JOIN_TREE_NODE].cursor_start */
extern void T439f18(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] JOIN_TREE_NODE].cursor_off */
extern T1 T439f12(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] JOIN_TREE_NODE].off */
extern T1 T465f6(GE_context* ac, T0* C);
/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE].wipe_out */
extern void T462f33(GE_context* ac, T0* C);
/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE].slots_wipe_out */
extern void T462f43(GE_context* ac, T0* C);
/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE].clashes_wipe_out */
extern void T462f42(GE_context* ac, T0* C);
/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE].key_storage_wipe_out */
extern void T462f41(GE_context* ac, T0* C);
/* [detachable] SPECIAL [[attached] XPLAIN_TYPE].keep_head */
extern void T479f5(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_TYPE].set_count */
extern void T479f8(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE].item_storage_wipe_out */
extern void T462f40(GE_context* ac, T0* C);
/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE].unset_found_item */
extern void T462f39(GE_context* ac, T0* C);
/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE].move_all_cursors_after */
extern void T462f38(GE_context* ac, T0* C);
/* detachable DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] XPLAIN_TYPE].set_next_cursor */
extern void T478f8(GE_context* ac, T0* C, T0* a1);
/* detachable DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] XPLAIN_TYPE].set_after */
extern void T478f7(GE_context* ac, T0* C);
/* [detachable] JOIN_TREE_NODE.table_counter */
extern T0* T438f12(GE_context* ac, T0* C);
/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE].make */
extern T0* T462c32(GE_context* ac, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE].make_with_equality_testers */
extern void T462f37(GE_context* ac, T0* C, T6 a1, T0* a2, T0* a3);
/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE].make_sparse_container */
extern void T462f49(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE].set_internal_cursor */
extern void T462f54(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE].new_cursor */
extern T0* T462f30(GE_context* ac, T0* C);
/* detachable DS_HASH_TABLE_CURSOR [[attached] INTEGER_32, [attached] XPLAIN_TYPE].make */
extern T0* T478c6(GE_context* ac, T0* a1);
/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE].make_slots */
extern void T462f53(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE].new_modulus */
extern T6 T462f29(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE].make_clashes */
extern void T462f52(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE].make_key_storage */
extern void T462f51(GE_context* ac, T0* C, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_TYPE].make */
extern T0* T480f1(GE_context* ac, T0* C, T6 a1);
/* [detachable] SPECIAL [[attached] XPLAIN_TYPE].make_empty */
extern T0* T479c4(GE_context* ac, T6 a1);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_TYPE].default_create */
extern T0* T480c2(GE_context* ac);
/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE].make_item_storage */
extern void T462f50(GE_context* ac, T0* C, T6 a1);
/* detachable KL_EQUALITY_TESTER [[attached] XPLAIN_TYPE].default_create */
extern T0* T476c2(GE_context* ac);
/* [detachable] JOIN_TREE_NODE.set_prefix_table */
extern void T438f18(GE_context* ac, T0* C, T0* a1);
/* detachable DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE_NAME_NODE].forth */
extern void T463f14(GE_context* ac, T0* C);
/* detachable DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE_NAME_NODE].cursor_forth */
extern void T463f17(GE_context* ac, T0* C, T0* a1);
/* detachable DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE_NAME_NODE].add_traversing_cursor */
extern void T463f18(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_ATTRIBUTE_NAME_NODE].set_next_cursor */
extern void T481f9(GE_context* ac, T0* C, T0* a1);
/* detachable DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE_NAME_NODE].remove_traversing_cursor */
extern void T463f19(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_ATTRIBUTE_NAME_NODE].set */
extern void T481f8(GE_context* ac, T0* C, T0* a1, T1 a2, T1 a3);
/* detachable DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE_NAME_NODE].item_for_iteration */
extern T0* T463f7(GE_context* ac, T0* C);
/* detachable DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE_NAME_NODE].cursor_item */
extern T0* T463f5(GE_context* ac, T0* C, T0* a1);
/* detachable DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE_NAME_NODE].after */
extern T1 T463f8(GE_context* ac, T0* C);
/* detachable DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE_NAME_NODE].cursor_after */
extern T1 T463f9(GE_context* ac, T0* C, T0* a1);
/* detachable DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE_NAME_NODE].start */
extern void T463f13(GE_context* ac, T0* C);
/* detachable DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE_NAME_NODE].cursor_start */
extern void T463f16(GE_context* ac, T0* C, T0* a1);
/* detachable DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE_NAME_NODE].cursor_off */
extern T1 T463f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_ATTRIBUTE_NAME_NODE].off */
extern T1 T481f6(GE_context* ac, T0* C);
/* [detachable] JOIN_TREE_NODE.increment_usage */
extern void T438f17(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE].put */
extern void T462f36(GE_context* ac, T0* C, T6 a1, T0* a2);
/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE].key_storage_put */
extern void T462f48(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] KL_SPECIAL_ROUTINES [[attached] XPLAIN_TYPE].force */
extern void T480f3(GE_context* ac, T0* C, T0* a1, T0* a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_TYPE].fill_with */
extern void T479f7(GE_context* ac, T0* C, T0* a1, T6 a2, T6 a3);
/* [detachable] SPECIAL [[attached] XPLAIN_TYPE].extend */
extern void T479f9(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE].slots_put */
extern void T462f47(GE_context* ac, T0* C, T6 a1, T6 a2);
/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE].clashes_put */
extern void T462f46(GE_context* ac, T0* C, T6 a1, T6 a2);
/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE].slots_item */
extern T6 T462f22(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE].clashes_item */
extern T6 T462f20(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE].item_storage_put */
extern void T462f45(GE_context* ac, T0* C, T6 a1, T6 a2);
/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE].search_position */
extern void T462f44(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE].hash_position */
extern T6 T462f25(GE_context* ac, T0* C, T0* a1);
/* detachable KL_EQUALITY_TESTER [[attached] XPLAIN_TYPE].test */
extern T1 T476f1(GE_context* ac, T0* C, T0* a1, T0* a2);
extern T1 T476f1oe1(GE_context* ac, T0* a1, T0* a2);
/* detachable XPLAIN_ASSERTION.is_equal */
extern T1 T146f31(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_EXTENSION.is_equal */
extern T1 T160f31(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_TYPE.is_equal */
extern T1 T132f30(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE].key_storage_item */
extern T0* T462f24(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE].replace */
extern void T462f35(GE_context* ac, T0* C, T6 a1, T0* a2);
/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE].found_item */
extern T6 T462f27(GE_context* ac, T0* C);
/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE].item_storage_item */
extern T6 T462f26(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE].found */
extern T1 T462f28(GE_context* ac, T0* C);
/* [detachable] DS_HASH_TABLE [[attached] INTEGER_32, [attached] XPLAIN_TYPE].search */
extern void T462f34(GE_context* ac, T0* C, T0* a1);
/* detachable JOIN_NODE.make */
extern T0* T353c4(GE_context* ac, T0* a1, T0* a2);
/* [detachable] JOIN.make */
extern T0* T423c11(GE_context* ac, T0* a1, T0* a2, T0* a3, T0* a4, T0* a5, T0* a6, T1 a7, T1 a8, T1 a9);
/* [detachable] XPLAIN_ATTRIBUTE_NAME.sqlcolumnidentifier */
extern T0* T141f9(GE_context* ac, T0* C, T0* a1);
/* [detachable] JOIN_TREE_NODE.alias_name */
extern T0* T438f8(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_ATTRIBUTE_NAME.type */
extern T0* T141f10(GE_context* ac, T0* C);
extern T1 T141f10ot1(T0* a1, T0** a2);
/* [detachable] JOIN_TREE_NODE.has_children */
extern T1 T438f9(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] JOIN_TREE_NODE].is_empty */
extern T1 T439f11(GE_context* ac, T0* C);
/* detachable XPLAIN_SELECTION_LIST.add_to_join */
extern void T164f13(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] JOIN_LIST.extend */
extern void T352f16(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] JOIN_TREE_NODE.get_immediate_child */
extern T0* T438f10(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] JOIN_TREE_NODE.append_child */
extern void T438f16(GE_context* ac, T0* C, T0* a1, T1 a2, T1 a3, T1 a4);
/* [detachable] DS_LINKED_LIST [[attached] JOIN_TREE_NODE].put_last */
extern void T439f16(GE_context* ac, T0* C, T0* a1);
/* detachable DS_LINKABLE [[attached] JOIN_TREE_NODE].put_right */
extern void T464f4(GE_context* ac, T0* C, T0* a1);
/* detachable DS_LINKABLE [[attached] JOIN_TREE_NODE].make */
extern T0* T464c3(GE_context* ac, T0* a1);
/* [detachable] JOIN_TREE_NODE.make */
extern T0* T438c14(GE_context* ac, T0* a1, T1 a2, T1 a3, T1 a4);
/* [detachable] DS_LINKED_LIST [[attached] JOIN_TREE_NODE].make */
extern T0* T439c13(GE_context* ac);
/* [detachable] DS_LINKED_LIST [[attached] JOIN_TREE_NODE].set_internal_cursor */
extern void T439f17(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] JOIN_TREE_NODE].new_cursor */
extern T0* T439f10(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] JOIN_TREE_NODE].make */
extern T0* T465c7(GE_context* ac, T0* a1);
/* [detachable] JOIN_LIST.existential_join_optimisation */
extern T1 T352f11(GE_context* ac, T0* C);
/* [detachable] JOIN_TREE_NODE.add_additional_item */
extern void T438f15(GE_context* ac, T0* C, T0* a1);
/* detachable DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE_NAME_NODE].put_last */
extern void T463f12(GE_context* ac, T0* C, T0* a1);
/* detachable DS_LINKABLE [[attached] XPLAIN_ATTRIBUTE_NAME_NODE].put_right */
extern void T482f4(GE_context* ac, T0* C, T0* a1);
/* detachable DS_LINKABLE [[attached] XPLAIN_ATTRIBUTE_NAME_NODE].make */
extern T0* T482c3(GE_context* ac, T0* a1);
/* detachable DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE_NAME_NODE].make */
extern T0* T463c11(GE_context* ac);
/* detachable DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE_NAME_NODE].set_internal_cursor */
extern void T463f15(GE_context* ac, T0* C, T0* a1);
/* detachable DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE_NAME_NODE].new_cursor */
extern T0* T463f6(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_ATTRIBUTE_NAME_NODE].make */
extern T0* T481c7(GE_context* ac, T0* a1);
/* [detachable] JOIN_TREE_NODE.has_immediate_child */
extern T1 T438f13(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] XPLAIN_ATTRIBUTE_NAME.is_equal */
extern T1 T141f12(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_ATTRIBUTE_NAME.string_ */
extern T0* T141f13(GE_context* ac, T0* C);
/* detachable XPLAIN_SELECTION_LIST.add_to_join */
extern void T164f13p1(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] JOIN_LIST.disable_existential_join_optimisation */
extern void T352f19(GE_context* ac, T0* C);
/* [detachable] JOIN_LIST.enable_existential_join_optimisation */
extern void T352f18(GE_context* ac, T0* C);
/* [detachable] JOIN_LIST.make */
extern T0* T352c14(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_ATTRIBUTE_NAME.quoted_name */
extern T0* T141f8(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.std */
extern T0* T66f23(GE_context* ac, T0* C);
/* detachable XPLAIN_SELECTION_LIST.type */
extern T0* T164f6(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.write_get_insert */
extern void T65f342(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_TSQL70.create_get_insert */
extern void T65f343(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_TSQL70.do_create_select_list */
extern void T65f344(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_TSQL70.commandseparator */
extern T0* T65f43(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.do_do_create_select_list */
extern void T65f345(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_TSQL70.create_predicate */
extern void T65f308(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL70.sql_predicate */
extern T0* T65f99(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL70.sql_select_joins */
extern T0* T65f61(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.sql_select_joins_oldstyle */
extern T0* T65f147(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.equal */
extern T1 T65f186(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL70.sql_select_joins_ansi92 */
extern T0* T65f146(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.string_ */
extern T0* T65f185(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.ansi92joinssupported */
extern T1 T65f145(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.sql_cast_to_iso_8601_date */
extern T0* T65f158(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.std */
extern T0* T65f23(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.write_get_insert */
extern void T64f342(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_TSQL2000.create_get_insert */
extern void T64f343(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_TSQL2000.do_create_select_list */
extern void T64f344(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_TSQL2000.commandseparator */
extern T0* T64f43(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.do_do_create_select_list */
extern void T64f345(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_TSQL2000.create_predicate */
extern void T64f308(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL2000.sql_predicate */
extern T0* T64f99(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL2000.sql_select_joins */
extern T0* T64f61(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.sql_select_joins_oldstyle */
extern T0* T64f147(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.equal */
extern T1 T64f186(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL2000.sql_select_joins_ansi92 */
extern T0* T64f146(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.string_ */
extern T0* T64f185(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.ansi92joinssupported */
extern T1 T64f145(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.sql_cast_to_iso_8601_date */
extern T0* T64f158(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.std */
extern T0* T64f23(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.write_get_insert */
extern void T63f338(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_TSQL2016.create_get_insert */
extern void T63f339(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_TSQL2016.do_create_select_list */
extern void T63f340(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_TSQL2016.commandseparator */
extern T0* T63f43(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.do_do_create_select_list */
extern void T63f341(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_TSQL2016.create_predicate */
extern void T63f308(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL2016.sql_predicate */
extern T0* T63f99(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL2016.sql_select_joins */
extern T0* T63f61(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.sql_select_joins_oldstyle */
extern T0* T63f147(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.equal */
extern T1 T63f186(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL2016.sql_select_joins_ansi92 */
extern T0* T63f146(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.string_ */
extern T0* T63f185(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.ansi92joinssupported */
extern T1 T63f145(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.sql_cast_to_iso_8601_date */
extern T0* T63f158(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.std */
extern T0* T63f23(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.write_get_insert */
extern void T62f316(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_get_insert */
extern void T62f317(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_SQLITE_3.do_create_select_list */
extern void T62f318(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_SQLITE_3.commandseparator */
extern T0* T62f44(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.do_do_create_select_list */
extern void T62f319(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_predicate */
extern void T62f299(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_SQLITE_3.sql_predicate */
extern T0* T62f76(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_SQLITE_3.sql_select_joins */
extern T0* T62f56(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.sql_select_joins_oldstyle */
extern T0* T62f135(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.equal */
extern T1 T62f168(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_SQLITE_3.sql_select_joins_ansi92 */
extern T0* T62f134(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.string_ */
extern T0* T62f167(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.ansi92joinssupported */
extern T1 T62f133(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.sql_cast_to_iso_8601_date */
extern T0* T62f145(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.std */
extern T0* T62f23(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.write_get_insert */
extern void T61f363(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_get_insert */
extern void T61f364(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_PGSQL_72.do_create_select_list */
extern void T61f365(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_PGSQL_72.do_do_create_select_list */
extern void T61f366(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_predicate */
extern void T61f327(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_72.sql_predicate */
extern T0* T61f97(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_72.sql_select_joins */
extern T0* T61f57(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.sql_select_joins_oldstyle */
extern T0* T61f160(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.equal */
extern T1 T61f201(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_72.sql_select_joins_ansi92 */
extern T0* T61f159(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.string_ */
extern T0* T61f200(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.ansi92joinssupported */
extern T1 T61f158(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.sql_cast_to_iso_8601_date */
extern T0* T61f179(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.write_get_insert */
extern void T60f368(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_get_insert */
extern void T60f369(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_PGSQL_73.do_create_select_list */
extern void T60f370(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_PGSQL_73.do_do_create_select_list */
extern void T60f371(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_predicate */
extern void T60f330(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_73.sql_predicate */
extern T0* T60f97(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_73.sql_select_joins */
extern T0* T60f57(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.sql_select_joins_oldstyle */
extern T0* T60f163(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.equal */
extern T1 T60f202(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_73.sql_select_joins_ansi92 */
extern T0* T60f162(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.string_ */
extern T0* T60f201(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.ansi92joinssupported */
extern T1 T60f161(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.sql_cast_to_iso_8601_date */
extern T0* T60f180(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.write_get_insert */
extern void T59f369(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_get_insert */
extern void T59f370(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_PGSQL_81.do_create_select_list */
extern void T59f371(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_PGSQL_81.do_do_create_select_list */
extern void T59f372(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_predicate */
extern void T59f331(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_81.sql_predicate */
extern T0* T59f97(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_81.sql_select_joins */
extern T0* T59f57(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.sql_select_joins_oldstyle */
extern T0* T59f164(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.equal */
extern T1 T59f203(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_81.sql_select_joins_ansi92 */
extern T0* T59f163(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.string_ */
extern T0* T59f202(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.ansi92joinssupported */
extern T1 T59f162(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.sql_cast_to_iso_8601_date */
extern T0* T59f180(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.write_get_insert */
extern void T58f369(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_get_insert */
extern void T58f370(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_PGSQL_82.do_create_select_list */
extern void T58f371(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_PGSQL_82.do_do_create_select_list */
extern void T58f372(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_predicate */
extern void T58f331(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_82.sql_predicate */
extern T0* T58f97(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_82.sql_select_joins */
extern T0* T58f57(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.sql_select_joins_oldstyle */
extern T0* T58f164(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.equal */
extern T1 T58f203(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_82.sql_select_joins_ansi92 */
extern T0* T58f163(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.string_ */
extern T0* T58f202(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.ansi92joinssupported */
extern T1 T58f162(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.sql_cast_to_iso_8601_date */
extern T0* T58f180(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.write_get_insert */
extern void T57f367(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_get_insert */
extern void T57f368(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_PGSQL_95.do_create_select_list */
extern void T57f369(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_PGSQL_95.do_do_create_select_list */
extern void T57f370(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_predicate */
extern void T57f331(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_95.sql_predicate */
extern T0* T57f96(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_95.sql_select_joins */
extern T0* T57f57(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.sql_select_joins_oldstyle */
extern T0* T57f164(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.equal */
extern T1 T57f203(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_95.sql_select_joins_ansi92 */
extern T0* T57f163(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.string_ */
extern T0* T57f202(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.ansi92joinssupported */
extern T1 T57f162(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.sql_cast_to_iso_8601_date */
extern T0* T57f178(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.write_get_insert */
extern void T56f357(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_ORACLE.create_get_insert */
extern void T56f358(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_ORACLE.do_create_select_list */
extern void T56f359(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_ORACLE.commandseparator */
extern T0* T56f46(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.do_do_create_select_list */
extern void T56f360(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_ORACLE.create_predicate */
extern void T56f326(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_ORACLE.sql_predicate */
extern T0* T56f99(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_ORACLE.sql_select_joins */
extern T0* T56f58(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.sql_select_joins_oldstyle */
extern T0* T56f158(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.equal */
extern T1 T56f199(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_ORACLE.sql_select_joins_ansi92 */
extern T0* T56f157(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.string_ */
extern T0* T56f198(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.ansi92joinssupported */
extern T1 T56f156(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.sql_cast_to_iso_8601_date */
extern T0* T56f171(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.std */
extern T0* T56f167(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.write_get_insert */
extern void T55f303(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_MYSQL322.create_get_insert */
extern void T55f304(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_MYSQL322.do_create_select_list */
extern void T55f305(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_MYSQL322.commandseparator */
extern T0* T55f44(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.do_do_create_select_list */
extern void T55f306(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_MYSQL322.create_predicate */
extern void T55f286(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL322.sql_predicate */
extern T0* T55f75(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL322.sql_select_joins */
extern T0* T55f56(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.sql_select_joins_oldstyle */
extern T0* T55f134(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.equal */
extern T1 T55f166(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL322.sql_select_joins_ansi92 */
extern T0* T55f133(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.string_ */
extern T0* T55f165(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.ansi92joinssupported */
extern T1 T55f132(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.sql_cast_to_iso_8601_date */
extern T0* T55f143(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.std */
extern T0* T55f23(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.write_get_insert */
extern void T54f303(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_MYSQL4.create_get_insert */
extern void T54f304(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_MYSQL4.do_create_select_list */
extern void T54f305(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_MYSQL4.commandseparator */
extern T0* T54f44(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.do_do_create_select_list */
extern void T54f306(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_MYSQL4.create_predicate */
extern void T54f286(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL4.sql_predicate */
extern T0* T54f75(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL4.sql_select_joins */
extern T0* T54f56(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.sql_select_joins_oldstyle */
extern T0* T54f134(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.equal */
extern T1 T54f166(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL4.sql_select_joins_ansi92 */
extern T0* T54f133(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.string_ */
extern T0* T54f165(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.ansi92joinssupported */
extern T1 T54f132(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.sql_cast_to_iso_8601_date */
extern T0* T54f143(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.std */
extern T0* T54f23(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.write_get_insert */
extern void T53f346(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_MYSQL5.create_get_insert */
extern void T53f347(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_MYSQL5.do_create_select_list */
extern void T53f348(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_MYSQL5.commandseparator */
extern T0* T53f43(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.do_do_create_select_list */
extern void T53f349(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_MYSQL5.create_predicate */
extern void T53f317(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL5.sql_predicate */
extern T0* T53f96(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL5.sql_select_joins */
extern T0* T53f59(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.sql_select_joins_oldstyle */
extern T0* T53f150(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.equal */
extern T1 T53f189(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL5.sql_select_joins_ansi92 */
extern T0* T53f149(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.string_ */
extern T0* T53f188(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.ansi92joinssupported */
extern T1 T53f148(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.sql_cast_to_iso_8601_date */
extern T0* T53f163(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.std */
extern T0* T53f23(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.write_get_insert */
extern void T52f302(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_MSQL.create_get_insert */
extern void T52f303(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_MSQL.do_create_select_list */
extern void T52f304(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_MSQL.commandseparator */
extern T0* T52f44(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.do_do_create_select_list */
extern void T52f305(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_MSQL.create_predicate */
extern void T52f285(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSQL.sql_predicate */
extern T0* T52f75(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSQL.sql_select_joins */
extern T0* T52f56(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.sql_select_joins_oldstyle */
extern T0* T52f134(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.equal */
extern T1 T52f165(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSQL.sql_select_joins_ansi92 */
extern T0* T52f133(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.string_ */
extern T0* T52f164(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.ansi92joinssupported */
extern T1 T52f132(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.sql_cast_to_iso_8601_date */
extern T0* T52f143(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.std */
extern T0* T52f23(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.write_get_insert */
extern void T51f302(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_MSACCESS2000.create_get_insert */
extern void T51f303(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_MSACCESS2000.do_create_select_list */
extern void T51f304(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_MSACCESS2000.commandseparator */
extern T0* T51f44(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.do_do_create_select_list */
extern void T51f305(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_MSACCESS2000.create_predicate */
extern void T51f285(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSACCESS2000.sql_predicate */
extern T0* T51f75(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSACCESS2000.sql_select_joins */
extern T0* T51f56(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.sql_select_joins_oldstyle */
extern T0* T51f134(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.equal */
extern T1 T51f165(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSACCESS2000.sql_select_joins_ansi92 */
extern T0* T51f133(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.string_ */
extern T0* T51f164(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.ansi92joinssupported */
extern T1 T51f132(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.sql_cast_to_iso_8601_date */
extern T0* T51f143(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.std */
extern T0* T51f23(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.write_get_insert */
extern void T50f302(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_MSACCESS97.create_get_insert */
extern void T50f303(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_MSACCESS97.do_create_select_list */
extern void T50f304(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_MSACCESS97.commandseparator */
extern T0* T50f44(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.do_do_create_select_list */
extern void T50f305(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_MSACCESS97.create_predicate */
extern void T50f285(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSACCESS97.sql_predicate */
extern T0* T50f75(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSACCESS97.sql_select_joins */
extern T0* T50f56(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.sql_select_joins_oldstyle */
extern T0* T50f134(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.equal */
extern T1 T50f165(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSACCESS97.sql_select_joins_ansi92 */
extern T0* T50f133(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.string_ */
extern T0* T50f164(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.ansi92joinssupported */
extern T1 T50f132(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.sql_cast_to_iso_8601_date */
extern T0* T50f143(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.std */
extern T0* T50f23(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.write_get_insert */
extern void T49f355(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_get_insert */
extern void T49f356(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.do_create_select_list */
extern void T49f357(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.commandseparator */
extern T0* T49f44(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.do_do_create_select_list */
extern void T49f358(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_predicate */
extern void T49f321(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.sql_predicate */
extern T0* T49f99(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.sql_select_joins */
extern T0* T49f59(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.sql_select_joins_oldstyle */
extern T0* T49f157(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.equal */
extern T1 T49f195(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.sql_select_joins_ansi92 */
extern T0* T49f156(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.string_ */
extern T0* T49f194(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.ansi92joinssupported */
extern T1 T49f155(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.sql_cast_to_iso_8601_date */
extern T0* T49f170(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.std */
extern T0* T49f166(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.write_get_insert */
extern void T48f354(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_get_insert */
extern void T48f355(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_INTERBASE6.do_create_select_list */
extern void T48f356(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_INTERBASE6.commandseparator */
extern T0* T48f44(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.do_do_create_select_list */
extern void T48f357(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_predicate */
extern void T48f321(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_INTERBASE6.sql_predicate */
extern T0* T48f99(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_INTERBASE6.sql_select_joins */
extern T0* T48f59(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.sql_select_joins_oldstyle */
extern T0* T48f157(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.equal */
extern T1 T48f195(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_INTERBASE6.sql_select_joins_ansi92 */
extern T0* T48f156(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.string_ */
extern T0* T48f194(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.ansi92joinssupported */
extern T1 T48f155(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.sql_cast_to_iso_8601_date */
extern T0* T48f170(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.std */
extern T0* T48f166(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.write_get_insert */
extern void T47f302(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_FOXPRO.create_get_insert */
extern void T47f303(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_FOXPRO.do_create_select_list */
extern void T47f304(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_FOXPRO.commandseparator */
extern T0* T47f44(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.do_do_create_select_list */
extern void T47f305(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_FOXPRO.create_predicate */
extern void T47f285(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_FOXPRO.sql_predicate */
extern T0* T47f75(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_FOXPRO.sql_select_joins */
extern T0* T47f56(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.sql_select_joins_oldstyle */
extern T0* T47f134(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.equal */
extern T1 T47f165(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_FOXPRO.sql_select_joins_ansi92 */
extern T0* T47f133(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.string_ */
extern T0* T47f164(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.ansi92joinssupported */
extern T1 T47f132(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.sql_cast_to_iso_8601_date */
extern T0* T47f143(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.std */
extern T0* T47f23(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.write_get_insert */
extern void T46f353(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_DB2_71.create_get_insert */
extern void T46f354(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_DB2_71.do_create_select_list */
extern void T46f355(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_DB2_71.commandseparator */
extern T0* T46f43(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.do_do_create_select_list */
extern void T46f356(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_DB2_71.create_predicate */
extern void T46f322(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_DB2_71.sql_predicate */
extern T0* T46f95(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_DB2_71.sql_select_joins */
extern T0* T46f58(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.sql_select_joins_oldstyle */
extern T0* T46f157(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.equal */
extern T1 T46f195(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_DB2_71.sql_select_joins_ansi92 */
extern T0* T46f156(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.string_ */
extern T0* T46f194(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.ansi92joinssupported */
extern T1 T46f155(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.sql_cast_to_iso_8601_date */
extern T0* T46f169(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.std */
extern T0* T46f167(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.write_get_insert */
extern void T45f304(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_DB2_6.create_get_insert */
extern void T45f305(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_DB2_6.do_create_select_list */
extern void T45f306(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_DB2_6.commandseparator */
extern T0* T45f41(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.do_do_create_select_list */
extern void T45f307(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_DB2_6.create_predicate */
extern void T45f287(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_DB2_6.sql_predicate */
extern T0* T45f75(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_DB2_6.sql_select_joins */
extern T0* T45f56(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.sql_select_joins_oldstyle */
extern T0* T45f135(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.equal */
extern T1 T45f167(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_DB2_6.sql_select_joins_ansi92 */
extern T0* T45f134(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.string_ */
extern T0* T45f166(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.ansi92joinssupported */
extern T1 T45f133(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.sql_cast_to_iso_8601_date */
extern T0* T45f144(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.std */
extern T0* T45f23(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.write_get_insert */
extern void T44f303(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_BASIC.create_get_insert */
extern void T44f304(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_BASIC.do_create_select_list */
extern void T44f305(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_BASIC.commandseparator */
extern T0* T44f44(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.do_do_create_select_list */
extern void T44f306(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_BASIC.create_predicate */
extern void T44f286(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_BASIC.sql_predicate */
extern T0* T44f75(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_BASIC.sql_select_joins */
extern T0* T44f56(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.sql_select_joins_oldstyle */
extern T0* T44f134(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.equal */
extern T1 T44f166(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_BASIC.sql_select_joins_ansi92 */
extern T0* T44f133(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.string_ */
extern T0* T44f165(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.ansi92joinssupported */
extern T1 T44f132(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.sql_cast_to_iso_8601_date */
extern T0* T44f143(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.std */
extern T0* T44f23(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.write_get_insert */
extern void T43f303(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_ANSI.create_get_insert */
extern void T43f304(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3, T0* a4);
/* [detachable] SQL_GENERATOR_ANSI.do_create_select_list */
extern void T43f305(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_ANSI.commandseparator */
extern T0* T43f44(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.do_do_create_select_list */
extern void T43f306(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_ANSI.create_predicate */
extern void T43f286(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_ANSI.sql_predicate */
extern T0* T43f75(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_ANSI.sql_select_joins */
extern T0* T43f56(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.sql_select_joins_oldstyle */
extern T0* T43f134(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.equal */
extern T1 T43f166(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_ANSI.sql_select_joins_ansi92 */
extern T0* T43f133(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.string_ */
extern T0* T43f165(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.ansi92joinssupported */
extern T1 T43f132(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.sql_cast_to_iso_8601_date */
extern T0* T43f143(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.std */
extern T0* T43f23(GE_context* ac, T0* C);
/* [detachable] XML_GENERATOR.write_constant_assignment */
extern void T69f25(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XML_GENERATOR.write_constant */
extern void T69f26(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_VARIABLE.sqlname */
extern T0* T185f4(GE_context* ac, T0* C, T0* a1);
/* [detachable] XML_GENERATOR.write_delete */
extern void T69f28(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XML_GENERATOR.write_update */
extern void T69f29(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] XML_GENERATOR.write_select_value */
extern void T69f30(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_VALUE.sqlname */
extern T0* T264f4(GE_context* ac, T0* C, T0* a1);
/* [detachable] XML_GENERATOR.write_select_function */
extern void T69f39(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_SELECTION_FUNCTION.type */
extern T0* T157f7(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.write_select_function */
extern void T66f348(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.create_select_function */
extern void T66f351(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.write_select_function */
extern void T65f346(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.create_select_function */
extern void T65f349(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.write_select_function */
extern void T64f346(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.create_select_function */
extern void T64f349(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.write_select_function */
extern void T63f342(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.create_select_function */
extern void T63f345(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.write_select_function */
extern void T62f320(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_select_function */
extern void T62f323(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.write_select_function */
extern void T61f367(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_select_function */
extern void T61f370(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.sp_return_cursor */
extern void T61f355(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_select_function */
extern void T61f370p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.sp_start_cursor */
extern void T61f354(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.write_select_function */
extern void T60f372(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_select_function */
extern void T60f375(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.sp_return_cursor */
extern void T60f358(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_select_function */
extern void T60f375p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.sp_start_cursor */
extern void T60f357(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.write_select_function */
extern void T59f373(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_select_function */
extern void T59f376(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.sp_return_cursor */
extern void T59f359(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_select_function */
extern void T59f376p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.sp_start_cursor */
extern void T59f358(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.write_select_function */
extern void T58f373(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_select_function */
extern void T58f376(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.sp_return_cursor */
extern void T58f359(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_select_function */
extern void T58f376p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.sp_start_cursor */
extern void T58f358(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.write_select_function */
extern void T57f371(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_select_function */
extern void T57f374(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.sp_return_cursor */
extern void T57f377(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_select_function */
extern void T57f374p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.sp_start_cursor */
extern void T57f376(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.write_select_function */
extern void T56f361(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.create_select_function */
extern void T56f364(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.create_select_function */
extern void T56f364p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.write_select_function */
extern void T55f307(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.create_select_function */
extern void T55f310(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.write_select_function */
extern void T54f307(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.create_select_function */
extern void T54f310(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.write_select_function */
extern void T53f350(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.create_select_function */
extern void T53f353(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.write_select_function */
extern void T52f306(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.create_select_function */
extern void T52f309(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.write_select_function */
extern void T51f306(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.create_select_function */
extern void T51f309(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.write_select_function */
extern void T50f306(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.create_select_function */
extern void T50f309(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.write_select_function */
extern void T49f359(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_select_function */
extern void T49f362(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_select_function */
extern void T49f362p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.write_select_function */
extern void T48f358(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_select_function */
extern void T48f361(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_select_function */
extern void T48f361p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.write_select_function */
extern void T47f306(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.create_select_function */
extern void T47f309(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.write_select_function */
extern void T46f357(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.create_select_function */
extern void T46f360(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.create_select_function */
extern void T46f360p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.write_select_function */
extern void T45f308(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.create_select_function */
extern void T45f311(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.write_select_function */
extern void T44f307(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.create_select_function */
extern void T44f310(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.write_select_function */
extern void T43f307(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.create_select_function */
extern void T43f310(GE_context* ac, T0* C, T0* a1);
/* [detachable] XML_GENERATOR.write_select */
extern void T69f31(GE_context* ac, T0* C, T0* a1);
/* [detachable] XML_GENERATOR.write_extend */
extern void T69f32(GE_context* ac, T0* C, T0* a1);
/* [detachable] XML_GENERATOR.write_select_list */
extern void T69f40(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.write_select_list */
extern void T66f349(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.create_select_list */
extern void T66f352(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.write_select_list */
extern void T65f347(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.create_select_list */
extern void T65f350(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.write_select_list */
extern void T64f347(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.create_select_list */
extern void T64f350(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.write_select_list */
extern void T63f343(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.create_select_list */
extern void T63f346(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.write_select_list */
extern void T62f321(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_select_list */
extern void T62f324(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.write_select_list */
extern void T61f368(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_select_list */
extern void T61f371(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_select_list */
extern void T61f371p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.write_select_list */
extern void T60f373(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_select_list */
extern void T60f376(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_select_list */
extern void T60f376p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.write_select_list */
extern void T59f374(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_select_list */
extern void T59f377(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_select_list */
extern void T59f377p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.write_select_list */
extern void T58f374(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_select_list */
extern void T58f377(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_select_list */
extern void T58f377p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.write_select_list */
extern void T57f372(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_select_list */
extern void T57f375(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_select_list */
extern void T57f375p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.write_select_list */
extern void T56f362(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.create_select_list */
extern void T56f365(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.create_select_list */
extern void T56f365p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.write_select_list */
extern void T55f308(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.create_select_list */
extern void T55f311(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.write_select_list */
extern void T54f308(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.create_select_list */
extern void T54f311(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.write_select_list */
extern void T53f351(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.create_select_list */
extern void T53f354(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.write_select_list */
extern void T52f307(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.create_select_list */
extern void T52f310(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.write_select_list */
extern void T51f307(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.create_select_list */
extern void T51f310(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.write_select_list */
extern void T50f307(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.create_select_list */
extern void T50f310(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.write_select_list */
extern void T49f360(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_select_list */
extern void T49f363(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_select_list */
extern void T49f363p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.write_select_list */
extern void T48f359(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_select_list */
extern void T48f362(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_select_list */
extern void T48f362p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.write_select_list */
extern void T47f307(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.create_select_list */
extern void T47f310(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.write_select_list */
extern void T46f358(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.create_select_list */
extern void T46f361(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.create_select_list */
extern void T46f361p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.write_select_list */
extern void T45f309(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.create_select_list */
extern void T45f312(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.write_select_list */
extern void T44f308(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.create_select_list */
extern void T44f311(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.write_select_list */
extern void T43f308(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.create_select_list */
extern void T43f311(GE_context* ac, T0* C, T0* a1);
/* [detachable] XML_GENERATOR.write_procedure */
extern void T69f33(GE_context* ac, T0* C, T0* a1);
extern T1 T69f33ot1(T0* a1, T0** a2);
extern T1 T69f33ot2(T0* a1, T0** a2);
/* detachable XPLAIN_SELECTION_LIST.write_select */
extern void T164f12(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_SELECTION_FUNCTION.write_select */
extern void T157f12(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PROCEDURE.last_get_statement */
extern T0* T333f16(GE_context* ac, T0* C);
extern T1 T333f16ot1(T0* a1, T0** a2);
/* [detachable] DS_BILINKED_LIST_CURSOR [[attached] XPLAIN_STATEMENT].back */
extern void T414f10(GE_context* ac, T0* C);
/* [detachable] DS_BILINKED_LIST [[attached] XPLAIN_STATEMENT].cursor_back */
extern void T360f23(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_BILINKED_LIST_CURSOR [[attached] XPLAIN_STATEMENT].finish */
extern void T414f9(GE_context* ac, T0* C);
/* [detachable] DS_BILINKED_LIST [[attached] XPLAIN_STATEMENT].cursor_finish */
extern void T360f22(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_BILINKED_LIST_CURSOR [[attached] XPLAIN_STATEMENT].forth */
extern void T414f12(GE_context* ac, T0* C);
/* [detachable] DS_BILINKED_LIST_CURSOR [[attached] XPLAIN_STATEMENT].item */
extern T0* T414f6(GE_context* ac, T0* C);
/* [detachable] DS_BILINKED_LIST [[attached] XPLAIN_STATEMENT].new_iterator */
extern T0* T360f12(GE_context* ac, T0* C);
/* [detachable] DS_BILINKED_LIST_CURSOR [[attached] XPLAIN_STATEMENT].start */
extern void T414f11(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_PARAMETER].forth */
extern void T359f10(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_PARAMETER].cursor_forth */
extern void T358f17(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_PARAMETER].add_traversing_cursor */
extern void T358f18(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_PARAMETER].set_next_cursor */
extern void T359f12(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_PARAMETER].remove_traversing_cursor */
extern void T358f19(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_PARAMETER].set */
extern void T359f11(GE_context* ac, T0* C, T0* a1, T1 a2, T1 a3);
/* [detachable] XPLAIN_PARAMETER.full_name */
extern T0* T170f6(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PARAMETER.abstracttype */
extern T0* T170f7(GE_context* ac, T0* C);
extern T1 T170f7ot1(T0* a1, T0** a2);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_PARAMETER].item */
extern T0* T359f6(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_PARAMETER].start */
extern void T359f9(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_PARAMETER].cursor_start */
extern void T358f16(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_PARAMETER].cursor_off */
extern T1 T358f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_PARAMETER].off */
extern T1 T359f7(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PROCEDURE.last_value_selection_statement */
extern T0* T333f9(GE_context* ac, T0* C);
extern T1 T333f9ot1(T0* a1, T0** a2);
/* [detachable] XPLAIN_PROCEDURE.sqlname */
extern T0* T333f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] XML_GENERATOR.write_if */
extern void T69f34(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.write_if */
extern void T66f350(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.write_if */
extern void T65f348(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.write_if */
extern void T64f348(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.write_if */
extern void T63f344(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.write_if */
extern void T62f322(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.write_if */
extern void T61f369(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.write_if */
extern void T60f374(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.write_if */
extern void T59f375(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.write_if */
extern void T58f375(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.write_if */
extern void T57f373(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_BILINKED_LIST [[attached] XPLAIN_STATEMENT].is_empty */
extern T1 T360f11(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.sp_body_statements */
extern void T57f348(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.write_if */
extern void T56f363(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.write_if */
extern void T55f309(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.write_if */
extern void T54f309(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.write_if */
extern void T53f352(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.write_if */
extern void T52f308(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.write_if */
extern void T51f308(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.write_if */
extern void T50f308(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.write_if */
extern void T49f361(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.write_if */
extern void T48f360(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.write_if */
extern void T47f308(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.write_if */
extern void T46f359(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.write_if */
extern void T45f310(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.write_if */
extern void T44f309(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.write_if */
extern void T43f309(GE_context* ac, T0* C, T0* a1);
/* [detachable] XML_GENERATOR.write_sql */
extern void T69f38(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_EXTENSION.hack_anode */
extern void T160f34(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_ATTRIBUTE_NAME_NODE.set_next */
extern void T140f6(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_ASSERTION.hack_anode */
extern void T146f34(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_TYPE.hack_anode */
extern void T132f34(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_BASE.hack_anode */
extern void T131f13(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PROCEDURE.write_drop */
extern void T333f24(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_VALUE.write_drop */
extern void T264f10(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_VARIABLE.write_drop */
extern void T185f10(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_TYPE.write_drop */
extern void T132f35(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_BASE.write_drop */
extern void T131f12(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_EXTENSION_ATTRIBUTE.set_init_default */
extern void T469f20(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_VIRTUAL_ATTRIBUTE.set_init_default */
extern void T468f20(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_ATTRIBUTE.set_init_default */
extern void T145f21(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_EXTENSION_ATTRIBUTE.set_init */
extern void T469f21(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_VIRTUAL_ATTRIBUTE.set_init */
extern void T468f21(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_ATTRIBUTE.set_init */
extern void T145f22(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_EXTENSION_ATTRIBUTE.write_drop */
extern void T469f19(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_VIRTUAL_ATTRIBUTE.write_drop */
extern void T468f19(GE_context* ac, T0* C, T0* a1, T0* a2);
/* detachable XPLAIN_ASSERTION.write_drop */
extern void T146f35(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_ATTRIBUTE.write_drop */
extern void T145f23(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_EXTENSION_ATTRIBUTE.set_required */
extern void T469f22(GE_context* ac, T0* C, T1 a1);
/* [detachable] XPLAIN_VIRTUAL_ATTRIBUTE.set_required */
extern void T468f22(GE_context* ac, T0* C, T1 a1);
/* [detachable] XPLAIN_ATTRIBUTE.set_required */
extern void T145f24(GE_context* ac, T0* C, T1 a1);
/* [detachable] XPLAIN_T_REPRESENTATION.set_domain_restriction */
extern void T253f13(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_R_REPRESENTATION.set_domain_restriction */
extern void T252f14(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_P_REPRESENTATION.set_domain_restriction */
extern void T251f13(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_M_REPRESENTATION.set_domain_restriction */
extern void T250f12(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_I_REPRESENTATION.set_domain_restriction */
extern void T249f13(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_D_REPRESENTATION.set_domain_restriction */
extern void T248f12(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_C_REPRESENTATION.set_domain_restriction */
extern void T247f14(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_B_REPRESENTATION.set_domain_restriction */
extern void T246f13(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_A_REPRESENTATION.set_domain_restriction */
extern void T204f14(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_ASSERTION_EXPRESSION.add_to_join */
extern void T471f29(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_ATTRIBUTE_EXPRESSION.add_to_join */
extern void T410f29(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_NOT_LIKE_EXPRESSION.add_to_join */
extern void T369f31(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_LIKE_EXPRESSION.add_to_join */
extern void T368f31(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_LAST_AUTO_PK_EXPRESSION.add_to_join */
extern void T297f28(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_SELECTION_EXPRESSION.add_to_join */
extern void T295f28(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_VALUE_EXPRESSION.add_to_join */
extern void T265f28(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_VARIABLE_EXPRESSION.add_to_join */
extern void T263f28(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_SQL_EXPRESSION.add_to_join */
extern void T240f28(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_USER_FUNCTION.add_to_join */
extern void T239f29(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_UNMANAGED_PARAMETER_EXPRESSION.add_to_join */
extern void T238f28(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_NUMBER_EXPRESSION.add_to_join */
extern void T237f28(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_ID_FUNCTION.add_to_join */
extern void T236f28(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_NEWDATE_FUNCTION.add_to_join */
extern void T235f30(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_STRING_EXPRESSION.add_to_join */
extern void T234f28(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_STRING_FUNCTION.add_to_join */
extern void T233f28(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_REAL_FUNCTION.add_to_join */
extern void T232f28(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_INTEGER_FUNCTION.add_to_join */
extern void T231f28(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_DATEF_FUNCTION.add_to_join */
extern void T230f28(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_COMBINE_FUNCTION.add_to_join */
extern void T229f28(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_PREFIX_EXPRESSION.add_to_join */
extern void T227f30(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_IF_EXPRESSION.add_to_join */
extern void T226f30(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_LOGINNAME_EXPRESSION.add_to_join */
extern void T225f27(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_SYSTEMDATE_EXPRESSION.add_to_join */
extern void T222f27(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_PARENTHESIS_EXPRESSION.add_to_join */
extern void T219f28(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_INFIX_EXPRESSION.add_to_join */
extern void T218f31(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_NE_NULL_EXPRESSION.add_to_join */
extern void T217f29(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_EQ_NULL_EXPRESSION.add_to_join */
extern void T216f29(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_PARAMETER_EXPRESSION.add_to_join */
extern void T215f28(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_NOT_EXPRESSION.add_to_join */
extern void T214f30(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_NOTNOT_EXPRESSION.add_to_join */
extern void T213f28(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_LOGICAL_INFIX_EXPRESSION.add_to_join */
extern void T210f32(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_LOGICAL_INFIX_EXPRESSION.add_to_join */
extern void T210f32p1(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_LOGICAL_INFIX_EXPRESSION.string_ */
extern T0* T210f21(GE_context* ac, T0* C);
/* detachable XPLAIN_EXTENSION_EXPRESSION.add_to_join */
extern void T162f29(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_LOGICAL_VALUE_EXPRESSION.add_to_join */
extern void T138f28(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_LOGICAL_EXPRESSION.add_to_join */
extern void T136f27(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_GET_INSERT_STATEMENT.write_value_declare_inside_sp */
extern void T281f9(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_INSERT_STATEMENT.write_value_declare_inside_sp */
extern void T279f8(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_BASE_STATEMENT.write_value_declare_inside_sp */
extern void T183f6(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_TYPE_STATEMENT.write_value_declare_inside_sp */
extern void T180f6(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_IF_STATEMENT.write_value_declare_inside_sp */
extern void T173f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_SQL_STATEMENT.write_value_declare_inside_sp */
extern void T171f6(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_PROCEDURE_STATEMENT.write_value_declare_inside_sp */
extern void T168f6(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_EXTEND_STATEMENT.write_value_declare_inside_sp */
extern void T159f6(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_GET_STATEMENT.write_value_declare_inside_sp */
extern void T158f6(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_VALUE_SELECTION_STATEMENT.write_value_declare_inside_sp */
extern void T155f6(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_VALUE_STATEMENT.write_value_declare_inside_sp */
extern void T154f7(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_UPDATE_STATEMENT.write_value_declare_inside_sp */
extern void T153f8(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_DELETE_STATEMENT.write_value_declare_inside_sp */
extern void T151f7(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_CASCADE_STATEMENT.write_value_declare_inside_sp */
extern void T149f16(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_CONSTANT_STATEMENT.write_value_declare_inside_sp */
extern void T133f6(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_CONSTANT_ASSIGNMENT_STATEMENT.write_value_declare_inside_sp */
extern void T130f7(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_GET_INSERT_STATEMENT.optimize_for_procedure */
extern void T281f10(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_INSERT_STATEMENT.optimize_for_procedure */
extern void T279f9(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_BASE_STATEMENT.optimize_for_procedure */
extern void T183f7(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_TYPE_STATEMENT.optimize_for_procedure */
extern void T180f7(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_IF_STATEMENT.optimize_for_procedure */
extern void T173f11(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_SQL_STATEMENT.optimize_for_procedure */
extern void T171f7(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_PROCEDURE_STATEMENT.optimize_for_procedure */
extern void T168f7(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_EXTEND_STATEMENT.optimize_for_procedure */
extern void T159f7(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_EXTENSION.set_no_update_optimization */
extern void T160f35(GE_context* ac, T0* C, T1 a1);
/* [detachable] XPLAIN_EXTENSION_FUNCTION_EXPRESSION.is_update_optimization_supported */
extern T1 T315f20(GE_context* ac, T0* C);
/* detachable XPLAIN_GET_STATEMENT.optimize_for_procedure */
extern void T158f7(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_VALUE_SELECTION_STATEMENT.optimize_for_procedure */
extern void T155f7(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_VALUE_STATEMENT.optimize_for_procedure */
extern void T154f8(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_UPDATE_STATEMENT.optimize_for_procedure */
extern void T153f9(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_DELETE_STATEMENT.optimize_for_procedure */
extern void T151f8(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_CASCADE_STATEMENT.optimize_for_procedure */
extern void T149f17(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_CONSTANT_STATEMENT.optimize_for_procedure */
extern void T133f7(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_CONSTANT_ASSIGNMENT_STATEMENT.optimize_for_procedure */
extern void T130f8(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_GET_INSERT_STATEMENT.cleanup */
extern void T281f11(GE_context* ac, T0* C);
/* [detachable] XPLAIN_INSERT_STATEMENT.cleanup */
extern void T279f10(GE_context* ac, T0* C);
/* [detachable] XPLAIN_BASE_STATEMENT.cleanup */
extern void T183f8(GE_context* ac, T0* C);
/* [detachable] XPLAIN_TYPE_STATEMENT.cleanup */
extern void T180f8(GE_context* ac, T0* C);
/* detachable XPLAIN_IF_STATEMENT.cleanup */
extern void T173f12(GE_context* ac, T0* C);
/* [detachable] XPLAIN_SQL_STATEMENT.cleanup */
extern void T171f8(GE_context* ac, T0* C);
/* detachable XPLAIN_PROCEDURE_STATEMENT.cleanup */
extern void T168f8(GE_context* ac, T0* C);
/* detachable XPLAIN_EXTEND_STATEMENT.cleanup */
extern void T159f8(GE_context* ac, T0* C);
/* detachable XPLAIN_TYPE.remove_extension */
extern void T132f39(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE].remove_at */
extern void T361f30(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE].remove_at_cursor */
extern void T361f31(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE].move_all_cursors */
extern void T361f34(GE_context* ac, T0* C, T0* a1, T0* a2);
/* detachable DS_LINKABLE [[attached] XPLAIN_ATTRIBUTE].put */
extern void T442f6(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE].remove_last */
extern void T361f33(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE].move_last_cursors_after */
extern void T361f36(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE].wipe_out */
extern void T361f35(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_ATTRIBUTE].is_last */
extern T1 T362f9(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE].cursor_is_last */
extern T1 T361f16(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE].remove_first */
extern void T361f32(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST_CURSOR [[attached] XPLAIN_ATTRIBUTE].is_first */
extern T1 T362f8(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE].cursor_is_first */
extern T1 T361f15(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE].forth */
extern void T361f20(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE].item_for_iteration */
extern T0* T361f11(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE].cursor_item */
extern T0* T361f9(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE].after */
extern T1 T361f10(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE].cursor_after */
extern T1 T361f8(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE].start */
extern void T361f19(GE_context* ac, T0* C);
/* detachable XPLAIN_GET_STATEMENT.cleanup */
extern void T158f8(GE_context* ac, T0* C);
/* detachable XPLAIN_VALUE_SELECTION_STATEMENT.cleanup */
extern void T155f8(GE_context* ac, T0* C);
/* [detachable] XPLAIN_VALUE_STATEMENT.cleanup */
extern void T154f9(GE_context* ac, T0* C);
/* [detachable] XPLAIN_VALUE_STATEMENT.universe */
extern T0* T154f3(GE_context* ac, T0* C);
/* detachable XPLAIN_UPDATE_STATEMENT.cleanup */
extern void T153f10(GE_context* ac, T0* C);
/* detachable XPLAIN_DELETE_STATEMENT.cleanup */
extern void T151f9(GE_context* ac, T0* C);
/* detachable XPLAIN_CASCADE_STATEMENT.cleanup */
extern void T149f18(GE_context* ac, T0* C);
/* detachable XPLAIN_CONSTANT_STATEMENT.cleanup */
extern void T133f8(GE_context* ac, T0* C);
/* detachable XPLAIN_CONSTANT_ASSIGNMENT_STATEMENT.cleanup */
extern void T130f9(GE_context* ac, T0* C);
/* [detachable] XPLAIN_GET_INSERT_STATEMENT.write */
extern void T281f8(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_INSERT_STATEMENT.write */
extern void T279f7(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_BASE_STATEMENT.write */
extern void T183f5(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_TYPE_STATEMENT.write */
extern void T180f5(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_IF_STATEMENT.write */
extern void T173f9(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_SQL_STATEMENT.write */
extern void T171f5(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_PROCEDURE_STATEMENT.write */
extern void T168f5(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_EXTEND_STATEMENT.write */
extern void T159f5(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_GET_STATEMENT.write */
extern void T158f5(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_VALUE_SELECTION_STATEMENT.write */
extern void T155f5(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_VALUE_STATEMENT.write */
extern void T154f6(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_UPDATE_STATEMENT.write */
extern void T153f7(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_DELETE_STATEMENT.write */
extern void T151f6(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_CASCADE_STATEMENT.write */
extern void T149f15(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_CONSTANT_STATEMENT.write */
extern void T133f5(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_CONSTANT_ASSIGNMENT_STATEMENT.write */
extern void T130f6(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_B_RESTRICTION.check_attachment */
extern void T365f5(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_TRAJECTORY.check_attachment */
extern void T207f9(GE_context* ac, T0* C, T0* a1, T0* a2);
extern T1 T207f9ot1(T0* a1, T0** a2);
extern T1 T207f9ot2(T0* a1, T0** a2);
/* [detachable] XPLAIN_TRAJECTORY.std */
extern T0* T207f5(GE_context* ac, T0* C);
/* [detachable] XPLAIN_I_ENUMERATION.check_attachment */
extern void T202f14(GE_context* ac, T0* C, T0* a1, T0* a2);
extern T1 T202f14ot1(T0* a1, T0** a2);
extern T1 T202f14ot2(T0* a1, T0** a2);
/* [detachable] XPLAIN_I_ENUMERATION.double_cell */
extern T0* T202f8(GE_context* ac, T0* C, T13 a1);
/* [detachable] DS_CELL [[attached] REAL_64].make */
extern T0* T435c4(GE_context* ac, T13 a1);
/* [detachable] XPLAIN_I_ENUMERATION.format */
extern T0* T202f7(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_I_ENUMERATION.formatter */
extern T0* T202f9(GE_context* ac, T0* C);
/* [detachable] XPLAIN_I_ENUMERATION.scientific_formatter */
extern T0* T202f10(GE_context* ac, T0* C);
/* [detachable] XPLAIN_I_ENUMERATION.integer_cell */
extern T0* T202f6(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_CELL [[attached] INTEGER_32].make */
extern T0* T378c4(GE_context* ac, T6 a1);
/* INTEGER_32.make_from_reference */
extern T6 T6c34(GE_context* ac, T0* a1);
/* [detachable] XPLAIN_I_ENUMERATION.std */
extern T0* T202f5(GE_context* ac, T0* C);
/* [detachable] KL_INTEGER_ROUTINES.power */
extern T6 T432f1(GE_context* ac, T0* C, T6 a1, T6 a2);
/* [detachable] KL_INTEGER_ROUTINES.is_even */
extern T1 T432f4(GE_context* ac, T0* C, T6 a1);
/* [detachable] XPLAIN_I_ENUMERATION.integer_ */
extern T0* T202f4(GE_context* ac, T0* C);
/* [detachable] XPLAIN_A_ENUMERATION.check_attachment */
extern void T200f8(GE_context* ac, T0* C, T0* a1, T0* a2);
extern T1 T200f8ot1(T0* a1, T0** a2);
/* [detachable] XPLAIN_A_NODE.to_sqlcode */
extern T0* T134f3(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_A_ENUMERATION.std */
extern T0* T200f4(GE_context* ac, T0* C);
/* [detachable] XPLAIN_A_PATTERN.check_attachment */
extern void T198f7(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_REQUIRED.check_attachment */
extern void T123f5(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL65.set_options */
extern void T66f251(GE_context* ac, T0* C, T1 a1, T1 a2, T1 a3, T1 a4, T1 a5, T1 a6, T1 a7, T1 a8, T1 a9, T1 a10, T1 a11, T1 a12, T1 a13, T1 a14);
/* [detachable] SQL_GENERATOR_TSQL70.set_options */
extern void T65f250(GE_context* ac, T0* C, T1 a1, T1 a2, T1 a3, T1 a4, T1 a5, T1 a6, T1 a7, T1 a8, T1 a9, T1 a10, T1 a11, T1 a12, T1 a13, T1 a14);
/* [detachable] SQL_GENERATOR_TSQL2000.set_options */
extern void T64f250(GE_context* ac, T0* C, T1 a1, T1 a2, T1 a3, T1 a4, T1 a5, T1 a6, T1 a7, T1 a8, T1 a9, T1 a10, T1 a11, T1 a12, T1 a13, T1 a14);
/* [detachable] SQL_GENERATOR_TSQL2016.set_options */
extern void T63f250(GE_context* ac, T0* C, T1 a1, T1 a2, T1 a3, T1 a4, T1 a5, T1 a6, T1 a7, T1 a8, T1 a9, T1 a10, T1 a11, T1 a12, T1 a13, T1 a14);
/* [detachable] SQL_GENERATOR_SQLITE_3.set_options */
extern void T62f249(GE_context* ac, T0* C, T1 a1, T1 a2, T1 a3, T1 a4, T1 a5, T1 a6, T1 a7, T1 a8, T1 a9, T1 a10, T1 a11, T1 a12, T1 a13, T1 a14);
/* [detachable] SQL_GENERATOR_PGSQL_72.set_options */
extern void T61f273(GE_context* ac, T0* C, T1 a1, T1 a2, T1 a3, T1 a4, T1 a5, T1 a6, T1 a7, T1 a8, T1 a9, T1 a10, T1 a11, T1 a12, T1 a13, T1 a14);
/* [detachable] SQL_GENERATOR_PGSQL_73.set_options */
extern void T60f276(GE_context* ac, T0* C, T1 a1, T1 a2, T1 a3, T1 a4, T1 a5, T1 a6, T1 a7, T1 a8, T1 a9, T1 a10, T1 a11, T1 a12, T1 a13, T1 a14);
/* [detachable] SQL_GENERATOR_PGSQL_81.set_options */
extern void T59f277(GE_context* ac, T0* C, T1 a1, T1 a2, T1 a3, T1 a4, T1 a5, T1 a6, T1 a7, T1 a8, T1 a9, T1 a10, T1 a11, T1 a12, T1 a13, T1 a14);
/* [detachable] SQL_GENERATOR_PGSQL_82.set_options */
extern void T58f277(GE_context* ac, T0* C, T1 a1, T1 a2, T1 a3, T1 a4, T1 a5, T1 a6, T1 a7, T1 a8, T1 a9, T1 a10, T1 a11, T1 a12, T1 a13, T1 a14);
/* [detachable] SQL_GENERATOR_PGSQL_95.set_options */
extern void T57f277(GE_context* ac, T0* C, T1 a1, T1 a2, T1 a3, T1 a4, T1 a5, T1 a6, T1 a7, T1 a8, T1 a9, T1 a10, T1 a11, T1 a12, T1 a13, T1 a14);
/* [detachable] SQL_GENERATOR_ORACLE.set_options */
extern void T56f271(GE_context* ac, T0* C, T1 a1, T1 a2, T1 a3, T1 a4, T1 a5, T1 a6, T1 a7, T1 a8, T1 a9, T1 a10, T1 a11, T1 a12, T1 a13, T1 a14);
/* [detachable] SQL_GENERATOR_MYSQL322.set_options */
extern void T55f237(GE_context* ac, T0* C, T1 a1, T1 a2, T1 a3, T1 a4, T1 a5, T1 a6, T1 a7, T1 a8, T1 a9, T1 a10, T1 a11, T1 a12, T1 a13, T1 a14);
/* [detachable] SQL_GENERATOR_MYSQL4.set_options */
extern void T54f237(GE_context* ac, T0* C, T1 a1, T1 a2, T1 a3, T1 a4, T1 a5, T1 a6, T1 a7, T1 a8, T1 a9, T1 a10, T1 a11, T1 a12, T1 a13, T1 a14);
/* [detachable] SQL_GENERATOR_MYSQL5.set_options */
extern void T53f262(GE_context* ac, T0* C, T1 a1, T1 a2, T1 a3, T1 a4, T1 a5, T1 a6, T1 a7, T1 a8, T1 a9, T1 a10, T1 a11, T1 a12, T1 a13, T1 a14);
/* [detachable] SQL_GENERATOR_MSQL.set_options */
extern void T52f236(GE_context* ac, T0* C, T1 a1, T1 a2, T1 a3, T1 a4, T1 a5, T1 a6, T1 a7, T1 a8, T1 a9, T1 a10, T1 a11, T1 a12, T1 a13, T1 a14);
/* [detachable] SQL_GENERATOR_MSACCESS2000.set_options */
extern void T51f236(GE_context* ac, T0* C, T1 a1, T1 a2, T1 a3, T1 a4, T1 a5, T1 a6, T1 a7, T1 a8, T1 a9, T1 a10, T1 a11, T1 a12, T1 a13, T1 a14);
/* [detachable] SQL_GENERATOR_MSACCESS97.set_options */
extern void T50f236(GE_context* ac, T0* C, T1 a1, T1 a2, T1 a3, T1 a4, T1 a5, T1 a6, T1 a7, T1 a8, T1 a9, T1 a10, T1 a11, T1 a12, T1 a13, T1 a14);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.set_options */
extern void T49f266(GE_context* ac, T0* C, T1 a1, T1 a2, T1 a3, T1 a4, T1 a5, T1 a6, T1 a7, T1 a8, T1 a9, T1 a10, T1 a11, T1 a12, T1 a13, T1 a14);
/* [detachable] SQL_GENERATOR_INTERBASE6.set_options */
extern void T48f266(GE_context* ac, T0* C, T1 a1, T1 a2, T1 a3, T1 a4, T1 a5, T1 a6, T1 a7, T1 a8, T1 a9, T1 a10, T1 a11, T1 a12, T1 a13, T1 a14);
/* [detachable] SQL_GENERATOR_FOXPRO.set_options */
extern void T47f236(GE_context* ac, T0* C, T1 a1, T1 a2, T1 a3, T1 a4, T1 a5, T1 a6, T1 a7, T1 a8, T1 a9, T1 a10, T1 a11, T1 a12, T1 a13, T1 a14);
/* [detachable] SQL_GENERATOR_DB2_71.set_options */
extern void T46f267(GE_context* ac, T0* C, T1 a1, T1 a2, T1 a3, T1 a4, T1 a5, T1 a6, T1 a7, T1 a8, T1 a9, T1 a10, T1 a11, T1 a12, T1 a13, T1 a14);
/* [detachable] SQL_GENERATOR_DB2_6.set_options */
extern void T45f238(GE_context* ac, T0* C, T1 a1, T1 a2, T1 a3, T1 a4, T1 a5, T1 a6, T1 a7, T1 a8, T1 a9, T1 a10, T1 a11, T1 a12, T1 a13, T1 a14);
/* [detachable] SQL_GENERATOR_BASIC.set_options */
extern void T44f237(GE_context* ac, T0* C, T1 a1, T1 a2, T1 a3, T1 a4, T1 a5, T1 a6, T1 a7, T1 a8, T1 a9, T1 a10, T1 a11, T1 a12, T1 a13, T1 a14);
/* [detachable] SQL_GENERATOR_ANSI.set_options */
extern void T43f237(GE_context* ac, T0* C, T1 a1, T1 a2, T1 a3, T1 a4, T1 a5, T1 a6, T1 a7, T1 a8, T1 a9, T1 a10, T1 a11, T1 a12, T1 a13, T1 a14);
/* [detachable] SQL_GENERATOR_TSQL65.set_primary_key_format */
extern void T66f252(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.set_primary_key_format */
extern void T65f251(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.set_primary_key_format */
extern void T64f251(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.set_primary_key_format */
extern void T63f251(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.set_primary_key_format */
extern void T62f250(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.set_primary_key_format */
extern void T61f274(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.set_primary_key_format */
extern void T60f277(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.set_primary_key_format */
extern void T59f278(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.set_primary_key_format */
extern void T58f278(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.set_primary_key_format */
extern void T57f278(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.set_primary_key_format */
extern void T56f272(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.set_primary_key_format */
extern void T55f238(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.set_primary_key_format */
extern void T54f238(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.set_primary_key_format */
extern void T53f263(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.set_primary_key_format */
extern void T52f237(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.set_primary_key_format */
extern void T51f237(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.set_primary_key_format */
extern void T50f237(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.set_primary_key_format */
extern void T49f267(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.set_primary_key_format */
extern void T48f267(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.set_primary_key_format */
extern void T47f237(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.set_primary_key_format */
extern void T46f268(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.set_primary_key_format */
extern void T45f239(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.set_primary_key_format */
extern void T44f238(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.set_primary_key_format */
extern void T43f238(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.set_sequence_name_format */
extern void T66f253(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.valid_format_and_parameters */
extern T1 T66f64(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL65.formatter */
extern T0* T66f30(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.scientific_formatter */
extern T0* T66f32(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.set_sequence_name_format */
extern void T65f252(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.valid_format_and_parameters */
extern T1 T65f64(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL70.formatter */
extern T0* T65f30(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.scientific_formatter */
extern T0* T65f32(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.set_sequence_name_format */
extern void T64f252(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.valid_format_and_parameters */
extern T1 T64f64(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL2000.formatter */
extern T0* T64f30(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.scientific_formatter */
extern T0* T64f32(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.set_sequence_name_format */
extern void T63f252(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.valid_format_and_parameters */
extern T1 T63f64(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL2016.formatter */
extern T0* T63f30(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.scientific_formatter */
extern T0* T63f32(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.set_sequence_name_format */
extern void T62f251(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.valid_format_and_parameters */
extern T1 T62f84(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_SQLITE_3.formatter */
extern T0* T62f30(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.scientific_formatter */
extern T0* T62f32(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.set_sequence_name_format */
extern void T61f275(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.valid_format_and_parameters */
extern T1 T61f63(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_72.formatter */
extern T0* T61f30(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.scientific_formatter */
extern T0* T61f32(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.set_sequence_name_format */
extern void T60f278(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.valid_format_and_parameters */
extern T1 T60f63(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_73.formatter */
extern T0* T60f30(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.scientific_formatter */
extern T0* T60f32(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.set_sequence_name_format */
extern void T59f279(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.valid_format_and_parameters */
extern T1 T59f63(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_81.formatter */
extern T0* T59f30(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.scientific_formatter */
extern T0* T59f32(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.set_sequence_name_format */
extern void T58f279(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.valid_format_and_parameters */
extern T1 T58f63(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_82.formatter */
extern T0* T58f30(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.scientific_formatter */
extern T0* T58f32(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.set_sequence_name_format */
extern void T57f279(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.valid_format_and_parameters */
extern T1 T57f63(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_95.formatter */
extern T0* T57f30(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.scientific_formatter */
extern T0* T57f32(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.set_sequence_name_format */
extern void T56f273(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.valid_format_and_parameters */
extern T1 T56f64(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_ORACLE.formatter */
extern T0* T56f30(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.scientific_formatter */
extern T0* T56f32(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.set_sequence_name_format */
extern void T55f239(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.valid_format_and_parameters */
extern T1 T55f83(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL322.formatter */
extern T0* T55f30(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.scientific_formatter */
extern T0* T55f32(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.set_sequence_name_format */
extern void T54f239(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.valid_format_and_parameters */
extern T1 T54f83(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL4.formatter */
extern T0* T54f30(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.scientific_formatter */
extern T0* T54f32(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.set_sequence_name_format */
extern void T53f264(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.valid_format_and_parameters */
extern T1 T53f65(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL5.formatter */
extern T0* T53f30(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.scientific_formatter */
extern T0* T53f32(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.set_sequence_name_format */
extern void T52f238(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.valid_format_and_parameters */
extern T1 T52f83(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSQL.formatter */
extern T0* T52f30(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.scientific_formatter */
extern T0* T52f32(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.set_sequence_name_format */
extern void T51f238(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.valid_format_and_parameters */
extern T1 T51f83(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSACCESS2000.formatter */
extern T0* T51f30(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.scientific_formatter */
extern T0* T51f32(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.set_sequence_name_format */
extern void T50f238(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.valid_format_and_parameters */
extern T1 T50f83(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSACCESS97.formatter */
extern T0* T50f30(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.scientific_formatter */
extern T0* T50f32(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.set_sequence_name_format */
extern void T49f268(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.valid_format_and_parameters */
extern T1 T49f65(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.formatter */
extern T0* T49f31(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.scientific_formatter */
extern T0* T49f33(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.set_sequence_name_format */
extern void T48f268(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.valid_format_and_parameters */
extern T1 T48f65(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_INTERBASE6.formatter */
extern T0* T48f31(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.scientific_formatter */
extern T0* T48f33(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.set_sequence_name_format */
extern void T47f238(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.valid_format_and_parameters */
extern T1 T47f83(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_FOXPRO.formatter */
extern T0* T47f30(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.scientific_formatter */
extern T0* T47f32(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.set_sequence_name_format */
extern void T46f269(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.valid_format_and_parameters */
extern T1 T46f64(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_DB2_71.formatter */
extern T0* T46f30(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.scientific_formatter */
extern T0* T46f32(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.set_sequence_name_format */
extern void T45f240(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.valid_format_and_parameters */
extern T1 T45f83(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_DB2_6.formatter */
extern T0* T45f30(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.scientific_formatter */
extern T0* T45f32(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.set_sequence_name_format */
extern void T44f239(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.valid_format_and_parameters */
extern T1 T44f83(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_BASIC.formatter */
extern T0* T44f30(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.scientific_formatter */
extern T0* T44f32(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.set_sequence_name_format */
extern void T43f239(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.valid_format_and_parameters */
extern T1 T43f83(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_ANSI.formatter */
extern T0* T43f30(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.scientific_formatter */
extern T0* T43f32(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.write_end */
extern void T66f255(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.create_end */
extern void T66f276(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.write_end */
extern void T65f254(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.create_end */
extern void T65f275(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.write_end */
extern void T64f254(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.create_end */
extern void T64f275(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.write_end */
extern void T63f254(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.create_end */
extern void T63f275(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.write_end */
extern void T62f253(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_end */
extern void T62f274(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.write_end */
extern void T61f277(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_end */
extern void T61f298(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.write_end */
extern void T60f280(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_end */
extern void T60f301(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.write_end */
extern void T59f281(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_end */
extern void T59f302(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.write_end */
extern void T58f281(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_end */
extern void T58f302(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.write_end */
extern void T57f281(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_end */
extern void T57f302(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.write_end */
extern void T56f275(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.create_end */
extern void T56f296(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.write_end */
extern void T55f241(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.create_end */
extern void T55f262(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.write_end */
extern void T54f241(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.create_end */
extern void T54f262(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.write_end */
extern void T53f266(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.create_end */
extern void T53f287(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.write_end */
extern void T52f240(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.create_end */
extern void T52f261(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.write_end */
extern void T51f240(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.create_end */
extern void T51f261(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.write_end */
extern void T50f240(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.create_end */
extern void T50f261(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.write_end */
extern void T49f270(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_end */
extern void T49f291(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.write_end */
extern void T48f270(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_end */
extern void T48f291(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.write_end */
extern void T47f240(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.create_end */
extern void T47f261(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.write_end */
extern void T46f271(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.create_end */
extern void T46f292(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.write_end */
extern void T45f242(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.create_end */
extern void T45f263(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.write_end */
extern void T44f241(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.create_end */
extern void T44f262(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.write_end */
extern void T43f241(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.create_end */
extern void T43f262(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.write_use_database */
extern void T66f256(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.create_use_database */
extern void T66f277(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.end_with_go */
extern void T66f303(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.write_use_database */
extern void T65f255(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.create_use_database */
extern void T65f276(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.end_with_go */
extern void T65f302(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.write_use_database */
extern void T64f255(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.create_use_database */
extern void T64f276(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.end_with_go */
extern void T64f302(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.write_use_database */
extern void T63f255(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.create_use_database */
extern void T63f276(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.end_with_go */
extern void T63f302(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.write_use_database */
extern void T62f254(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_use_database */
extern void T62f275(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.write_use_database */
extern void T61f278(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_use_database */
extern void T61f299(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.write_use_database */
extern void T60f281(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_use_database */
extern void T60f302(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.write_use_database */
extern void T59f282(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_use_database */
extern void T59f303(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.write_use_database */
extern void T58f282(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_use_database */
extern void T58f303(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.write_use_database */
extern void T57f282(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_use_database */
extern void T57f303(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.write_use_database */
extern void T56f276(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.create_use_database */
extern void T56f297(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.write_use_database */
extern void T55f242(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.create_use_database */
extern void T55f263(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.write_use_database */
extern void T54f242(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.create_use_database */
extern void T54f263(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.write_use_database */
extern void T53f267(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.create_use_database */
extern void T53f288(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.write_use_database */
extern void T52f241(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.create_use_database */
extern void T52f262(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.write_use_database */
extern void T51f241(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.create_use_database */
extern void T51f262(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.write_use_database */
extern void T50f241(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.create_use_database */
extern void T50f262(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.write_use_database */
extern void T49f271(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_use_database */
extern void T49f292(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_use_database */
extern void T49f292p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.write_use_database */
extern void T48f271(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_use_database */
extern void T48f292(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_use_database */
extern void T48f292p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.write_use_database */
extern void T47f241(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.create_use_database */
extern void T47f262(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.write_use_database */
extern void T46f272(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.create_use_database */
extern void T46f293(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.write_use_database */
extern void T45f243(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.create_use_database */
extern void T45f264(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.write_use_database */
extern void T44f242(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.create_use_database */
extern void T44f263(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.write_use_database */
extern void T43f242(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.create_use_database */
extern void T43f263(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.write_base */
extern void T66f257(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.create_domain */
extern void T66f278(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.rule_identifier */
extern T0* T66f46(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.createdomaincheck */
extern T1 T66f45(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.domain_identifier */
extern T0* T66f44(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.no_space_identifier */
extern T0* T66f143(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.maxidentifierlength */
extern T6 T66f184(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.domainnameprefix */
extern T0* T66f142(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.createdomains */
extern T1 T66f33(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.domainssupported */
extern T1 T66f137(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.write_base */
extern void T65f256(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.create_domain */
extern void T65f277(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.rule_identifier */
extern T0* T65f46(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.createdomaincheck */
extern T1 T65f45(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.domain_identifier */
extern T0* T65f44(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.no_space_identifier */
extern T0* T65f142(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.maxidentifierlength */
extern T6 T65f183(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.domainnameprefix */
extern T0* T65f141(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.createdomains */
extern T1 T65f33(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.domainssupported */
extern T1 T65f136(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.write_base */
extern void T64f256(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.create_domain */
extern void T64f277(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.rule_identifier */
extern T0* T64f46(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.createdomaincheck */
extern T1 T64f45(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.domain_identifier */
extern T0* T64f44(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.no_space_identifier */
extern T0* T64f142(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.maxidentifierlength */
extern T6 T64f183(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.domainnameprefix */
extern T0* T64f141(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.createdomains */
extern T1 T64f33(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.domainssupported */
extern T1 T64f136(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.write_base */
extern void T63f256(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.create_domain */
extern void T63f277(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.rule_identifier */
extern T0* T63f46(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.createdomaincheck */
extern T1 T63f45(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.domain_identifier */
extern T0* T63f44(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.no_space_identifier */
extern T0* T63f142(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.maxidentifierlength */
extern T6 T63f183(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.domainnameprefix */
extern T0* T63f141(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.createdomains */
extern T1 T63f33(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.domainssupported */
extern T1 T63f136(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.write_base */
extern void T62f255(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_domain */
extern void T62f276(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.domain_null_or_not_null */
extern T0* T62f43(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.domainnullallowed */
extern T1 T62f132(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.domainnotnullallowed */
extern T1 T62f131(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.domainnulldefault */
extern T1 T62f130(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.domain_identifier */
extern T0* T62f41(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.domainnameprefix */
extern T0* T62f128(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.createdomains */
extern T1 T62f33(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.domainssupported */
extern T1 T62f124(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.write_base */
extern void T61f279(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_domain */
extern void T61f300(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.domain_null_or_not_null */
extern T0* T61f44(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.domainnullallowed */
extern T1 T61f157(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.domainnotnullallowed */
extern T1 T61f156(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.domainnulldefault */
extern T1 T61f155(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.domain_identifier */
extern T0* T61f42(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.domainnameprefix */
extern T0* T61f154(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.createdomains */
extern T1 T61f33(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.domainssupported */
extern T1 T61f150(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.write_base */
extern void T60f282(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_domain */
extern void T60f303(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.domain_null_or_not_null */
extern T0* T60f44(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.domainnullallowed */
extern T1 T60f160(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.domainnotnullallowed */
extern T1 T60f159(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.domainnulldefault */
extern T1 T60f158(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.domain_identifier */
extern T0* T60f42(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.domainnameprefix */
extern T0* T60f157(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.createdomains */
extern T1 T60f33(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.domainssupported */
extern T1 T60f153(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.write_base */
extern void T59f283(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_domain */
extern void T59f304(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.domain_null_or_not_null */
extern T0* T59f44(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.domainnullallowed */
extern T1 T59f161(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.domainnotnullallowed */
extern T1 T59f160(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.domainnulldefault */
extern T1 T59f159(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.domain_identifier */
extern T0* T59f42(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.domainnameprefix */
extern T0* T59f158(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.createdomains */
extern T1 T59f33(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.domainssupported */
extern T1 T59f154(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.write_base */
extern void T58f283(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_domain */
extern void T58f304(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.domain_null_or_not_null */
extern T0* T58f44(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.domainnullallowed */
extern T1 T58f161(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.domainnotnullallowed */
extern T1 T58f160(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.domainnulldefault */
extern T1 T58f159(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.domain_identifier */
extern T0* T58f42(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.domainnameprefix */
extern T0* T58f158(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.createdomains */
extern T1 T58f33(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.domainssupported */
extern T1 T58f154(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.write_base */
extern void T57f283(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_domain */
extern void T57f304(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.domain_null_or_not_null */
extern T0* T57f44(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.domainnullallowed */
extern T1 T57f161(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.domainnotnullallowed */
extern T1 T57f160(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.domainnulldefault */
extern T1 T57f159(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.domain_identifier */
extern T0* T57f42(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.domainnameprefix */
extern T0* T57f157(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.createdomains */
extern T1 T57f33(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.domainssupported */
extern T1 T57f153(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.write_base */
extern void T56f277(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.create_domain */
extern void T56f298(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.domain_null_or_not_null */
extern T0* T56f45(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.domainnullallowed */
extern T1 T56f155(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.domainnotnullallowed */
extern T1 T56f154(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.domainnulldefault */
extern T1 T56f153(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.domain_identifier */
extern T0* T56f43(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.domainnameprefix */
extern T0* T56f151(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.createdomains */
extern T1 T56f33(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.domainssupported */
extern T1 T56f146(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.write_base */
extern void T55f243(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.create_domain */
extern void T55f264(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.domain_null_or_not_null */
extern T0* T55f43(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.domainnullallowed */
extern T1 T55f131(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.domainnotnullallowed */
extern T1 T55f130(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.domainnulldefault */
extern T1 T55f129(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.domain_identifier */
extern T0* T55f41(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.domainnameprefix */
extern T0* T55f127(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.createdomains */
extern T1 T55f33(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.domainssupported */
extern T1 T55f123(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.write_base */
extern void T54f243(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.create_domain */
extern void T54f264(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.domain_null_or_not_null */
extern T0* T54f43(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.domainnullallowed */
extern T1 T54f131(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.domainnotnullallowed */
extern T1 T54f130(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.domainnulldefault */
extern T1 T54f129(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.domain_identifier */
extern T0* T54f41(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.domainnameprefix */
extern T0* T54f127(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.createdomains */
extern T1 T54f33(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.domainssupported */
extern T1 T54f123(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.write_base */
extern void T53f268(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.create_domain */
extern void T53f289(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.domain_null_or_not_null */
extern T0* T53f47(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.domainnullallowed */
extern T1 T53f147(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.domainnotnullallowed */
extern T1 T53f146(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.domainnulldefault */
extern T1 T53f145(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.domain_identifier */
extern T0* T53f45(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.domainnameprefix */
extern T0* T53f143(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.createdomains */
extern T1 T53f33(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.domainssupported */
extern T1 T53f138(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.write_base */
extern void T52f242(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.create_domain */
extern void T52f263(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.domain_null_or_not_null */
extern T0* T52f43(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.domainnullallowed */
extern T1 T52f131(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.domainnotnullallowed */
extern T1 T52f130(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.domainnulldefault */
extern T1 T52f129(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.domain_identifier */
extern T0* T52f41(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.domainnameprefix */
extern T0* T52f127(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.createdomains */
extern T1 T52f33(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.domainssupported */
extern T1 T52f123(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.write_base */
extern void T51f242(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.create_domain */
extern void T51f263(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.domain_null_or_not_null */
extern T0* T51f43(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.domainnullallowed */
extern T1 T51f131(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.domainnotnullallowed */
extern T1 T51f130(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.domainnulldefault */
extern T1 T51f129(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.domain_identifier */
extern T0* T51f41(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.domainnameprefix */
extern T0* T51f127(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.createdomains */
extern T1 T51f33(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.domainssupported */
extern T1 T51f123(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.write_base */
extern void T50f242(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.create_domain */
extern void T50f263(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.domain_null_or_not_null */
extern T0* T50f43(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.domainnullallowed */
extern T1 T50f131(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.domainnotnullallowed */
extern T1 T50f130(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.domainnulldefault */
extern T1 T50f129(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.domain_identifier */
extern T0* T50f41(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.domainnameprefix */
extern T0* T50f127(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.createdomains */
extern T1 T50f33(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.domainssupported */
extern T1 T50f123(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.write_base */
extern void T49f272(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_domain */
extern void T49f293(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.domain_null_or_not_null */
extern T0* T49f47(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.domainnullallowed */
extern T1 T49f154(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.domainnotnullallowed */
extern T1 T49f153(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.domainnulldefault */
extern T1 T49f152(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.domain_identifier */
extern T0* T49f45(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.domainnameprefix */
extern T0* T49f150(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.createdomains */
extern T1 T49f34(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.domainssupported */
extern T1 T49f145(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.write_base */
extern void T48f272(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_domain */
extern void T48f293(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.domain_null_or_not_null */
extern T0* T48f47(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.domainnullallowed */
extern T1 T48f154(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.domainnotnullallowed */
extern T1 T48f153(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.domainnulldefault */
extern T1 T48f152(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.domain_identifier */
extern T0* T48f45(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.domainnameprefix */
extern T0* T48f150(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.createdomains */
extern T1 T48f34(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.domainssupported */
extern T1 T48f145(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.write_base */
extern void T47f242(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.create_domain */
extern void T47f263(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.domain_null_or_not_null */
extern T0* T47f43(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.domainnullallowed */
extern T1 T47f131(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.domainnotnullallowed */
extern T1 T47f130(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.domainnulldefault */
extern T1 T47f129(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.domain_identifier */
extern T0* T47f41(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.domainnameprefix */
extern T0* T47f127(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.createdomains */
extern T1 T47f33(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.domainssupported */
extern T1 T47f123(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.write_base */
extern void T46f273(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.create_domain */
extern void T46f294(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.domain_null_or_not_null */
extern T0* T46f46(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.domainnullallowed */
extern T1 T46f153(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.domainnotnullallowed */
extern T1 T46f152(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.domainnulldefault */
extern T1 T46f151(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.domain_identifier */
extern T0* T46f44(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.domainnameprefix */
extern T0* T46f149(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.createdomains */
extern T1 T46f33(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.domainssupported */
extern T1 T46f142(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.write_base */
extern void T45f244(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.create_domain */
extern void T45f265(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.domain_null_or_not_null */
extern T0* T45f44(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.domainnullallowed */
extern T1 T45f131(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.domainnotnullallowed */
extern T1 T45f130(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.domainnulldefault */
extern T1 T45f129(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.domain_identifier */
extern T0* T45f42(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.domainnameprefix */
extern T0* T45f127(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.createdomains */
extern T1 T45f33(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.domainssupported */
extern T1 T45f123(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.write_base */
extern void T44f243(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.create_domain */
extern void T44f264(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.domain_null_or_not_null */
extern T0* T44f43(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.domainnullallowed */
extern T1 T44f131(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.domainnotnullallowed */
extern T1 T44f130(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.domainnulldefault */
extern T1 T44f129(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.domain_identifier */
extern T0* T44f41(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.domainnameprefix */
extern T0* T44f127(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.createdomains */
extern T1 T44f33(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.domainssupported */
extern T1 T44f123(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.write_base */
extern void T43f243(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.create_domain */
extern void T43f264(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.domain_null_or_not_null */
extern T0* T43f43(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.domainnullallowed */
extern T1 T43f131(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.domainnotnullallowed */
extern T1 T43f130(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.domainnulldefault */
extern T1 T43f129(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.domain_identifier */
extern T0* T43f41(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.domainnameprefix */
extern T0* T43f127(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.createdomains */
extern T1 T43f33(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.domainssupported */
extern T1 T43f123(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.write_constant_assignment */
extern void T66f258(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL65.create_constant_assignment */
extern void T66f279(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL70.write_constant_assignment */
extern void T65f257(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL70.create_constant_assignment */
extern void T65f278(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL2000.write_constant_assignment */
extern void T64f257(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL2000.create_constant_assignment */
extern void T64f278(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL2016.write_constant_assignment */
extern void T63f257(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL2016.create_constant_assignment */
extern void T63f278(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_SQLITE_3.write_constant_assignment */
extern void T62f256(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_constant_assignment */
extern void T62f277(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_72.write_constant_assignment */
extern void T61f280(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_constant_assignment */
extern void T61f301(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_73.write_constant_assignment */
extern void T60f283(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_constant_assignment */
extern void T60f304(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_81.write_constant_assignment */
extern void T59f284(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_constant_assignment */
extern void T59f305(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_82.write_constant_assignment */
extern void T58f284(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_constant_assignment */
extern void T58f305(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_95.write_constant_assignment */
extern void T57f284(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_constant_assignment */
extern void T57f305(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_ORACLE.write_constant_assignment */
extern void T56f278(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_ORACLE.create_constant_assignment */
extern void T56f299(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL322.write_constant_assignment */
extern void T55f244(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL322.create_constant_assignment */
extern void T55f265(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL4.write_constant_assignment */
extern void T54f244(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL4.create_constant_assignment */
extern void T54f265(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL5.write_constant_assignment */
extern void T53f269(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL5.create_constant_assignment */
extern void T53f290(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSQL.write_constant_assignment */
extern void T52f243(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSQL.create_constant_assignment */
extern void T52f264(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSACCESS2000.write_constant_assignment */
extern void T51f243(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSACCESS2000.create_constant_assignment */
extern void T51f264(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSACCESS97.write_constant_assignment */
extern void T50f243(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSACCESS97.create_constant_assignment */
extern void T50f264(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.write_constant_assignment */
extern void T49f273(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_constant_assignment */
extern void T49f294(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_INTERBASE6.write_constant_assignment */
extern void T48f273(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_constant_assignment */
extern void T48f294(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_FOXPRO.write_constant_assignment */
extern void T47f243(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_FOXPRO.create_constant_assignment */
extern void T47f264(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_DB2_71.write_constant_assignment */
extern void T46f274(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_DB2_71.create_constant_assignment */
extern void T46f295(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_DB2_6.write_constant_assignment */
extern void T45f245(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_DB2_6.create_constant_assignment */
extern void T45f266(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_BASIC.write_constant_assignment */
extern void T44f244(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_BASIC.create_constant_assignment */
extern void T44f265(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_ANSI.write_constant_assignment */
extern void T43f244(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_ANSI.create_constant_assignment */
extern void T43f265(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL65.write_constant */
extern void T66f259(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.create_constant */
extern void T66f280(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.create_constant */
extern void T66f280p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.write_constant */
extern void T65f258(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.create_constant */
extern void T65f279(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.create_constant */
extern void T65f279p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.write_constant */
extern void T64f258(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.create_constant */
extern void T64f279(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.create_constant */
extern void T64f279p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.write_constant */
extern void T63f258(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.create_constant */
extern void T63f279(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.create_constant */
extern void T63f279p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.write_constant */
extern void T62f257(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_constant */
extern void T62f278(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.write_constant */
extern void T61f281(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_constant */
extern void T61f302(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.write_constant */
extern void T60f284(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_constant */
extern void T60f305(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.write_constant */
extern void T59f285(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_constant */
extern void T59f306(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.write_constant */
extern void T58f285(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_constant */
extern void T58f306(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.write_constant */
extern void T57f285(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_constant */
extern void T57f306(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.write_constant */
extern void T56f279(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.create_constant */
extern void T56f300(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.write_constant */
extern void T55f245(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.create_constant */
extern void T55f266(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.write_constant */
extern void T54f245(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.create_constant */
extern void T54f266(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.write_constant */
extern void T53f270(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.create_constant */
extern void T53f291(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.write_constant */
extern void T52f244(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.create_constant */
extern void T52f265(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.write_constant */
extern void T51f244(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.create_constant */
extern void T51f265(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.write_constant */
extern void T50f244(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.create_constant */
extern void T50f265(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.write_constant */
extern void T49f274(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_constant */
extern void T49f295(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.write_constant */
extern void T48f274(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_constant */
extern void T48f295(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.write_constant */
extern void T47f244(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.create_constant */
extern void T47f265(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.write_constant */
extern void T46f275(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.create_constant */
extern void T46f296(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.write_constant */
extern void T45f246(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.create_constant */
extern void T45f267(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.write_constant */
extern void T44f245(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.create_constant */
extern void T44f266(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.write_constant */
extern void T43f245(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.create_constant */
extern void T43f266(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.write_drop */
extern void T66f260(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.write_drop */
extern void T65f259(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.write_drop */
extern void T64f259(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.write_drop */
extern void T63f259(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.write_drop */
extern void T62f258(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.write_drop */
extern void T61f282(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.write_drop */
extern void T60f285(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.write_drop */
extern void T59f286(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.write_drop */
extern void T58f286(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.write_drop */
extern void T57f286(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.write_drop */
extern void T56f280(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.write_drop */
extern void T55f246(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.write_drop */
extern void T54f246(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.write_drop */
extern void T53f271(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.write_drop */
extern void T52f245(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.write_drop */
extern void T51f245(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.write_drop */
extern void T50f245(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.write_drop */
extern void T49f275(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.write_drop */
extern void T48f275(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.write_drop */
extern void T47f245(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.write_drop */
extern void T46f276(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.write_drop */
extern void T45f247(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.write_drop */
extern void T44f246(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.write_drop */
extern void T43f246(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.write_assertion */
extern void T66f261(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.create_assertion */
extern void T66f281(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.create_assertion */
extern void T66f281p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_EXTENSION_FUNCTION_EXPRESSION.sqlselect */
extern T0* T315f5(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_EXTENSION_EXPRESSION_EXPRESSION.sqlselect */
extern T0* T313f7(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL65.viewcolumnssupported */
extern T1 T66f54(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.createviewsql */
extern T0* T66f52(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.drop_view_if_exist */
extern void T66f304(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.viewssupported */
extern T1 T66f35(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.createassert */
extern T1 T66f34(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.write_assertion */
extern void T65f260(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.create_assertion */
extern void T65f280(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.create_assertion */
extern void T65f280p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.viewcolumnssupported */
extern T1 T65f54(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.createviewsql */
extern T0* T65f52(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.drop_view_if_exist */
extern void T65f303(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.viewssupported */
extern T1 T65f35(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.createassert */
extern T1 T65f34(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.write_assertion */
extern void T64f260(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.create_assertion */
extern void T64f280(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.create_assertion */
extern void T64f280p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.viewcolumnssupported */
extern T1 T64f54(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.createviewsql */
extern T0* T64f52(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.drop_view_if_exist */
extern void T64f303(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.viewssupported */
extern T1 T64f35(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.createassert */
extern T1 T64f34(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.write_assertion */
extern void T63f260(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.create_assertion */
extern void T63f280(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.create_assertion */
extern void T63f280p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.viewcolumnssupported */
extern T1 T63f54(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.createviewsql */
extern T0* T63f52(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.drop_view_if_exist */
extern void T63f303(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.viewssupported */
extern T1 T63f35(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.createassert */
extern T1 T63f34(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.write_assertion */
extern void T62f259(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_assertion */
extern void T62f279(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.createviewsql */
extern T0* T62f49(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.drop_view_if_exist */
extern void T62f296(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.viewssupported */
extern T1 T62f35(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.createassert */
extern T1 T62f34(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.write_assertion */
extern void T61f283(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_assertion */
extern void T61f303(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.viewcolumnssupported */
extern T1 T61f52(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.createviewsql */
extern T0* T61f50(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.drop_view_if_exist */
extern void T61f324(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.viewssupported */
extern T1 T61f35(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.createassert */
extern T1 T61f34(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.write_assertion */
extern void T60f286(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_assertion */
extern void T60f306(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.viewcolumnssupported */
extern T1 T60f52(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.drop_view_if_exist */
extern void T60f327(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.viewssupported */
extern T1 T60f35(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.createassert */
extern T1 T60f34(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.write_assertion */
extern void T59f287(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_assertion */
extern void T59f307(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.viewcolumnssupported */
extern T1 T59f52(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.drop_view_if_exist */
extern void T59f328(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.viewssupported */
extern T1 T59f35(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.createassert */
extern T1 T59f34(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.write_assertion */
extern void T58f287(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_assertion */
extern void T58f307(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.viewcolumnssupported */
extern T1 T58f52(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.drop_view_if_exist */
extern void T58f328(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.viewssupported */
extern T1 T58f35(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.createassert */
extern T1 T58f34(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.write_assertion */
extern void T57f287(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_assertion */
extern void T57f307(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.viewcolumnssupported */
extern T1 T57f52(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.drop_view_if_exist */
extern void T57f328(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.viewssupported */
extern T1 T57f35(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.createassert */
extern T1 T57f34(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.write_assertion */
extern void T56f281(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.create_assertion */
extern void T56f301(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.viewcolumnssupported */
extern T1 T56f53(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.createviewsql */
extern T0* T56f51(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.drop_view_if_exist */
extern void T56f323(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.viewssupported */
extern T1 T56f35(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.createassert */
extern T1 T56f34(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.write_assertion */
extern void T55f247(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.create_assertion */
extern void T55f267(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.viewcolumnssupported */
extern T1 T55f51(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.createviewsql */
extern T0* T55f49(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.drop_view_if_exist */
extern void T55f283(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.viewssupported */
extern T1 T55f35(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.createassert */
extern T1 T55f34(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.write_assertion */
extern void T54f247(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.create_assertion */
extern void T54f267(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.viewcolumnssupported */
extern T1 T54f51(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.createviewsql */
extern T0* T54f49(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.drop_view_if_exist */
extern void T54f283(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.viewssupported */
extern T1 T54f35(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.createassert */
extern T1 T54f34(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.write_assertion */
extern void T53f272(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.create_assertion */
extern void T53f292(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.viewcolumnssupported */
extern T1 T53f54(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.createviewsql */
extern T0* T53f52(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.drop_view_if_exist */
extern void T53f314(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.viewssupported */
extern T1 T53f35(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.createassert */
extern T1 T53f34(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.write_assertion */
extern void T52f246(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.create_assertion */
extern void T52f266(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.viewcolumnssupported */
extern T1 T52f51(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.createviewsql */
extern T0* T52f49(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.drop_view_if_exist */
extern void T52f282(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.viewssupported */
extern T1 T52f35(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.createassert */
extern T1 T52f34(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.write_assertion */
extern void T51f246(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.create_assertion */
extern void T51f266(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.viewcolumnssupported */
extern T1 T51f51(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.createviewsql */
extern T0* T51f49(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.drop_view_if_exist */
extern void T51f282(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.viewssupported */
extern T1 T51f35(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.createassert */
extern T1 T51f34(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.write_assertion */
extern void T50f246(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.create_assertion */
extern void T50f266(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.viewcolumnssupported */
extern T1 T50f51(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.createviewsql */
extern T0* T50f49(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.drop_view_if_exist */
extern void T50f282(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.viewssupported */
extern T1 T50f35(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.createassert */
extern T1 T50f34(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.write_assertion */
extern void T49f276(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_assertion */
extern void T49f296(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.viewcolumnssupported */
extern T1 T49f54(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.createviewsql */
extern T0* T49f52(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.drop_view_if_exist */
extern void T49f318(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.viewssupported */
extern T1 T49f36(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.createassert */
extern T1 T49f35(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.write_assertion */
extern void T48f276(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_assertion */
extern void T48f296(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.viewcolumnssupported */
extern T1 T48f54(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.createviewsql */
extern T0* T48f52(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.drop_view_if_exist */
extern void T48f318(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.viewssupported */
extern T1 T48f36(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.createassert */
extern T1 T48f35(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.write_assertion */
extern void T47f246(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.create_assertion */
extern void T47f266(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.viewcolumnssupported */
extern T1 T47f51(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.createviewsql */
extern T0* T47f49(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.drop_view_if_exist */
extern void T47f282(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.viewssupported */
extern T1 T47f35(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.createassert */
extern T1 T47f34(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.write_assertion */
extern void T46f277(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.create_assertion */
extern void T46f297(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.viewcolumnssupported */
extern T1 T46f53(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.createviewsql */
extern T0* T46f51(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.drop_view_if_exist */
extern void T46f319(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.viewssupported */
extern T1 T46f35(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.createassert */
extern T1 T46f34(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.write_assertion */
extern void T45f248(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.create_assertion */
extern void T45f268(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.viewcolumnssupported */
extern T1 T45f51(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.createviewsql */
extern T0* T45f49(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.drop_view_if_exist */
extern void T45f284(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.viewssupported */
extern T1 T45f35(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.createassert */
extern T1 T45f34(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.write_assertion */
extern void T44f247(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.create_assertion */
extern void T44f267(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.viewcolumnssupported */
extern T1 T44f51(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.createviewsql */
extern T0* T44f49(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.drop_view_if_exist */
extern void T44f283(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.viewssupported */
extern T1 T44f35(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.createassert */
extern T1 T44f34(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.write_assertion */
extern void T43f247(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.create_assertion */
extern void T43f267(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.viewcolumnssupported */
extern T1 T43f51(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.createviewsql */
extern T0* T43f49(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.drop_view_if_exist */
extern void T43f283(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.viewssupported */
extern T1 T43f35(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.createassert */
extern T1 T43f34(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.write_index */
extern void T66f262(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.create_index */
extern void T66f282(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.create_index */
extern void T66f282p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.index_name */
extern T0* T66f57(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.createindexifnotexistsclause */
extern T0* T66f56(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.clusteredindexsupported */
extern T1 T66f55(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.createindex */
extern T1 T66f36(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.indexsupported */
extern T1 T66f138(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.write_index */
extern void T65f261(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.create_index */
extern void T65f281(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.create_index */
extern void T65f281p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.index_name */
extern T0* T65f57(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.createindexifnotexistsclause */
extern T0* T65f56(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.clusteredindexsupported */
extern T1 T65f55(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.createindex */
extern T1 T65f36(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.indexsupported */
extern T1 T65f137(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.write_index */
extern void T64f261(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.create_index */
extern void T64f281(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.create_index */
extern void T64f281p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.index_name */
extern T0* T64f57(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.createindexifnotexistsclause */
extern T0* T64f56(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.clusteredindexsupported */
extern T1 T64f55(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.createindex */
extern T1 T64f36(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.indexsupported */
extern T1 T64f137(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.write_index */
extern void T63f261(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.create_index */
extern void T63f281(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.create_index */
extern void T63f281p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.index_name */
extern T0* T63f57(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.createindexifnotexistsclause */
extern T0* T63f56(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.clusteredindexsupported */
extern T1 T63f55(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.createindex */
extern T1 T63f36(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.indexsupported */
extern T1 T63f137(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.write_index */
extern void T62f260(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_index */
extern void T62f280(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.index_name */
extern T0* T62f54(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.createindexifnotexistsclause */
extern T0* T62f53(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.clusteredindexsupported */
extern T1 T62f52(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.createindex */
extern T1 T62f36(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.indexsupported */
extern T1 T62f125(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.write_index */
extern void T61f284(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_index */
extern void T61f304(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.index_name */
extern T0* T61f55(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.createindexifnotexistsclause */
extern T0* T61f54(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.clusteredindexsupported */
extern T1 T61f53(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.createindex */
extern T1 T61f36(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.indexsupported */
extern T1 T61f151(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.write_index */
extern void T60f287(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_index */
extern void T60f307(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.index_name */
extern T0* T60f55(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.createindexifnotexistsclause */
extern T0* T60f54(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.clusteredindexsupported */
extern T1 T60f53(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.createindex */
extern T1 T60f36(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.indexsupported */
extern T1 T60f154(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.write_index */
extern void T59f288(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_index */
extern void T59f308(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.index_name */
extern T0* T59f55(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.createindexifnotexistsclause */
extern T0* T59f54(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.clusteredindexsupported */
extern T1 T59f53(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.createindex */
extern T1 T59f36(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.indexsupported */
extern T1 T59f155(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.write_index */
extern void T58f288(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_index */
extern void T58f308(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.index_name */
extern T0* T58f55(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.createindexifnotexistsclause */
extern T0* T58f54(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.clusteredindexsupported */
extern T1 T58f53(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.createindex */
extern T1 T58f36(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.indexsupported */
extern T1 T58f155(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.write_index */
extern void T57f288(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_index */
extern void T57f308(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.index_name */
extern T0* T57f55(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.clusteredindexsupported */
extern T1 T57f53(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.createindex */
extern T1 T57f36(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.indexsupported */
extern T1 T57f154(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.write_index */
extern void T56f282(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.create_index */
extern void T56f302(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.index_name */
extern T0* T56f56(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.createindexifnotexistsclause */
extern T0* T56f55(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.clusteredindexsupported */
extern T1 T56f54(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.createindex */
extern T1 T56f36(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.indexsupported */
extern T1 T56f147(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.write_index */
extern void T55f248(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.create_index */
extern void T55f268(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.index_name */
extern T0* T55f54(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.createindexifnotexistsclause */
extern T0* T55f53(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.clusteredindexsupported */
extern T1 T55f52(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.createindex */
extern T1 T55f36(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.indexsupported */
extern T1 T55f124(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.write_index */
extern void T54f248(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.create_index */
extern void T54f268(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.index_name */
extern T0* T54f54(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.createindexifnotexistsclause */
extern T0* T54f53(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.clusteredindexsupported */
extern T1 T54f52(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.createindex */
extern T1 T54f36(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.indexsupported */
extern T1 T54f124(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.write_index */
extern void T53f273(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.create_index */
extern void T53f293(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.index_name */
extern T0* T53f57(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.createindexifnotexistsclause */
extern T0* T53f56(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.clusteredindexsupported */
extern T1 T53f55(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.createindex */
extern T1 T53f36(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.indexsupported */
extern T1 T53f139(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.write_index */
extern void T52f247(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.create_index */
extern void T52f267(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.index_name */
extern T0* T52f54(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.createindexifnotexistsclause */
extern T0* T52f53(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.clusteredindexsupported */
extern T1 T52f52(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.createindex */
extern T1 T52f36(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.indexsupported */
extern T1 T52f124(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.write_index */
extern void T51f247(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.create_index */
extern void T51f267(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.index_name */
extern T0* T51f54(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.createindexifnotexistsclause */
extern T0* T51f53(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.clusteredindexsupported */
extern T1 T51f52(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.createindex */
extern T1 T51f36(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.indexsupported */
extern T1 T51f124(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.write_index */
extern void T50f247(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.create_index */
extern void T50f267(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.index_name */
extern T0* T50f54(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.createindexifnotexistsclause */
extern T0* T50f53(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.clusteredindexsupported */
extern T1 T50f52(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.createindex */
extern T1 T50f36(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.indexsupported */
extern T1 T50f124(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.write_index */
extern void T49f277(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_index */
extern void T49f297(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.index_name */
extern T0* T49f57(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.createindexifnotexistsclause */
extern T0* T49f56(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.clusteredindexsupported */
extern T1 T49f55(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.createindex */
extern T1 T49f37(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.indexsupported */
extern T1 T49f146(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.write_index */
extern void T48f277(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_index */
extern void T48f297(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.index_name */
extern T0* T48f57(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.createindexifnotexistsclause */
extern T0* T48f56(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.clusteredindexsupported */
extern T1 T48f55(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.createindex */
extern T1 T48f37(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.indexsupported */
extern T1 T48f146(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.write_index */
extern void T47f247(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.create_index */
extern void T47f267(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.index_name */
extern T0* T47f54(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.createindexifnotexistsclause */
extern T0* T47f53(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.clusteredindexsupported */
extern T1 T47f52(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.createindex */
extern T1 T47f36(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.indexsupported */
extern T1 T47f124(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.write_index */
extern void T46f278(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.create_index */
extern void T46f298(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.index_name */
extern T0* T46f56(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.createindexifnotexistsclause */
extern T0* T46f55(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.clusteredindexsupported */
extern T1 T46f54(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.createindex */
extern T1 T46f36(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.indexsupported */
extern T1 T46f143(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.write_index */
extern void T45f249(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.create_index */
extern void T45f269(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.index_name */
extern T0* T45f54(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.createindexifnotexistsclause */
extern T0* T45f53(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.clusteredindexsupported */
extern T1 T45f52(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.createindex */
extern T1 T45f36(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.indexsupported */
extern T1 T45f124(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.write_index */
extern void T44f248(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.create_index */
extern void T44f268(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.index_name */
extern T0* T44f54(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.createindexifnotexistsclause */
extern T0* T44f53(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.clusteredindexsupported */
extern T1 T44f52(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.createindex */
extern T1 T44f36(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.indexsupported */
extern T1 T44f124(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.write_index */
extern void T43f248(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.create_index */
extern void T43f268(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.index_name */
extern T0* T43f54(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.createindexifnotexistsclause */
extern T0* T43f53(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.clusteredindexsupported */
extern T1 T43f52(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.createindex */
extern T1 T43f36(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.indexsupported */
extern T1 T43f124(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.write_insert */
extern void T66f263(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_TSQL65.create_insert */
extern void T66f283(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_TSQL65.set_nocount_off */
extern void T66f306(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.create_insert */
extern void T66f283p1(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T66f283p1ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_TSQL65.create_sync_auto_generated_primary_key_to_highest_value */
extern void T66f308(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.create_sync_auto_generated_primary_key_with_supplied_value */
extern void T66f307(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SQL_GENERATOR_TSQL65.table_has_identity_column */
extern T1 T66f59(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.set_nocount_on */
extern void T66f305(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.write_insert */
extern void T65f262(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_TSQL70.create_insert */
extern void T65f282(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_TSQL70.set_nocount_off */
extern void T65f305(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.create_insert */
extern void T65f282p1(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T65f282p1ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_TSQL70.create_sync_auto_generated_primary_key_to_highest_value */
extern void T65f307(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.create_sync_auto_generated_primary_key_with_supplied_value */
extern void T65f306(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SQL_GENERATOR_TSQL70.table_has_identity_column */
extern T1 T65f59(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.set_nocount_on */
extern void T65f304(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.write_insert */
extern void T64f262(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_TSQL2000.create_insert */
extern void T64f282(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_TSQL2000.set_nocount_off */
extern void T64f305(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.create_insert */
extern void T64f282p1(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T64f282p1ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_TSQL2000.create_sync_auto_generated_primary_key_to_highest_value */
extern void T64f307(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.create_sync_auto_generated_primary_key_with_supplied_value */
extern void T64f306(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SQL_GENERATOR_TSQL2000.table_has_identity_column */
extern T1 T64f59(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.set_nocount_on */
extern void T64f304(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.write_insert */
extern void T63f262(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_TSQL2016.create_insert */
extern void T63f282(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_TSQL2016.set_nocount_off */
extern void T63f305(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.create_insert */
extern void T63f282p1(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T63f282p1ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_TSQL2016.create_sync_auto_generated_primary_key_to_highest_value */
extern void T63f307(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.create_sync_auto_generated_primary_key_with_supplied_value */
extern void T63f306(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SQL_GENERATOR_TSQL2016.table_has_identity_column */
extern T1 T63f59(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.set_nocount_on */
extern void T63f304(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.write_insert */
extern void T62f261(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_insert */
extern void T62f281(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T62f281ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_sync_auto_generated_primary_key_to_highest_value */
extern void T62f298(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_sync_auto_generated_primary_key_with_supplied_value */
extern void T62f297(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SQL_GENERATOR_PGSQL_72.write_insert */
extern void T61f285(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_insert */
extern void T61f305(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T61f305ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_sync_auto_generated_primary_key_to_highest_value */
extern void T61f326(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_sync_auto_generated_primary_key_with_supplied_value */
extern void T61f325(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SQL_GENERATOR_PGSQL_73.write_insert */
extern void T60f288(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_insert */
extern void T60f308(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T60f308ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_sync_auto_generated_primary_key_to_highest_value */
extern void T60f329(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_sync_auto_generated_primary_key_with_supplied_value */
extern void T60f328(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SQL_GENERATOR_PGSQL_81.write_insert */
extern void T59f289(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_insert */
extern void T59f309(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T59f309ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_sync_auto_generated_primary_key_to_highest_value */
extern void T59f330(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_sync_auto_generated_primary_key_with_supplied_value */
extern void T59f329(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SQL_GENERATOR_PGSQL_82.write_insert */
extern void T58f289(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_insert */
extern void T58f309(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T58f309ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_sync_auto_generated_primary_key_to_highest_value */
extern void T58f330(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_sync_auto_generated_primary_key_with_supplied_value */
extern void T58f329(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SQL_GENERATOR_PGSQL_95.write_insert */
extern void T57f289(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_insert */
extern void T57f309(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T57f309ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_sync_auto_generated_primary_key_to_highest_value */
extern void T57f330(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_sync_auto_generated_primary_key_with_supplied_value */
extern void T57f329(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SQL_GENERATOR_ORACLE.write_insert */
extern void T56f283(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_ORACLE.create_insert */
extern void T56f303(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T56f303ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_ORACLE.create_sync_auto_generated_primary_key_to_highest_value */
extern void T56f325(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.create_sync_auto_generated_primary_key_with_supplied_value */
extern void T56f324(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SQL_GENERATOR_MYSQL322.write_insert */
extern void T55f249(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_MYSQL322.create_insert */
extern void T55f269(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T55f269ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_MYSQL322.create_sync_auto_generated_primary_key_to_highest_value */
extern void T55f285(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.create_sync_auto_generated_primary_key_with_supplied_value */
extern void T55f284(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SQL_GENERATOR_MYSQL4.write_insert */
extern void T54f249(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_MYSQL4.create_insert */
extern void T54f269(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T54f269ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_MYSQL4.create_sync_auto_generated_primary_key_to_highest_value */
extern void T54f285(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.create_sync_auto_generated_primary_key_with_supplied_value */
extern void T54f284(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SQL_GENERATOR_MYSQL5.write_insert */
extern void T53f274(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_MYSQL5.create_insert */
extern void T53f294(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T53f294ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_MYSQL5.create_sync_auto_generated_primary_key_to_highest_value */
extern void T53f316(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.create_sync_auto_generated_primary_key_with_supplied_value */
extern void T53f315(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SQL_GENERATOR_MSQL.write_insert */
extern void T52f248(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_MSQL.create_insert */
extern void T52f268(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T52f268ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_MSQL.create_sync_auto_generated_primary_key_to_highest_value */
extern void T52f284(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.create_sync_auto_generated_primary_key_with_supplied_value */
extern void T52f283(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SQL_GENERATOR_MSACCESS2000.write_insert */
extern void T51f248(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_MSACCESS2000.create_insert */
extern void T51f268(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T51f268ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_MSACCESS2000.create_sync_auto_generated_primary_key_to_highest_value */
extern void T51f284(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.create_sync_auto_generated_primary_key_with_supplied_value */
extern void T51f283(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SQL_GENERATOR_MSACCESS97.write_insert */
extern void T50f248(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_MSACCESS97.create_insert */
extern void T50f268(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T50f268ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_MSACCESS97.create_sync_auto_generated_primary_key_to_highest_value */
extern void T50f284(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.create_sync_auto_generated_primary_key_with_supplied_value */
extern void T50f283(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.write_insert */
extern void T49f278(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_insert */
extern void T49f298(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T49f298ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_sync_auto_generated_primary_key_to_highest_value */
extern void T49f320(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_sync_auto_generated_primary_key_with_supplied_value */
extern void T49f319(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SQL_GENERATOR_INTERBASE6.write_insert */
extern void T48f278(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_insert */
extern void T48f298(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T48f298ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_sync_auto_generated_primary_key_to_highest_value */
extern void T48f320(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_sync_auto_generated_primary_key_with_supplied_value */
extern void T48f319(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SQL_GENERATOR_FOXPRO.write_insert */
extern void T47f248(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_FOXPRO.create_insert */
extern void T47f268(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T47f268ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_FOXPRO.create_sync_auto_generated_primary_key_to_highest_value */
extern void T47f284(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.create_sync_auto_generated_primary_key_with_supplied_value */
extern void T47f283(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SQL_GENERATOR_DB2_71.write_insert */
extern void T46f279(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_DB2_71.create_insert */
extern void T46f299(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T46f299ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_DB2_71.create_sync_auto_generated_primary_key_to_highest_value */
extern void T46f321(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.create_sync_auto_generated_primary_key_with_supplied_value */
extern void T46f320(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SQL_GENERATOR_DB2_6.write_insert */
extern void T45f250(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_DB2_6.create_insert */
extern void T45f270(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T45f270ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_DB2_6.create_sync_auto_generated_primary_key_to_highest_value */
extern void T45f286(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.create_sync_auto_generated_primary_key_with_supplied_value */
extern void T45f285(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SQL_GENERATOR_BASIC.write_insert */
extern void T44f249(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_BASIC.create_insert */
extern void T44f269(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T44f269ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_BASIC.create_sync_auto_generated_primary_key_to_highest_value */
extern void T44f285(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.create_sync_auto_generated_primary_key_with_supplied_value */
extern void T44f284(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SQL_GENERATOR_ANSI.write_insert */
extern void T43f249(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_ANSI.create_insert */
extern void T43f269(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T43f269ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_ANSI.create_sync_auto_generated_primary_key_to_highest_value */
extern void T43f285(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.create_sync_auto_generated_primary_key_with_supplied_value */
extern void T43f284(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] SQL_GENERATOR_TSQL65.write_delete */
extern void T66f264(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL65.create_delete */
extern void T66f284(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL70.write_delete */
extern void T65f263(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL70.create_delete */
extern void T65f283(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL2000.write_delete */
extern void T64f263(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL2000.create_delete */
extern void T64f283(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL2016.write_delete */
extern void T63f263(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL2016.create_delete */
extern void T63f283(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_SQLITE_3.write_delete */
extern void T62f262(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_delete */
extern void T62f282(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_72.write_delete */
extern void T61f286(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_delete */
extern void T61f306(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_73.write_delete */
extern void T60f289(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_delete */
extern void T60f309(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_81.write_delete */
extern void T59f290(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_delete */
extern void T59f310(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_82.write_delete */
extern void T58f290(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_delete */
extern void T58f310(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_95.write_delete */
extern void T57f290(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_delete */
extern void T57f310(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_ORACLE.write_delete */
extern void T56f284(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_ORACLE.create_delete */
extern void T56f304(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL322.write_delete */
extern void T55f250(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL322.create_delete */
extern void T55f270(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL4.write_delete */
extern void T54f250(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL4.create_delete */
extern void T54f270(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL5.write_delete */
extern void T53f275(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL5.create_delete */
extern void T53f295(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSQL.write_delete */
extern void T52f249(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSQL.create_delete */
extern void T52f269(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSACCESS2000.write_delete */
extern void T51f249(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSACCESS2000.create_delete */
extern void T51f269(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSACCESS97.write_delete */
extern void T50f249(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSACCESS97.create_delete */
extern void T50f269(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.write_delete */
extern void T49f279(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_delete */
extern void T49f299(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_INTERBASE6.write_delete */
extern void T48f279(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_delete */
extern void T48f299(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_FOXPRO.write_delete */
extern void T47f249(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_FOXPRO.create_delete */
extern void T47f269(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_DB2_71.write_delete */
extern void T46f280(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_DB2_71.create_delete */
extern void T46f300(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_DB2_6.write_delete */
extern void T45f251(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_DB2_6.create_delete */
extern void T45f271(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_BASIC.write_delete */
extern void T44f250(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_BASIC.create_delete */
extern void T44f270(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_ANSI.write_delete */
extern void T43f250(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_ANSI.create_delete */
extern void T43f270(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL65.write_update */
extern void T66f265(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_TSQL65.create_update */
extern void T66f285(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T66f285ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_TSQL65.output_update_from_clause */
extern void T66f313(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] JOIN_LIST.is_empty */
extern T1 T352f13(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.output_update_join_clause */
extern void T66f312(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL65.output_update_extend_from_clause */
extern void T66f311(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL65.supportsqualifiedsetinupdate */
extern T1 T66f66(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.output_update_extend_join_clause */
extern void T66f310(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_TSQL65.supportsjoininupdate */
extern T1 T66f65(GE_context* ac, T0* C);
/* detachable XPLAIN_EXTENSION_EXPRESSION.make */
extern T0* T162c28(GE_context* ac, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL70.write_update */
extern void T65f264(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_TSQL70.create_update */
extern void T65f284(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T65f284ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_TSQL70.output_update_from_clause */
extern void T65f312(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL70.output_update_join_clause */
extern void T65f311(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL70.output_update_extend_from_clause */
extern void T65f310(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL70.supportsqualifiedsetinupdate */
extern T1 T65f66(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.output_update_extend_join_clause */
extern void T65f309(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_TSQL70.supportsjoininupdate */
extern T1 T65f65(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.write_update */
extern void T64f264(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_TSQL2000.create_update */
extern void T64f284(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T64f284ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_TSQL2000.output_update_from_clause */
extern void T64f312(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL2000.output_update_join_clause */
extern void T64f311(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL2000.output_update_extend_from_clause */
extern void T64f310(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL2000.supportsqualifiedsetinupdate */
extern T1 T64f66(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.output_update_extend_join_clause */
extern void T64f309(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_TSQL2000.supportsjoininupdate */
extern T1 T64f65(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.write_update */
extern void T63f264(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_TSQL2016.create_update */
extern void T63f284(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T63f284ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_TSQL2016.output_update_from_clause */
extern void T63f312(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL2016.output_update_join_clause */
extern void T63f311(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL2016.output_update_extend_from_clause */
extern void T63f310(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL2016.supportsqualifiedsetinupdate */
extern T1 T63f66(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.output_update_extend_join_clause */
extern void T63f309(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_TSQL2016.supportsjoininupdate */
extern T1 T63f65(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.write_update */
extern void T62f263(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_update */
extern void T62f283(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T62f283ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_SQLITE_3.output_update_from_clause */
extern void T62f303(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_SQLITE_3.output_update_join_clause */
extern void T62f302(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_SQLITE_3.output_update_extend_from_clause */
extern void T62f301(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_SQLITE_3.supportsqualifiedsetinupdate */
extern T1 T62f61(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.output_update_extend_join_clause */
extern void T62f300(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_SQLITE_3.supportsjoininupdate */
extern T1 T62f60(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.write_update */
extern void T61f287(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_update */
extern void T61f307(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T61f307ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_PGSQL_72.output_update_from_clause */
extern void T61f331(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_72.output_update_join_clause */
extern void T61f330(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_72.output_update_extend_from_clause */
extern void T61f329(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_72.supportsqualifiedsetinupdate */
extern T1 T61f62(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.output_update_extend_join_clause */
extern void T61f328(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_PGSQL_72.supportsjoininupdate */
extern T1 T61f61(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.write_update */
extern void T60f290(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_update */
extern void T60f310(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T60f310ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_PGSQL_73.output_update_from_clause */
extern void T60f334(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_73.output_update_join_clause */
extern void T60f333(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_73.output_update_extend_from_clause */
extern void T60f332(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_73.supportsqualifiedsetinupdate */
extern T1 T60f62(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.output_update_extend_join_clause */
extern void T60f331(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_PGSQL_73.supportsjoininupdate */
extern T1 T60f61(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.write_update */
extern void T59f291(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_update */
extern void T59f311(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T59f311ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_PGSQL_81.output_update_from_clause */
extern void T59f335(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_81.sql_update_joins */
extern T0* T59f98(GE_context* ac, T0* C, T1 a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_81.output_update_join_clause */
extern void T59f334(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_81.output_update_extend_from_clause */
extern void T59f333(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_81.supportsqualifiedsetinupdate */
extern T1 T59f62(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.output_update_extend_join_clause */
extern void T59f332(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_PGSQL_81.supportsjoininupdate */
extern T1 T59f61(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.write_update */
extern void T58f291(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_update */
extern void T58f311(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T58f311ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_PGSQL_82.output_update_from_clause */
extern void T58f335(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_82.sql_update_joins */
extern T0* T58f98(GE_context* ac, T0* C, T1 a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_82.output_update_join_clause */
extern void T58f334(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_82.output_update_extend_from_clause */
extern void T58f333(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_82.supportsqualifiedsetinupdate */
extern T1 T58f62(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.output_update_extend_join_clause */
extern void T58f332(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_PGSQL_82.supportsjoininupdate */
extern T1 T58f61(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.write_update */
extern void T57f291(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_update */
extern void T57f311(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T57f311ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_PGSQL_95.output_update_from_clause */
extern void T57f335(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_95.sql_update_joins */
extern T0* T57f97(GE_context* ac, T0* C, T1 a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_95.output_update_join_clause */
extern void T57f334(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_95.output_update_extend_from_clause */
extern void T57f333(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_95.supportsqualifiedsetinupdate */
extern T1 T57f62(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.output_update_extend_join_clause */
extern void T57f332(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_PGSQL_95.supportsjoininupdate */
extern T1 T57f61(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.write_update */
extern void T56f285(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_ORACLE.create_update */
extern void T56f305(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T56f305ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_ORACLE.output_update_from_clause */
extern void T56f330(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_ORACLE.output_update_join_clause */
extern void T56f329(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_ORACLE.output_update_extend_from_clause */
extern void T56f328(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_ORACLE.supportsqualifiedsetinupdate */
extern T1 T56f63(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.output_update_extend_join_clause */
extern void T56f327(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_ORACLE.supportsjoininupdate */
extern T1 T56f62(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.write_update */
extern void T55f251(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_MYSQL322.create_update */
extern void T55f271(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T55f271ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_MYSQL322.output_update_from_clause */
extern void T55f290(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL322.output_update_join_clause */
extern void T55f289(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL322.output_update_extend_from_clause */
extern void T55f288(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL322.supportsqualifiedsetinupdate */
extern T1 T55f61(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.output_update_extend_join_clause */
extern void T55f287(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_MYSQL322.supportsjoininupdate */
extern T1 T55f60(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.write_update */
extern void T54f251(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_MYSQL4.create_update */
extern void T54f271(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T54f271ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_MYSQL4.output_update_from_clause */
extern void T54f290(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL4.output_update_join_clause */
extern void T54f289(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL4.output_update_extend_from_clause */
extern void T54f288(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL4.supportsqualifiedsetinupdate */
extern T1 T54f61(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.output_update_extend_join_clause */
extern void T54f287(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_MYSQL4.supportsjoininupdate */
extern T1 T54f60(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.write_update */
extern void T53f276(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_MYSQL5.create_update */
extern void T53f296(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T53f296ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_MYSQL5.output_update_from_clause */
extern void T53f321(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL5.output_update_join_clause */
extern void T53f320(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL5.output_update_extend_from_clause */
extern void T53f319(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL5.supportsqualifiedsetinupdate */
extern T1 T53f64(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.output_update_extend_join_clause */
extern void T53f318(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_MYSQL5.output_update_extend_join_clause */
extern void T53f318p1(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_MYSQL5.supportsjoininupdate */
extern T1 T53f63(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.write_update */
extern void T52f250(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_MSQL.create_update */
extern void T52f270(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T52f270ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_MSQL.output_update_from_clause */
extern void T52f289(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSQL.output_update_join_clause */
extern void T52f288(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSQL.output_update_extend_from_clause */
extern void T52f287(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSQL.supportsqualifiedsetinupdate */
extern T1 T52f61(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.output_update_extend_join_clause */
extern void T52f286(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_MSQL.supportsjoininupdate */
extern T1 T52f60(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.write_update */
extern void T51f250(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_MSACCESS2000.create_update */
extern void T51f270(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T51f270ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_MSACCESS2000.output_update_from_clause */
extern void T51f289(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSACCESS2000.output_update_join_clause */
extern void T51f288(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSACCESS2000.output_update_extend_from_clause */
extern void T51f287(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSACCESS2000.supportsqualifiedsetinupdate */
extern T1 T51f61(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.output_update_extend_join_clause */
extern void T51f286(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_MSACCESS2000.supportsjoininupdate */
extern T1 T51f60(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.write_update */
extern void T50f250(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_MSACCESS97.create_update */
extern void T50f270(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T50f270ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_MSACCESS97.output_update_from_clause */
extern void T50f289(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSACCESS97.output_update_join_clause */
extern void T50f288(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSACCESS97.output_update_extend_from_clause */
extern void T50f287(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSACCESS97.supportsqualifiedsetinupdate */
extern T1 T50f61(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.output_update_extend_join_clause */
extern void T50f286(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_MSACCESS97.supportsjoininupdate */
extern T1 T50f60(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.write_update */
extern void T49f280(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_update */
extern void T49f300(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T49f300ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.output_update_from_clause */
extern void T49f325(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.output_update_join_clause */
extern void T49f324(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.output_update_extend_from_clause */
extern void T49f323(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.supportsqualifiedsetinupdate */
extern T1 T49f64(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.output_update_extend_join_clause */
extern void T49f322(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.supportsjoininupdate */
extern T1 T49f63(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.write_update */
extern void T48f280(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_update */
extern void T48f300(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T48f300ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_INTERBASE6.output_update_from_clause */
extern void T48f325(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_INTERBASE6.output_update_join_clause */
extern void T48f324(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_INTERBASE6.output_update_extend_from_clause */
extern void T48f323(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_INTERBASE6.supportsqualifiedsetinupdate */
extern T1 T48f64(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.output_update_extend_join_clause */
extern void T48f322(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_INTERBASE6.supportsjoininupdate */
extern T1 T48f63(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.write_update */
extern void T47f250(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_FOXPRO.create_update */
extern void T47f270(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T47f270ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_FOXPRO.output_update_from_clause */
extern void T47f289(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_FOXPRO.output_update_join_clause */
extern void T47f288(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_FOXPRO.output_update_extend_from_clause */
extern void T47f287(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_FOXPRO.supportsqualifiedsetinupdate */
extern T1 T47f61(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.output_update_extend_join_clause */
extern void T47f286(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_FOXPRO.supportsjoininupdate */
extern T1 T47f60(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.write_update */
extern void T46f281(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_DB2_71.create_update */
extern void T46f301(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T46f301ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_DB2_71.output_update_from_clause */
extern void T46f326(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_DB2_71.output_update_join_clause */
extern void T46f325(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_DB2_71.output_update_extend_from_clause */
extern void T46f324(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_DB2_71.supportsqualifiedsetinupdate */
extern T1 T46f63(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.output_update_extend_join_clause */
extern void T46f323(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_DB2_71.supportsjoininupdate */
extern T1 T46f62(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.write_update */
extern void T45f252(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_DB2_6.create_update */
extern void T45f272(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T45f272ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_DB2_6.output_update_from_clause */
extern void T45f291(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_DB2_6.output_update_join_clause */
extern void T45f290(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_DB2_6.output_update_extend_from_clause */
extern void T45f289(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_DB2_6.supportsqualifiedsetinupdate */
extern T1 T45f61(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.output_update_extend_join_clause */
extern void T45f288(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_DB2_6.supportsjoininupdate */
extern T1 T45f60(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.write_update */
extern void T44f251(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_BASIC.create_update */
extern void T44f271(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T44f271ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_BASIC.output_update_from_clause */
extern void T44f290(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_BASIC.output_update_join_clause */
extern void T44f289(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_BASIC.output_update_extend_from_clause */
extern void T44f288(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_BASIC.supportsqualifiedsetinupdate */
extern T1 T44f61(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.output_update_extend_join_clause */
extern void T44f287(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_BASIC.supportsjoininupdate */
extern T1 T44f60(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.write_update */
extern void T43f251(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_ANSI.create_update */
extern void T43f271(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
extern T1 T43f271ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_ANSI.output_update_from_clause */
extern void T43f290(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_ANSI.output_update_join_clause */
extern void T43f289(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_ANSI.output_update_extend_from_clause */
extern void T43f288(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_ANSI.supportsqualifiedsetinupdate */
extern T1 T43f61(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.output_update_extend_join_clause */
extern void T43f287(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_ANSI.supportsjoininupdate */
extern T1 T43f60(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.write_value */
extern void T66f266(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.create_value */
extern void T66f286(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.create_value_assign */
extern void T66f315(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.create_value_assign_outside_sp */
extern void T66f335(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.do_create_value_assign */
extern void T66f341(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.create_value_assign_inside_sp */
extern void T66f334(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.optional_create_value_declare */
extern void T66f314(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.create_value_declare */
extern void T66f333(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.create_value_declare_outside_sp */
extern void T66f340(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.do_create_value_declare */
extern void T66f343(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.create_value_declare_inside_sp */
extern void T66f339(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.is_value_declared */
extern T1 T66f100(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8].has */
extern T1 T116f31(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.write_value */
extern void T65f265(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.create_value */
extern void T65f285(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.create_value_assign */
extern void T65f314(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.create_value_assign_outside_sp */
extern void T65f334(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.do_create_value_assign */
extern void T65f340(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.create_value_assign_inside_sp */
extern void T65f333(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.optional_create_value_declare */
extern void T65f313(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.create_value_declare */
extern void T65f332(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.create_value_declare_outside_sp */
extern void T65f339(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.do_create_value_declare */
extern void T65f341(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.create_value_declare_inside_sp */
extern void T65f338(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.is_value_declared */
extern T1 T65f100(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.write_value */
extern void T64f265(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.create_value */
extern void T64f285(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.create_value_assign */
extern void T64f314(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.create_value_assign_outside_sp */
extern void T64f334(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.do_create_value_assign */
extern void T64f340(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.create_value_assign_inside_sp */
extern void T64f333(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.optional_create_value_declare */
extern void T64f313(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.create_value_declare */
extern void T64f332(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.create_value_declare_outside_sp */
extern void T64f339(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.do_create_value_declare */
extern void T64f341(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.create_value_declare_inside_sp */
extern void T64f338(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.is_value_declared */
extern T1 T64f100(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.write_value */
extern void T63f265(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.create_value */
extern void T63f285(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.create_value_assign */
extern void T63f314(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.create_value_assign_outside_sp */
extern void T63f332(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.do_create_value_assign */
extern void T63f336(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.create_value_assign_inside_sp */
extern void T63f331(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.optional_create_value_declare */
extern void T63f313(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.create_value_declare */
extern void T63f330(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.create_value_declare_outside_sp */
extern void T63f335(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.do_create_value_declare */
extern void T63f337(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.create_value_declare_inside_sp */
extern void T63f334(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.is_value_declared */
extern T1 T63f100(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.write_value */
extern void T62f264(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_value */
extern void T62f284(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_value_assign */
extern void T62f305(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_value_assign_outside_sp */
extern void T62f311(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.valuetablename */
extern T0* T62f79(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.createtemporaryvaluetable */
extern T1 T62f85(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.temporarytablessupported */
extern T1 T62f142(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.optional_create_value_declare */
extern void T62f304(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_value_declare */
extern void T62f310(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_value_declare_outside_sp */
extern void T62f313(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_value_alter_table */
extern void T62f315(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_value_create_table */
extern void T62f314(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.finishtemporaryvaluetablestatement */
extern T0* T62f86(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.finishtemporarytablestatement */
extern T0* T62f82(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.createtemporarytablestatement */
extern T0* T62f80(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.is_value_data_type_changed */
extern T1 T62f78(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8].item */
extern T0* T116f32(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8].item_storage_item */
extern T0* T116f34(GE_context* ac, T0* C, T6 a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.is_value_declared */
extern T1 T62f77(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.write_value */
extern void T61f288(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_value */
extern void T61f308(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_value_assign */
extern void T61f333(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_value_assign_outside_sp */
extern void T61f353(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_value_assign_inside_sp */
extern void T61f352(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.optional_create_value_declare */
extern void T61f332(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.write_value */
extern void T60f291(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_value */
extern void T60f311(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_value_assign */
extern void T60f336(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_value_assign_outside_sp */
extern void T60f356(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_value_assign_inside_sp */
extern void T60f355(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.optional_create_value_declare */
extern void T60f335(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.write_value */
extern void T59f292(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_value */
extern void T59f312(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_value_assign */
extern void T59f337(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_value_assign_outside_sp */
extern void T59f357(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_value_assign_inside_sp */
extern void T59f356(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.optional_create_value_declare */
extern void T59f336(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.write_value */
extern void T58f292(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_value */
extern void T58f312(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_value_assign */
extern void T58f337(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_value_assign_outside_sp */
extern void T58f357(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_value_assign_inside_sp */
extern void T58f356(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.optional_create_value_declare */
extern void T58f336(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.write_value */
extern void T57f292(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_value */
extern void T57f312(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_value_assign */
extern void T57f337(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_value_assign_outside_sp */
extern void T57f357(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_value_assign_inside_sp */
extern void T57f356(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.optional_create_value_declare */
extern void T57f336(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.write_value */
extern void T56f286(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.create_value */
extern void T56f306(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.create_value_assign */
extern void T56f332(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.create_value_assign_outside_sp */
extern void T56f352(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.valuetablename */
extern T0* T56f101(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.createtemporaryvaluetable */
extern T1 T56f106(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.create_value_assign_inside_sp */
extern void T56f351(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.optional_create_value_declare */
extern void T56f331(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.create_value_declare */
extern void T56f350(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.create_value_declare_outside_sp */
extern void T56f354(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.create_value_alter_table */
extern void T56f356(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.create_value_create_table */
extern void T56f355(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.finishtemporaryvaluetablestatement */
extern T0* T56f107(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.create_value_declare_inside_sp */
extern void T56f353(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.is_value_declared */
extern T1 T56f100(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.write_value */
extern void T55f252(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.create_value */
extern void T55f272(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.create_value_assign */
extern void T55f292(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.create_value_assign_outside_sp */
extern void T55f298(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.valuetablename */
extern T0* T55f78(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.createtemporaryvaluetable */
extern T1 T55f84(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.temporarytablessupported */
extern T1 T55f140(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.optional_create_value_declare */
extern void T55f291(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.create_value_declare */
extern void T55f297(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.create_value_declare_outside_sp */
extern void T55f300(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.create_value_alter_table */
extern void T55f302(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.create_value_create_table */
extern void T55f301(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.finishtemporaryvaluetablestatement */
extern T0* T55f85(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.finishtemporarytablestatement */
extern T0* T55f81(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.createtemporarytablestatement */
extern T0* T55f79(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.is_value_data_type_changed */
extern T1 T55f77(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.is_value_declared */
extern T1 T55f76(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.write_value */
extern void T54f252(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.create_value */
extern void T54f272(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.create_value_assign */
extern void T54f292(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.create_value_assign_outside_sp */
extern void T54f298(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.valuetablename */
extern T0* T54f78(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.createtemporaryvaluetable */
extern T1 T54f84(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.temporarytablessupported */
extern T1 T54f140(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.optional_create_value_declare */
extern void T54f291(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.create_value_declare */
extern void T54f297(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.create_value_declare_outside_sp */
extern void T54f300(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.create_value_alter_table */
extern void T54f302(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.create_value_create_table */
extern void T54f301(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.finishtemporaryvaluetablestatement */
extern T0* T54f85(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.finishtemporarytablestatement */
extern T0* T54f81(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.createtemporarytablestatement */
extern T0* T54f79(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.is_value_data_type_changed */
extern T1 T54f77(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.is_value_declared */
extern T1 T54f76(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.write_value */
extern void T53f277(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.create_value */
extern void T53f297(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.create_value_assign */
extern void T53f323(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.create_value_assign_outside_sp */
extern void T53f343(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.format */
extern T0* T53f68(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL5.create_value_assign_inside_sp */
extern void T53f342(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.optional_create_value_declare */
extern void T53f322(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.create_value_declare */
extern void T53f341(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.create_value_declare_outside_sp */
extern void T53f345(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.create_value_declare_inside_sp */
extern void T53f344(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.is_value_declared */
extern T1 T53f97(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.write_value */
extern void T52f251(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.create_value */
extern void T52f271(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.create_value_assign */
extern void T52f291(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.create_value_assign_outside_sp */
extern void T52f297(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.valuetablename */
extern T0* T52f78(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.createtemporaryvaluetable */
extern T1 T52f84(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.temporarytablessupported */
extern T1 T52f140(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.optional_create_value_declare */
extern void T52f290(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.create_value_declare */
extern void T52f296(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.create_value_declare_outside_sp */
extern void T52f299(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.create_value_alter_table */
extern void T52f301(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.create_value_create_table */
extern void T52f300(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.finishtemporaryvaluetablestatement */
extern T0* T52f85(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.finishtemporarytablestatement */
extern T0* T52f81(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.createtemporarytablestatement */
extern T0* T52f79(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.is_value_data_type_changed */
extern T1 T52f77(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.is_value_declared */
extern T1 T52f76(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.write_value */
extern void T51f251(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.create_value */
extern void T51f271(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.create_value_assign */
extern void T51f291(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.create_value_assign_outside_sp */
extern void T51f297(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.valuetablename */
extern T0* T51f78(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.createtemporaryvaluetable */
extern T1 T51f84(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.temporarytablessupported */
extern T1 T51f140(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.optional_create_value_declare */
extern void T51f290(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.create_value_declare */
extern void T51f296(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.create_value_declare_outside_sp */
extern void T51f299(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.create_value_alter_table */
extern void T51f301(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.create_value_create_table */
extern void T51f300(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.finishtemporaryvaluetablestatement */
extern T0* T51f85(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.finishtemporarytablestatement */
extern T0* T51f81(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.createtemporarytablestatement */
extern T0* T51f79(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.is_value_data_type_changed */
extern T1 T51f77(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.is_value_declared */
extern T1 T51f76(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.write_value */
extern void T50f251(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.create_value */
extern void T50f271(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.create_value_assign */
extern void T50f291(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.create_value_assign_outside_sp */
extern void T50f297(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.valuetablename */
extern T0* T50f78(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.createtemporaryvaluetable */
extern T1 T50f84(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.temporarytablessupported */
extern T1 T50f140(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.optional_create_value_declare */
extern void T50f290(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.create_value_declare */
extern void T50f296(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.create_value_declare_outside_sp */
extern void T50f299(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.create_value_alter_table */
extern void T50f301(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.create_value_create_table */
extern void T50f300(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.finishtemporaryvaluetablestatement */
extern T0* T50f85(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.finishtemporarytablestatement */
extern T0* T50f81(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.createtemporarytablestatement */
extern T0* T50f79(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.is_value_data_type_changed */
extern T1 T50f77(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.is_value_declared */
extern T1 T50f76(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.write_value */
extern void T49f281(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_value */
extern void T49f301(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_value_assign */
extern void T49f327(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_value_assign_outside_sp */
extern void T49f350(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.valuetablename */
extern T0* T49f101(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.createtemporaryvaluetable */
extern T1 T49f106(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_value_assign_inside_sp */
extern void T49f349(GE_context* ac, T0* C, T0* a1);
extern T1 T49f349ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.optional_create_value_declare */
extern void T49f326(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_value_declare */
extern void T49f348(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_value_declare_outside_sp */
extern void T49f352(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_value_alter_table */
extern void T49f354(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_value_create_table */
extern void T49f353(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.finishtemporaryvaluetablestatement */
extern T0* T49f107(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.createtemporarytablestatement */
extern T0* T49f102(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_value_declare_inside_sp */
extern void T49f351(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.is_value_declared */
extern T1 T49f100(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.write_value */
extern void T48f281(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_value */
extern void T48f301(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_value_assign */
extern void T48f327(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_value_assign_outside_sp */
extern void T48f349(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.valuetablename */
extern T0* T48f101(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.createtemporaryvaluetable */
extern T1 T48f106(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.temporarytablessupported */
extern T1 T48f167(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_value_assign_inside_sp */
extern void T48f348(GE_context* ac, T0* C, T0* a1);
extern T1 T48f348ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_INTERBASE6.optional_create_value_declare */
extern void T48f326(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_value_declare */
extern void T48f347(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_value_declare_outside_sp */
extern void T48f351(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_value_alter_table */
extern void T48f353(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_value_create_table */
extern void T48f352(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.finishtemporaryvaluetablestatement */
extern T0* T48f107(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.finishtemporarytablestatement */
extern T0* T48f104(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.createtemporarytablestatement */
extern T0* T48f102(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_value_declare_inside_sp */
extern void T48f350(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.is_value_declared */
extern T1 T48f100(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.write_value */
extern void T47f251(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.create_value */
extern void T47f271(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.create_value_assign */
extern void T47f291(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.create_value_assign_outside_sp */
extern void T47f297(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.valuetablename */
extern T0* T47f78(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.createtemporaryvaluetable */
extern T1 T47f84(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.temporarytablessupported */
extern T1 T47f140(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.optional_create_value_declare */
extern void T47f290(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.create_value_declare */
extern void T47f296(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.create_value_declare_outside_sp */
extern void T47f299(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.create_value_alter_table */
extern void T47f301(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.create_value_create_table */
extern void T47f300(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.finishtemporaryvaluetablestatement */
extern T0* T47f85(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.finishtemporarytablestatement */
extern T0* T47f81(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.createtemporarytablestatement */
extern T0* T47f79(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.is_value_data_type_changed */
extern T1 T47f77(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.is_value_declared */
extern T1 T47f76(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.write_value */
extern void T46f282(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.create_value */
extern void T46f302(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.create_value_assign */
extern void T46f328(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.create_value_assign_outside_sp */
extern void T46f348(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.valuetablename */
extern T0* T46f97(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.createtemporaryvaluetable */
extern T1 T46f102(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.create_value_assign_inside_sp */
extern void T46f347(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.optional_create_value_declare */
extern void T46f327(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.create_value_declare */
extern void T46f346(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.create_value_declare_outside_sp */
extern void T46f350(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.create_value_alter_table */
extern void T46f352(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.create_value_create_table */
extern void T46f351(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.finishtemporaryvaluetablestatement */
extern T0* T46f103(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.finishtemporarytablestatement */
extern T0* T46f100(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.createtemporarytablestatement */
extern T0* T46f98(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.create_value_declare_inside_sp */
extern void T46f349(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.is_value_declared */
extern T1 T46f96(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.write_value */
extern void T45f253(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.create_value */
extern void T45f273(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.create_value_assign */
extern void T45f293(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.create_value_assign_outside_sp */
extern void T45f299(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.valuetablename */
extern T0* T45f78(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.createtemporaryvaluetable */
extern T1 T45f84(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.temporarytablessupported */
extern T1 T45f141(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.optional_create_value_declare */
extern void T45f292(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.create_value_declare */
extern void T45f298(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.create_value_declare_outside_sp */
extern void T45f301(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.create_value_alter_table */
extern void T45f303(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.create_value_create_table */
extern void T45f302(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.finishtemporaryvaluetablestatement */
extern T0* T45f85(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.finishtemporarytablestatement */
extern T0* T45f81(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.createtemporarytablestatement */
extern T0* T45f79(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.is_value_data_type_changed */
extern T1 T45f77(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.is_value_declared */
extern T1 T45f76(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.write_value */
extern void T44f252(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.create_value */
extern void T44f272(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.create_value_assign */
extern void T44f292(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.create_value_assign_outside_sp */
extern void T44f298(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.valuetablename */
extern T0* T44f78(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.createtemporaryvaluetable */
extern T1 T44f84(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.temporarytablessupported */
extern T1 T44f140(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.optional_create_value_declare */
extern void T44f291(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.create_value_declare */
extern void T44f297(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.create_value_declare_outside_sp */
extern void T44f300(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.create_value_alter_table */
extern void T44f302(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.create_value_create_table */
extern void T44f301(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.finishtemporaryvaluetablestatement */
extern T0* T44f85(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.finishtemporarytablestatement */
extern T0* T44f81(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.createtemporarytablestatement */
extern T0* T44f79(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.is_value_data_type_changed */
extern T1 T44f77(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.is_value_declared */
extern T1 T44f76(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.write_value */
extern void T43f252(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.create_value */
extern void T43f272(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.create_value_assign */
extern void T43f292(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.create_value_assign_outside_sp */
extern void T43f298(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.valuetablename */
extern T0* T43f78(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.createtemporaryvaluetable */
extern T1 T43f84(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.temporarytablessupported */
extern T1 T43f140(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.optional_create_value_declare */
extern void T43f291(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.create_value_declare */
extern void T43f297(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.create_value_declare_outside_sp */
extern void T43f300(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.create_value_alter_table */
extern void T43f302(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.create_value_create_table */
extern void T43f301(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.finishtemporaryvaluetablestatement */
extern T0* T43f85(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.finishtemporarytablestatement */
extern T0* T43f81(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.createtemporarytablestatement */
extern T0* T43f79(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.is_value_data_type_changed */
extern T1 T43f77(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.is_value_declared */
extern T1 T43f76(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.write_select_value */
extern void T66f267(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.create_select_value */
extern void T66f287(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.create_select_value_outside_sp */
extern void T66f317(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.do_create_select_value */
extern void T66f336(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.create_select_value_inside_sp */
extern void T66f316(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.write_select_value */
extern void T65f266(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.create_select_value */
extern void T65f286(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.create_select_value_outside_sp */
extern void T65f316(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.do_create_select_value */
extern void T65f335(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.create_select_value_inside_sp */
extern void T65f315(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.write_select_value */
extern void T64f266(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.create_select_value */
extern void T64f286(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.create_select_value_outside_sp */
extern void T64f316(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.do_create_select_value */
extern void T64f335(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.create_select_value_inside_sp */
extern void T64f315(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.write_select_value */
extern void T63f266(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.create_select_value */
extern void T63f286(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.create_select_value_outside_sp */
extern void T63f316(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.do_create_select_value */
extern void T63f333(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.create_select_value_inside_sp */
extern void T63f315(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.write_select_value */
extern void T62f265(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_select_value */
extern void T62f285(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_select_value_outside_sp */
extern void T62f306(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.format */
extern T0* T62f65(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_72.write_select_value */
extern void T61f289(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_select_value */
extern void T61f309(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_select_value_outside_sp */
extern void T61f335(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.format */
extern T0* T61f79(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_select_value_inside_sp */
extern void T61f334(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.write_select_value */
extern void T60f292(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_select_value */
extern void T60f312(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_select_value_outside_sp */
extern void T60f338(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.format */
extern T0* T60f79(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_select_value_inside_sp */
extern void T60f337(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.write_select_value */
extern void T59f293(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_select_value */
extern void T59f313(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_select_value_outside_sp */
extern void T59f339(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.format */
extern T0* T59f79(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_select_value_inside_sp */
extern void T59f338(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.write_select_value */
extern void T58f293(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_select_value */
extern void T58f313(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_select_value_outside_sp */
extern void T58f339(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.format */
extern T0* T58f79(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_select_value_inside_sp */
extern void T58f338(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.write_select_value */
extern void T57f293(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_select_value */
extern void T57f313(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_select_value_outside_sp */
extern void T57f339(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.format */
extern T0* T57f79(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_select_value_inside_sp */
extern void T57f338(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.write_select_value */
extern void T56f287(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.create_select_value */
extern void T56f307(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.create_select_value_outside_sp */
extern void T56f334(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.format */
extern T0* T56f67(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_ORACLE.create_select_value_inside_sp */
extern void T56f333(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.write_select_value */
extern void T55f253(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.create_select_value */
extern void T55f273(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.create_select_value_outside_sp */
extern void T55f293(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.format */
extern T0* T55f65(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL4.write_select_value */
extern void T54f253(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.create_select_value */
extern void T54f273(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.create_select_value_outside_sp */
extern void T54f293(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.format */
extern T0* T54f65(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL5.write_select_value */
extern void T53f278(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.create_select_value */
extern void T53f298(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.create_select_value_outside_sp */
extern void T53f325(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.create_select_value_inside_sp */
extern void T53f324(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.write_select_value */
extern void T52f252(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.create_select_value */
extern void T52f272(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.create_select_value_outside_sp */
extern void T52f292(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.format */
extern T0* T52f65(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSACCESS2000.write_select_value */
extern void T51f252(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.create_select_value */
extern void T51f272(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.create_select_value_outside_sp */
extern void T51f292(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.format */
extern T0* T51f65(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MSACCESS97.write_select_value */
extern void T50f252(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.create_select_value */
extern void T50f272(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.create_select_value_outside_sp */
extern void T50f292(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.format */
extern T0* T50f65(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.write_select_value */
extern void T49f282(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_select_value */
extern void T49f302(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_select_value_outside_sp */
extern void T49f329(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.format */
extern T0* T49f68(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_select_value_inside_sp */
extern void T49f328(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.write_select_value */
extern void T48f282(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_select_value */
extern void T48f302(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_select_value_outside_sp */
extern void T48f329(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.format */
extern T0* T48f68(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_select_value_inside_sp */
extern void T48f328(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.write_select_value */
extern void T47f252(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.create_select_value */
extern void T47f272(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.create_select_value_outside_sp */
extern void T47f292(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.format */
extern T0* T47f65(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_DB2_71.write_select_value */
extern void T46f283(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.create_select_value */
extern void T46f303(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.create_select_value_outside_sp */
extern void T46f330(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.format */
extern T0* T46f67(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_DB2_71.create_select_value_inside_sp */
extern void T46f329(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.write_select_value */
extern void T45f254(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.create_select_value */
extern void T45f274(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.create_select_value_outside_sp */
extern void T45f294(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.format */
extern T0* T45f65(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_BASIC.write_select_value */
extern void T44f253(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.create_select_value */
extern void T44f273(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.create_select_value_outside_sp */
extern void T44f293(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.format */
extern T0* T44f65(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_ANSI.write_select_value */
extern void T43f253(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.create_select_value */
extern void T43f273(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.create_select_value_outside_sp */
extern void T43f293(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.format */
extern T0* T43f65(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL65.write_select */
extern void T66f268(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.write_select */
extern void T65f267(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.write_select */
extern void T64f267(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.write_select */
extern void T63f267(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.write_select */
extern void T62f266(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.write_select */
extern void T61f290(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.write_select */
extern void T60f293(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.write_select */
extern void T59f294(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.write_select */
extern void T58f294(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.write_select */
extern void T57f294(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.write_select */
extern void T56f288(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.write_select */
extern void T55f254(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.write_select */
extern void T54f254(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.write_select */
extern void T53f279(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.write_select */
extern void T52f253(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.write_select */
extern void T51f253(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.write_select */
extern void T50f253(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.write_select */
extern void T49f283(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.write_select */
extern void T48f283(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.write_select */
extern void T47f253(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.write_select */
extern void T46f284(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.write_select */
extern void T45f255(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.write_select */
extern void T44f254(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.write_select */
extern void T43f254(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.write_extend */
extern void T66f269(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.create_extend_create_index */
extern void T66f290(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.format */
extern T0* T66f69(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL65.extension_index_name */
extern T0* T66f68(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.createextendindex */
extern T1 T66f38(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.extendindexsupported */
extern T1 T66f139(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.create_extend */
extern void T66f289(GE_context* ac, T0* C, T0* a1);
extern T1 T66f289ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_TSQL65.tabtab */
extern T0* T66f67(GE_context* ac, T0* C);
/* detachable XPLAIN_EXTENSION.q_sql_insert_name */
extern T0* T160f13(GE_context* ac, T0* C, T0* a1, T0* a2);
/* detachable XPLAIN_EXTENSION.sql_insert_name */
extern T0* T160f23(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] XPLAIN_EXTENSION_EXPRESSION_EXPRESSION.outer_sqlvalue */
extern T0* T313f6(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_EXTENSION_EXPRESSION_EXPRESSION.sqlfromaliasname */
extern T0* T313f5(GE_context* ac, T0* C);
/* [detachable] XPLAIN_EXTENSION_FUNCTION_EXPRESSION.add_to_join */
extern void T315f24(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] JOIN_LIST.add_join_to_aggregate */
extern void T352f17(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3, T1 a4);
/* detachable XPLAIN_SELECTION_FUNCTION.add_to_join */
extern void T157f13(GE_context* ac, T0* C, T0* a1, T0* a2);
/* detachable XPLAIN_SELECTION_FUNCTION.add_to_join */
extern void T157f13p1(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] DS_LINKED_LIST [[attached] JOIN_TREE_NODE].last */
extern T0* T439f5(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_ATTRIBUTE].first */
extern T0* T361f14(GE_context* ac, T0* C);
/* [detachable] JOIN_LIST.copy_role_to_next */
extern void T352f20(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_ATTRIBUTE_NAME.set_role */
extern void T141f20(GE_context* ac, T0* C, T0* a1);
/* [detachable] JOIN_LIST.revert_alist */
extern T0* T352f12(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_EXTENSION_EXPRESSION_EXPRESSION.add_to_join */
extern void T313f16(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL65.create_extend_view */
extern void T66f288(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.can_write_extend_as_view */
extern T1 T66f37(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.can_write_extend_as_view */
extern T1 T66f37p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.write_extend */
extern void T65f268(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.create_extend_create_index */
extern void T65f289(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.format */
extern T0* T65f69(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL70.extension_index_name */
extern T0* T65f68(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.createextendindex */
extern T1 T65f38(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.extendindexsupported */
extern T1 T65f138(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.create_extend */
extern void T65f288(GE_context* ac, T0* C, T0* a1);
extern T1 T65f288ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_TSQL70.tabtab */
extern T0* T65f67(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.create_extend_view */
extern void T65f287(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.can_write_extend_as_view */
extern T1 T65f37(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.can_write_extend_as_view */
extern T1 T65f37p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.write_extend */
extern void T64f268(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.create_extend_create_index */
extern void T64f289(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.format */
extern T0* T64f69(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL2000.extension_index_name */
extern T0* T64f68(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.createextendindex */
extern T1 T64f38(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.extendindexsupported */
extern T1 T64f138(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.create_extend */
extern void T64f288(GE_context* ac, T0* C, T0* a1);
extern T1 T64f288ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_TSQL2000.tabtab */
extern T0* T64f67(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.create_extend_view */
extern void T64f287(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.can_write_extend_as_view */
extern T1 T64f37(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.can_write_extend_as_view */
extern T1 T64f37p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.write_extend */
extern void T63f268(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.create_extend_create_index */
extern void T63f289(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.format */
extern T0* T63f69(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL2016.extension_index_name */
extern T0* T63f68(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.createextendindex */
extern T1 T63f38(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.extendindexsupported */
extern T1 T63f138(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.create_extend */
extern void T63f288(GE_context* ac, T0* C, T0* a1);
extern T1 T63f288ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_TSQL2016.tabtab */
extern T0* T63f67(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.create_extend_view */
extern void T63f287(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.can_write_extend_as_view */
extern T1 T63f37(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.can_write_extend_as_view */
extern T1 T63f37p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.write_extend */
extern void T62f267(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_extend_create_index */
extern void T62f288(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.extension_index_name */
extern T0* T62f64(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.createextendindex */
extern T1 T62f38(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.extendindexsupported */
extern T1 T62f126(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_extend */
extern void T62f287(GE_context* ac, T0* C, T0* a1);
extern T1 T62f287ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_SQLITE_3.coalescesupported */
extern T1 T62f63(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_extend_create_table */
extern void T62f308(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.temporarytableprimarykey */
extern T0* T62f81(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.primarykeyconstraint */
extern T0* T62f143(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.drop_temporary_table_if_exist */
extern void T62f307(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_extend_view */
extern void T62f286(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.can_write_extend_as_view */
extern T1 T62f37(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.write_extend */
extern void T61f291(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_extend */
extern void T61f311(GE_context* ac, T0* C, T0* a1);
extern T1 T61f311ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_PGSQL_72.coalescesupported */
extern T1 T61f38(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_extend_create_table */
extern void T61f337(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.end_if */
extern void T61f357(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_extend_create_index */
extern void T61f356(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.extension_index_name */
extern T0* T61f108(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.createextendindex */
extern T1 T61f101(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.extendindexsupported */
extern T1 T61f175(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_extend_create_table */
extern void T61f337p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.temporarytableprimarykey */
extern T0* T61f103(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.primarykeyconstraint */
extern T0* T61f177(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.safe_sql */
extern T0* T61f88(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.safe_string */
extern T0* T61f165(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.drop_temporary_table_if_exist */
extern void T61f336(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_extend_view */
extern void T61f310(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.can_write_extend_as_view */
extern T1 T61f37(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.can_write_extend_as_view */
extern T1 T61f37p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.write_extend */
extern void T60f294(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_extend */
extern void T60f314(GE_context* ac, T0* C, T0* a1);
extern T1 T60f314ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_PGSQL_73.coalescesupported */
extern T1 T60f38(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_extend_create_table */
extern void T60f340(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.end_if */
extern void T60f360(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_extend_create_index */
extern void T60f359(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.extension_index_name */
extern T0* T60f108(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.createextendindex */
extern T1 T60f101(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.extendindexsupported */
extern T1 T60f176(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_extend_create_table */
extern void T60f340p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.temporarytableprimarykey */
extern T0* T60f103(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.primarykeyconstraint */
extern T0* T60f178(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.safe_sql */
extern T0* T60f88(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.safe_string */
extern T0* T60f168(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.drop_temporary_table_if_exist */
extern void T60f339(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_extend_view */
extern void T60f313(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.can_write_extend_as_view */
extern T1 T60f37(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.can_write_extend_as_view */
extern T1 T60f37p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.write_extend */
extern void T59f295(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_extend */
extern void T59f315(GE_context* ac, T0* C, T0* a1);
extern T1 T59f315ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_PGSQL_81.coalescesupported */
extern T1 T59f38(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_extend_create_table */
extern void T59f341(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.end_if */
extern void T59f361(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_extend_create_index */
extern void T59f360(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.extension_index_name */
extern T0* T59f109(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.createextendindex */
extern T1 T59f102(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.extendindexsupported */
extern T1 T59f176(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_extend_create_table */
extern void T59f341p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.temporarytableprimarykey */
extern T0* T59f104(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.primarykeyconstraint */
extern T0* T59f178(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.safe_sql */
extern T0* T59f88(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.drop_temporary_table_if_exist */
extern void T59f340(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_extend_view */
extern void T59f314(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.can_write_extend_as_view */
extern T1 T59f37(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.can_write_extend_as_view */
extern T1 T59f37p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.write_extend */
extern void T58f295(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_extend */
extern void T58f315(GE_context* ac, T0* C, T0* a1);
extern T1 T58f315ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_PGSQL_82.coalescesupported */
extern T1 T58f38(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_extend_create_table */
extern void T58f341(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.end_if */
extern void T58f361(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_extend_create_index */
extern void T58f360(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.extension_index_name */
extern T0* T58f109(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.createextendindex */
extern T1 T58f102(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.extendindexsupported */
extern T1 T58f176(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_extend_create_table */
extern void T58f341p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.temporarytableprimarykey */
extern T0* T58f104(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.primarykeyconstraint */
extern T0* T58f178(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.safe_sql */
extern T0* T58f88(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.drop_temporary_table_if_exist */
extern void T58f340(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_extend_view */
extern void T58f314(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.can_write_extend_as_view */
extern T1 T58f37(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.can_write_extend_as_view */
extern T1 T58f37p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.write_extend */
extern void T57f295(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_extend */
extern void T57f315(GE_context* ac, T0* C, T0* a1);
extern T1 T57f315ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_PGSQL_95.coalescesupported */
extern T1 T57f38(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_extend_create_table */
extern void T57f341(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.end_if */
extern void T57f359(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_extend_create_index */
extern void T57f358(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.extension_index_name */
extern T0* T57f108(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.createextendindex */
extern T1 T57f101(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.extendindexsupported */
extern T1 T57f173(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_extend_create_table */
extern void T57f341p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.temporarytableprimarykey */
extern T0* T57f103(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.primarykeyconstraint */
extern T0* T57f175(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.safe_sql */
extern T0* T57f88(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.drop_temporary_table_if_exist */
extern void T57f340(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_extend_view */
extern void T57f314(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.can_write_extend_as_view */
extern T1 T57f37(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.can_write_extend_as_view */
extern T1 T57f37p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.write_extend */
extern void T56f289(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.create_extend_create_index */
extern void T56f310(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.extension_index_name */
extern T0* T56f66(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.createextendindex */
extern T1 T56f38(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.extendindexsupported */
extern T1 T56f148(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.create_extend */
extern void T56f309(GE_context* ac, T0* C, T0* a1);
extern T1 T56f309ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_ORACLE.coalescesupported */
extern T1 T56f39(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.create_extend_create_table */
extern void T56f336(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.finishtemporarytablestatement */
extern T0* T56f104(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.temporarytableprimarykey */
extern T0* T56f103(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.primarykeyconstraint */
extern T0* T56f168(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.drop_temporary_table_if_exist */
extern void T56f335(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.create_extend_view */
extern void T56f308(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.can_write_extend_as_view */
extern T1 T56f37(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.can_write_extend_as_view */
extern T1 T56f37p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.write_extend */
extern void T55f255(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.create_extend_create_index */
extern void T55f276(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.extension_index_name */
extern T0* T55f64(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.createextendindex */
extern T1 T55f38(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.extendindexsupported */
extern T1 T55f125(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.create_extend */
extern void T55f275(GE_context* ac, T0* C, T0* a1);
extern T1 T55f275ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_MYSQL322.coalescesupported */
extern T1 T55f63(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.create_extend_create_table */
extern void T55f295(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.temporarytableprimarykey */
extern T0* T55f80(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.primarykeyconstraint */
extern T0* T55f141(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.drop_temporary_table_if_exist */
extern void T55f294(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.create_extend_view */
extern void T55f274(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.can_write_extend_as_view */
extern T1 T55f37(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.write_extend */
extern void T54f255(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.create_extend_create_index */
extern void T54f276(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.extension_index_name */
extern T0* T54f64(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.createextendindex */
extern T1 T54f38(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.extendindexsupported */
extern T1 T54f125(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.create_extend */
extern void T54f275(GE_context* ac, T0* C, T0* a1);
extern T1 T54f275ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_MYSQL4.coalescesupported */
extern T1 T54f63(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.create_extend_create_table */
extern void T54f295(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.temporarytableprimarykey */
extern T0* T54f80(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.primarykeyconstraint */
extern T0* T54f141(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.drop_temporary_table_if_exist */
extern void T54f294(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.create_extend_view */
extern void T54f274(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.can_write_extend_as_view */
extern T1 T54f37(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.write_extend */
extern void T53f280(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.create_extend_create_index */
extern void T53f301(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.extension_index_name */
extern T0* T53f67(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.createextendindex */
extern T1 T53f38(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.extendindexsupported */
extern T1 T53f140(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.create_extend */
extern void T53f300(GE_context* ac, T0* C, T0* a1);
extern T1 T53f300ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_MYSQL5.coalescesupported */
extern T1 T53f39(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.create_extend_create_table */
extern void T53f327(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.finishtemporarytablestatement */
extern T0* T53f100(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.temporarytableprimarykey */
extern T0* T53f99(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.primarykeyconstraint */
extern T0* T53f161(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.createtemporarytablestatement */
extern T0* T53f98(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.temporarytablessupported */
extern T1 T53f160(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.drop_temporary_table_if_exist */
extern void T53f326(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.create_extend_view */
extern void T53f299(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.can_write_extend_as_view */
extern T1 T53f37(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.can_write_extend_as_view */
extern T1 T53f37p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.write_extend */
extern void T52f254(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.create_extend_create_index */
extern void T52f275(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.extension_index_name */
extern T0* T52f64(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.createextendindex */
extern T1 T52f38(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.extendindexsupported */
extern T1 T52f125(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.create_extend */
extern void T52f274(GE_context* ac, T0* C, T0* a1);
extern T1 T52f274ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_MSQL.coalescesupported */
extern T1 T52f63(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.create_extend_create_table */
extern void T52f294(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.temporarytableprimarykey */
extern T0* T52f80(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.primarykeyconstraint */
extern T0* T52f141(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.drop_temporary_table_if_exist */
extern void T52f293(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.create_extend_view */
extern void T52f273(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.can_write_extend_as_view */
extern T1 T52f37(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.write_extend */
extern void T51f254(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.create_extend_create_index */
extern void T51f275(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.extension_index_name */
extern T0* T51f64(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.createextendindex */
extern T1 T51f38(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.extendindexsupported */
extern T1 T51f125(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.create_extend */
extern void T51f274(GE_context* ac, T0* C, T0* a1);
extern T1 T51f274ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_MSACCESS2000.coalescesupported */
extern T1 T51f63(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.create_extend_create_table */
extern void T51f294(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.temporarytableprimarykey */
extern T0* T51f80(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.primarykeyconstraint */
extern T0* T51f141(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.drop_temporary_table_if_exist */
extern void T51f293(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.create_extend_view */
extern void T51f273(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.can_write_extend_as_view */
extern T1 T51f37(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.write_extend */
extern void T50f254(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.create_extend_create_index */
extern void T50f275(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.extension_index_name */
extern T0* T50f64(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.createextendindex */
extern T1 T50f38(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.extendindexsupported */
extern T1 T50f125(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.create_extend */
extern void T50f274(GE_context* ac, T0* C, T0* a1);
extern T1 T50f274ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_MSACCESS97.coalescesupported */
extern T1 T50f63(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.create_extend_create_table */
extern void T50f294(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.temporarytableprimarykey */
extern T0* T50f80(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.primarykeyconstraint */
extern T0* T50f141(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.drop_temporary_table_if_exist */
extern void T50f293(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.create_extend_view */
extern void T50f273(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.can_write_extend_as_view */
extern T1 T50f37(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.write_extend */
extern void T49f284(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_extend_create_index */
extern void T49f305(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_extend_create_index */
extern void T49f305p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.extension_index_name */
extern T0* T49f67(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.commit */
extern void T49f332(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.createextendindex */
extern T1 T49f39(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.extendindexsupported */
extern T1 T49f147(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_extend */
extern void T49f304(GE_context* ac, T0* C, T0* a1);
extern T1 T49f304ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.coalescesupported */
extern T1 T49f40(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_extend_create_table */
extern void T49f331(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.temporarytableprimarykey */
extern T0* T49f103(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.primarykeyconstraint */
extern T0* T49f168(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.drop_temporary_table_if_exist */
extern void T49f330(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_extend_view */
extern void T49f303(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.can_write_extend_as_view */
extern T1 T49f38(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.can_write_extend_as_view */
extern T1 T49f38p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.write_extend */
extern void T48f284(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_extend_create_index */
extern void T48f305(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.extension_index_name */
extern T0* T48f67(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.createextendindex */
extern T1 T48f39(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.extendindexsupported */
extern T1 T48f147(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_extend */
extern void T48f304(GE_context* ac, T0* C, T0* a1);
extern T1 T48f304ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_INTERBASE6.coalescesupported */
extern T1 T48f40(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_extend_create_table */
extern void T48f331(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.temporarytableprimarykey */
extern T0* T48f103(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.primarykeyconstraint */
extern T0* T48f168(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.drop_temporary_table_if_exist */
extern void T48f330(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_extend_view */
extern void T48f303(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.can_write_extend_as_view */
extern T1 T48f38(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.can_write_extend_as_view */
extern T1 T48f38p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.write_extend */
extern void T47f254(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.create_extend_create_index */
extern void T47f275(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.extension_index_name */
extern T0* T47f64(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.createextendindex */
extern T1 T47f38(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.extendindexsupported */
extern T1 T47f125(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.create_extend */
extern void T47f274(GE_context* ac, T0* C, T0* a1);
extern T1 T47f274ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_FOXPRO.coalescesupported */
extern T1 T47f63(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.create_extend_create_table */
extern void T47f294(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.temporarytableprimarykey */
extern T0* T47f80(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.primarykeyconstraint */
extern T0* T47f141(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.drop_temporary_table_if_exist */
extern void T47f293(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.create_extend_view */
extern void T47f273(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.can_write_extend_as_view */
extern T1 T47f37(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.write_extend */
extern void T46f285(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.create_extend_create_index */
extern void T46f306(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.extension_index_name */
extern T0* T46f66(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.createextendindex */
extern T1 T46f38(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.extendindexsupported */
extern T1 T46f144(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.create_extend */
extern void T46f305(GE_context* ac, T0* C, T0* a1);
extern T1 T46f305ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_DB2_71.coalescesupported */
extern T1 T46f39(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.create_extend_create_table */
extern void T46f332(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.temporarytableprimarykey */
extern T0* T46f99(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.drop_temporary_table_if_exist */
extern void T46f331(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.create_extend_view */
extern void T46f304(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.can_write_extend_as_view */
extern T1 T46f37(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.can_write_extend_as_view */
extern T1 T46f37p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.write_extend */
extern void T45f256(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.create_extend_create_index */
extern void T45f277(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.extension_index_name */
extern T0* T45f64(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.createextendindex */
extern T1 T45f38(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.extendindexsupported */
extern T1 T45f125(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.create_extend */
extern void T45f276(GE_context* ac, T0* C, T0* a1);
extern T1 T45f276ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_DB2_6.coalescesupported */
extern T1 T45f63(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.create_extend_create_table */
extern void T45f296(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.temporarytableprimarykey */
extern T0* T45f80(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.primarykeyconstraint */
extern T0* T45f142(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.drop_temporary_table_if_exist */
extern void T45f295(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.create_extend_view */
extern void T45f275(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.can_write_extend_as_view */
extern T1 T45f37(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.write_extend */
extern void T44f255(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.create_extend_create_index */
extern void T44f276(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.extension_index_name */
extern T0* T44f64(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.createextendindex */
extern T1 T44f38(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.extendindexsupported */
extern T1 T44f125(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.create_extend */
extern void T44f275(GE_context* ac, T0* C, T0* a1);
extern T1 T44f275ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_BASIC.coalescesupported */
extern T1 T44f63(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.create_extend_create_table */
extern void T44f295(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.temporarytableprimarykey */
extern T0* T44f80(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.primarykeyconstraint */
extern T0* T44f141(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.drop_temporary_table_if_exist */
extern void T44f294(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.create_extend_view */
extern void T44f274(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.can_write_extend_as_view */
extern T1 T44f37(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.write_extend */
extern void T43f255(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.create_extend_create_index */
extern void T43f276(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.extension_index_name */
extern T0* T43f64(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.createextendindex */
extern T1 T43f38(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.extendindexsupported */
extern T1 T43f125(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.create_extend */
extern void T43f275(GE_context* ac, T0* C, T0* a1);
extern T1 T43f275ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_ANSI.coalescesupported */
extern T1 T43f63(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.create_extend_create_table */
extern void T43f295(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.temporarytableprimarykey */
extern T0* T43f80(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.primarykeyconstraint */
extern T0* T43f141(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.drop_temporary_table_if_exist */
extern void T43f294(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.create_extend_view */
extern void T43f274(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.can_write_extend_as_view */
extern T1 T43f37(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.echo */
extern void T66f270(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.echo */
extern void T65f269(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.echo */
extern void T64f269(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.echo */
extern void T63f269(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.echo */
extern void T62f268(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.echo */
extern void T61f292(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.echo */
extern void T60f295(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.echo */
extern void T59f296(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.echo */
extern void T58f296(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.echo */
extern void T57f296(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.echo */
extern void T56f290(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.echo */
extern void T55f256(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.echo */
extern void T54f256(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.echo */
extern void T53f281(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.echo */
extern void T52f255(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.echo */
extern void T51f255(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.echo */
extern void T50f255(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.echo */
extern void T49f285(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.echo */
extern void T48f285(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.echo */
extern void T47f255(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.echo */
extern void T46f286(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.echo */
extern void T45f257(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.echo */
extern void T44f256(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.echo */
extern void T43f256(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.write_echo */
extern void T66f271(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.create_echo */
extern void T66f291(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.write_echo */
extern void T65f270(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.create_echo */
extern void T65f290(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.write_echo */
extern void T64f270(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.create_echo */
extern void T64f290(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.write_echo */
extern void T63f270(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.create_echo */
extern void T63f290(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.write_echo */
extern void T62f269(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_echo */
extern void T62f289(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.write_echo */
extern void T61f293(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_echo */
extern void T61f312(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.write_echo */
extern void T60f296(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_echo */
extern void T60f315(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.write_echo */
extern void T59f297(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_echo */
extern void T59f316(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.write_echo */
extern void T58f297(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_echo */
extern void T58f316(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.write_echo */
extern void T57f297(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_echo */
extern void T57f316(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.write_echo */
extern void T56f291(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.create_echo */
extern void T56f311(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.write_echo */
extern void T55f257(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.create_echo */
extern void T55f277(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.write_echo */
extern void T54f257(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.create_echo */
extern void T54f277(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.write_echo */
extern void T53f282(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.create_echo */
extern void T53f302(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.write_echo */
extern void T52f256(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.create_echo */
extern void T52f276(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.write_echo */
extern void T51f256(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.create_echo */
extern void T51f276(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.write_echo */
extern void T50f256(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.create_echo */
extern void T50f276(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.write_echo */
extern void T49f286(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_echo */
extern void T49f306(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.write_echo */
extern void T48f286(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_echo */
extern void T48f306(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.write_echo */
extern void T47f256(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.create_echo */
extern void T47f276(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.write_echo */
extern void T46f287(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.create_echo */
extern void T46f307(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.write_echo */
extern void T45f258(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.create_echo */
extern void T45f278(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.write_echo */
extern void T44f257(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.create_echo */
extern void T44f277(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.write_echo */
extern void T43f257(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.create_echo */
extern void T43f277(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.write_procedure */
extern void T66f272(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PROCEDURE.warn_about_unused_parameters */
extern void T333f22(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_PARAMETER].forth */
extern void T358f13(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PROCEDURE.std */
extern T0* T333f15(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_PARAMETER].item_for_iteration */
extern T0* T358f9(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_PARAMETER].cursor_item */
extern T0* T358f7(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_PARAMETER].after */
extern T1 T358f8(GE_context* ac, T0* C);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_PARAMETER].cursor_after */
extern T1 T358f6(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_LINKED_LIST [[attached] XPLAIN_PARAMETER].start */
extern void T358f12(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.create_procedure */
extern void T66f293(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PROCEDURE.cleanup_after_write */
extern void T333f21(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.sp_end */
extern void T66f325(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.sp_body_statements */
extern void T66f324(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.sp_body_start */
extern void T66f323(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.sp_user_declaration */
extern void T66f322(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.storedprocedureuserdeclarationbeforebody */
extern T1 T66f76(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.sp_header_end */
extern void T66f321(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.sp_user_result */
extern void T66f320(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.sp_result_parameter */
extern void T66f319(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.storedprocedureparamlistend */
extern T0* T66f75(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.storedprocedureparamliststart */
extern T0* T66f71(GE_context* ac, T0* C);
/* [detachable] XPLAIN_PROCEDURE.quoted_name */
extern T0* T333f14(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.sp_start */
extern void T66f318(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.sp_start */
extern void T66f318p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_HASH_TABLE [[attached] XPLAIN_VALUE, [attached] STRING_8].make */
extern T0* T116c38(GE_context* ac, T6 a1);
/* [detachable] XPLAIN_PROCEDURE.optimize_statements */
extern void T333f20(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.coalescesupported */
extern T1 T66f39(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.conditional_drop_procedure */
extern void T66f292(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.write_procedure */
extern void T65f271(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.create_procedure */
extern void T65f292(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.sp_end */
extern void T65f325(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.sp_body_statements */
extern void T65f324(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.sp_body_start */
extern void T65f323(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.sp_user_declaration */
extern void T65f322(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.storedprocedureuserdeclarationbeforebody */
extern T1 T65f76(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.sp_header_end */
extern void T65f321(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.sp_user_result */
extern void T65f320(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.sp_result_parameter */
extern void T65f319(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.storedprocedureparamlistend */
extern T0* T65f75(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.storedprocedureparamliststart */
extern T0* T65f71(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.sp_start */
extern void T65f318(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.sp_start */
extern void T65f318p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.coalescesupported */
extern T1 T65f39(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.conditional_drop_procedure */
extern void T65f291(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.do_drop_procedure */
extern void T65f317(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.write_procedure */
extern void T64f271(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.create_procedure */
extern void T64f292(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.sp_end */
extern void T64f325(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.sp_body_statements */
extern void T64f324(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.sp_body_start */
extern void T64f323(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.sp_user_declaration */
extern void T64f322(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.storedprocedureuserdeclarationbeforebody */
extern T1 T64f76(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.sp_header_end */
extern void T64f321(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.sp_user_result */
extern void T64f320(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.sp_result_parameter */
extern void T64f319(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.storedprocedureparamlistend */
extern T0* T64f75(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.storedprocedureparamliststart */
extern T0* T64f71(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.sp_start */
extern void T64f318(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.sp_start */
extern void T64f318p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.coalescesupported */
extern T1 T64f39(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.conditional_drop_procedure */
extern void T64f291(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.do_drop_procedure */
extern void T64f317(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.write_procedure */
extern void T63f271(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.create_procedure */
extern void T63f292(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.sp_end */
extern void T63f324(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.sp_body_statements */
extern void T63f323(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.sp_body_start */
extern void T63f322(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.sp_user_declaration */
extern void T63f321(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.storedprocedureuserdeclarationbeforebody */
extern T1 T63f76(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.sp_header_end */
extern void T63f320(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.sp_user_result */
extern void T63f319(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.sp_result_parameter */
extern void T63f318(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.storedprocedureparamlistend */
extern T0* T63f75(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.storedprocedureparamliststart */
extern T0* T63f71(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.sp_start */
extern void T63f317(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.sp_start */
extern void T63f317p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.coalescesupported */
extern T1 T63f39(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.conditional_drop_procedure */
extern void T63f291(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_STDOUT_FILE.put_line */
extern void T37f8(GE_context* ac, T0* C, T0* a1);
/* [detachable] KL_STDOUT_FILE.put_new_line */
extern void T37f9(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.write_procedure */
extern void T62f270(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.write_procedure */
extern void T61f294(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_procedure */
extern void T61f314(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.sp_end */
extern void T61f345(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.plpgsql_block_demarcation */
extern T0* T61f72(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.sp_body_statements */
extern void T61f344(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.sp_body_start */
extern void T61f343(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.sp_user_declaration */
extern void T61f342(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.sp_type_name */
extern T0* T61f105(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PROCEDURE.returns_rows */
extern T1 T333f12(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.sp_user_declaration */
extern void T61f342p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PARAMETER.sqlcolumnidentifier */
extern T0* T170f5(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.storedprocedureuserdeclarationbeforebody */
extern T1 T61f71(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.sp_header_end */
extern void T61f341(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.sp_user_result */
extern void T61f340(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.sp_result_parameter */
extern void T61f339(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.storedprocedureparamlistend */
extern T0* T61f70(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.storedprocedureparamliststart */
extern T0* T61f66(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.storedprocedureparamliststartrequired */
extern T1 T61f65(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.sp_start */
extern void T61f338(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.conditional_drop_procedure */
extern void T61f313(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.write_procedure */
extern void T60f297(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_procedure */
extern void T60f317(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.sp_end */
extern void T60f348(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.plpgsql_block_demarcation */
extern T0* T60f72(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.sp_body_statements */
extern void T60f347(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.sp_body_start */
extern void T60f346(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.sp_user_declaration */
extern void T60f345(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.sp_type_name */
extern T0* T60f105(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.sp_user_declaration */
extern void T60f345p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.storedprocedureuserdeclarationbeforebody */
extern T1 T60f71(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.sp_header_end */
extern void T60f344(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.sp_user_result */
extern void T60f343(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.sp_result_parameter */
extern void T60f342(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PROCEDURE.returns_value */
extern T1 T333f13(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.storedprocedureparamlistend */
extern T0* T60f70(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.storedprocedureparamliststart */
extern T0* T60f66(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.storedprocedureparamliststartrequired */
extern T1 T60f65(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.sp_start */
extern void T60f341(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_sp_type */
extern void T60f362(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PROCEDURE.sp_function_type */
extern T0* T333f11(GE_context* ac, T0* C, T0* a1, T1 a2);
/* [detachable] SQL_GENERATOR_PGSQL_73.drop_sp_type */
extern void T60f361(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.conditional_drop_procedure */
extern void T60f316(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.write_procedure */
extern void T59f298(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_procedure */
extern void T59f318(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.sp_end */
extern void T59f349(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.sp_body_statements */
extern void T59f348(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.sp_body_start */
extern void T59f347(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.sp_user_declaration */
extern void T59f346(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.sp_type_name */
extern T0* T59f106(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.sp_user_declaration */
extern void T59f346p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.storedprocedureuserdeclarationbeforebody */
extern T1 T59f71(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.sp_header_end */
extern void T59f345(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.sp_user_result */
extern void T59f344(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.sp_result_parameter */
extern void T59f343(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.storedprocedureparamlistend */
extern T0* T59f70(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.storedprocedureparamliststart */
extern T0* T59f66(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.storedprocedureparamliststartrequired */
extern T1 T59f65(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.sp_start */
extern void T59f342(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_sp_type */
extern void T59f363(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.drop_sp_type */
extern void T59f362(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.conditional_drop_procedure */
extern void T59f317(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.write_procedure */
extern void T58f298(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_procedure */
extern void T58f318(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.sp_end */
extern void T58f349(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.sp_body_statements */
extern void T58f348(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.sp_body_start */
extern void T58f347(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.sp_user_declaration */
extern void T58f346(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.sp_type_name */
extern T0* T58f106(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.sp_user_declaration */
extern void T58f346p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.storedprocedureuserdeclarationbeforebody */
extern T1 T58f71(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.sp_header_end */
extern void T58f345(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.sp_user_result */
extern void T58f344(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.sp_result_parameter */
extern void T58f343(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.storedprocedureparamlistend */
extern T0* T58f70(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.storedprocedureparamliststart */
extern T0* T58f66(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.storedprocedureparamliststartrequired */
extern T1 T58f65(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.sp_start */
extern void T58f342(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_sp_type */
extern void T58f363(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.drop_sp_type */
extern void T58f362(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.conditional_drop_procedure */
extern void T58f317(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.write_procedure */
extern void T57f298(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_procedure */
extern void T57f318(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.sp_end */
extern void T57f349(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.sp_body_start */
extern void T57f347(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.sp_user_declaration */
extern void T57f346(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.sp_type_name */
extern T0* T57f105(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.sp_user_declaration */
extern void T57f346p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.storedprocedureuserdeclarationbeforebody */
extern T1 T57f71(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.sp_header_end */
extern void T57f345(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.sp_user_result */
extern void T57f344(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.sp_result_parameter */
extern void T57f343(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.storedprocedureparamlistend */
extern T0* T57f70(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.storedprocedureparamliststart */
extern T0* T57f66(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.storedprocedureparamliststartrequired */
extern T1 T57f65(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.sp_start */
extern void T57f342(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_sp_type */
extern void T57f361(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.drop_sp_type */
extern void T57f360(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.conditional_drop_procedure */
extern void T57f317(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.write_procedure */
extern void T56f292(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.create_procedure */
extern void T56f313(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.sp_end */
extern void T56f344(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.sp_body_statements */
extern void T56f343(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.sp_body_start */
extern void T56f342(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.sp_user_declaration */
extern void T56f341(GE_context* ac, T0* C, T0* a1);
extern T1 T56f341ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_ORACLE.storedprocedureuserdeclarationbeforebody */
extern T1 T56f74(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.sp_header_end */
extern void T56f340(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.sp_user_result */
extern void T56f339(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.sp_result_parameter */
extern void T56f338(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.storedprocedureparamlistend */
extern T0* T56f73(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.storedprocedureparamliststart */
extern T0* T56f69(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.sp_start */
extern void T56f337(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.conditional_drop_procedure */
extern void T56f312(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.write_procedure */
extern void T55f258(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.write_procedure */
extern void T54f258(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.write_procedure */
extern void T53f283(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.create_procedure */
extern void T53f304(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.sp_end */
extern void T53f335(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.sp_end */
extern void T53f335p1(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.sp_body_statements */
extern void T53f334(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.sp_body_start */
extern void T53f333(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.sp_user_declaration */
extern void T53f332(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.storedprocedureuserdeclarationbeforebody */
extern T1 T53f75(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.sp_header_end */
extern void T53f331(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.sp_user_result */
extern void T53f330(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.sp_result_parameter */
extern void T53f329(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.storedprocedureparamlistend */
extern T0* T53f74(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.storedprocedureparamliststart */
extern T0* T53f70(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.storedprocedureparamliststartrequired */
extern T1 T53f69(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.sp_start */
extern void T53f328(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.sp_start */
extern void T53f328p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.conditional_drop_procedure */
extern void T53f303(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.write_procedure */
extern void T52f257(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.write_procedure */
extern void T51f257(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.write_procedure */
extern void T50f257(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.write_procedure */
extern void T49f287(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_procedure */
extern void T49f308(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.sp_end */
extern void T49f340(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.end_new_separator */
extern void T49f342(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.sp_body_statements */
extern void T49f339(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.sp_body_start */
extern void T49f338(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.sp_user_declaration */
extern void T49f337(GE_context* ac, T0* C, T0* a1);
extern T1 T49f337ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.storedprocedureuserdeclarationbeforebody */
extern T1 T49f75(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.sp_header_end */
extern void T49f336(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.sp_user_result */
extern void T49f335(GE_context* ac, T0* C, T0* a1);
extern T1 T49f335ot1(T0* a1, T0** a2);
extern T1 T49f335ot2(T0* a1, T0** a2);
/* detachable XPLAIN_SELECTION_FUNCTION.representation */
extern T0* T157f6(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.sp_result_parameter */
extern void T49f334(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.storedprocedureparamlistend */
extern T0* T49f74(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.storedprocedureparamliststart */
extern T0* T49f70(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.sp_start */
extern void T49f333(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.sp_start */
extern void T49f333p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.begin_new_separator */
extern void T49f341(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.conditional_drop_procedure */
extern void T49f307(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.write_procedure */
extern void T48f287(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_procedure */
extern void T48f308(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.sp_end */
extern void T48f339(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.end_new_separator */
extern void T48f341(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.sp_body_statements */
extern void T48f338(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.sp_body_start */
extern void T48f337(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.sp_user_declaration */
extern void T48f336(GE_context* ac, T0* C, T0* a1);
extern T1 T48f336ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_INTERBASE6.storedprocedureuserdeclarationbeforebody */
extern T1 T48f75(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.sp_header_end */
extern void T48f335(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.sp_user_result */
extern void T48f334(GE_context* ac, T0* C, T0* a1);
extern T1 T48f334ot1(T0* a1, T0** a2);
extern T1 T48f334ot2(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_INTERBASE6.sp_result_parameter */
extern void T48f333(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.storedprocedureparamlistend */
extern T0* T48f74(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.storedprocedureparamliststart */
extern T0* T48f70(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.sp_start */
extern void T48f332(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.sp_start */
extern void T48f332p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.begin_new_separator */
extern void T48f340(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.conditional_drop_procedure */
extern void T48f307(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.write_procedure */
extern void T47f257(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.write_procedure */
extern void T46f288(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.create_procedure */
extern void T46f309(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.sp_end */
extern void T46f340(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.sp_body_statements */
extern void T46f339(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.sp_body_start */
extern void T46f338(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.sp_user_declaration */
extern void T46f337(GE_context* ac, T0* C, T0* a1);
extern T1 T46f337ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_DB2_71.sp_header_end */
extern void T46f336(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.sp_user_result */
extern void T46f335(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.sp_result_parameter */
extern void T46f334(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.storedprocedureparamlistend */
extern T0* T46f73(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.storedprocedureparamliststart */
extern T0* T46f69(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.storedprocedureparamliststartrequired */
extern T1 T46f68(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.sp_start */
extern void T46f333(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.conditional_drop_procedure */
extern void T46f308(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.write_procedure */
extern void T45f259(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.write_procedure */
extern void T44f258(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.write_procedure */
extern void T43f258(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.write_sql */
extern void T66f273(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.write_sql */
extern void T65f272(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.write_sql */
extern void T64f272(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.write_sql */
extern void T63f272(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.write_sql */
extern void T62f271(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.write_sql */
extern void T61f295(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.write_sql */
extern void T60f298(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.write_sql */
extern void T59f299(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.write_sql */
extern void T58f299(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.write_sql */
extern void T57f299(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.write_sql */
extern void T56f293(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.write_sql */
extern void T55f259(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.write_sql */
extern void T54f259(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.write_sql */
extern void T53f284(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.write_sql */
extern void T52f258(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.write_sql */
extern void T51f258(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.write_sql */
extern void T50f258(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.write_sql */
extern void T49f288(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.write_sql */
extern void T48f288(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.write_sql */
extern void T47f258(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.write_sql */
extern void T46f289(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.write_sql */
extern void T45f260(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.write_sql */
extern void T44f259(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.write_sql */
extern void T43f259(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.write_one_line_comment */
extern void T66f254(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.multilinecommentpostfix */
extern T0* T66f26(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.onelinecommentprefix */
extern T0* T66f25(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.multilinecommentprefix */
extern T0* T66f24(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.noonelinecommentssupported */
extern T1 T66f157(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.write_one_line_comment */
extern void T65f253(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.multilinecommentpostfix */
extern T0* T65f26(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.onelinecommentprefix */
extern T0* T65f25(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.multilinecommentprefix */
extern T0* T65f24(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.noonelinecommentssupported */
extern T1 T65f156(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.write_one_line_comment */
extern void T64f253(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.multilinecommentpostfix */
extern T0* T64f26(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.onelinecommentprefix */
extern T0* T64f25(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.multilinecommentprefix */
extern T0* T64f24(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.noonelinecommentssupported */
extern T1 T64f156(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.write_one_line_comment */
extern void T63f253(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.multilinecommentpostfix */
extern T0* T63f26(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.onelinecommentprefix */
extern T0* T63f25(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.multilinecommentprefix */
extern T0* T63f24(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.noonelinecommentssupported */
extern T1 T63f156(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.write_one_line_comment */
extern void T62f252(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.multilinecommentpostfix */
extern T0* T62f26(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.onelinecommentprefix */
extern T0* T62f25(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.multilinecommentprefix */
extern T0* T62f24(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.noonelinecommentssupported */
extern T1 T62f141(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.write_one_line_comment */
extern void T61f276(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.multilinecommentpostfix */
extern T0* T61f26(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.onelinecommentprefix */
extern T0* T61f25(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.multilinecommentprefix */
extern T0* T61f174(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.noonelinecommentssupported */
extern T1 T61f107(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.write_one_line_comment */
extern void T60f279(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.multilinecommentpostfix */
extern T0* T60f26(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.onelinecommentprefix */
extern T0* T60f175(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.multilinecommentprefix */
extern T0* T60f110(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.noonelinecommentssupported */
extern T1 T60f107(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.write_one_line_comment */
extern void T59f280(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.multilinecommentpostfix */
extern T0* T59f26(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.onelinecommentprefix */
extern T0* T59f175(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.multilinecommentprefix */
extern T0* T59f111(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.noonelinecommentssupported */
extern T1 T59f108(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.write_one_line_comment */
extern void T58f280(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.multilinecommentpostfix */
extern T0* T58f26(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.onelinecommentprefix */
extern T0* T58f175(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.multilinecommentprefix */
extern T0* T58f111(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.noonelinecommentssupported */
extern T1 T58f108(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.write_one_line_comment */
extern void T57f280(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.multilinecommentpostfix */
extern T0* T57f26(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.onelinecommentprefix */
extern T0* T57f172(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.multilinecommentprefix */
extern T0* T57f110(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.noonelinecommentssupported */
extern T1 T57f107(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.write_one_line_comment */
extern void T56f274(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.multilinecommentpostfix */
extern T0* T56f26(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.onelinecommentprefix */
extern T0* T56f25(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.multilinecommentprefix */
extern T0* T56f24(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.noonelinecommentssupported */
extern T1 T56f105(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.write_one_line_comment */
extern void T55f240(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.multilinecommentpostfix */
extern T0* T55f26(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.onelinecommentprefix */
extern T0* T55f25(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.multilinecommentprefix */
extern T0* T55f24(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.noonelinecommentssupported */
extern T1 T55f139(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.write_one_line_comment */
extern void T54f240(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.multilinecommentpostfix */
extern T0* T54f26(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.onelinecommentprefix */
extern T0* T54f25(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.multilinecommentprefix */
extern T0* T54f24(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.noonelinecommentssupported */
extern T1 T54f139(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.write_one_line_comment */
extern void T53f265(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.multilinecommentpostfix */
extern T0* T53f26(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.onelinecommentprefix */
extern T0* T53f25(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.multilinecommentprefix */
extern T0* T53f24(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.noonelinecommentssupported */
extern T1 T53f159(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.write_one_line_comment */
extern void T52f239(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.multilinecommentpostfix */
extern T0* T52f26(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.onelinecommentprefix */
extern T0* T52f25(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.multilinecommentprefix */
extern T0* T52f24(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSQL.noonelinecommentssupported */
extern T1 T52f139(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.write_one_line_comment */
extern void T51f239(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.multilinecommentpostfix */
extern T0* T51f26(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.onelinecommentprefix */
extern T0* T51f25(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.multilinecommentprefix */
extern T0* T51f24(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.noonelinecommentssupported */
extern T1 T51f139(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.write_one_line_comment */
extern void T50f239(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.multilinecommentpostfix */
extern T0* T50f26(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.onelinecommentprefix */
extern T0* T50f25(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.multilinecommentprefix */
extern T0* T50f24(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.noonelinecommentssupported */
extern T1 T50f139(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.write_one_line_comment */
extern void T49f269(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.multilinecommentpostfix */
extern T0* T49f27(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.onelinecommentprefix */
extern T0* T49f26(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.multilinecommentprefix */
extern T0* T49f25(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.noonelinecommentssupported */
extern T1 T49f105(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.write_one_line_comment */
extern void T48f269(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.multilinecommentpostfix */
extern T0* T48f27(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.onelinecommentprefix */
extern T0* T48f26(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.multilinecommentprefix */
extern T0* T48f25(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_INTERBASE6.noonelinecommentssupported */
extern T1 T48f105(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.write_one_line_comment */
extern void T47f239(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.multilinecommentpostfix */
extern T0* T47f26(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.onelinecommentprefix */
extern T0* T47f25(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.multilinecommentprefix */
extern T0* T47f24(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FOXPRO.noonelinecommentssupported */
extern T1 T47f139(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.write_one_line_comment */
extern void T46f270(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.multilinecommentpostfix */
extern T0* T46f26(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.onelinecommentprefix */
extern T0* T46f25(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.multilinecommentprefix */
extern T0* T46f24(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.noonelinecommentssupported */
extern T1 T46f101(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.write_one_line_comment */
extern void T45f241(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.multilinecommentpostfix */
extern T0* T45f26(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.onelinecommentprefix */
extern T0* T45f25(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.multilinecommentprefix */
extern T0* T45f24(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_6.noonelinecommentssupported */
extern T1 T45f140(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.write_one_line_comment */
extern void T44f240(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.multilinecommentpostfix */
extern T0* T44f26(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.onelinecommentprefix */
extern T0* T44f25(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.multilinecommentprefix */
extern T0* T44f24(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_BASIC.noonelinecommentssupported */
extern T1 T44f139(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.write_one_line_comment */
extern void T43f240(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.multilinecommentpostfix */
extern T0* T43f26(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.onelinecommentprefix */
extern T0* T43f25(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.multilinecommentprefix */
extern T0* T43f24(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.noonelinecommentssupported */
extern T1 T43f139(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.write_init */
extern void T66f274(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.create_init */
extern void T66f294(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_INIT_ATTRIBUTES_CURSOR [[attached] XPLAIN_ATTRIBUTE].forth */
extern void T430f12(GE_context* ac, T0* C);
/* [detachable] XPLAIN_INIT_ATTRIBUTES_CURSOR [[attached] XPLAIN_ATTRIBUTE].is_initialized_attribute */
extern T1 T430f9(GE_context* ac, T0* C);
/* [detachable] XPLAIN_INIT_ATTRIBUTES_CURSOR [[attached] XPLAIN_ATTRIBUTE].forth */
extern void T430f12p1(GE_context* ac, T0* C);
/* [detachable] XPLAIN_INIT_ATTRIBUTES_CURSOR [[attached] XPLAIN_ATTRIBUTE].item */
extern T0* T430f8(GE_context* ac, T0* C);
/* [detachable] XPLAIN_INIT_ATTRIBUTES_CURSOR [[attached] XPLAIN_ATTRIBUTE].start */
extern void T430f11(GE_context* ac, T0* C);
/* [detachable] XPLAIN_INIT_ATTRIBUTES_CURSOR [[attached] XPLAIN_ATTRIBUTE].start */
extern void T430f11p1(GE_context* ac, T0* C);
/* detachable XPLAIN_TYPE.new_init_attributes_cursor */
extern T0* T132f15(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_INIT_ATTRIBUTES_CURSOR [[attached] XPLAIN_ATTRIBUTE].make */
extern T0* T430c10(GE_context* ac, T0* a1, T0* a2);
/* [detachable] XPLAIN_INIT_ATTRIBUTES_CURSOR [[attached] XPLAIN_ATTRIBUTE].ds_make */
extern void T430f13(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.init_trigger_name */
extern T0* T66f77(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.init_necessary */
extern T1 T66f40(GE_context* ac, T0* C, T0* a1);
/* detachable XPLAIN_TYPE.has_non_literal_init */
extern T1 T132f17(GE_context* ac, T0* C);
/* detachable XPLAIN_TYPE.has_non_constant_init */
extern T1 T132f16(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.expressionsindefaultclausesupported */
extern T1 T66f141(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.write_init */
extern void T65f273(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.create_init */
extern void T65f293(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.init_trigger_name */
extern T0* T65f77(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.init_necessary */
extern T1 T65f40(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.expressionsindefaultclausesupported */
extern T1 T65f140(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.write_init */
extern void T64f273(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.create_init */
extern void T64f293(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.init_trigger_name */
extern T0* T64f77(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.init_necessary */
extern T1 T64f40(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.expressionsindefaultclausesupported */
extern T1 T64f140(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.write_init */
extern void T63f273(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.create_init */
extern void T63f293(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.init_trigger_name */
extern T0* T63f77(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.init_necessary */
extern T1 T63f40(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.expressionsindefaultclausesupported */
extern T1 T63f140(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.write_init */
extern void T62f272(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_init */
extern void T62f290(GE_context* ac, T0* C, T0* a1);
/* [detachable] STRING_8.remove_tail */
extern void T17f51(GE_context* ac, T0* C, T6 a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.sql_init_expression */
extern T0* T62f66(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.init_necessary */
extern T1 T62f39(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.write_init */
extern void T61f296(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_init */
extern void T61f315(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.sql_init_expression */
extern T0* T61f73(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.init_necessary */
extern T1 T61f39(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.expressionsindefaultclausesupported */
extern T1 T61f153(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.write_init */
extern void T60f299(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_init */
extern void T60f318(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.sql_init_expression */
extern T0* T60f73(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.init_necessary */
extern T1 T60f39(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.expressionsindefaultclausesupported */
extern T1 T60f156(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.write_init */
extern void T59f300(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_init */
extern void T59f319(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.sql_init_expression */
extern T0* T59f73(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.init_necessary */
extern T1 T59f39(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.expressionsindefaultclausesupported */
extern T1 T59f157(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.write_init */
extern void T58f300(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_init */
extern void T58f319(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.sql_init_expression */
extern T0* T58f73(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.init_necessary */
extern T1 T58f39(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.expressionsindefaultclausesupported */
extern T1 T58f157(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.write_init */
extern void T57f300(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_init */
extern void T57f319(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.sql_init_expression */
extern T0* T57f73(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.init_necessary */
extern T1 T57f39(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.expressionsindefaultclausesupported */
extern T1 T57f156(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.write_init */
extern void T56f294(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.create_init */
extern void T56f314(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.sql_init_expression */
extern T0* T56f75(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.init_necessary */
extern T1 T56f40(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.expressionsindefaultclausesupported */
extern T1 T56f150(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.write_init */
extern void T55f260(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.init_necessary */
extern T1 T55f39(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.expressionsindefaultclausesupported */
extern T1 T55f126(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.write_init */
extern void T54f260(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.init_necessary */
extern T1 T54f39(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.expressionsindefaultclausesupported */
extern T1 T54f126(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL5.write_init */
extern void T53f285(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.create_init */
extern void T53f305(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.sql_init_expression */
extern T0* T53f76(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL5.init_necessary */
extern T1 T53f40(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.write_init */
extern void T52f259(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.init_necessary */
extern T1 T52f39(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSQL.expressionsindefaultclausesupported */
extern T1 T52f126(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS2000.write_init */
extern void T51f259(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.init_necessary */
extern T1 T51f39(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS2000.expressionsindefaultclausesupported */
extern T1 T51f126(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MSACCESS97.write_init */
extern void T50f259(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.init_necessary */
extern T1 T50f39(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MSACCESS97.expressionsindefaultclausesupported */
extern T1 T50f126(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.write_init */
extern void T49f289(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.create_init */
extern void T49f309(GE_context* ac, T0* C, T0* a1);
extern T1 T49f309ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.sql_init_expression */
extern T0* T49f76(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FIREBIRD_21.init_necessary */
extern T1 T49f41(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.write_init */
extern void T48f289(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.create_init */
extern void T48f309(GE_context* ac, T0* C, T0* a1);
extern T1 T48f309ot1(T0* a1, T0** a2);
/* [detachable] SQL_GENERATOR_INTERBASE6.sql_init_expression */
extern T0* T48f76(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_INTERBASE6.init_necessary */
extern T1 T48f41(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.write_init */
extern void T47f259(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.init_necessary */
extern T1 T47f39(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_FOXPRO.expressionsindefaultclausesupported */
extern T1 T47f126(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_DB2_71.write_init */
extern void T46f290(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.create_init */
extern void T46f310(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.sql_init_expression */
extern T0* T46f75(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_71.init_necessary */
extern T1 T46f40(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.write_init */
extern void T45f261(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_DB2_6.init_necessary */
extern T1 T45f39(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.write_init */
extern void T44f260(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.init_necessary */
extern T1 T44f39(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_BASIC.expressionsindefaultclausesupported */
extern T1 T44f126(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ANSI.write_init */
extern void T43f260(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.init_necessary */
extern T1 T43f39(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ANSI.expressionsindefaultclausesupported */
extern T1 T43f126(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.write_type */
extern void T66f275(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.create_sp_delete */
extern void T66f297(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.sp_before_end_delete */
extern void T66f330(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.sp_delete_declaration */
extern void T66f329(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL65.table_ts_name */
extern T0* T66f88(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.createtimestamp */
extern T1 T66f87(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.create_sp_update */
extern void T66f296(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.sp_before_end_update */
extern void T66f328(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.sp_update_declaration */
extern void T66f327(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* detachable XPLAIN_TYPE.has_updatable_attributes */
extern T1 T132f8(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.create_sp_insert */
extern void T66f295(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.sp_get_auto_generated_primary_key */
extern T0* T66f85(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.supportsdefaultvalues */
extern T1 T66f84(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.sp_insert_declaration */
extern void T66f326(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3);
/* [detachable] SQL_GENERATOR_TSQL65.sp_define_out_param */
extern T0* T66f82(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.sp_return_parameter_format_string */
extern T0* T66f81(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.storedprocedurereturnparamliststart */
extern T0* T66f80(GE_context* ac, T0* C);
/* detachable XPLAIN_TYPE.has_auto_pk */
extern T1 T132f13(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.outputparameterssupported */
extern T1 T66f79(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.write_type */
extern void T66f275p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.forbid_update_of_primary_key */
extern void T66f302(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.create_primary_key_generator */
extern void T66f301(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.create_view */
extern void T66f300(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.createviews */
extern T1 T66f42(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.create_table */
extern void T66f299(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.create_table */
extern void T66f299p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.timestampdatatype */
extern T0* T66f96(GE_context* ac, T0* C);
/* [detachable] XPLAIN_ALL_ATTRIBUTES_CURSOR [[attached] XPLAIN_ATTRIBUTE].forth */
extern void T407f13(GE_context* ac, T0* C);
/* [detachable] XPLAIN_ALL_ATTRIBUTES_CURSOR [[attached] XPLAIN_ATTRIBUTE].is_generated_attribute */
extern T1 T407f9(GE_context* ac, T0* C);
extern T1 T407f9ot1(T0* a1, T0** a2);
/* detachable XPLAIN_ASSERTION.is_simple */
extern T1 T146f26(GE_context* ac, T0* C);
/* detachable XPLAIN_ASSERTION.is_literal */
extern T1 T146f25(GE_context* ac, T0* C);
/* [detachable] XPLAIN_EXTENSION_EXPRESSION_EXPRESSION.is_literal */
extern T1 T313f10(GE_context* ac, T0* C);
/* [detachable] XPLAIN_ALL_ATTRIBUTES_CURSOR [[attached] XPLAIN_ATTRIBUTE].forth */
extern void T407f13p1(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.column_constraint_name */
extern T0* T66f95(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] DS_HASH_SET [[attached] STRING_8].force */
extern void T424f32(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_HASH_SET [[attached] STRING_8].slots_put */
extern void T424f39(GE_context* ac, T0* C, T6 a1, T6 a2);
/* [detachable] DS_HASH_SET [[attached] STRING_8].clashes_put */
extern void T424f38(GE_context* ac, T0* C, T6 a1, T6 a2);
/* [detachable] DS_HASH_SET [[attached] STRING_8].slots_item */
extern T6 T424f26(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_SET [[attached] STRING_8].clashes_item */
extern T6 T424f25(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_SET [[attached] STRING_8].hash_position */
extern T6 T424f27(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_HASH_SET [[attached] STRING_8].resize */
extern void T424f37(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_SET [[attached] STRING_8].clashes_resize */
extern void T424f48(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_SET [[attached] STRING_8].special_integer_ */
extern T0* T424f28(GE_context* ac, T0* C);
/* [detachable] DS_HASH_SET [[attached] STRING_8].key_storage_resize */
extern void T424f47(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_SET [[attached] STRING_8].item_storage_resize */
extern void T424f46(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_SET [[attached] STRING_8].key_storage_item */
extern T0* T424f19(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_SET [[attached] STRING_8].item_storage_item */
extern T0* T424f29(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_SET [[attached] STRING_8].slots_resize */
extern void T424f45(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_SET [[attached] STRING_8].new_modulus */
extern T6 T424f20(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_SET [[attached] STRING_8].new_capacity */
extern T6 T424f23(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_SET [[attached] STRING_8].item_storage_put */
extern void T424f36(GE_context* ac, T0* C, T0* a1, T6 a2);
/* [detachable] DS_HASH_SET [[attached] STRING_8].search_position */
extern void T424f33(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_HASH_SET [[attached] STRING_8].key_equality_tester */
extern T0* T424f17(GE_context* ac, T0* C);
/* [detachable] DS_HASH_SET [[attached] STRING_8].unset_found_item */
extern void T424f35(GE_context* ac, T0* C);
/* [detachable] DS_HASH_SET [[attached] STRING_8].has */
extern T1 T424f22(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.constraint_names */
extern T0* T66f156(GE_context* ac, T0* C);
/* [detachable] DS_HASH_SET [[attached] STRING_8].make_equal */
extern T0* T424c31(GE_context* ac, T6 a1);
/* [detachable] DS_HASH_SET [[attached] STRING_8].make */
extern void T424f34(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_SET [[attached] STRING_8].set_internal_cursor */
extern void T424f44(GE_context* ac, T0* C, T0* a1);
/* [detachable] DS_HASH_SET [[attached] STRING_8].new_cursor */
extern T0* T424f21(GE_context* ac, T0* C);
/* [detachable] DS_HASH_SET_CURSOR [[attached] STRING_8].make */
extern T0* T461c4(GE_context* ac, T0* a1);
/* [detachable] DS_HASH_SET [[attached] STRING_8].make_slots */
extern void T424f43(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_SET [[attached] STRING_8].make_clashes */
extern void T424f42(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_SET [[attached] STRING_8].make_key_storage */
extern void T424f41(GE_context* ac, T0* C, T6 a1);
/* [detachable] DS_HASH_SET [[attached] STRING_8].make_item_storage */
extern void T424f40(GE_context* ac, T0* C, T6 a1);
/* [detachable] SQL_GENERATOR_TSQL65.make_valid_constraint_identifier */
extern T0* T66f155(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.maxconstraintidentifierlength */
extern T6 T66f188(GE_context* ac, T0* C);
/* detachable XPLAIN_TYPE.next_constraint_number */
extern void T132f32(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.constraintnamesupported */
extern T1 T66f94(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.inlineuniqueconstraintsupported */
extern T1 T66f93(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.uniqueconstraintsupported */
extern T1 T66f92(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.init_forced_default */
extern T1 T66f91(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.checksnullaftertrigger */
extern T1 T66f154(GE_context* ac, T0* C);
/* [detachable] XPLAIN_ALL_ATTRIBUTES_CURSOR [[attached] XPLAIN_ATTRIBUTE].item */
extern T0* T407f8(GE_context* ac, T0* C);
/* [detachable] XPLAIN_ALL_ATTRIBUTES_CURSOR [[attached] XPLAIN_ATTRIBUTE].start */
extern void T407f12(GE_context* ac, T0* C);
/* [detachable] XPLAIN_ALL_ATTRIBUTES_CURSOR [[attached] XPLAIN_ATTRIBUTE].start */
extern void T407f12p1(GE_context* ac, T0* C);
/* detachable XPLAIN_TYPE.new_all_attributes_cursor */
extern T0* T132f14(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_ALL_ATTRIBUTES_CURSOR [[attached] XPLAIN_ATTRIBUTE].make */
extern T0* T407c11(GE_context* ac, T0* a1, T0* a2);
/* [detachable] XPLAIN_ALL_ATTRIBUTES_CURSOR [[attached] XPLAIN_ATTRIBUTE].ds_make */
extern void T407f14(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PK_I_REPRESENTATION.pkdatatype */
extern T0* T257f5(GE_context* ac, T0* C, T0* a1);
/* [detachable] XPLAIN_PK_A_REPRESENTATION.pkdatatype */
extern T0* T254f3(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.createtablestatement */
extern T0* T66f90(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.drop_table_if_exist */
extern void T66f298(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL65.print */
extern void T66f331(GE_context* ac, T0* C, T0* a1);
/* [detachable] STD_FILES.put_string */
extern void T351f8(GE_context* ac, T0* C, T0* a1);
/* [detachable] STD_FILES.standard_default */
extern T0* T351f4(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL65.io */
extern T0* T66f101(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.write_type */
extern void T65f274(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.create_sp_delete */
extern void T65f296(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.sp_before_end_delete */
extern void T65f330(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.sp_delete_declaration */
extern void T65f329(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL70.table_ts_name */
extern T0* T65f88(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.createtimestamp */
extern T1 T65f87(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.create_sp_update */
extern void T65f295(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.sp_before_end_update */
extern void T65f328(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.sp_update_declaration */
extern void T65f327(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_TSQL70.create_sp_insert */
extern void T65f294(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.sp_get_auto_generated_primary_key */
extern T0* T65f85(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.supportsdefaultvalues */
extern T1 T65f84(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.sp_insert_declaration */
extern void T65f326(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3);
/* [detachable] SQL_GENERATOR_TSQL70.sp_define_out_param */
extern T0* T65f82(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.sp_return_parameter_format_string */
extern T0* T65f81(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.storedprocedurereturnparamliststart */
extern T0* T65f80(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.outputparameterssupported */
extern T1 T65f79(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.write_type */
extern void T65f274p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.forbid_update_of_primary_key */
extern void T65f301(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.create_primary_key_generator */
extern void T65f300(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.create_view */
extern void T65f299(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.createviews */
extern T1 T65f42(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.create_table */
extern void T65f298(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.create_table */
extern void T65f298p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.timestampdatatype */
extern T0* T65f96(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.column_constraint_name */
extern T0* T65f95(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL70.constraint_names */
extern T0* T65f155(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.make_valid_constraint_identifier */
extern T0* T65f154(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.maxconstraintidentifierlength */
extern T6 T65f187(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.constraintnamesupported */
extern T1 T65f94(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.inlineuniqueconstraintsupported */
extern T1 T65f93(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.uniqueconstraintsupported */
extern T1 T65f92(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.init_forced_default */
extern T1 T65f91(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL70.checksnullaftertrigger */
extern T1 T65f153(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.createtablestatement */
extern T0* T65f90(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL70.drop_table_if_exist */
extern void T65f297(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.write_type */
extern void T64f274(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.create_sp_delete */
extern void T64f296(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.sp_before_end_delete */
extern void T64f330(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.sp_delete_declaration */
extern void T64f329(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL2000.table_ts_name */
extern T0* T64f88(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.createtimestamp */
extern T1 T64f87(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.create_sp_update */
extern void T64f295(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.sp_before_end_update */
extern void T64f328(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.sp_update_declaration */
extern void T64f327(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_TSQL2000.create_sp_insert */
extern void T64f294(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.sp_get_auto_generated_primary_key */
extern T0* T64f85(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.supportsdefaultvalues */
extern T1 T64f84(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.sp_insert_declaration */
extern void T64f326(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3);
/* [detachable] SQL_GENERATOR_TSQL2000.sp_define_out_param */
extern T0* T64f82(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.sp_return_parameter_format_string */
extern T0* T64f81(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.storedprocedurereturnparamliststart */
extern T0* T64f80(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.outputparameterssupported */
extern T1 T64f79(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.write_type */
extern void T64f274p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.forbid_update_of_primary_key */
extern void T64f301(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.create_primary_key_generator */
extern void T64f300(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.create_view */
extern void T64f299(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.createviews */
extern T1 T64f42(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.create_table */
extern void T64f298(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.create_table */
extern void T64f298p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.timestampdatatype */
extern T0* T64f96(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.column_constraint_name */
extern T0* T64f95(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL2000.constraint_names */
extern T0* T64f155(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.make_valid_constraint_identifier */
extern T0* T64f154(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.maxconstraintidentifierlength */
extern T6 T64f187(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.constraintnamesupported */
extern T1 T64f94(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.inlineuniqueconstraintsupported */
extern T1 T64f93(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.uniqueconstraintsupported */
extern T1 T64f92(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.init_forced_default */
extern T1 T64f91(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2000.checksnullaftertrigger */
extern T1 T64f153(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.createtablestatement */
extern T0* T64f90(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2000.drop_table_if_exist */
extern void T64f297(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.write_type */
extern void T63f274(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.create_sp_delete */
extern void T63f296(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.sp_before_end_delete */
extern void T63f329(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.sp_delete_declaration */
extern void T63f328(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL2016.table_ts_name */
extern T0* T63f88(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.createtimestamp */
extern T1 T63f87(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.create_sp_update */
extern void T63f295(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.sp_before_end_update */
extern void T63f327(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.sp_update_declaration */
extern void T63f326(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_TSQL2016.create_sp_insert */
extern void T63f294(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.sp_get_auto_generated_primary_key */
extern T0* T63f85(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.supportsdefaultvalues */
extern T1 T63f84(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.sp_insert_declaration */
extern void T63f325(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3);
/* [detachable] SQL_GENERATOR_TSQL2016.sp_define_out_param */
extern T0* T63f82(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.sp_return_parameter_format_string */
extern T0* T63f81(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.storedprocedurereturnparamliststart */
extern T0* T63f80(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.outputparameterssupported */
extern T1 T63f79(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.write_type */
extern void T63f274p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.forbid_update_of_primary_key */
extern void T63f301(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.create_primary_key_generator */
extern void T63f300(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.create_view */
extern void T63f299(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.createviews */
extern T1 T63f42(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.create_table */
extern void T63f298(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.create_table */
extern void T63f298p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.timestampdatatype */
extern T0* T63f96(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.column_constraint_name */
extern T0* T63f95(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_TSQL2016.constraint_names */
extern T0* T63f155(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.make_valid_constraint_identifier */
extern T0* T63f154(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.maxconstraintidentifierlength */
extern T6 T63f187(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.constraintnamesupported */
extern T1 T63f94(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.inlineuniqueconstraintsupported */
extern T1 T63f93(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.uniqueconstraintsupported */
extern T1 T63f92(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.init_forced_default */
extern T1 T63f91(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_TSQL2016.checksnullaftertrigger */
extern T1 T63f153(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.createtablestatement */
extern T0* T63f90(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_TSQL2016.drop_table_if_exist */
extern void T63f297(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.write_type */
extern void T62f273(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.forbid_update_of_primary_key */
extern void T62f295(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_primary_key_generator */
extern void T62f294(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_view */
extern void T62f293(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.createviews */
extern T1 T62f40(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.create_table */
extern void T62f292(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.timestampdatatype */
extern T0* T62f75(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.table_ts_name */
extern T0* T62f74(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.createtimestamp */
extern T1 T62f73(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.timestampsupported */
extern T1 T62f140(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.column_constraint_name */
extern T0* T62f72(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_SQLITE_3.constraint_names */
extern T0* T62f139(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.make_valid_constraint_identifier */
extern T0* T62f138(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.maxconstraintidentifierlength */
extern T6 T62f169(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.constraintnamesupported */
extern T1 T62f71(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.inlineuniqueconstraintsupported */
extern T1 T62f70(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.uniqueconstraintsupported */
extern T1 T62f69(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.init_forced_default */
extern T1 T62f68(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_SQLITE_3.checksnullaftertrigger */
extern T1 T62f137(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.createtablestatement */
extern T0* T62f67(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_SQLITE_3.drop_table_if_exist */
extern void T62f291(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.write_type */
extern void T61f297(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_sp_delete */
extern void T61f318(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.sp_before_end_delete */
extern void T61f350(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.sp_delete_declaration */
extern void T61f349(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_72.table_ts_name */
extern T0* T61f85(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.createtimestamp */
extern T1 T61f84(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.timestampsupported */
extern T1 T61f163(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_sp_update */
extern void T61f317(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.sp_before_end_update */
extern void T61f348(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.sp_update_declaration */
extern void T61f347(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_sp_insert */
extern void T61f316(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.sp_get_auto_generated_primary_key */
extern T0* T61f82(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.supportsdefaultvalues */
extern T1 T61f81(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.sp_insert_declaration */
extern void T61f346(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3);
/* [detachable] SQL_GENERATOR_PGSQL_72.sp_define_out_param */
extern T0* T61f78(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.sp_return_parameter_format_string */
extern T0* T61f77(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.storedprocedurereturnparamliststart */
extern T0* T61f76(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.write_type */
extern void T61f297p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.forbid_update_of_primary_key */
extern void T61f323(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_primary_key_generator */
extern void T61f322(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_view */
extern void T61f321(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.createviews */
extern T1 T61f41(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.create_table */
extern void T61f320(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.timestampdatatype */
extern T0* T61f95(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.column_constraint_name */
extern T0* T61f94(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_72.constraint_names */
extern T0* T61f168(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.make_valid_constraint_identifier */
extern T0* T61f167(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.maxconstraintidentifierlength */
extern T6 T61f202(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.maxidentifierlength */
extern T6 T61f172(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.constraintnamesupported */
extern T1 T61f93(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.inlineuniqueconstraintsupported */
extern T1 T61f92(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.uniqueconstraintsupported */
extern T1 T61f91(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.init_forced_default */
extern T1 T61f90(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_72.checksnullaftertrigger */
extern T1 T61f166(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.createtablestatement */
extern T0* T61f89(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_72.drop_table_if_exist */
extern void T61f319(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.write_type */
extern void T60f300(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_sp_delete */
extern void T60f321(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.sp_before_end_delete */
extern void T60f353(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.sp_delete_declaration */
extern void T60f352(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_73.table_ts_name */
extern T0* T60f85(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.createtimestamp */
extern T1 T60f84(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.timestampsupported */
extern T1 T60f166(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_sp_update */
extern void T60f320(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.sp_before_end_update */
extern void T60f351(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.sp_update_declaration */
extern void T60f350(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_sp_insert */
extern void T60f319(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.sp_get_auto_generated_primary_key */
extern T0* T60f82(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.supportsdefaultvalues */
extern T1 T60f81(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.sp_insert_declaration */
extern void T60f349(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3);
/* [detachable] SQL_GENERATOR_PGSQL_73.sp_define_out_param */
extern T0* T60f78(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.sp_return_parameter_format_string */
extern T0* T60f77(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.storedprocedurereturnparamliststart */
extern T0* T60f76(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.write_type */
extern void T60f300p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.forbid_update_of_primary_key */
extern void T60f326(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_primary_key_generator */
extern void T60f325(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_view */
extern void T60f324(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.createviews */
extern T1 T60f41(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.create_table */
extern void T60f323(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.timestampdatatype */
extern T0* T60f95(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.column_constraint_name */
extern T0* T60f94(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_73.constraint_names */
extern T0* T60f171(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.make_valid_constraint_identifier */
extern T0* T60f170(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.maxconstraintidentifierlength */
extern T6 T60f203(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.maxidentifierlength */
extern T6 T60f173(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.constraintnamesupported */
extern T1 T60f93(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.inlineuniqueconstraintsupported */
extern T1 T60f92(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.uniqueconstraintsupported */
extern T1 T60f91(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.init_forced_default */
extern T1 T60f90(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_73.checksnullaftertrigger */
extern T1 T60f169(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.createtablestatement */
extern T0* T60f89(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_73.drop_table_if_exist */
extern void T60f322(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.write_type */
extern void T59f301(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_sp_delete */
extern void T59f322(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.sp_before_end_delete */
extern void T59f354(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.sp_delete_declaration */
extern void T59f353(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_81.table_ts_name */
extern T0* T59f85(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.createtimestamp */
extern T1 T59f84(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.timestampsupported */
extern T1 T59f167(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_sp_update */
extern void T59f321(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.sp_before_end_update */
extern void T59f352(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.sp_update_declaration */
extern void T59f351(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_sp_insert */
extern void T59f320(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.sp_get_auto_generated_primary_key */
extern T0* T59f82(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.supportsdefaultvalues */
extern T1 T59f81(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.sp_insert_declaration */
extern void T59f350(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3);
/* [detachable] SQL_GENERATOR_PGSQL_81.sp_define_out_param */
extern T0* T59f78(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.sp_return_parameter_format_string */
extern T0* T59f77(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.storedprocedurereturnparamliststart */
extern T0* T59f76(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.write_type */
extern void T59f301p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.forbid_update_of_primary_key */
extern void T59f327(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_primary_key_generator */
extern void T59f326(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_view */
extern void T59f325(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.createviews */
extern T1 T59f41(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.create_table */
extern void T59f324(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.timestampdatatype */
extern T0* T59f95(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.column_constraint_name */
extern T0* T59f94(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_81.constraint_names */
extern T0* T59f171(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.make_valid_constraint_identifier */
extern T0* T59f170(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.maxconstraintidentifierlength */
extern T6 T59f204(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.maxidentifierlength */
extern T6 T59f173(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.constraintnamesupported */
extern T1 T59f93(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.inlineuniqueconstraintsupported */
extern T1 T59f92(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.uniqueconstraintsupported */
extern T1 T59f91(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.init_forced_default */
extern T1 T59f90(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_81.checksnullaftertrigger */
extern T1 T59f169(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.createtablestatement */
extern T0* T59f89(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_81.drop_table_if_exist */
extern void T59f323(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.write_type */
extern void T58f301(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_sp_delete */
extern void T58f322(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.sp_before_end_delete */
extern void T58f354(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.sp_delete_declaration */
extern void T58f353(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_82.table_ts_name */
extern T0* T58f85(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.createtimestamp */
extern T1 T58f84(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.timestampsupported */
extern T1 T58f167(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_sp_update */
extern void T58f321(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.sp_before_end_update */
extern void T58f352(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.sp_update_declaration */
extern void T58f351(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_sp_insert */
extern void T58f320(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.sp_get_auto_generated_primary_key */
extern T0* T58f82(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.supportsdefaultvalues */
extern T1 T58f81(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.sp_insert_declaration */
extern void T58f350(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3);
/* [detachable] SQL_GENERATOR_PGSQL_82.sp_define_out_param */
extern T0* T58f78(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.sp_return_parameter_format_string */
extern T0* T58f77(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.storedprocedurereturnparamliststart */
extern T0* T58f76(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.write_type */
extern void T58f301p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.forbid_update_of_primary_key */
extern void T58f327(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_primary_key_generator */
extern void T58f326(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_view */
extern void T58f325(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.createviews */
extern T1 T58f41(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.create_table */
extern void T58f324(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.timestampdatatype */
extern T0* T58f95(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.column_constraint_name */
extern T0* T58f94(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_82.constraint_names */
extern T0* T58f171(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.make_valid_constraint_identifier */
extern T0* T58f170(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.maxconstraintidentifierlength */
extern T6 T58f204(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.maxidentifierlength */
extern T6 T58f173(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.constraintnamesupported */
extern T1 T58f93(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.inlineuniqueconstraintsupported */
extern T1 T58f92(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.uniqueconstraintsupported */
extern T1 T58f91(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.init_forced_default */
extern T1 T58f90(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_82.checksnullaftertrigger */
extern T1 T58f169(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.createtablestatement */
extern T0* T58f89(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_82.drop_table_if_exist */
extern void T58f323(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.write_type */
extern void T57f301(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_sp_delete */
extern void T57f322(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.sp_before_end_delete */
extern void T57f354(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.sp_delete_declaration */
extern void T57f353(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_95.table_ts_name */
extern T0* T57f85(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.createtimestamp */
extern T1 T57f84(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.timestampsupported */
extern T1 T57f167(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_sp_update */
extern void T57f321(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.sp_before_end_update */
extern void T57f352(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.sp_update_declaration */
extern void T57f351(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_sp_insert */
extern void T57f320(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.sp_get_auto_generated_primary_key */
extern T0* T57f82(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.supportsdefaultvalues */
extern T1 T57f81(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.sp_insert_declaration */
extern void T57f350(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3);
/* [detachable] SQL_GENERATOR_PGSQL_95.sp_define_out_param */
extern T0* T57f78(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.sp_return_parameter_format_string */
extern T0* T57f77(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.storedprocedurereturnparamliststart */
extern T0* T57f76(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.write_type */
extern void T57f301p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.forbid_update_of_primary_key */
extern void T57f327(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_primary_key_generator */
extern void T57f326(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_view */
extern void T57f325(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.createviews */
extern T1 T57f41(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.create_table */
extern void T57f324(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.timestampdatatype */
extern T0* T57f95(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.column_constraint_name */
extern T0* T57f94(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_PGSQL_95.constraint_names */
extern T0* T57f171(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.make_valid_constraint_identifier */
extern T0* T57f170(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.maxconstraintidentifierlength */
extern T6 T57f204(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.maxidentifierlength */
extern T6 T57f201(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.constraintnamesupported */
extern T1 T57f93(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.inlineuniqueconstraintsupported */
extern T1 T57f92(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.uniqueconstraintsupported */
extern T1 T57f91(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.init_forced_default */
extern T1 T57f90(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_PGSQL_95.checksnullaftertrigger */
extern T1 T57f169(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.createtablestatement */
extern T0* T57f89(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_PGSQL_95.drop_table_if_exist */
extern void T57f323(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.write_type */
extern void T56f295(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.create_sp_delete */
extern void T56f317(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.sp_before_end_delete */
extern void T56f349(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.sp_delete_declaration */
extern void T56f348(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_ORACLE.table_ts_name */
extern T0* T56f87(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.createtimestamp */
extern T1 T56f86(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.timestampsupported */
extern T1 T56f162(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.create_sp_update */
extern void T56f316(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.sp_before_end_update */
extern void T56f347(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.sp_update_declaration */
extern void T56f346(GE_context* ac, T0* C, T0* a1, T0* a2, T0* a3);
/* [detachable] SQL_GENERATOR_ORACLE.create_sp_insert */
extern void T56f315(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.sp_get_auto_generated_primary_key */
extern T0* T56f84(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.supportsdefaultvalues */
extern T1 T56f83(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.sp_insert_declaration */
extern void T56f345(GE_context* ac, T0* C, T0* a1, T0* a2, T1 a3);
/* [detachable] SQL_GENERATOR_ORACLE.sp_define_out_param */
extern T0* T56f81(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.sp_return_parameter_format_string */
extern T0* T56f80(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.storedprocedurereturnparamliststart */
extern T0* T56f79(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.outputparameterssupported */
extern T1 T56f78(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.write_type */
extern void T56f295p1(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.forbid_update_of_primary_key */
extern void T56f322(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.trigger_primary_key_update_name */
extern T0* T56f98(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.create_primary_key_generator */
extern void T56f321(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.sequence_name */
extern T0* T56f96(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.create_view */
extern void T56f320(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.createviews */
extern T1 T56f42(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.create_table */
extern void T56f319(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.timestampdatatype */
extern T0* T56f95(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.column_constraint_name */
extern T0* T56f94(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_ORACLE.constraint_names */
extern T0* T56f166(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.make_valid_constraint_identifier */
extern T0* T56f165(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.maxconstraintidentifierlength */
extern T6 T56f200(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.maxidentifierlength */
extern T6 T56f196(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.constraintnamesupported */
extern T1 T56f93(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.inlineuniqueconstraintsupported */
extern T1 T56f92(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.uniqueconstraintsupported */
extern T1 T56f91(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.init_forced_default */
extern T1 T56f90(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_ORACLE.checksnullaftertrigger */
extern T1 T56f164(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.createtablestatement */
extern T0* T56f89(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_ORACLE.drop_table_if_exist */
extern void T56f318(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.write_type */
extern void T55f261(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.forbid_update_of_primary_key */
extern void T55f282(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.create_primary_key_generator */
extern void T55f281(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.create_view */
extern void T55f280(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.createviews */
extern T1 T55f40(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.create_table */
extern void T55f279(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.timestampdatatype */
extern T0* T55f74(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.table_ts_name */
extern T0* T55f73(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.createtimestamp */
extern T1 T55f72(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.timestampsupported */
extern T1 T55f138(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.column_constraint_name */
extern T0* T55f71(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL322.constraint_names */
extern T0* T55f137(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.make_valid_constraint_identifier */
extern T0* T55f136(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.maxconstraintidentifierlength */
extern T6 T55f167(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.maxidentifierlength */
extern T6 T55f163(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.constraintnamesupported */
extern T1 T55f70(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.inlineuniqueconstraintsupported */
extern T1 T55f69(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.uniqueconstraintsupported */
extern T1 T55f68(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.init_forced_default */
extern T1 T55f67(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL322.createtablestatement */
extern T0* T55f66(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL322.drop_table_if_exist */
extern void T55f278(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.write_type */
extern void T54f261(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.forbid_update_of_primary_key */
extern void T54f282(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.create_primary_key_generator */
extern void T54f281(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.create_view */
extern void T54f280(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.createviews */
extern T1 T54f40(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.create_table */
extern void T54f279(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.timestampdatatype */
extern T0* T54f74(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.table_ts_name */
extern T0* T54f73(GE_context* ac, T0* C, T0* a1);
/* [detachable] SQL_GENERATOR_MYSQL4.createtimestamp */
extern T1 T54f72(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.timestampsupported */
extern T1 T54f138(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.column_constraint_name */
extern T0* T54f71(GE_context* ac, T0* C, T0* a1, T0* a2);
/* [detachable] SQL_GENERATOR_MYSQL4.constraint_names */
extern T0* T54f137(GE_context* ac, T0* C);
/* [detachable] SQL_GENERATOR_MYSQL4.make_valid_constraint_